/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.ui.action;


import java.util.ArrayList;
import java.util.List;

import jp.valtech.bts.data.Attachment;
import jp.valtech.bts.data.Issue;
import jp.valtech.bts.data.IssueType;
import jp.valtech.bts.facade.ModifyIssueFacade;
import jp.valtech.bts.ui.BtsPlugin;
import jp.valtech.bts.ui.IBtsListener;
import jp.valtech.bts.ui.editor.IssueMultiPageEditor;
import jp.valtech.bts.ui.editor.IssueMultiPageEditorInput;
import jp.valtech.bts.util.IssueUtility;
import jp.valtech.bts.util.Logging;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.resource.ImageDescriptor;


/**
 * {@link UpdateReport クライアント用の課題票更新コマンド}を実行するアクションです。
 * 
 * @author		<A href="mailto:m_sugitou@valtech.jp">M.Sugito</A>
 * @version	Ver.0.8
 */
public class SaveIssueAction extends Action implements Logging {

	
	/** 活性時のアイコン */
	private static final ImageDescriptor THIS_IMAGE = BtsPlugin.getInstance().getImageDescriptor("save.gif"); 
	/** 非活性時のアイコン */
	private static final ImageDescriptor THIS_IMAGE_DISABLE = BtsPlugin.getInstance().getImageDescriptor("save_disable.gif"); 

	/**
	 * 課題票を設定します。
	 */
	public SaveIssueAction() {
		super();
		setText(Messages.getString("SaveIssueAction.0")); //$NON-NLS-1$
        setToolTipText(Messages.getString("SaveIssueAction.1")); //$NON-NLS-1$

		setImageDescriptor(THIS_IMAGE);
		setDisabledImageDescriptor(THIS_IMAGE_DISABLE);
	}

	
	/**
	 * 課題票を保存します。
	 */
	public void run() {

		// アクティブな課題票エディタを取得する
    	IssueMultiPageEditor editorPart = BtsPlugin.getInstance().getActiveEditor();
		if(editorPart == null) {
			logger.debug(Messages.getString("SaveIssueAction.2")); //$NON-NLS-1$
			return;
		}

		try {
			IssueMultiPageEditorInput issueEditorInput
				= (IssueMultiPageEditorInput)editorPart.getEditorInput();

			// 更新情報を格納した課題票を取得する
	    	Issue newIssue = editorPart.getModifiedIssue();
			if(newIssue==null) {
				return;
			}

			// 更新する前の課題票種別を保持しておく
	    	String typeUpdateBefore = newIssue.getType();

	    	// 課題票情報をDBに保存する値に更新する
	    	updateIssueInfo(newIssue);
	    	
			// 課題票更新時に削除された課題票情報のリストを取得
			ArrayList oldAttachList = issueEditorInput.getSavedAttachment();
			ArrayList newAttachList = newIssue.getAttachmentList();
			Attachment[] deleteList = IssueUtility.getAttachmentDeleteList(oldAttachList, newAttachList);

			// 保存する前に、課題票種別'D'の課題票が既にあるかどうかを調べる
			ModifyIssueFacade issueModifyFacade = new ModifyIssueFacade();
			Issue registedDraftIssue 
				= issueModifyFacade.getByFingerPrint(newIssue.getFingerPrint(), IssueType.DRAFT_VALUE);

			if(registedDraftIssue==null) {

				// 課題票を新規登録
				if(typeUpdateBefore.equals(IssueType.RELEASE_VALUE)) {
					//###  新規登録処理 CASE:'R'→'D'('D'がまだない。'R'を元に'D'を作成) ### 
					issueModifyFacade.addCloneIssue(newIssue);
				} else {
					//### 新規登録処理 CASE:新規登録 ###
					issueModifyFacade.addNewDraftIssue(newIssue);
				}
				
				// ビュー表示を更新
				List listeners = BtsPlugin.getInstance().getListeners();
				for (int idx = 0; idx < listeners.size(); idx++) {
					IBtsListener listnener = (IBtsListener)listeners.get(idx);
					listnener.addIssue(newIssue);
				}

				// 最初の下書き保存時だけメッセージを出す
				MessageDialog.openInformation
					( BtsPlugin.getInstance().getShell()
					, Messages.getString("SaveIssueAction.3") //$NON-NLS-1$
					, Messages.getString("SaveIssueAction.4"));  //$NON-NLS-1$

			} else {
				
				//### CASE:'D'→'D' ###
				if( IssueType.DRAFT_VALUE.equals(typeUpdateBefore) ) {
					
					// 課題票を更新
					issueModifyFacade.updateIssue(newIssue, deleteList);

					// 更新前の課題票を取得する。
					Issue oldIssue = issueEditorInput.getIssue();
					
					// ビュー表示を更新
					List listeners = BtsPlugin.getInstance().getListeners();
					for (int idx = 0; idx < listeners.size(); idx++) {
						IBtsListener listnener = (IBtsListener)listeners.get(idx);
						listnener.updateIssue(oldIssue, newIssue);
					}
				}
				
				//### CASE:'R'→'D'('D'がすでにある)  ###
				// (1).RELEASE版はそのまま保持して更新しない
				// (2).DB上の既存のDRAFT版を新規更新情報で上書き
				else if( IssueType.RELEASE_VALUE.equals(typeUpdateBefore) ) {

					// 課題票を更新
					issueModifyFacade.replaceIssue(newIssue, deleteList);

					// ビュー表示を更新
					List listeners = BtsPlugin.getInstance().getListeners();
					for (int idx = 0; idx < listeners.size(); idx++) {
						IBtsListener listnener = (IBtsListener)listeners.get(idx);

						listnener.updateIssue(registedDraftIssue, newIssue);
					}

					// 'R'→'D'の時、既に'D'の課題票が開いているときはそれを閉じる
					closeIssueEditor(newIssue);
				}
			}
			
			
			// IssueMultiPageEditor#doSave()を呼ぶ
			editorPart.postIssueSave(newIssue);
			
		} catch (Exception e) {
			String msg = Messages.getString("SaveIssueAction.5"); //$NON-NLS-1$
			logger.fatal(msg, e);
			BtsPlugin.getInstance().error(msg, e);
		}
	}

	/**
	 * 課題票に「下書き保存」の情報を格納する。
	 * 
	 * @param		newIssue		更新済み課題票
	 */
	private void updateIssueInfo(Issue newIssue) {
		
    	// 課題票情報を「DRAFT」に更新する。
		newIssue.setType(IssueType.DRAFT_VALUE);

		// 課題票の添付ファイル情報を更新する
		Attachment[] attachments = newIssue.getAttachments();
		if(attachments==null) {
			return;
		}
       	for (int i = 0; i < attachments.length; i++) {
       		// 最初の起票時はFingerPrintが設定されてないので、ここで設定する
			attachments[i].setFingerPrint(newIssue.getFingerPrint());
			attachments[i].setType(IssueType.DRAFT_VALUE);
		}

		
	}
	
	
	/**
	 * エディタを閉じます。<br>
	 * RELEASE→DRAFTとするとき、以前のDRAFT課題票に情報を上書きします。
	 * したがって上書きされる予定のDRAFT課題票がエディタで開かれているときには、
	 * そのエディタを自動的に閉じるようにします。
	 * 
	 * @param			issue				送信する課題票
	 */
	private void closeIssueEditor(Issue issue) {
	
		// 送信する課題票がDRAFTの場合のみ
		if( IssueType.DRAFT_VALUE.equals(issue.getType()) ) {
			// 同じFingerPrintを持つ課題票で、DRAFT版のものが開いていたら閉じます
			BtsPlugin.getInstance().closeIssueEditor(issue.getFingerPrint(), IssueType.DRAFT_VALUE);
		}
	}

}
