/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.util;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;

import jp.valtech.bts.data.Attachment;
import jp.valtech.bts.data.CurrentProject;
import jp.valtech.bts.data.MessagePacket;

/**
 * 課題票に設定する添付ファイルを操作するユーティリティクラスです。
 * 
 * @author		<A href="mailto:m_sugitou@valtech.jp">M.Sugito</A>
 * @version	Ver.0.8
 */
public class AttachmentUtility implements Logging {

	/** 添付ファイルを保管するディレクトリのPATH文字列 */
	private String registBaseDir	= CurrentProject.getInsance().getBaseDir();
	
	/**
	 * 当ユーティリティはインスタンスを生成する必要があります。
	 */
	public AttachmentUtility() {
		;
	}

	
	/**
	 *  添付ファイルをワークスペースに出力します。
	 *  <ul>
	 *    <li>引数<code>attachment</code>にはローカル端末上のファイル情報が格納されてます。
	 *    <li>ローカル端末上のファイルをワークスペース内に出力します。
	 *    <li>出力する際には「ファイルのGZIP圧縮」「ファイル名を一意の名前に変更」を行います。
	 *    <li>登録した情報のみをリストにして返します。
	 *  </ul>
	 * @param		attachmentList		添付ファイル情報のリスト
	 * @throws		IOException			出力失敗
	 * @return		登録した添付ファイル情報のリスト
	 */
	public Attachment[] writeWorkspaceFromLocal( Attachment[] attachments)  throws IOException {

		if(attachments ==null || attachments.length==0) {
			return null;
		}
		ArrayList addNewList = new ArrayList();

		for (int i = 0; i < attachments.length; i++) {
			if(!attachments[i].isRegisted()) {
				Attachment res = writeWorkspaceFromLocal(attachments[i]);
				if(res != null) {
					addNewList.add(res);
				}
			}
		}
		
		return (Attachment[])addNewList.toArray(new Attachment[0]);
	}


	/**
	 *  添付ファイルをワークスペースに出力します。
	 *  <ul>
	 *    <li>引数<code>attachment</code>にはローカル端末上のファイル情報が格納されてます。
	 *    <li>ローカル端末上のファイルをワークスペース内に出力します。
	 *    <li>出力する際には「ファイルのGZIP圧縮」「ファイル名を一意の名前に変更」を行います。
	 *    <li>登録した情報を{@link Attachment}の形式で返します。
	 *  </ul>
	 * @param		attachment			添付ファイル情報
	 * @throws		IOException			出力失敗
	 * @return		登録した添付ファイル情報
	 */
	public Attachment writeWorkspaceFromLocal( Attachment attachment)  throws IOException {
		
		// 既に登録済みの場合は何もしない。
		if(attachment.isRegisted()) {
			return null;
		}

		// 対象ファイルをバイト配列で取得
		byte[]	data	= IOUtility.read(attachment.getRealPath());
		// 取得したファイルをGZIP圧縮
		byte[]	compress	= IOUtility.compress(data);

		// ファイル名決定。必ず一意になるファイル名にする。
		// （「ファイル名」＋「そのときのミリ秒」でダイジェストを生成）
		String fileName = CipherUtility.hash( attachment.getRealPath() + System.currentTimeMillis() );
		File filePath	= new File( registBaseDir + fileName );
		
		// ファイルを書き出す
		IOUtility.write( filePath, compress );

		// 生成したファイル名を設定する。
		attachment.setRealPath(fileName);
		
		// 「保存済み」を設定する
		attachment.setRegisted(true);
		
		// 設定しなおしたインスタンスを返す
		return attachment;
	}

	
	/**
	 * 添付ファイルをローカルのファイルシステム上に出力します。<br>
	 * 当システムで管理していない添付ファイルを指定の場所に出力します。<br>
	 * <br>
	 * このメソッドは、課題票エディタにて添付ファイルを設定した後、更新情報を保存する前に
	 * 「添付ファイルを名前をつけて保存」を選んだときに呼ばれます。<br>
	 * 
	 * @param		attachment			添付ファイル情報
	 * @param		writePath			出力先PATH文字列
	 * @throws		IOException			出力失敗
	 */
	public void writeLocalFromLocal( Attachment attachment, String writePath)  throws IOException {

		// ファイルの実体を取得
		byte[]	data	= IOUtility.read(attachment.getRealPath());
		
		// ファイルシステムに出力
		File file = new File(writePath);
		IOUtility.write( file, data );
	}

	
	/**
	 * 添付ファイルをワークスペースに出力します。<br>
	 * 当システムでの管理下にある添付ファイルを指定の場所に出力します。<br>
	 * <br>
	 * このメソッドは、システムに登録済みの添付ファイルに対して「添付ファイルを名前をつけて保存」
	 * を選んだときに呼ばれます。<br>
	 * 
	 * @param		attachment			添付ファイル情報
	 * @param		writePath			出力先PATH文字列
	 * @throws		IOException			出力失敗
	 */
	public void writeLocalFromWorkspace( Attachment attachment, String writePath)  throws IOException {

		// ファイルの実体を取得
		byte[] data	= IOUtility.read( registBaseDir + attachment.getRealPath() );
		// 取得した実体を解凍する
		byte[]	decompress	= IOUtility.decompress(data);

		// ファイルシステムに出力
		File file = new File(writePath);
		IOUtility.write( file, decompress );
	}


	/**
	 * 添付ファイルをワークスペースに出力します。
	 * {@link Attachment#getAttachmentFile() バイト配列}で持つ添付ファイル情報をワークスペースに出力します。
	 * 
	 * @param		attachment			添付ファイル情報
	 * @throws		IOException			出力失敗
	 */
	public void writeWorkspaceFromEntity( Attachment[] attachments)  throws IOException {

		// 空の場合は何もしない。
		if(attachments==null) {
			return;
		}
		
		// 添付情報の数分繰り返す。
		for (int i = 0; i < attachments.length; i++) {
			byte[]	data	= attachments[i].getAttachmentFile();
			File filePath	= new File( registBaseDir + attachments[i].getRealPath() );
			IOUtility.write( filePath, data );
		}
	}

	
	/**
	 * 指定の添付ファイル情報の中から、ワークスペース上に登録済みのファイルのみクローンを作成します。
	 * クローンファイルは、ファイル名は違う名前ですが中身は全く同じです。
	 *  
	 * @param		attachments		添付ファイル情報の配列
	 * @return		クローン添付ファイル情報の配列
	 * @throws		IOException
	 */
	public Attachment[] clone( Attachment[] attachments)  throws IOException {

		// 空の場合はなにもしない
		if(attachments ==null || attachments.length==0) {
			return null;
		}

		ArrayList cloneList = new ArrayList();

		// 添付ファイル情報の数分繰り返します。
		for (int i = 0; i < attachments.length; i++) {
			
			// 登録済みのみが対象。
			if(attachments[i].isRegisted()) {
				
				// クローン作成。
				Attachment res = clone(attachments[i]);
				if(res != null) {
					// null以外が返ったもののみを登録する
					cloneList.add(res);
				}
			}
		}
		
		return (Attachment[])cloneList.toArray(new Attachment[0]);
	}


	/**
	 * 指定の添付ファイル情報の中から、ワークスペース上に登録済みのファイルのみクローンを作成します。
	 * クローンファイルは、ファイル名は違う名前ですが中身は全く同じです。
	 *  
	 * @param		attachments		添付ファイル情報
	 * @return		クローン添付ファイル情報
	 * @throws		IOException
	 */
	public Attachment clone( Attachment attachment)  throws IOException {
		
		// 登録済みのもののみ対象。
		if(!attachment.isRegisted()) {
			return null;
		}

		// 対象ファイルをバイト配列で取得
		byte[]	data = IOUtility.read( registBaseDir + attachment.getRealPath() );

		// ファイル名決定。必ず一意になるファイル名にする。
		// （「ファイル名」＋「そのときのミリ秒」でダイジェストを生成）
		String fileName = CipherUtility.hash( attachment.getRealPath() + System.currentTimeMillis() );
		
		// ファイルを書き出す
		File filePath	= new File( registBaseDir + fileName );
		IOUtility.write( filePath, data );

		// 生成したファイル名で上書きする。
		attachment.setRealPath(fileName);
		
		// 設定しなおしたインスタンスを返す
		return attachment;
	}

	
	/**
	 * 指定の添付ファイル情報にてワークスペース上のファイルを削除します。
	 * 
	 * @param		attachments			添付ファイル情報
	 * @throws		IOException		
	 */
	public void delete( Attachment[] attachments)  throws IOException {
		
		// 空の場合は何もしない。
		if(attachments==null) {
			return;
		}
		
		// 添付情報の数分繰り返す。
		for (int i = 0; i < attachments.length; i++) {
			File filePath	= new File( registBaseDir + attachments[i].getRealPath() );
			filePath.delete();
		}
	}

	
	/**
	 * 添付ファイルをワークスペースから読み込みバイト配列で取得します。
	 * 取得したバイト配列を{@link Attachment#setAttachmentFile(byte[])}にて{@link Attachment}に格納します。
	 *  
	 * @param		attachments			添付ファイル情報
	 * @throws		IOException
	 */
	public void read(Attachment[] attachments)  throws IOException {
		// 空の場合は何もしない。
		if(attachments==null) {
			return;
		}

		// 添付情報の数分繰り返す。
		for (int i = 0; i < attachments.length; i++) {

			// 登録済みのもののみ対象。
			if(attachments[i].isRegisted()) {
				// 対象ファイルをバイト配列で取得
				byte[]	data = IOUtility.read( registBaseDir + attachments[i].getRealPath() );
				
				// インスタンスに設定する。
				attachments[i].setAttachmentFile(data);
			}
		}
	}
	
	public void read(ArrayList attachmentList)  throws IOException {
		// 空の場合は何もしない。
		if(attachmentList==null || attachmentList.isEmpty()) {
			return;
		}

		// 添付情報の数分繰り返す。
		Iterator it = attachmentList.iterator();
		while(it.hasNext()) {
			Attachment attachment = (Attachment)it.next();
			// 登録済みのもののみ対象。
			if(attachment.isRegisted()) {
				// 対象ファイルをバイト配列で取得
				byte[]	data = IOUtility.read( registBaseDir + attachment.getRealPath() );
				
				// インスタンスに設定する。
				attachment.setAttachmentFile(data);
			}
		}
	}
	
	/**
	 * メッセージの添付ファイルをローカルからワークスペースに出力します。
	 * 
	 * @param		message			メッセージ情報
	 */
	public void writeWorkspaceFromLocal(MessagePacket message)  throws IOException {
		
		byte[] attach = message.getAttachFile();
		if(attach ==null) {
			return;
		}
		
		// 取得したファイルをGZIP圧縮
		byte[] compress = IOUtility.compress(attach);
		message.setAttachFile(compress);
		
		// ファイル名決定。必ず一意になるファイル名にする。
		// （「ファイル名」＋「添付した時のミリ秒」）
		File filePath	= new File(registBaseDir + message.getRealpath() );
		
		// ファイルを書き出す
		IOUtility.write( filePath, compress );
	}


	/**
	 * メッセージの添付ファイルをワークスペースに出力します。
	 * 
	 * @param		message			メッセージ情報
	 */
	public void writeWorkspaceFromEntity( MessagePacket message )  throws IOException {

		byte[]	data	= message.getAttachFile();
		if(data !=null) {
			File filePath	= new File( registBaseDir + message.getRealpath() );
			IOUtility.write( filePath, data );
		}
	}


	/**
	 * メッセージの添付ファイルをワークスペースからローカルに出力します。
	 * 
	 * @param		message			メッセージ情報
	 */
	public void writeLocalFromWorkspace( MessagePacket message, String writePath)  throws IOException {

		// ファイルの実体を取得
		byte[] data	= IOUtility.read( registBaseDir + message.getRealpath() );
		// 取得した実体を解凍する
		byte[]	decompress	= IOUtility.decompress(data);

		// ファイルシステムに出力
		File file = new File(writePath);
		IOUtility.write( file, decompress );
	}

	/**
	 * 指定のメッセージオブジェクトが持つ添付ファイル情報のにてワークスペース上のファイルを削除します。
	 * 
	 * @param		message			メッセージオブジェクト
	 * @throws		IOException		
	 */
	public void delete( MessagePacket message)  throws IOException {
		
		// 空の場合は何もしない。
		if(message==null) {
			return;
		}
		// 添付情報の数分繰り返す。
		File filePath	= new File( registBaseDir + message.getRealpath() );
		filePath.delete();
	}

}
