/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.ui.editor;

import jp.valtech.bts.data.IssueHistory;
import jp.valtech.bts.ui.BtsPlugin;
import jp.valtech.bts.util.BTSUtility;
import jp.valtech.bts.util.Logging;

import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CLabel;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.FontData;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.ui.forms.widgets.ScrolledForm;
import org.eclipse.ui.forms.widgets.Section;

/**
 * 課題票エディタの「履歴」ページ。
 * 
 * @author		<A href="mailto:m_sugitou@valtech.jp">M.Sugito</A>
 * @version	Ver.0.8
 */
public class HistoryPart extends AbstractIssueEditorPart implements Logging {

	/** このページのタブに設定する文字列 */
	public static final String TAB_TITLE=Messages.getString("HistoryPart.0"); //$NON-NLS-1$

	/** このページの部品で使うサイズの固定値 */
	private static final int COL1_WITH 	 =100;
	private static final int HEAD_HEIGHT = 18;
	private static final int ROW_HEIGHT  = 12; 
	private static final int ROW_MARGIN  = 4; 

	/** このページのフォーム */
	private ScrolledForm form ;
	/** セクションのコンポジットを保持 */
	private Composite sectionComposit;

	/** 表示する履歴情報 */
	private IssueHistory[] histories;
	
	
	/**
	 * このページを生成します。
	 */
	public HistoryPart() {
		super();
	}

	/* (非 Javadoc)
	 * @see org.eclipse.ui.part.WorkbenchPart#createPartControl(org.eclipse.swt.widgets.Composite)
	 */
	public void createPartControl(Composite parent) {
		try {
			// このページに表示する履歴を取得
			this.histories = this.issueDisplayValue.getIssue().getIssueHistories();
			
			// フォームボディ部生成
			Composite body = createBody(parent, Messages.getString("HistoryPart.1")); //$NON-NLS-1$
			// コメントセクションの生成
			createHistorySection(body);
			
			// レイアウトやスクロールバーの再処理を行う
			this.form.reflow(true);
		}catch (Exception e) {
			String msg = Messages.getString("HistoryPart.2"); //$NON-NLS-1$
			logger.fatal(msg, e);
			BtsPlugin.getInstance().error(msg, e);
		}
	}


	
	/**
	 * このページの基本となるボディ部を生成します。
	 * 
	 * @param		composite			親コンポジット
	 * @return		ボディ部のコンポジット
	 */
	private Composite createBody(Composite parent, String title) {
		if (toolkit == null) {
			getFormToolkit(parent);
		}
		
		GridLayout layout = new GridLayout();
		layout.marginHeight = 0;

		// スクロールフォーム生成
		form = toolkit.createScrolledForm(parent);
		form.setText(title);
		form.getBody().setLayout(layout);
		form.getBody().setLayoutData(new GridData(GridData.FILL_BOTH));
		
		// 基本レイアウト
		Composite body = toolkit.createComposite(form.getBody());
		body.setLayout(layout);
		body.setLayoutData(new GridData(GridData.FILL_BOTH));
		
		return body;
	}


	/**
	 * コメントセクションの生成
	 * 
	 * @param		parent				親コンポジット				
	 * @throws		Exception
	 */
	private void createHistorySection(Composite parent) throws Exception {
		
		// セクションインスタンス生成
		Section section = createSection(parent, Messages.getString("HistoryPart.3")); //$NON-NLS-1$
		section.setDescription(Messages.getString("HistoryPart.4")); //$NON-NLS-1$

		// セクションのレイアウト設定
		GridData gridData = new GridData(GridData.FILL_HORIZONTAL);
		section.setLayoutData(gridData);
		
		// このセクションに乗せる基本コンポジット生成
		sectionComposit = toolkit.createComposite(section);
		sectionComposit.setLayout(new GridLayout());
		sectionComposit.setLayoutData(new GridData(GridData.FILL_BOTH));
		section.setClient(sectionComposit);

		// 履歴テーブルを生成
		createHistoryTableArea();
	}

	

	/**
	 * 履歴を表示するテーブル生成。
	 */
	private void createHistoryTableArea() {
		// この部品の基本コンポジットを生成
		Composite historyTable = toolkit.createComposite(sectionComposit);
		toolkit.paintBordersFor(historyTable);
		
		// レイアウト・グリッドレイアウト設定
		GridLayout gridLayout = new GridLayout();
		gridLayout.numColumns = 5;
		gridLayout.marginWidth = 0;
		gridLayout.horizontalSpacing = 0;
		gridLayout.verticalSpacing = 0;
		historyTable.setLayout(gridLayout);

		GridData gridData = new GridData(GridData.FILL_BOTH);
		historyTable.setLayoutData(gridData);

		// テーブルのヘッダ部生成
		createHistoryHeader(historyTable);
		// テーブルのデータ部生成
		createHistoryData(historyTable);

        // ただの線を生成
        CLabel line = new CLabel(historyTable, SWT.HORIZONTAL);
        GridData gd = new GridData(GridData.FILL_HORIZONTAL);
        gd.horizontalSpan = 5;
        gd.heightHint = 1;
        line.setLayoutData(gd);
        line.setBackground(new Color(Display.getCurrent(), new RGB(153, 153, 255)));
	}
	

	/**
	 * 当ページを最新表示に更新します。
	 * 課題票配信処理後に呼ばれます。 
	 * 
	 * @param		histories		課題票履歴
	 */
	public void postSave(IssueHistory[] histories) {
		this.histories = histories;
		
		// 履歴テーブルを一旦消去
		Control[] children = sectionComposit.getChildren();
		for (int i = 0; i < children.length; i++) {
			children[i].dispose();
			children[i] = null;
		}
		
		// 履歴テーブルを生成
		createHistoryTableArea();
		reflow();
	}

	/**
	 * セクションコンポジット以下を再描画する。
	 *
	 */
	private void reflow() {
		Composite c = sectionComposit;
		while (c != null) {
			c.setRedraw(false);
			c = c.getParent();
			if (c instanceof ScrolledForm) {
				break;
			}
		}
		c = sectionComposit;
		while (c != null) {
			c.layout(true);
			c = c.getParent();
			if (c instanceof ScrolledForm) {
				((ScrolledForm) c).reflow(true);
				break;
			}
		}
		c = sectionComposit;
		while (c != null) {
			c.setRedraw(true);
			c = c.getParent();
			if (c instanceof ScrolledForm) {
				break;
			}
		}
	}
	
	
	/**
	 * 履歴テーブルのヘッダ部を生成します。
	 * 
	 * @param		historyTable		履歴テーブルのコンポジット
	 */
	private void createHistoryHeader(Composite historyTable) {

		// テーブルヘッダで使う色。
		Color headBackground = new Color(Display.getCurrent(), new RGB(195, 195, 255));
		Color headForeground = new Color(Display.getCurrent(), new RGB(255, 255, 255));

		// テーブルヘッダで使うフォント。
		Font font = new Font(Display.getCurrent(), new FontData("", 10, SWT.BOLD));
		
		// １カラム目のヘッダ
		CLabel head1 = new CLabel(historyTable, SWT.CENTER);
		head1.setText(Messages.getString("HistoryPart.5")); //$NON-NLS-1$
		head1.setBackground(headBackground);
		head1.setForeground(headForeground);
		head1.setFont(font);
		
		GridData headGrid = new GridData();
		headGrid.widthHint  = COL1_WITH;
		headGrid.heightHint = HEAD_HEIGHT;
		head1.setLayoutData(headGrid);
		
		// ２カラム目のヘッダ
		CLabel head2 = new CLabel(historyTable, SWT.CENTER);
		head2.setText(Messages.getString("HistoryPart.6")); //$NON-NLS-1$
		head2.setBackground(headBackground);
		head2.setForeground(headForeground);
		head2.setFont(font);
		
		headGrid = new GridData();
		headGrid.widthHint  = COL1_WITH;
		headGrid.heightHint = HEAD_HEIGHT;
		head2.setLayoutData(headGrid);
		
		// ３カラム目のヘッダ
		CLabel head3 = new CLabel(historyTable, SWT.CENTER);
		head3.setText(Messages.getString("HistoryPart.7")); //$NON-NLS-1$
		head3.setBackground(headBackground);
		head3.setForeground(headForeground);
		head3.setFont(font);
		
		headGrid = new GridData();
		headGrid.widthHint  = COL1_WITH;
		headGrid.heightHint = HEAD_HEIGHT;
		head3.setLayoutData(headGrid);

		// ４カラム目のヘッダ
		CLabel head4 = new CLabel(historyTable, SWT.CENTER);
		head4.setText(Messages.getString("HistoryPart.8")); //$NON-NLS-1$
		head4.setBackground(headBackground);
		head4.setForeground(headForeground);
		head4.setFont(font);
		
		headGrid = new GridData(GridData.FILL_HORIZONTAL);
		headGrid.heightHint = HEAD_HEIGHT;
		head4.setLayoutData(headGrid);

		// ５カラム目のヘッダ
		CLabel head5 = new CLabel(historyTable, SWT.CENTER);
		head5.setText(Messages.getString("HistoryPart.9")); //$NON-NLS-1$
		head5.setBackground(headBackground);
		head5.setForeground(headForeground);
		head5.setFont(font);
		
		headGrid = new GridData(GridData.FILL_HORIZONTAL);
		headGrid.heightHint = HEAD_HEIGHT;
		head5.setLayoutData(headGrid);	
	}


	/**
	 * 履歴テーブルのデータ部を生成します。
	 * 
	 * @param		historyTable		履歴テーブルのコンポジット
	 */
	private void createHistoryData(Composite historyTable) {
		
		// 課題票履歴取得
		if( this.histories == null || this.histories.length == 0) {
			return;
		}

		// テーブルに設定する色
		Color headBackground1 = new Color(Display.getCurrent(), new RGB(255, 255, 255));
		Color headBackground2 = new Color(Display.getCurrent(), new RGB(240, 242, 255));

		// 取得した履歴でテーブル生成
		for (int idx = 0; idx < histories.length; idx++) {
			
			// バックグラウンドカラーの設定。（偶数行：薄いブルー。奇数行：白）
			Color headBackground = headBackground2;
			if(idx % 2 == 0) {
				headBackground = headBackground1;
			}
			
			// 行の高さを取得
			int height = computeHeightHint(histories[idx]);

			// 幅の狭いカラムのレイアウト
			GridData colM = new GridData();
			colM.widthHint  = COL1_WITH;
			colM.verticalSpan = 0;
			colM.heightHint = height;


			// 幅の広いカラムのレイアウト
			GridData colL = new GridData(GridData.FILL_HORIZONTAL);
			colL.verticalSpan = 0;
			colL.heightHint = height;

			// カラムで使うラベルインスタンス生成
			CLabel col1 = new CLabel(historyTable, SWT.LEFT); 
 			CLabel col2 = new CLabel(historyTable, SWT.LEFT); 
 			CLabel col3 = new CLabel(historyTable, SWT.LEFT); 
 			Label col4 = new Label(historyTable, SWT.LEFT); 
 			Label col5 = new Label(historyTable, SWT.LEFT); 

 			// 更新日時取得
			String dateStr = BTSUtility.formatDate( histories[idx].getUpdateDate(), "yyyy/MM/dd HH:mm" );
			col1.setText(dateStr);
			col1.setBackground(headBackground);
			col1.setLayoutData(colM);
			
			
			// 更新者名取得
			String userName = histories[idx].getUpdateUser();
			col2.setText(userName);
			col2.setBackground(headBackground);
			col2.setLayoutData(colM);
			
			// 更新した属性
			String updAttribute = getUpdateAttribute(histories[idx]);
			col3.setText(updAttribute);
			col3.setBackground(headBackground);
			col3.setLayoutData(colM);

			// 変更前詳細
			String updBefore = getUpdateBefore(histories[idx]);
			col4.setText(updBefore);
			col4.setBackground(headBackground);
			col4.setLayoutData(colL);

			// 変更後詳細
			String updAfter = getUpdateAfter(histories[idx]);
			col5.setText(updAfter);
			col5.setBackground(headBackground);
			col5.setLayoutData(colL);
		}
	}
	
	
    /**
     * 課題票履歴の「属性」項目を返す。
     * 
     * @param		issueHistory		課題票履歴
     * @return		課題票履歴の「属性」項目。nullの場合は空文字を返す。
     */
	private String getUpdateAttribute(IssueHistory issueHistory) {
		String updAttribute = issueHistory.getUpdateAttribute();
		if(updAttribute ==null) {
			return "";
		}
		return updAttribute;
	}
	

    /**
     * 課題票履歴の「更新後」項目を返す。
     * 
     * @param		issueHistory		課題票履歴
     * @return		課題票履歴の「更新後」項目。nullの場合は空文字を返す。
     */
	private String getUpdateAfter(IssueHistory issueHistory) {
		String updateAfter = issueHistory.getUpdateAfter();
		if(updateAfter ==null) {
			return "";
		}
		return updateAfter;
	}
	

    /**
     * 課題票履歴の「更新前」項目を返す。
     * 
     * @param		issueHistory		課題票履歴
     * @return		課題票履歴の「更新前」項目。nullの場合は空文字を返す。
     */
	private String getUpdateBefore(IssueHistory issueHistory) {
		String updateBefore = issueHistory.getUpdateBefore();
		if(updateBefore ==null) {
			return "";
		}
		return updateBefore;
	}


	/**
	 * 履歴テーブルに設定する1行の高さをint値で返します。
	 * 課題票情報を欠けることなく表示するための、1行の高さを計算して返します。
	 * 
	 * @param		issueHistory		課題票履歴
	 * @return		1行の高さ
	 */
	private int computeHeightHint(IssueHistory issueHistory) {
		int row1Cnt = countRow(issueHistory.getUpdateBefore());
		int row2Cnt = countRow(issueHistory.getUpdateAfter());
		
		int ret = row1Cnt;
		if(row1Cnt < row2Cnt) {
			ret = row2Cnt;
		}
		return ret * ROW_HEIGHT + ROW_MARGIN;
	}

	
	/**
	 * 指定の文字列の行数を返します。
	 *  
	 * @param		str			指定の文字列
	 * @return		指定の文字列の行数。nullの場合は1行とし「１」を返す。
	 */
	private int countRow(String str) {
		if(str == null) {
			return 1;
		}
		String[] tmp = str.split("\n");
		return tmp.length;
	}
	

	/**
	 * 常にfalseを返します。
	 * 
	 * @return		常にfalse
	 */
    public boolean isDirty() {
    	return false;
    }


    /*
	 *  (非 Javadoc)
	 * @see org.eclipse.ui.IWorkbenchPart#setFocus()
	 */
	public void setFocus() {
		sectionComposit.setFocus();
	}
	
	
    /*
     *  (非 Javadoc)
     * @see jp.valtech.bts.ui.editor.AbstractIssueEditorPart#notifyModifyInput()
     */
	protected void fireIssueChange() {
		// 何もしない。
	}
	
}
