/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.dao;

import java.util.ArrayList;
import java.util.List;

import jp.valtech.bts.connection.DBConnection;
import jp.valtech.bts.data.Attachment;

import org.apache.commons.dbutils.QueryRunner;
import org.apache.commons.dbutils.ResultSetHandler;
import org.apache.commons.dbutils.handlers.BeanListHandler;

/**
 * 添付ファイル用のDAOです。
 * ファイルの実体はDBには保存しません。ファイルシステム上に保存します。
 * 
 * @author		<A href="mailto:m_sugitou@valtech.jp">M.Sugito</A>
 * @version	Ver.0.8
 */
public class AttachmentDAO extends DataBaseAccessor{
	
	/** テーブル作成SQL */
	private static final String CREATE_TABLE
			= "CREATE TABLE ATTACHMENT" 
			+  "( FINGERPRINT  VARCHAR(40)"
			+  ", TYPE         CHAR(1)"
			+  ", DISPLAYNAME  VARCHAR(40)"
			+  ", REALPATH     VARCHAR(300)"
			+  ") ;"
			+ "CREATE INDEX IDX_ATTACHMENT ON ATTACHMENT(FINGERPRINT) ;";

	/** テーブル削除SQL */
	private static final String DROP_TABLE
			= "DROP TABLE ATTACHMENT CASCADE";

	/** データ登録SQL */
	private static final String INSERT_SQL
			= "INSERT INTO ATTACHMENT("
			+  " FINGERPRINT"
			+ ", TYPE"
			+ ", DISPLAYNAME"
			+ ", REALPATH "
			+ ") VALUES(?, ?, ?, ?)";

	/** データ削除SQL */
	private static final String DELETE_SQL
			= "DELETE FROM ATTACHMENT WHERE REALPATH=?";

	/** FingerPrintによるデータ削除SQL */
	private static final String DELETE_BY_FINGERPRINT
			= "DELETE FROM ATTACHMENT WHERE FINGERPRINT=? AND TYPE=?";
	
	/** ごみ箱に移動するSQL */
	private static final String UPDATE_GARBAGE	
			= "UPDATE ATTACHMENT SET TYPE='G' WHERE FINGERPRINT=? AND TYPE='R'"; 

	/** 課題表種別をDRAFTからRELEASEに置き換える */
	private static final String REPLACE_ISSUE_TYPE
			= "UPDATE ATTACHMENT SET TYPE='R' WHERE FINGERPRINT=? AND TYPE='D'";

	/** データ全件取得 */
	private static final String SELECT_ALL
			= "SELECT"
			+  " FINGERPRINT"
			+ ", TYPE"
			+ ", DISPLAYNAME"
			+ ", REALPATH"
			+ " FROM ATTACHMENT";

	/** FingerPrintによるデータ取得SQL */
	private static final String SELECT_BY_FINGERPRINT
			= SELECT_ALL + " WHERE FINGERPRINT=? AND TYPE=? ";

	/**
	 * <DL><DT><B>コンストラクタ</B>
	 *  <DD></DD>
	 * </DL>
	 * @param connection
	 */
	public AttachmentDAO( DBConnection connection ) {
		super( connection );
	}


	/**
	 * テーブルを生成します。
	 * 
	 * @throws		BtsDBException
	 */
	public void createTable() throws BtsDBException {
		try {
			// テーブル生成用SQL実行
			new QueryRunner().update( this.con, CREATE_TABLE );
			
		} catch (Exception e) {
			throw new BtsDBException(e);
		}
	}

	/**
	 * テーブルを削除します。
	 * 
	 * @throws		BtsDBException
	 */
	public void dropTable() throws BtsDBException {
		try {
			// テーブル削除用SQL実行
			new QueryRunner().update( this.con, DROP_TABLE );	
		} catch (Exception e) {
			throw new BtsDBException(e);
		}
	}

	/**
	 * <DL><DT><B>指定の添付ファイル情報を登録します。</B></DT>
	 *  <DD></DD>
	 * </DL>
	 * @param attachment
	 * @throws BtsDBException
	 */
	public void addAttachment( Attachment attachment ) throws BtsDBException {
		try {
			List param = new ArrayList();
			param.add(attachment.getFingerPrint());
			param.add(attachment.getType());
			param.add(attachment.getDisplayName());
			param.add(attachment.getRealPath());

			new QueryRunner().update( this.con, INSERT_SQL, param.toArray() );	
		} catch (Exception e) {
			throw new BtsDBException(e);
		}
	}

	/**
	 * <DL><DT><B>指定の添付ファイル情報を削除します。</B></DT>
	 *  <DD></DD>
	 * </DL>
	 * @param attachment
	 * @throws BtsDBException
	 */
	public void delete( Attachment attachment ) throws BtsDBException{
		try {
			new QueryRunner().update( this.con, DELETE_SQL, attachment.getRealPath() );	
		} catch (Exception e) {
			throw new BtsDBException(e);
		}
	}

	/**
	 * <DL><DT><B>指定の添付ファイル情報を削除します。</B></DT>
	 *  <DD></DD>
	 * </DL>
	 * @param attachment
	 * @throws BtsDBException
	 */
	public void deleteByFingerPrint(String fingerPrint , String type) throws BtsDBException{
		try {
			List param = new ArrayList();
			param.add(fingerPrint);
			param.add(type);

			new QueryRunner().update( this.con, DELETE_BY_FINGERPRINT, param.toArray() );	
		} catch (Exception e) {
			throw new BtsDBException(e);
		}
	}

	
	/**
	 * <DL><DT><B>指定のFingerPrintの課題票種別を'D'から'R'に置き換えます。</B></DT>
	 *  <DD></DD>
	 * </DL>
	 * @param fingerPrint
	 * @throws BtsDBException
	 */
	public int replaceReleaseVersion(String fingerPrint) throws BtsDBException{
		try {
			return new QueryRunner().update( this.con, REPLACE_ISSUE_TYPE, fingerPrint );	
		} catch (Exception e) {
			throw new BtsDBException(e);
		}
	}
	

	/**
	 * <DL><DT><B></B></DL>
	 *  <DD></DD>
	 * </DL>
	 * @param issueSystemID
	 * @return			課題票管理IDに属する添付ファイルデータ全件
	 * @throws BtsDBException
	 */
	public ArrayList getByFingerPrint(String fingerPrint , String type) throws BtsDBException {

		try {
			List param = new ArrayList();
			param.add(fingerPrint);
			param.add(type);

			ResultSetHandler rsh = new BeanListHandler(Attachment.class);
			
			return (ArrayList)new QueryRunner().query( this.con, SELECT_BY_FINGERPRINT, param.toArray(), rsh );
			
		} catch (Exception e) {
			throw new BtsDBException(e);
		}
	}

	/**
	 * <DL><DT><B>指定の課題票をごみ箱に移動します。</B></DT>
	 *  <DD></DD>
	 * </DL>
	 * @param		fingerPrint			削除対象の課題票のFingerPrint
	 * @throws 		BtsDBException
	 */
	public void changeGarbageVersion(String fingerPrint) throws BtsDBException{
		try {
			List param = new ArrayList();
			param.add(fingerPrint);

			new QueryRunner().update( this.con, UPDATE_GARBAGE, param.toArray() );	
		} catch (Exception e) {
			throw new BtsDBException(e);
		}
	}

}
