/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.dao;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.List;

import jp.valtech.bts.connection.DBConnection;
import jp.valtech.bts.data.Issue;
import jp.valtech.bts.data.IssueSyncHeader;
import jp.valtech.bts.data.IssueType;

import org.apache.commons.dbutils.QueryRunner;
import org.apache.commons.dbutils.ResultSetHandler;
import org.apache.commons.dbutils.handlers.ArrayListHandler;
import org.apache.commons.dbutils.handlers.BeanHandler;
import org.apache.commons.dbutils.handlers.BeanListHandler;
import org.apache.commons.dbutils.handlers.ScalarHandler;

/**
 * <dl><dt><b>課題票用のDAO</b></dt>
 * <dd>正確にはDBAO</dd>
 * <dt><b>使い方</b></dt>
 * <dd>
 * </dd>
 * </dl>
 * 
 * @author		<A href="mailto:m_sugitou@valtech.jp">M.Sugito</A>
 * @version	Ver.0.8
 */
public class IssueDAO extends DataBaseAccessor {

	/** テーブル作成SQL */
	private static final String CREATE_TABLE
			= "CREATE TABLE ISSUE" 
			+ " ( ISSUEID        INTEGER DEFAULT 0" 
			+ " , FINGERPRINT    CHAR(40)" 
			+ " , TYPE           CHAR(1)" 
			+ " , CATEGORY       VARCHAR(40)" 
			+ " , TITLE          VARCHAR(90)" 
			+ " , DESCRIPTION    LONGVARCHAR"  
			+ " , PRIORITY       VARCHAR(40)"  
			+ " , DEADLINE       TIMESTAMP"  
			+ " , STATUS         CHAR(1) DEFAULT '1'"  
			+ " , ASSIGNED       VARCHAR(40)"  
			+ " , SECRETBUG      BOOLEAN DEFAULT false"  
			+ " , SUSPEND        BOOLEAN DEFAULT false"  
			+ " , VERSION        INTEGER DEFAULT 0 "  
			+ " , CREATEUSER     VARCHAR(40) "  
			+ " , CREATEDATE     TIMESTAMP"  
			+ " , UPDATEDATE     TIMESTAMP"  
			+ " ) ;"
			+ "CREATE INDEX IDX_ISSUE ON ISSUE(FINGERPRINT) ;";

	/** テーブル削除SQL */
	private static final String DROP_TABLE
			= "DROP TABLE ISSUE CASCADE";

	/** データ登録SQL */
	private static final String INSERT_SQL
			= "INSERT INTO ISSUE"
			+ " ( ISSUEID"
			+ " , FINGERPRINT"
			+ " , TYPE"
			+ " , CATEGORY"
			+ " , TITLE"
			+ " , DESCRIPTION"
			+ " , PRIORITY"
			+ " , DEADLINE"
			+ " , STATUS"
			+ " , ASSIGNED"
			+ " , SECRETBUG"
			+ " , SUSPEND"
			+ " , VERSION"
			+ " , CREATEUSER"
			+ " , CREATEDATE"
			+ " , UPDATEDATE"
			+ " ) "
			+ " VALUES(?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";

	/** キーを基準にデータを更新 */
	private static final String UPDATE_TABLE
			= "UPDATE ISSUE "
			+ "   SET ISSUEID=?"
			+ "     , TYPE=?"
			+ "     , CATEGORY=?"
			+ "     , TITLE=?"
			+ "     , DESCRIPTION=?"
			+ "     , PRIORITY=?"
			+ "     , DEADLINE=?"
			+ "     , STATUS=?"
			+ "     , ASSIGNED=?"
			+ "     , SECRETBUG=?"
			+ "     , SUSPEND=?"
			+ "     , VERSION=?"
			+ "     , CREATEUSER=?"
			+ "     , CREATEDATE=?"
			+ "     , UPDATEDATE=?"
			+ " WHERE FINGERPRINT=? AND TYPE=?";

	/** 課題票種別を更新します。 */
	private static final String UPDATTE_RELEASE	
			= "UPDATE ISSUE SET TYPE='R' WHERE FINGERPRINT=? AND TYPE='D'"; 

	/** 課題票種別を更新します。 */
	private static final String UPDATE_GARBAGE	
			= "UPDATE ISSUE SET TYPE='G' WHERE FINGERPRINT=? AND TYPE='R'"; 

	/** データを全件取得するSQL */
	private static final String SELECT_ALL
			= "SELECT" 
			+ "    nvl(ISSUEID, 0) ISSUEID" 
			+ "  , FINGERPRINT" 
			+ "  , TYPE" 
			+ "  , CATEGORY" 
			+ "  , TITLE" 
			+ "  , DESCRIPTION" 
			+ "  , PRIORITY" 
			+ "  , DEADLINE" 
			+ "  , nvl(STATUS, '1') STATUS" 
			+ "  , ASSIGNED" 
			+ "  , nvl(SECRETBUG, false) SECRETBUG" 
			+ "  , nvl(SUSPEND, false) SUSPEND" 
			+ "  , VERSION" 
			+ "  , CREATEUSER" 
			+ "  , CREATEDATE" 
			+ "  , UPDATEDATE " 
			+ "  FROM ISSUE";
			

	private static final String SELECT_ALL_ORDER
			= SELECT_ALL + " ORDER BY ISSUEID DESC";

	private static final String SELECT_BY_FINGERPRINT
			= SELECT_ALL + " WHERE FINGERPRINT=? AND TYPE=?";

	private static final String GET_MAX_ISSUEID
			= "SELECT MAX(ISSUEID) FROM ISSUE ";

	private static final String DELETE_BY_FINGERPRINT
			= "DELETE FROM ISSUE WHERE FINGERPRINT=? AND TYPE=?";
	
	private static final String SELECT_BY_TYPE
			= SELECT_ALL + " WHERE TYPE=?";
	
	/** 課題票種別がRのみ取得するSQL */
	private static final String SELECT_BY_TYPE_R
			= SELECT_ALL + " WHERE TYPE='R'";
	
	/** 同期処理で使う課題票ヘッダ情報の取得 */
	private static final String GET_SYNC_HEADER
			= "SELECT" 
			+ "   FINGERPRINT" 
			+ " , VERSION" 
			+ " , UPDATEDATE " 
			+ " FROM ISSUE"
			+ " WHERE TYPE=?";

	/** 担当者一覧を取得するSQL（課題票フィルタダイアログ用） */
	private static final String GET_ASSIGNED
			= "SELECT DISTINCT ASSIGNED FROM ISSUE";
	
	/** 課題票種別がRの担当者一覧を取得するSQL（担当者別課題票出力用） */
	private static final String GET_ASSIGNED_BY_TYPE_R
			= "SELECT DISTINCT ASSIGNED FROM (" + SELECT_BY_TYPE_R + ")";
	
	/** 課題票種別がRのカテゴリ一覧を取得するSQL（バグ累計ダイアログ・帳票出力ダイアログ用） */
	private static final String GET_CATEGORY_BY_TYPE_R
			= "SELECT DISTINCT CATEGORY FROM (" + SELECT_BY_TYPE_R + ")";
	
	/** 優先度一覧を取得するSQL（課題票フィルタダイアログ用） */
	private static final String GET_PRIORITY
			= "SELECT DISTINCT PRIORITY FROM ISSUE";
	

	/**
	 * <DL><DT><B>コンストラクタ</B>
	 *  <DD></DD>
	 * </DL>
	 * @param connection
	 */
	public IssueDAO( DBConnection connection ) {
		super( connection );
	}

	/**
	 * テーブルを生成します。
	 * 
	 * @throws		BtsDBException
	 */
	public void createTable() throws BtsDBException {
		try {
			// テーブル生成用SQL実行
			new QueryRunner().update( this.con , CREATE_TABLE );
			
		} catch (Exception e) {
			throw new BtsDBException(e);
		}
	}

	/**
	 * テーブルを削除します。
	 * 
	 * @throws		BtsDBException
	 */
	public void dropTable() throws BtsDBException {
		try {
			// テーブル削除用SQL実行
			new QueryRunner().update( this.con , DROP_TABLE );	
		} catch (Exception e) {
			throw new BtsDBException(e);
		}
	}

	/**
	 * <DL><DT><B>データを登録します。</B></DL>
	 *  <DD></DD>
	 * </DL>
	 * @param issue
	 * @throws BtsDBException
	 */
	public void addIssue( Issue issue ) throws BtsDBException {
		try {
			List param = new ArrayList();
			param.add( new Integer(issue.getIssueID()) );
			param.add( issue.getFingerPrint() );
			param.add( issue.getType() );
			param.add( issue.getCategory() );
			param.add( issue.getTitle() );
			param.add( issue.getDescription() );
			param.add( issue.getPriority() );
			if( issue.getDeadline() != null ){
				param.add( new Timestamp( issue.getDeadline().getTime() ) );
			}else{
				param.add( null );
			}
			param.add( issue.getStatus() );
			param.add( issue.getAssigned() );
			param.add( issue.isSecretBug() );
			param.add( issue.isSuspend() );
			param.add( issue.getVersion() );
			param.add( issue.getCreateUser() );
			if( issue.getCreateDate() != null ){
				param.add( new Timestamp( issue.getCreateDate().getTime() ) );
			}else{
				param.add( null );
			}
			if( issue.getUpdateDate() != null ){
				param.add( new Timestamp( issue.getUpdateDate().getTime() ) );
			}else{
				param.add( null );
			}

			new QueryRunner().update( this.con, INSERT_SQL, param.toArray() );	
		} catch (Exception e) {
			throw new BtsDBException(e);
		}
	}

	/**
	 * <DL><DT><B></B></DL>
	 *  <DD></DD>
	 * </DL>
	 * @param issue
	 * @throws BtsDBException
	 */
	public int modifyIssue( Issue issue , String type) throws BtsDBException{
		try {
			List param = new ArrayList();
			param.add( new Integer(issue.getIssueID()) );
			param.add( issue.getType() );
			param.add( issue.getCategory() );
			param.add( issue.getTitle() );
			param.add( issue.getDescription() );
			param.add( issue.getPriority() );
			if( issue.getDeadline() != null ){
				param.add( new Timestamp( issue.getDeadline().getTime() ) );
			}else{
				param.add( null );
			}
			param.add( issue.getStatus() );
			param.add( issue.getAssigned() );
			param.add( issue.isSecretBug() );
			param.add( issue.isSuspend() );
			param.add( issue.getVersion() );		
			param.add( issue.getCreateUser() );
			if( issue.getCreateDate() != null ){
				param.add( new Timestamp( issue.getCreateDate().getTime() ) );
			}else{
				param.add( null );
			}
			if( issue.getUpdateDate() != null ){
				param.add( new Timestamp( issue.getUpdateDate().getTime() ) );
			}else{
				param.add( null );
			}
			param.add( issue.getFingerPrint() );
			param.add( type );

			return new QueryRunner().update( this.con, UPDATE_TABLE, param.toArray() );	
		
		} catch (Exception e) {
			throw new BtsDBException(e);
		}
	}

	/**
	 * <DL><DT><B></B></DL>
	 *  <DD></DD>
	 * </DL>
	 * @param issue
	 * @throws BtsDBException
	 */
	public void changeReleaseVersion(String fingerPrint) throws BtsDBException{
		try {
			List param = new ArrayList();
			param.add(fingerPrint);

			new QueryRunner().update( this.con, UPDATTE_RELEASE, param.toArray() );	
		} catch (Exception e) {
			throw new BtsDBException(e);
		}
	}


	
	/**
	 * <DL><DT>課題票を全件取得します。<B></B></DL>
	 *  <DD></DD>
	 * </DL>
	 * @return		課題票全件
	 * @throws		BtsDBException
	 */
	public Issue[] getAll() throws BtsDBException {

		List result = getIssueList();
		return (Issue[])result.toArray(new Issue[0]);
	}

	/**
	 * <DL><DT>課題票を全件取得します。<B></B></DL>
	 *  <DD></DD>
	 * </DL>
	 * @return		課題票全件
	 * @throws		BtsDBException
	 */
	public List getIssueList() throws BtsDBException {

		try {
			ResultSetHandler rsh = new BeanListHandler(Issue.class);
			List issueList = (List)new QueryRunner().query( this.con, SELECT_ALL_ORDER, rsh );
			return issueList;

		} catch (Exception e) {
			throw new BtsDBException(e);
		}
	}


	/**
	 * <DL><DT><B>課題票のフィンガープリントを使用して一意の課題票を取得します。</B></DT>
	 *  <DD></DD>
	 * </DL>
	 * @param fingerprint
	 * @return					指定されたフィンガープリントの課題票(1件)
	 * @throws BtsDBException
	 */
	public Issue getByFingerPrint( String fingerprint, String type ) throws BtsDBException{
		Issue issue = null;
		List param = new ArrayList();
		param.add(fingerprint);
		param.add(type);

		try {
			ResultSetHandler rsh = new BeanHandler(Issue.class);
			
			issue = (Issue)new QueryRunner().query( this.con, SELECT_BY_FINGERPRINT, param.toArray(), rsh );

		} catch (Exception e) {
			throw new BtsDBException(e);
		}
		return issue;
	}

	/**
	 * <DL><DT><B>課題票のフィンガープリントを使用して一意の課題票を削除します。</B></DT>
	 *  <DD></DD>
	 * </DL>
	 * @param			fingerprint			課題票のフィンガープリント
	 * @param			type				課題票の種別
	 * @throws			BtsDBException
	 */
	public int deleteByFingerPrint( String fingerprint, String type ) throws BtsDBException{
		try {
			List param = new ArrayList();
			param.add( fingerprint );
			param.add( type );
			return new QueryRunner().update( this.con, DELETE_BY_FINGERPRINT, param.toArray() );	
		} catch (Exception e) {
			throw new BtsDBException(e);
		}
	}

	
	/**
	 * <DL><DT><B>課題票の件数を返します。</B></DT>
	 *  <DD>「下書き」「コンフリクト」の課題は除外した件数です。</DD>
	 * </DL>
	 * @return					課題票の件数
	 * @throws BtsDBException
	 */
	public int getUsefulIssueCnt() throws BtsDBException{

		try {
			ResultSetHandler rsh = new ScalarHandler(1);
			
			Object result = new QueryRunner().query( this.con, GET_MAX_ISSUEID, rsh );
			if(result == null) {
				return 0;
			}
			return Integer.parseInt(result.toString());
			
		} catch (Exception e) {
			throw new BtsDBException(e);
		}
	}
	
	
	/**
	 * <DL><DT><B>課題票の種別を使用して課題票を取得します。</B></DL>
	 *  <DD></DD>
	 * </DL>
	 * @param type				課題票の種別
	 * @return					指定された種別の課題票全件
	 * @throws BtsDBException
	 */
	public List getByType( String type ) throws BtsDBException{
		List param = new ArrayList();
		param.add(type);
		
		try {
			ResultSetHandler rsh = new BeanListHandler(Issue.class);
			List issueList = (List)new QueryRunner().query( this.con, SELECT_BY_TYPE, param.toArray(), rsh );
			
			return issueList;
			
		} catch (Exception e) {
			throw new BtsDBException(e);
		}

	}
	
	
	/**
	 * <DL><DT><B>ごみ箱入りの課題票全件を取得します。</B></DT>
	 *  <DD></DD>
	 * </DL>
	 * @return		ごみ箱入りの課題票全件
	 * @throws		BtsDBException
	 */
	public List getGarbageList() throws BtsDBException{
		List param = new ArrayList();
		param.add(IssueType.GARBAGE_VALUE);
		
		try {
			ResultSetHandler rsh = new BeanListHandler(Issue.class);
			List issueList = (List)new QueryRunner().query( this.con, SELECT_BY_TYPE, param.toArray(), rsh );
			
			return issueList;
			
		} catch (Exception e) {
			throw new BtsDBException(e);
		}

	}

	
	/**
	 * <DL><DT><B>課題票の担当者一覧(重複なし)を取得します。</B></DL>
	 *  <DD></DD>
	 * </DL>
	 * @return					担当者一覧
	 * @throws BtsDBException
	 */
	public String[] getAssigned() throws BtsDBException{

		try {
			ResultSetHandler rsh = new ArrayListHandler();
			
			List list = (List) new QueryRunner().query( this.con, GET_ASSIGNED, rsh );
			
			List result = new ArrayList();

			for (int i = 0; i < list.size(); i++) {

				if(((Object[])list.get(i))[0] != null && !"".equals(((Object[])list.get(i))[0])) {
					result.add(((Object[])list.get(i))[0]);
				}

			}

			return (String[]) result.toArray(new String[0]);
			
		} catch (Exception e) {
			throw new BtsDBException(e);
		}

	}
	
	
	/**
	 * <DL><DT><B>課題票種別がRの担当者一覧(重複なし)を取得します。</B></DL>
	 *  <DD></DD>
	 * </DL>
	 * @return					種別がR担当者一覧
	 * @throws BtsDBException
	 */
	public String[] getAssignedTypeR() throws BtsDBException{

		try {
			ResultSetHandler rsh = new ArrayListHandler();
			
			List list = (List) new QueryRunner().query( this.con, GET_ASSIGNED_BY_TYPE_R, rsh );
			
			List result = new ArrayList();

			for (int i = 0; i < list.size(); i++) {

				if(((Object[])list.get(i))[0] != null && !"".equals(((Object[])list.get(i))[0])) {
					result.add(((Object[])list.get(i))[0]);
				}

			}

			return (String[]) result.toArray(new String[0]);
			
		} catch (Exception e) {
			throw new BtsDBException(e);
		}

	}
	
	
	/**
	 * <DL><DT><B>課題票種別がRのカテゴリ一覧(重複なし)を取得します。</B></DL>
	 *  <DD></DD>
	 * </DL>
	 * @return					種別がRのカテゴリ一覧
	 * @throws BtsDBException
	 */
	public String[] getCategoryTypeR() throws BtsDBException{

		try {
			ResultSetHandler rsh = new ArrayListHandler();
			
			List list = (List) new QueryRunner().query( this.con, GET_CATEGORY_BY_TYPE_R, rsh );
			
			List result = new ArrayList();

			for (int i = 0; i < list.size(); i++) {

				if(((Object[])list.get(i))[0] != null && !"".equals(((Object[])list.get(i))[0])) {
					result.add(((Object[])list.get(i))[0]);
				}

			}

			return (String[]) result.toArray(new String[0]);
			
		} catch (Exception e) {
			throw new BtsDBException(e);
		}

	}
	
	
	/**
	 * <DL><DT><B>課題票の優先度一覧(重複なし)を取得します。</B></DL>
	 *  <DD></DD>
	 * </DL>
	 * @return					優先度一覧
	 * @throws BtsDBException
	 */
	public String[] getPriority() throws BtsDBException{

		try {
			ResultSetHandler rsh = new ArrayListHandler();
			
			List list = (List) new QueryRunner().query( this.con, GET_PRIORITY, rsh );
			
			List result = new ArrayList();

			for (int i = 0; i < list.size(); i++) {

				if(((Object[])list.get(i))[0] != null && !"".equals(((Object[])list.get(i))[0])) {
					result.add(((Object[])list.get(i))[0]);
				}

			}

			return (String[]) result.toArray(new String[0]);
			
		} catch (Exception e) {
			throw new BtsDBException(e);
		}

	}
	
	
	/**
	 * <DL><DT><B>指定の課題票をごみ箱に移動します。</B></DT>
	 *  <DD></DD>
	 * </DL>
	 * @param		fingerPrint			削除対象の課題票のFingerPrint
	 * @throws 		BtsDBException
	 */
	public void changeGarbageVersion(String fingerPrint) throws BtsDBException{
		try {
			List param = new ArrayList();
			param.add(fingerPrint);

			new QueryRunner().update( this.con, UPDATE_GARBAGE, param.toArray() );	
		} catch (Exception e) {
			throw new BtsDBException(e);
		}
	}

	
	/**
	 * 同期用の課題票ヘッダ情報を取得します。
	 * 
	 * @return		課題票ヘッダ情報
	 * @throws		BtsDBException
	 */
	public List getIssueSyncHeaders(String type) throws BtsDBException {

		try {
			ResultSetHandler rsh = new BeanListHandler(IssueSyncHeader.class);
			return (List)new QueryRunner().query( this.con, GET_SYNC_HEADER, type, rsh);

		} catch (Exception e) {
			throw new BtsDBException(e);
		}
	}

	/**
	 * ごみ箱に入っている課題票のFingerPrintの配列を返します。
	 * 
	 * 
	 * @return		ごみ箱に入っている課題票のFingerPrintの配列
	 * @throws		BtsDBException
	 */
	public List getGarbageSyncList() throws BtsDBException {

		try {
			ResultSetHandler rsh = new BeanListHandler(IssueSyncHeader.class);
			
			List list = (List)new QueryRunner().query( this.con, GET_SYNC_HEADER, IssueType.GARBAGE_VALUE, rsh);
			
			if(list==null || list.isEmpty()) {
				return null;
			}
			
			List result = new ArrayList();
			for (int i = 0; i < list.size(); i++) {
				result.add( ((IssueSyncHeader)list.get(i)).getFingerPrint());
			}

			return result;
			
		} catch (Exception e) {
			throw new BtsDBException(e);
		}
	}

	
}
