/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.facade;

import java.util.List;

import jp.valtech.bts.connection.MessageDBConnection;
import jp.valtech.bts.dao.BtsDBException;
import jp.valtech.bts.dao.MessagePacketDAO;
import jp.valtech.bts.data.MessagePacket;
import jp.valtech.bts.ui.BtsPlugin;
import jp.valtech.bts.ui.IBtsListener;
import jp.valtech.bts.util.AttachmentUtility;
import jp.valtech.bts.util.Logging;

/**
 * {@link jp.valtech.bts.dao.MessagePacketDAO}用のFacadeクラスです。
 * 
 * @author		<A href="mailto:m_sugitou@valtech.jp">M.Sugito</A>
 * @version	Ver.0.8
 */
public class MessagePacketFacade implements Logging {

	/**
	 * 何もしない。
	 */
	public MessagePacketFacade() {
		;
	}
	
	/**
	 * メッセージ一覧（全件）を取得します。
	 * 
	 * @return			メッセージ一覧（全件）
	 */
	public List getMessageList() throws BtsDBException {
		
		// 戻り値
		List messageList = null;
		
		// DBコネクション取得
		MessageDBConnection dbcon = new MessageDBConnection();
		try {
			
			// 課題票一覧を取得
			MessagePacketDAO dao = new MessagePacketDAO( dbcon );
			messageList = dao.getMessageList();
			return messageList;
			
		}catch (Exception e) {
			String msg = Messages.getString("MessagePacketFacade.0"); //$NON-NLS-1$
			logger.fatal(msg, e);
			BtsPlugin.getInstance().error(msg, e);
			return null;
		} finally {
			dbcon.close();
			dbcon = null;
		}
	}

	
	/**
	 * 指定のメッセージIDのメッセージを「既読」に設定します。
	 * 失敗してもエラーはスローしません。ログに残すだけです。
	 * 
	 * @param		messageID		メッセージID
	 */
	public void setOpend(int messageID) {

		// DBコネクション取得
		MessageDBConnection dbcon = new MessageDBConnection();
		try {
			// ローカルDBに登録。
			MessagePacketDAO dao = new MessagePacketDAO( dbcon );
			dao.setOpend(messageID);

			dbcon.commit();
		} catch (Exception e) {
			dbcon.rollback();

			// エラー情報の出力
			String msg = Messages.getString("MessagePacketFacade.1"); //$NON-NLS-1$
			logger.fatal(msg, e);
			BtsPlugin.getInstance().error(msg, e);
			
		} finally {
			dbcon.close();
			dbcon = null;
		}
	}


	/**
	 * 指定のメッセージを。追加します。
	 * 失敗してもエラーはスローしません。ログに残すだけです。<br>
	 * 添付ファイルの登録処理はありませんよー。添付ファイルの登録は以下のメソッドでは行ってます。<br>
	 * ・{@link ReceiveMessageFacade#addMessage(MessagePacket)}<br>
	 * ・{@link SendMessageFacade#addMessage(MessagePacket)}<br>
	 * 
	 * @param		message		メッセージオブジェクト
	 */
	public void addMessage(MessagePacket message) {
		addMessage(message, true);
	}
	

	/**
	 * 指定のメッセージを。追加します。
	 * 失敗してもエラーはスローしません。ログに残すだけです。<br>
	 * 添付ファイルの登録処理はありませんよー。添付ファイルの登録は以下のメソッドでは行ってます。<br>
	 * ・{@link ReceiveMessageFacade#addMessage(MessagePacket)}<br>
	 * ・{@link SendMessageFacade#addMessage(MessagePacket)}<br>
	 * 
	 * @param		message		メッセージオブジェクト
	 * @param		updateView	ビューを更新するかどうか（true:更新する, false:更新しない）
	 */
	public void addMessage(MessagePacket message, boolean updateView) {
		// DBコネクション取得
		MessageDBConnection dbcon = new MessageDBConnection();
		try {
			// ローカルDBに登録。
			MessagePacketDAO dao = new MessagePacketDAO( dbcon );
			dao.addMessage(message);

			// メッセージIDはDBには設定されているがJAVAオブジェクトには格納されてないので、ここで入れる
			int messageID = dao.getMaxMessageID();
			message.setMessageID(messageID);

			// 現在プラグインが保持しているリスナクラスを取得します。
			List listeners = BtsPlugin.getInstance().getListeners();
			
			// 各リスナにビューを最新表示をするよう通知します。
			if(updateView) {
				for (int idx = 0; idx < listeners.size(); idx++) {
					IBtsListener listnener = (IBtsListener)listeners.get(idx);
					listnener.addMessage(message);
				}
			}
		
			dbcon.commit();
		} catch (Exception e) {
			dbcon.rollback();

			// エラー情報の出力
			String msg = Messages.getString("MessagePacketFacade.2"); //$NON-NLS-1$
			logger.fatal(msg, e);
			BtsPlugin.getInstance().error(msg, e);
			
		} finally {
			dbcon.close();
			dbcon = null;
		}
	}
	
	
	/**
	 * 指定のメッセージを削除します。
	 * メッセージの添付ファイルも削除されます。
	 * 
	 * @param		message		削除対象のメッセージ
	 */
	public void deleteByMessageID(MessagePacket message) {
		// DBコネクション取得
		MessageDBConnection dbcon = new MessageDBConnection();
		try {
			// DBから削除。
			MessagePacketDAO dao = new MessagePacketDAO( dbcon );
			dao.deleteByMessageID(message.getMessageID());

			// 添付ファイルを削除
			AttachmentUtility utility = new AttachmentUtility();
			utility.delete(message);
			
			// 現在プラグインが保持しているリスナクラスを取得します。
			List listeners = BtsPlugin.getInstance().getListeners();
			
			// 各リスナにビューを最新表示をするよう通知します。
			for (int idx = 0; idx < listeners.size(); idx++) {
				IBtsListener listnener = (IBtsListener)listeners.get(idx);
				listnener.deleteMessage(message);
			}
			dbcon.commit();
		} catch (Exception e) {
			dbcon.rollback();
			BtsPlugin.getInstance().errorlog(e);
		} finally {
			dbcon.close();
			dbcon = null;
		}
	}
	/**
	 * メッセージ送信者一覧(重複なし)を取得します。
	 * 
	 * @return			送信者一覧(重複なし)
	 */
	public String[] getFromUser() throws BtsDBException {
		
		// 戻り値
		String[] fromUsers;
		
		// DBコネクション取得
		MessageDBConnection dbcon = new MessageDBConnection();
		try {
			
			// 送信者一覧を取得
			MessagePacketDAO dao = new MessagePacketDAO( dbcon );
			fromUsers = dao.getFromUser();
			
		} finally {
			dbcon.close();
			dbcon = null;
		}
		return fromUsers;
	}

}
