/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.facade;

import java.util.ArrayList;

import jp.valtech.bts.connection.IssueDBConnection;
import jp.valtech.bts.connection.VoteDBConnection;
import jp.valtech.bts.dao.AttachmentDAO;
import jp.valtech.bts.dao.CommentHistoryDAO;
import jp.valtech.bts.dao.IssueDAO;
import jp.valtech.bts.dao.IssueHistoryDAO;
import jp.valtech.bts.dao.VoteDAO;
import jp.valtech.bts.data.CommentHistory;
import jp.valtech.bts.data.Issue;
import jp.valtech.bts.data.IssueHistory;
import jp.valtech.bts.data.IssueType;
import jp.valtech.bts.data.MessagePacket;
import jp.valtech.bts.data.MessageType;
import jp.valtech.bts.data.Vote;
import jp.valtech.bts.util.BTSUtility;
import jp.valtech.bts.util.Logging;

import org.eclipse.core.runtime.IProgressMonitor;

/**
 * クライアント側同期処理用のFacadeクラスです。
 * 
 * @author		<A href="mailto:m_sugitou@valtech.jp">M.Sugito</A>
 * @version	Ver.0.8
 */
public class SyncClientFacade implements Logging {

	/** 同期処理により更新した課題票 */
	private ArrayList updateList = new ArrayList();
	
	/** 同期処理により追加した課題票 */
	private ArrayList addList = new ArrayList();
	
	/** 同期処理によりごみ箱入りした課題票 */
	private ArrayList garbageList = new ArrayList();
	
	
	/**
	 * 何もしない。
	 */
	public SyncClientFacade() {
		;
	}
	
	/**
	 * 受け取った課題票情報でローカルDBを更新します。
	 * 
	 * @param			syncIssues		同期する課題票情報
	 * @param			syncGarbages	同期するごみ箱情報
	 * @param			monitor			プログレスモニタ
	 * @throws 			Exception
	 */
	public void applySyncIssues
		(Issue[] syncIssues, Issue[] syncGarbages, IProgressMonitor monitor) throws Exception {
		
		int taskCnt = 0; 
		
		// 更新する課題票
		if(syncIssues!=null) {
			taskCnt += syncIssues.length;
		}

		// ごみ箱に入る課題票
		if(syncGarbages!=null) {
			taskCnt += syncGarbages.length;
		}
		
		if(taskCnt == 0) {
			return;
		}
		
		monitor.beginTask(Messages.getString("SyncClientFacade.2"), taskCnt); //$NON-NLS-1$

		// コネクション取得
		IssueDBConnection issueDbcon = new IssueDBConnection();
		VoteDBConnection voteDbcon = new VoteDBConnection();
		
		try {
			// 課題票更新
			if(syncIssues!=null) {
				applyIssue(syncIssues, monitor, issueDbcon, voteDbcon);
			}

			// ごみ箱更新（ローカルデータが「リリースバージョン」だったときごみ箱にする）
			if(syncGarbages!=null) {
				applyGarbage(syncGarbages, monitor, issueDbcon);			
			}
			
			issueDbcon.commit();
			voteDbcon.commit();
		}catch (Exception e) {
			issueDbcon.rollback();
			voteDbcon.rollback();
			throw e;
		}finally {
			issueDbcon.close();
			issueDbcon = null;
			
			voteDbcon.close();
			voteDbcon = null;
		}
	}
	
	
	/**
	 * 受け取った同期対象の課題票をローカルに反映します。
	 * 
	 * @param		 syncIssues		同期対象の課題票
	 * @param		 monitor		プログレスモニタ
	 * @param		 issueDbcon		DBコネクション
	 * @param		 voteDbcon		DBコネクション
	 * @throws		 Exception		
	 */
	private void applyIssue( Issue[] 			syncIssues
							, IProgressMonitor 	monitor
							, IssueDBConnection issueDbcon
							, VoteDBConnection 	voteDbcon) throws Exception {
		monitor.subTask (Messages.getString("SyncClientFacade.3"));  //$NON-NLS-1$

		IssueDAO issueDAO = new IssueDAO( issueDbcon );
		IssueHistoryDAO historyDAO = new IssueHistoryDAO(issueDbcon);
		CommentHistoryDAO commentDAO = new CommentHistoryDAO(issueDbcon);
		VoteDAO voteDAO = new VoteDAO(voteDbcon);
		
		for (int i = 0; i < syncIssues.length; i++) {
			Issue localIssue 
				= issueDAO.getByFingerPrint(syncIssues[i].getFingerPrint(), IssueType.RELEASE_VALUE);
			
			if(localIssue == null) {
				//### 同期したデータを新規登録 ###
				addList.add("[" + syncIssues[i].getDisplayIssueID() + "] " + syncIssues[i].getTitle());

				// 課題票登録
				issueDAO.addIssue( syncIssues[i] );
				
				// 課題票履歴登録
				IssueHistory[] histories = syncIssues[i].getIssueHistories();
				for (int idx = 0; idx < histories.length; idx++) {
					historyDAO.addHistory( histories[idx] );
				}
				
				// コメント履歴登録
				CommentHistory[] comments = syncIssues[i].getCommentHistories();
				if(comments != null) {
					for (int idx = 0; idx < comments.length; idx++) {
						commentDAO.addComment( comments[idx] );
					}
				}
				
				// 添付ファイル登録
				AttachmentFacade.addFromEntity(syncIssues[i], issueDbcon);
				
				// 投票情報登録
				Vote[] votes = syncIssues[i].getVotes();
				if(votes != null) {
					for (int idx = 0; idx < votes.length; idx++) {
						voteDAO.insert(votes[idx]);
					}
				}

			} else {
				//### 同期したデータで更新 ###
				updateList.add("[" + syncIssues[i].getDisplayIssueID() + "] " + syncIssues[i].getTitle());

				// 課題票更新
				issueDAO.modifyIssue( syncIssues[i] , IssueType.RELEASE_VALUE );
				
				// 課題票履歴更新(一旦削除後に登録)
				historyDAO.deleteByFingerPrint(syncIssues[i].getFingerPrint(), IssueType.RELEASE_VALUE);
				// 登録
				IssueHistory[] histories = syncIssues[i].getIssueHistories();
				for (int idx = 0; idx < histories.length; idx++) {
					historyDAO.addHistory( histories[idx] );
				}
				
				// コメント履歴更新(一旦削除後に登録)
				commentDAO.deleteByFingerPrint(syncIssues[i].getFingerPrint(), IssueType.RELEASE_VALUE);
				// 登録
				CommentHistory[] comments = syncIssues[i].getCommentHistories();
				if(comments != null) {
					for (int idx = 0; idx < comments.length; idx++) {
						commentDAO.addComment( comments[idx] );
					}
				}
				
				// 添付ファイル更新(一旦削除後に登録)
				AttachmentFacade.deleteAll(syncIssues[i], IssueType.RELEASE_VALUE, issueDbcon);
				// 登録
				AttachmentFacade.addFromEntity(syncIssues[i], issueDbcon);
				
				// 投票情報更新(一旦削除後、登録)
				voteDAO.deleteByFingerPrint(syncIssues[i].getFingerPrint());
				// 登録
				Vote[] votes = syncIssues[i].getVotes();
				if(votes != null) {
					for (int idx = 0; idx < votes.length; idx++) {
						voteDAO.insert(votes[idx]);
					}
				}
			}
			
			
			// キャンセルボタンを押した場合
			if (monitor.isCanceled()) {
				throw new InterruptedException(Messages.getString("SyncClientFacade.4")); //$NON-NLS-1$
			}
			monitor.worked(1);
		}
	}


	/**
	 * 受け取ったごみ箱情報をローカルに反映します。
	 * <br><br>
	 * <b>（＊）ごみ箱の課題票は同期をとりません。</b><br>
	 * 同期相手の端末にごみ箱の課題票があり、それがローカルでは「配布済み」だった場合には
	 * それをごみ箱に移動します。しかし、ローカル上に「配布済み」「ごみ箱」いずれにも無い場合には
	 * そのまま何もしません。『同期相手からごみ箱課題票を受け取り、ローカルのごみ箱に入れる』という
	 * 処理は行いません。
	 * 
	 * @param		syncGarbages		ごみ箱課題票	
	 * @param		monitor				プログレスモニタ
	 * @param		issueDbcon			コネクション
	 * @throws		Exception
	 */
	private void applyGarbage( Issue[] 			syncGarbages
							   , IProgressMonitor 	monitor
							   , IssueDBConnection 	issueDbcon) throws Exception {
		monitor.subTask (Messages.getString("SyncClientFacade.5"));  //$NON-NLS-1$
		
		IssueDAO issueDAO = new IssueDAO( issueDbcon );
		IssueHistoryDAO historyDAO = new IssueHistoryDAO(issueDbcon);
		CommentHistoryDAO commentDAO = new CommentHistoryDAO(issueDbcon);
		AttachmentDAO attachmentDAO = new AttachmentDAO( issueDbcon );

		
		for (int i = 0; i < syncGarbages.length; i++) {


			String fingerPrint = syncGarbages[i].getFingerPrint();

			// リリースバージョンのものがある場合のみ
			Issue releaseIssue = issueDAO.getByFingerPrint(fingerPrint, IssueType.RELEASE_VALUE);
			if(releaseIssue != null) {
				//### ごみ箱入り ###
				garbageList.add("[" + syncGarbages[i].getDisplayIssueID() + "] " + syncGarbages[i].getTitle());

				// 課題票の種別をごみ箱に変更
				issueDAO.changeGarbageVersion(fingerPrint);
				
				// 添付の種別をごみ箱に変更
				attachmentDAO.changeGarbageVersion(fingerPrint);

				// 課題票履歴の種別をごみ箱に変更
				historyDAO.changeGarbageVersion(fingerPrint);

				// 課題票コメント履歴の種別をごみ箱に変更
				commentDAO.changeGarbageVersion(fingerPrint);
				
				// ごみ箱履歴を追加
		        IssueHistory[] histories = syncGarbages[i].getIssueHistories();
		        if(histories != null && histories.length != 0 ){
			        historyDAO.addHistory( histories[0] );
		        }
			}
			
			// キャンセルボタンを押した場合
			if (monitor.isCanceled()) {
				throw new InterruptedException(Messages.getString("SyncClientFacade.4")); //$NON-NLS-1$
			}
			monitor.worked(1);
		}

	}
	
	
	/**
	 * 同期メッセージ生成・登録。
	 * 
	 * @param		syncUserName		同期相手ユーザ名
	 * @param		syncAddress			同期相手アドレス
	 */
	public void saveMessage(String syncUserName, String syncAddress) {
		MessagePacket message = new MessagePacket();

		message.setFromUser(syncUserName);
		
		// 受信日時設定
		message.setSent(System.currentTimeMillis());
		
		// メッセージ種別設定
		message.setMessageType(MessageType.SYNC_VALUE);

		// メッセージ本文設定
		message.setMessage( createMessage(syncUserName, syncAddress) );
		
        // 保存 ＆ ビュー表示更新
        MessagePacketFacade facade = new MessagePacketFacade();
        facade.addMessage(message, false);
	}
	
	
	/**
	 * 同期メッセージの本文生成。
	 * 
	 * @param		syncUserName		同期相手ユーザ名
	 * @param		syncAddress			同期相手アドレス
	 * @return		メッセージ本文
	 */
	private String createMessage(String syncUserName, String syncAddress) {
		
		// メッセージ本文設定
		StringBuffer msg = new StringBuffer();
		
		// 最初の１行目
		msg.append(Messages.getString("SyncClientFacade.6")).append(BTSUtility.formatDate(System.currentTimeMillis())); //$NON-NLS-1$
		msg.append("\n\n");
	
		// 同期相手情報
		msg.append(Messages.getString("SyncClientFacade.7")); //$NON-NLS-1$
		msg.append("  (").append(syncAddress).append(")   ").append(syncUserName);
		msg.append("\n\n");

		// 同期処理結果
		msg.append(Messages.getString("SyncClientFacade.8")); //$NON-NLS-1$
		String indent = "  ";
		String returnCode = "\n";

		// 追加課題票
		msg.append(Messages.getString("SyncClientFacade.9")).append(this.addList.size()).append(Messages.getString("SyncClientFacade.10")); //$NON-NLS-1$ //$NON-NLS-2$
		for (int i = 0; i < addList.size(); i++) {
			msg.append(indent).append( (String)addList.get(i)).append(returnCode);
		}
		msg.append(returnCode);

		// 更新課題票
		msg.append(Messages.getString("SyncClientFacade.11")).append(this.updateList.size()).append(Messages.getString("SyncClientFacade.10")); //$NON-NLS-1$ //$NON-NLS-2$
		for (int i = 0; i < updateList.size(); i++) {
			msg.append(indent).append( (String)updateList.get(i)).append(returnCode);
		}
		msg.append(returnCode);

		// ごみ箱
		msg.append(Messages.getString("SyncClientFacade.12")).append(this.garbageList.size()).append(Messages.getString("SyncClientFacade.10")); //$NON-NLS-1$ //$NON-NLS-2$
		for (int i = 0; i < garbageList.size(); i++) {
			msg.append(indent).append( (String)garbageList.get(i)).append(returnCode);
		}
		msg.append(returnCode);

		return msg.toString();
	}
}