/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.ui.action;

import java.io.File;

import jp.valtech.bts.command.client.VoteAdd;
import jp.valtech.bts.data.CurrentProject;
import jp.valtech.bts.data.Issue;
import jp.valtech.bts.data.NetworkConfig;
import jp.valtech.bts.data.Vote;
import jp.valtech.bts.facade.VoteDAOFacade;
import jp.valtech.bts.ui.BtsPlugin;
import jp.valtech.bts.ui.navigator.ProjectConfig;
import jp.valtech.bts.util.Logging;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.forms.widgets.ImageHyperlink;
import org.eclipse.ui.part.WorkbenchPart;
import org.jfree.chart.ChartFactory;
import org.jfree.chart.ChartUtilities;
import org.jfree.chart.JFreeChart;
import org.jfree.chart.axis.NumberAxis;
import org.jfree.chart.labels.StandardCategoryItemLabelGenerator;
import org.jfree.chart.plot.CategoryPlot;
import org.jfree.chart.plot.PlotOrientation;
import org.jfree.chart.renderer.category.CategoryItemRenderer;
import org.jfree.chart.title.TextTitle;
import org.jfree.data.category.DefaultCategoryDataset;
import org.jfree.ui.HorizontalAlignment;

/**
 * プロジェクトを閉じるアクションです。
 * 
 * @author		<A href="mailto:s_imai@valtech.jp">S.Imai</A>
 * @version	Ver.0.8
 */
public class OpenVoteDlgAction extends Action implements Logging {

	/** このアクションの呼び出しもとのビューオブジェクト */
	private WorkbenchPart parent;

	/** 開いている課題票 */
	private Issue issue;

	/** カレントディレクトリ */
	private String currentDir = BtsPlugin.getInstance().getStateLocation() + "/";

	/** 投票総数 */
	private int voteCnt;

	/** エディタ基本情報セクションのイメージハイパーリンク */
	private ImageHyperlink voteLink;

	/**
	 * 呼び出し元のビューインスタンスを設定します。
	 * 
	 * @param parent 呼び出し元のビューインスタンス
	 */
	public OpenVoteDlgAction(WorkbenchPart parent, ImageHyperlink voteLink) {
		super();
		this.parent = parent;
		this.voteLink = voteLink;
		setImageDescriptor(BtsPlugin.getInstance().getImageDescriptor("vote.gif"));
		setText(Messages.getString("OpenVoteDlgAction.0")); //$NON-NLS-1$
		setToolTipText(Messages.getString("OpenVoteDlgAction.1")); //$NON-NLS-1$
	}

	/**
	 * グラフを作成し、投票ダイアログを表示します。
	 */
	public void run() {

		try {

			// 開いている課題票の投票データ取得
			VoteDAOFacade voteDAOFacade = new VoteDAOFacade();
			Vote[] votes = voteDAOFacade.get(issue.getFingerPrint());

			// グラフを作成
			createChart(votes);

			// ダイアログ生成
			VoteDlg myDialog = new VoteDlg(parent.getSite().getShell());

			// ダイアログオープン
			myDialog.open();

		} catch (Exception e) {
			String msg = Messages.getString("OpenVoteDlgAction.11"); //$NON-NLS-1$
			logger.fatal(msg, e);
			BtsPlugin.getInstance().error(msg, e);
		}
	}

	/**
	 * 投票グラフを作成します。
	 */
	private void createChart(Vote[] votes) throws Exception {

		// 投票総数カウンタ
		voteCnt = 0;

		// データセット生成
		DefaultCategoryDataset dataset = new DefaultCategoryDataset();

		for (int i = 0; i < votes.length; i++) {

			// データセットに値を設定
			dataset.addValue(votes[i].getVoteCnt(), "", votes[i].getVoteUser());

			// 投票総数カウント
			voteCnt += votes[i].getVoteCnt();

		}

		// JFreeChartオブジェクトの生成
		JFreeChart chart = ChartFactory.createBarChart(null, null, null, dataset, PlotOrientation.HORIZONTAL, false, true, true);

		// グラフタイトルを設定
		TextTitle textTitle = new TextTitle(Messages.getString("OpenVoteDlgAction.2")); //$NON-NLS-1$
		textTitle.setFont(new java.awt.Font("SansSerif", 0, 16));
		textTitle.setHorizontalAlignment(HorizontalAlignment.CENTER);
		chart.addSubtitle(textTitle);

		// 投票総数を設定
		TextTitle voteCntTitle = new TextTitle(
				Messages.getString("OpenVoteDlgAction.3") + voteCnt + Messages.getString("OpenVoteDlgAction.4")); //$NON-NLS-1$ //$NON-NLS-2$
		voteCntTitle.setFont(new java.awt.Font("SansSerif", 2, 11));
		voteCntTitle.setHorizontalAlignment(HorizontalAlignment.RIGHT);
		chart.addSubtitle(voteCntTitle);

		// グラフ全体の背景色を白に設定
		chart.setBackgroundPaint(java.awt.Color.WHITE);

		// グラフの描画領域取得
		CategoryPlot plot = chart.getCategoryPlot();

		// 縦軸うの目盛線を非表示
		plot.setRangeGridlinesVisible(false);

		// 棒グラフの色を設定
		plot.getRenderer().setPaint(new java.awt.Color(195, 195, 255));

		// 横軸の設定を取得
		NumberAxis axis = (NumberAxis) plot.getRangeAxis();

		// 横軸の目盛線を非表示
		axis.setVisible(false);

		// 横軸の値を右に表示
		CategoryItemRenderer categoryitemrenderer = plot.getRenderer();
		categoryitemrenderer.setItemLabelGenerator(new StandardCategoryItemLabelGenerator());
		categoryitemrenderer.setSeriesItemLabelsVisible(0, Boolean.TRUE);

		// ファイル出力
		File outputFile = new File(currentDir + "vote.png");

		// PNGで出力 縦のサイズは参加人数によって変動
		ChartUtilities.saveChartAsPNG(outputFile, chart, 400, votes.length * 25 + 45);

	}

	/**
	 * 課題票を設定
	 * 
	 * @param issue 課題票
	 */
	public void setIssue(Issue issue) {
		this.issue = issue;
	}

	/**
	 * 投票ダイアログを作成します。
	 * 
	 * @author <A href="mailto:s_imai@valtech.jp">$Author$</A>
	 * @version $Rev$
	 */
	public class VoteDlg extends Dialog {

		/** スクロール可能なコンポジット */
		private ScrolledComposite scrolledComposite;

		/** スクロール可能なコンポジットを親とするコンポジット */
		private Composite composite;

		/** グラフを表示するラベル */
		private Label chartLabel;

		/**
		 * 初期設定
		 * 
		 * @param parent シェル
		 */
		protected VoteDlg(Shell parent) {
			super(parent);
		}

		/*
		 * (非 Javadoc)
		 * 
		 * @see org.eclipse.jface.window.Window#configureShell(org.eclipse.swt.widgets.Shell)
		 */
		protected void configureShell(Shell newShell) {

			// ダイアログ名の設定
			newShell.setText(BtsPlugin.getInstance().getActiveEditor().getModifiedIssue().getTitle());

			super.configureShell(newShell);
		}

		/*
		 * (非 Javadoc)
		 * 
		 * @see org.eclipse.jface.dialogs.Dialog#createDialogArea(org.eclipse.swt.widgets.Composite)
		 */
		protected Control createDialogArea(Composite parent) {

			// スクロール可能なコンポジット生成
			scrolledComposite = new ScrolledComposite(parent, SWT.V_SCROLL);

			// スクロール可能なコンポジットを親とするコンポジット生成
			composite = (Composite) super.createDialogArea(scrolledComposite);

			// コンポジットの背景色設定
			scrolledComposite.setBackground(Display.getCurrent().getSystemColor(SWT.COLOR_WHITE));
			composite.setBackground(Display.getCurrent().getSystemColor(SWT.COLOR_WHITE));

			// ボタンエリア設定
			createBtnArea(composite);
			
			// グラフエリア設定
			createChartArea(composite);

			// スクロールバー設定
			scrolledComposite.setContent(composite);
			composite.setSize(composite.computeSize(SWT.DEFAULT, SWT.DEFAULT));

			return composite;

		}

		/*
		 * (非 Javadoc)
		 * 
		 * @see org.eclipse.jface.window.Window#createContents(org.eclipse.swt.widgets.Composite)
		 */
		protected Control createContents(Composite parent) {
			// create the top level composite for the dialog
			Composite composite = new Composite(parent, 0);
			GridLayout layout = new GridLayout();
			layout.marginHeight = 0;
			layout.marginWidth = 0;
			layout.verticalSpacing = 0;
			composite.setLayout(layout);
			composite.setLayoutData(new GridData(GridData.FILL_BOTH));
			applyDialogFont(composite);
			// initialize the dialog units
			initializeDialogUnits(composite);
			// create the dialog area and button bar
			dialogArea = createDialogArea(composite);
			// ボタンエリア非表示
			// buttonBar = createButtonBar(composite);
			return composite;
		}

		/**
		 * グラフエリアの設定を行います。
		 * 
		 * @param parent 親コンポジット
		 */
		private void createChartArea(Composite parent) {

			// このエリアのコンポジット生成
			Composite buttonArea = new Composite(parent, SWT.NONE);

			// 背景色を白に設定
			buttonArea.setBackground(Display.getCurrent().getSystemColor(SWT.COLOR_WHITE));

			// ラベル生成
			chartLabel = new Label(buttonArea, SWT.NONE);

			// グラフイメージ取得
			Image image = new Image(Display.getCurrent(), currentDir + "vote.png");

			// グラフイメージをラベルにセット
			chartLabel.setImage(image);

			// ラベルの大きさをイメージの大きさに合わせる
			Rectangle rectangle = image.getBounds();
			chartLabel.setBounds(10, 10, rectangle.width, rectangle.height+50);

		}

		/**
		 * 「投票する」・「閉じる」ボタンエリアの設定を行います。
		 * 
		 * @param parent 親コンポジット
		 */
		private void createBtnArea(Composite parent) {

			// このエリアのコンポジット生成
			Composite buttonArea = new Composite(parent, SWT.NONE);

			// 背景色を白に設定
			buttonArea.setBackground(Display.getCurrent().getSystemColor(SWT.COLOR_WHITE));

			// レイアウト設定。２列表示のグリッドレイアウトを使う
			GridLayout gridLayout = new GridLayout();
			gridLayout.numColumns = 2;
			buttonArea.setLayout(gridLayout);

			// 「投票する」ボタン設定
			Button voteBtn = new Button(buttonArea, SWT.PUSH | SWT.FLAT);
			voteBtn.setText(Messages.getString("OpenVoteDlgAction.5")); //$NON-NLS-1$
			GridData voteGrid = new GridData();
			voteGrid.horizontalIndent = 260;
			voteGrid.widthHint = 60;
			voteGrid.heightHint = 30;
			voteBtn.setLayoutData(voteGrid);

			// 「投票する」ボタンにセレクションリスナを追加
			voteBtn.addSelectionListener(new SelectionAdapter() {

				/*
				 * (非 Javadoc)
				 * 
				 * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
				 */
				public void widgetSelected(SelectionEvent e) {
					// グラフ再表示
					votePressed();
				}

			});

			// 「閉じる」ボタン設定
			Button closeBtn = new Button(buttonArea, SWT.PUSH | SWT.FLAT);
			closeBtn.setText(Messages.getString("OpenVoteDlgAction.6")); //$NON-NLS-1$
			GridData closeGrid = new GridData();
			closeGrid.widthHint = 60;
			closeGrid.heightHint = 30;
			closeBtn.setLayoutData(closeGrid);

			// 「閉じる」ボタンにセレクションリスナを追加
			closeBtn.addSelectionListener(new SelectionAdapter() {

				/*
				 * (非 Javadoc)
				 * 
				 * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
				 */
				public void widgetSelected(SelectionEvent e) {

					// ダイアログを閉じる
					close();
				}

			});

		}

		/**
		 * 「投票する」ボタンが押されたときの処理です。
		 */
		private void votePressed() {

			try {
				// 投票データセット
				Vote vote = new Vote();
				vote.setFingerPrint(issue.getFingerPrint());
				vote.setVoteUser(CurrentProject.getInsance().getProjectConfig().getUserName());
				vote.setVoteCnt(1);

				// 投票数を1増やし、新しい投票データを取得
				VoteDAOFacade voteDAOFacade = new VoteDAOFacade();
				Vote[] votes = voteDAOFacade.cntPlus(vote);

				// 投票グラフ再作成
				createChart(votes);

				// グラフイメージ取得
				Image image = new Image(Display.getCurrent(), currentDir + "vote.png");

				// グラフイメージをラベルにセット
				chartLabel.setImage(image);

				// ラベルの大きさをイメージの大きさに合わせる
				Rectangle rectangle = image.getBounds();
				chartLabel.setBounds(10, 10, rectangle.width, rectangle.height);

				// スクロールバー再設定
				scrolledComposite.setContent(composite);
				composite.setSize(composite.computeSize(SWT.DEFAULT, SWT.DEFAULT));

				// 他の端末に配信する
				VoteAdd voteAdd = new VoteAdd(new NetworkConfig());
				voteAdd.setVote(vote);
				voteAdd.execute();

			} catch (Exception e) {
				BtsPlugin.getInstance().errorlog(e);
			}

		}

		/*
		 * (非 Javadoc)
		 * 
		 * @see org.eclipse.jface.dialogs.Dialog#close()
		 */
		public boolean close() {

			// 投票リンクの表示設定
			String voteStr = Messages.getString("OpenVoteDlgAction.7") + voteCnt + Messages.getString("OpenVoteDlgAction.8"); //$NON-NLS-1$ //$NON-NLS-2$
			voteLink.setForeground(null);
			voteLink.setFont(null);
			
			// 現在開いているプロジェクト設定情報取得
			ProjectConfig config = CurrentProject.getInsance().getProjectConfig();
			
			// 投票数の警告表示をする場合
			if(config.isVoteCheck()) {
				
				// 投票数警告日数を取得
				int warningCnt = Integer.parseInt(config.getVote());
				
				// 投票数が投票警告日を超えている場合
				if(voteCnt > warningCnt) {
					
					// 警告メッセージ
					voteStr = Messages.getString("OpenVoteDlgAction.9") + voteCnt + Messages.getString("OpenVoteDlgAction.10"); //$NON-NLS-1$ //$NON-NLS-2$

					// 警告メッセージ色設定
					voteLink.setForeground(new Color(Display.getCurrent(), 255, 0, 0));

					// 警告メッセージフォント設定
					voteLink.setFont(new Font(Display.getCurrent(), "", 10, SWT.BOLD));
					
				}
				
			}
			
			// 投票リンクに文字列をセット
			voteLink.setText(voteStr);

			return super.close();
		}

	}

}
