/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.ui.editor;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.StyleRange;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.forms.widgets.ExpandableComposite;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.forms.widgets.Section;
import org.eclipse.ui.part.EditorPart;

/**
 * 課題票エディタの抽象クラスです。
 * 課題票の各ページ（タブ）は当クラスを親クラスとして持つ必要があります。
 * 
 * @author		<A href="mailto:m_sugitou@valtech.jp">M.Sugito</A>
 * @version	Ver.0.8
 */
public abstract class AbstractIssueEditorPart extends EditorPart {

	/** FormToolkit */
	protected FormToolkit toolkit;
	
	/** 編集したかどうかのフラグ */
	protected boolean dirty = false;
	
	/** 課題票表示用バリューオブジェクト */
	protected IssueDisplayValue issueDisplayValue;

	/** 表示で使う色：白 */
	protected static final Color COLOR_WHITE = Display.getDefault().getSystemColor(SWT.COLOR_WHITE);
	
	/**
	 * エディタ生成。 
	 *
	 */
	public AbstractIssueEditorPart() {
		super();
	}

	
	/* (非 Javadoc)
	 * @see org.eclipse.ui.IEditorPart#doSave(IProgressMonitor)
	 */
	public void doSave(IProgressMonitor monitor) {
		dirty = false;
		firePropertyChange(PROP_DIRTY);
		fireIssueChange();
	}

	
	/* (非 Javadoc)
	 * @see org.eclipse.ui.IEditorPart#doSaveAs()
	 */
	public void doSaveAs() {
		// 何もしない。保存処理は IssueMultiPageEditor#doSaveAs() で行う
	}


	/* (非 Javadoc)
	 * @see org.eclipse.ui.ISaveablePart#isSaveAsAllowed()
	 */
	public boolean isSaveAsAllowed() {
		// #doSaveAs()を使わない
		return false;
	}


	/* (非 Javadoc)
	 * @see org.eclipse.ui.IEditorPart#init(org.eclipse.ui.IEditorSite, org.eclipse.ui.IEditorInput)
	 */
	public void init(IEditorSite site, IEditorInput input)	throws PartInitException {
		setSite(site);
		setInput(input);
		
		// IssueMultiPageEditorInputのみ受け付けます。
		if(input instanceof IssueMultiPageEditorInput) {

			// 課題票表示用バリューオブジェクトをインスタンス変数に保持
			IssueMultiPageEditorInput issueEditorInput = (IssueMultiPageEditorInput)input;
			issueDisplayValue = issueEditorInput.getIssueDisplayValue();
		} else {
			throw new PartInitException(Messages.getString("AbstractIssueEditorPart.0")); //$NON-NLS-1$
		}
	}


	/**
	 * Dispose of the editor.
	 */
	public void dispose() {
		if (toolkit != null) {
			toolkit.dispose();
		}
	}

	/**
	 * Get a form toolkit to create widgets. It will automatically be disposed
	 * when the editor is disposed.
	 * 
	 * @param			composite			親コンポジット
	 * @return			FormToolkit
	 */
	protected FormToolkit getFormToolkit(Composite composite) {
		// FormTookKit生成
		if (toolkit == null) {
			toolkit = new FormToolkit(composite.getDisplay());
		}
		return toolkit;
	}
	
	
	/**
	 * {@link Section セクションインスタンス}を生成して返します。
	 * 
	 * @param			parent			親コンポジット
	 * @param			title			セクションに設定するタイトル
	 * @return			セクションインスタンス
	 */
	protected Section createSection(Composite parent, String title) {
		// FormTookKit生成
		if (toolkit == null) {
			getFormToolkit(parent);
		}
		
		Composite thisArea = toolkit.createComposite(parent);
		GridLayout layout = new GridLayout();
		layout.marginHeight = 5;
		layout.marginWidth = 5;
		thisArea.setLayout(layout);
		thisArea.setLayoutData(new GridData(GridData.FILL_BOTH));

		Section section = toolkit.createSection(thisArea, ExpandableComposite.TWISTIE | ExpandableComposite.EXPANDED | ExpandableComposite.TITLE_BAR | Section.DESCRIPTION | ExpandableComposite.FOCUS_TITLE);
		section.setLayoutData(new GridData(GridData.FILL_BOTH));
		if(title != null) {
			section.setText(title);
		}

		
		return section; 
	}
	
	/**
	 * {@link Text テキストボックス}を生成して返します。
	 * 
	 * @param			parent			親コンポジット
	 * @param			value			テキストボックスに表示する初期値
	 * @param			style			テキストボックスに指定する書式
	 * @return			テキストボックスインスタンス
	 */
	protected Text createText(Composite parent, String value, int style) {
		// FormTookKit生成
		if (toolkit == null) {
			getFormToolkit(parent);
		}
		Text text = toolkit.createText(parent, value, style);
		
		// 当該テキストボックスに何か入力したときの処理
		text.addModifyListener(new ModifyListener() {

			public void modifyText(ModifyEvent e) {
				// 「dirty」をtrueに更新し、それをEclipseに通知する
				dirty = true;
				firePropertyChange(PROP_DIRTY);
				
				// 課題票エディタ全体に更新を通知する
				fireIssueChange();
			}
		});
		return text;
	}
	
	
	/**
	 * {@link OverviewPart 課題票エディタ}になんらかの入力があった際に呼ばれるメソッドです。
	 */
	protected abstract void fireIssueChange();

	/**
	 * {@link Text テキストボックス}を生成して返します。
	 * 
	 * @param			parent			親コンポジット
	 * @param			value			テキストボックスに表示する初期値
	 * @return			テキストボックスインスタンス
	 */
	protected Text createText(Composite parent, String value) {
		return createText(parent, value, SWT.NONE);
	}
	
	
	/**
	 * {@link Combo コンボボックス}を生成して返します。
	 * 
	 * @param			parent			親コンポジット
	 * @param			values			コンボボックスのプルダウンメニュー
	 * @return			コンボボックスインスタンス
	 */
	protected Combo createCombo(Composite parent, String[] values) {
		return createCombo(parent, values, "");
	}
		

	/**
	 * {@link Combo コンボボックス}を生成して返します。
	 * 生成するコンボボックスには{@link SelectionListener}と{@link ModifyListener}を登録し、選択が変わったら通知するようにします。
	 * 
	 * @param			parent			親コンポジット
	 * @param			values			コンボボックスのプルダウンメニュー
	 * @param			value			コンボボックスに表示する初期値
	 * @return			コンボボックスインスタンス
	 */
	protected Combo createCombo(Composite parent, String[] values, String value) {

		// FormTookKit生成
		if (toolkit == null) {
			getFormToolkit(parent);
		}

		// Comboを使う
		Combo combo = new Combo(parent, SWT.DROP_DOWN);
		toolkit.adapt(combo, true, true);
		
		// コンボのプルダウンメニューを設定
		for (int idx = 0; idx < values.length; idx++) {
			combo.add(values[idx]);
		}

		// コンボの初期値を設定
		combo.setText(value);

		// 選択を変更したときのリスナ登録
		combo.addSelectionListener(new SelectionListener() {
			
			public void widgetSelected(SelectionEvent e) {
				// 「dirty」をtrueに更新し、それをEclipseに通知する
				dirty = true;
				firePropertyChange(PROP_DIRTY);

				// 課題票エディタ全体に更新を通知する
				fireIssueChange();
			}
			
			public void widgetDefaultSelected(SelectionEvent e) {
				// 何もしない
			}
			
		});

		// 入力したときのリスナ登録
		combo.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				// 「dirty」をtrueに更新し、それをEclipseに通知する
				dirty = true;
				firePropertyChange(PROP_DIRTY);

				// 課題票エディタ全体に更新を通知する
				fireIssueChange();
			}
		});
		
		return combo;
	}

	
	/**
	 * ラジオボタンを生成して返します。
	 * 生成するラジオボタンには{@link SelectionListener}を登録し、選択が変わったら通知するようにします。
	 * 
	 * @param			parent				親コンポジット
	 * @param			description			ラジオボタンに設定するテキスト
	 * @param			selection			ラジオボタンに設定する初期値
	 * @return			ラジオボタンインスタンス
	 */
	protected Button createRadio(Composite parent, String description, boolean selection) {

		// FormTookKit生成
		if (toolkit == null) {
			getFormToolkit(parent);
		}

		// ラジオボタン生成
		Button radio = toolkit.createButton(parent, description, SWT.RADIO);
		radio.setSelection(selection);

		// 選択を変更したときのリスナ登録
		radio.addSelectionListener(new SelectionListener() {

			public void widgetSelected(SelectionEvent e) {
				// 「dirty」をtrueに更新し、それをEclipseに通知する
				dirty = true;
				firePropertyChange(PROP_DIRTY);

				// 課題票エディタ全体に更新を通知する
				fireIssueChange();
			}
			
			public void widgetDefaultSelected(SelectionEvent e) {
				// 何もしない
			}
		});

		return radio;
	}

	/**
	 * チェックボックスを生成して返します。
	 * 生成するチェックボックスには{@link SelectionListener}を登録し、チェックが変わったら通知するようにします。
	 * 
	 * @param			parent				親コンポジット
	 * @param			description			チェックボックスに設定するテキスト
	 * @param			selection			チェックボックスに設定する初期値
	 * @return			チェックボックスインスタンス
	 */
	protected Button createCheckBox(Composite parent, String description, boolean selection) {

		// FormTookKit生成
		if (toolkit == null) {
			getFormToolkit(parent);
		}

		// ラジオボタン生成
		Button check = toolkit.createButton(parent, description, SWT.CHECK);
		check.setSelection(selection);

		// 	チェックを変更したときのリスナ登録
		check.addSelectionListener(new SelectionListener() {

			public void widgetSelected(SelectionEvent e) {
				// 「dirty」をtrueに更新し、それをEclipseに通知する
				dirty = true;
				firePropertyChange(PROP_DIRTY);

				// 課題票エディタ全体に更新を通知する
				fireIssueChange();
			}
			
			public void widgetDefaultSelected(SelectionEvent e) {
				// 何もしない
			}
		});

		return check;
	}


	/**
	 * {@link StyledText}に設定するスタイルを文字列単位で設定します。
	 * 
	 * @param		text			スタイルテキスト
	 * @param		str				スタイルテキストに追加する文字列
	 * @param		rgb				スタイルテキストに追加する文字列の色
	 * @param		fontStyle		スタイルテキストに追加する文字列のフォントスタイル
	 */
	public static void setStyle(StyledText text, String str, RGB rgb, int  fontStyle) {
		StyleRange styleRange = new StyleRange();
		
		// スタイル適用範囲（開始位置）
		styleRange.start = text.getText().length();
		// スタイル適用範囲（終了位置）
		styleRange.length = str.length();
		
		// 色指定
		styleRange.foreground = new Color(PlatformUI.getWorkbench().getDisplay(), rgb);// 色
		 
		// フォントスタイル
		styleRange.fontStyle = fontStyle;
		
		// テキスト追加
		text.append(str);

		// スタイル適用
		text.setStyleRange(styleRange);
	}
}
