/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.ui.editor;

import java.text.SimpleDateFormat;
import java.util.Date;

import jp.valtech.bts.data.CommentHistory;
import jp.valtech.bts.ui.BtsPlugin;
import jp.valtech.bts.util.Logging;

import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.SashForm;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Link;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.forms.widgets.ScrolledForm;
import org.eclipse.ui.forms.widgets.Section;

/**
 * 課題票エディタの「コメント」タブ。
 * 
 * @author		<A href="mailto:m_sugitou@valtech.jp">M.Sugito</A>
 * @version	Ver.0.8
 */
public class CommentPart extends AbstractIssueEditorPart implements Logging {

	/** このページのタブに設定する文字列 */
	public static final String TAB_TITLE=Messages.getString("CommentPart.0"); //$NON-NLS-1$

	/** このページで使う色：暗い青 */
	private static final RGB DARK_BLUE  = new RGB(0,0,204);

	/** このページで使う色：暗い緑 */
	private static final RGB DARK_GREEN = new RGB(0,128,0);
	
	/** 履歴表示での日付書式 */
	private static final SimpleDateFormat FORMATTER = new SimpleDateFormat(Messages.getString("CommentPart.1")); //$NON-NLS-1$
	
	/** 履歴表示での改行 */
	private static final String END_LINE = "\n\n";

	/** このページのフォーム */
	private ScrolledForm form ;

	/** このページの部品：コメント入力のテキストボックス */
	
	private Text commentTxt;
	/** このページの部品：コメント履歴のスタイルテキスト */
	private StyledText historyTxt;

	/** マルチページエディタの親 */
	IssueMultiPageEditor parentEditor;
	
    /**
     * 当該ページのインスタンスを生成します。
     */
	public CommentPart(IssueMultiPageEditor parentEditor) {
		super();
    	this.parentEditor = parentEditor;
	}

	/* (非 Javadoc)
	 * @see org.eclipse.ui.part.WorkbenchPart#createPartControl(org.eclipse.swt.widgets.Composite)
	 */
	public void createPartControl(Composite parent) {
		try {
			// フォームボディ部生成
			Composite body = createBody(parent, Messages.getString("CommentPart.2")); //$NON-NLS-1$
			// コメントセクションの生成
			createCommentSection(body);
			
			// レイアウトやスクロールバーの再処理を行う
			this.form.reflow(false);
		}catch (Exception e) {
			String msg = Messages.getString("CommentPart.3"); //$NON-NLS-1$
			logger.fatal(msg, e);
			BtsPlugin.getInstance().error(msg, e);
		}
	}

	/**
	 * このページのボディ部を生成します。
	 * 
	 * @param		parent			親コンポジット
	 * @param		title			ボディタイトル
	 * @return		生成したボディ部のコンポジット
	 */
	private Composite createBody(Composite parent, String title) {
		if (toolkit == null) {
			getFormToolkit(parent);
		}
		
		GridLayout layout = new GridLayout();
		layout.marginHeight = 0;

		form = toolkit.createScrolledForm(parent);
		form.setText(title);
		form.getBody().setLayout(layout);
		form.getBody().setLayoutData(new GridData(GridData.FILL_BOTH));
		
		// 基本レイアウト
		Composite body = toolkit.createComposite(form.getBody());
		body.setLayout(layout);
		body.setLayoutData(new GridData(GridData.FILL_BOTH));
		
		return body;
	
	}

	
	/**
	 * コメントセクションの生成
	 * 
	 * @param		parent				親コンポジット				
	 * @throws		Exception
	 */
	private void createCommentSection(Composite parent) throws Exception {
		
		// セクションインスタンス生成
		Section section = createSection(parent, Messages.getString("CommentPart.4")); //$NON-NLS-1$
		section.setDescription(Messages.getString("CommentPart.5")); //$NON-NLS-1$

		// このセクションに乗せる基本コンポジット生成
		Composite composite = toolkit.createComposite(section);
		composite.setLayout(new GridLayout());
		composite.setLayoutData(new GridData(GridData.FILL_BOTH));
		section.setClient(composite);
		
		// 左右分割する。
		SashForm sashForm = new SashForm(composite, SWT.HORIZONTAL);
		sashForm.setLayoutData(new GridData(GridData.FILL_BOTH));
		
		// コメント入力部生成
		createInputArea(sashForm);
		
		// コメント履歴表示部生成
		createHistoryArea(sashForm);
		
		sashForm.setWeights(new int[]{40, 60});
	}
	
	
	/**
	 * コメント入力の部品生成。
	 * 
	 * @param		parent				親コンポジット				
	 * @throws		Exception
	 */
	private void createInputArea(Composite composite) throws Exception {
		
		// この部品の基本コンポジットを生成
		Composite thisArea = toolkit.createComposite(composite);
		toolkit.paintBordersFor(thisArea);
		
		// レイアウト・グリッドレイアウト設定
		thisArea.setLayout(new GridLayout());
		thisArea.setLayoutData(new GridData(GridData.FILL_BOTH));
		
		// ラベル生成
		toolkit.createLabel(thisArea, Messages.getString("CommentPart.6")); //$NON-NLS-1$

		// コメント入力テキストボックス生成
		commentTxt = toolkit.createText(thisArea,"", SWT.MULTI | SWT.WRAP | SWT.H_SCROLL | SWT.V_SCROLL);
		GridData gridData = new GridData(GridData.FILL_HORIZONTAL);
		gridData.heightHint = 100;
		commentTxt.setLayoutData(gridData);

		// 概要ページへのリンク
		Link link = new Link(thisArea, SWT.NONE);
        link.setText(Messages.getString("CommentPart.7")); //$NON-NLS-1$
        link.addSelectionListener(new org.eclipse.swt.events.SelectionAdapter() {
            public void widgetSelected(org.eclipse.swt.events.SelectionEvent e) {
            	// 「概要ページ」に遷移します
                parentEditor.changeOverviewPage();
            }
        });
 		
        // リンクテキストのレイアウト
        GridData gridLink = new GridData(GridData.FILL_HORIZONTAL);
        gridLink.horizontalSpan = 2;
        gridLink.verticalIndent = 5;
        
        link.setLayoutData(gridLink);
        link.setBackground(COLOR_WHITE);
	}

	
	/**
	 * コメント履歴の部品生成。
	 * 
	 * @param		parent				親コンポジット				
	 * @throws		Exception
	 */
	private void createHistoryArea(Composite composite) throws Exception {

		// この部品の基本コンポジットを生成
		Composite thisArea = toolkit.createComposite(composite);
		toolkit.paintBordersFor(thisArea);
		
		// レイアウト・グリッドレイアウト設定
		thisArea.setLayout(new GridLayout());
		thisArea.setLayoutData(new GridData(GridData.FILL_BOTH));

		// ラベル生成
		toolkit.createLabel(thisArea, Messages.getString("CommentPart.8")); //$NON-NLS-1$


		// コメント履歴のスタイルテキストボックス生成
 		historyTxt = new StyledText(thisArea, SWT.MULTI | SWT.FLAT | SWT.READ_ONLY| SWT.V_SCROLL);
 		GridData gridData = new GridData(GridData.FILL_BOTH);
		historyTxt.setLayoutData(gridData);
 		historyTxt.setData(FormToolkit.KEY_DRAW_BORDER, FormToolkit.TEXT_BORDER);
 		
 		CommentHistory[] commentHistories = issueDisplayValue.getCommentHistories();
 		setHistoryTxt(commentHistories);

	}
	
	/**
	 * 追記したコメントを取得します。
	 * 
	 * @return		追記したコメント		
	 */
	String getNewComment() {
		return commentTxt.getText();
	}
	
	/**
	 * 当ページを最新表示に更新します。
	 * 課題票配信処理後に呼ばれます。 
	 * 
	 * @param		commentHistories		コメント履歴
	 */
	void postSave(CommentHistory[] commentHistories) {
		setHistoryTxt(commentHistories);
		commentTxt.setText("");
	}
	
	
	/**
	 * コメント履歴テキストボックスにコメント履歴情報を格納します。
	 * 
	 * @param		commentHistories		コメント履歴情報
	 */
	private void setHistoryTxt(CommentHistory[] commentHistories) {

		// 一旦テキストの中身をクリアする。
		historyTxt.setText("");

		// コメントが無いときは何もしない
 		if(commentHistories==null) {
 			return;
 		}

 		// コメントオブジェクトを元にコメントを書き込む
 		for (int idx = 0; idx < commentHistories.length; idx++) {
			
 			// 更新日時取得
 			Date date = commentHistories[idx].getUpdateDate();
			String dateStr = FORMATTER.format(date);
			setStyle(historyTxt, dateStr, DARK_BLUE, SWT.NONE);
			
			// 更新者名取得
			String userName = Messages.getString("CommentPart.9") + commentHistories[idx].getUpdateUser() + END_LINE; //$NON-NLS-1$
			setStyle(historyTxt, userName, DARK_GREEN, SWT.NONE);
			
			// 更新詳細取得（スタイル設定なし）
			String desc = commentHistories[idx].getDescription() + END_LINE;
			historyTxt.append(desc);
		}
	}
	
	

	/** 
	 * ページ初期表示時のフォーカス。
	 * コメント入力のテキストボックスにフォーカスする。
	 * 
	 * @see org.eclipse.ui.part.WorkbenchPart#setFocus()
	 */
	public void setFocus() {
		commentTxt.setFocus();
	}
	
	
	/**
	 * 常にfalseを返します。
	 * 
	 * @return		常にfalse
	 */
    public boolean isDirty() {
    	return false;
    }

    /*
     *  (非 Javadoc)
     * @see jp.valtech.bts.ui.editor.AbstractIssueEditorPart#notifyModifyInput()
     */
	protected void fireIssueChange() {
		// 何もしない。
	}

}
