/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.ui.editor;

import java.util.ArrayList;

import jp.valtech.bts.data.Attachment;
import jp.valtech.bts.data.CommentHistory;
import jp.valtech.bts.data.Issue;
import jp.valtech.bts.data.IssueStatus;
import jp.valtech.bts.util.BTSUtility;


/**
 * 課題票情報を課題票をエディタに表示する形式に変換します。
 * 
 * {@link Issue 課題票}の情報を{@link jp.valtech.bts.ui.editor.IssueMultiPageEditor 課題票エディタ}に表示する為のValueObjectです。
 * 
 * @author		<A href="mailto:m_sugitou@valtech.jp">M.Sugito</A>
 * @version	Ver.0.8
 */
public class IssueDisplayValue{

	private static final String TEMPOLARY_ISSUE_ID_LABLEL= Messages.getString("IssueDisplayValue.0"); //$NON-NLS-1$

	/** 課題票オブジェクト */
	private Issue issue = new Issue();
	
	/**
	 * 何もしません。 
	 */
	public IssueDisplayValue() {
		;
	}

	/**
	 * 課題票インスタンスを設定します。
	 * 
	 * @param		issue		課題票インスタンス
	 */
	public void setIssue(Issue issue){
		this.issue = issue;
	}

	/**
	 * 課題票インスタンスを取得します。
	 * 
	 * @return		課題票インスタンス
	 */
	public Issue getIssue() {
		return this.issue;
	}
	
	/**
	 * <DL>
	 *  <DT><B>課題票作製タイムスタンプ を取得。</B></DL>
	 *  <DD></DD>
	 * </DL>
	 * @return 課題票作製タイムスタンプ
	 */
	public String getCreateDate(){
		if(issue.getCreateDate()==null) {
			return "";
		}
		return BTSUtility.formatDate(issue.getCreateDate());
	}
	
	/**
	 * <DL>
	 *  <DT><B>課題票作製者を取得。</B></DL>
	 *  <DD></DD>
	 * </DL>
	 * @return 課題票作製者
	 */
	public String getCreateUser(){
		if(issue.getCreateUser() ==null) {
			return "";
		}
		return issue.getCreateUser();
	}

	/**
	 * <DL>
	 *  <DT><B>課題票更新タイムスタンプ を取得。</B></DL>
	 *  <DD></DD>
	 * </DL>
	 * @return 課題票更新タイムスタンプ
	 */
	public String getUpdateDate() {
		if(issue.getUpdateDate() ==null) {
			return "";
		}
		return BTSUtility.formatDate(issue.getUpdateDate());

	}

	/**
	 * <DL><DT><B>バージョン番号 を戻します。</B></B>
	 * <DD></DD>
	 * </DL>
	 * @return version バージョン番号
	 */
	public String getVersion() {
		return "Ver." + issue.getVersion().toString();
	}
	
	
	/**
	 * <DL>
	 * <DT><B>課題票ID を戻します。</B></B>
	 * <DD></DD>
	 * </DL>
	 * @return 課題票ID 
	 */
	public String getDisplayIssueID() {
		if( "".equals(issue.getDisplayIssueID()) ) {
			return TEMPOLARY_ISSUE_ID_LABLEL;
		}
		return issue.getDisplayIssueID();
	}

	/**
	 * <DL>
	 * <DT><B>カテゴリ を戻します。</B></B>
	 * <DD></DD>
	 * </DL>
	 * @return カテゴリ 
	 */
	public String getCategory() {
		if(issue.getCategory()==null) {
			return "";
		}
		return issue.getCategory(); 
	}


	/**
	 * <DL>
	 * <DT><B>タイトル を戻します。</B></B>
	 * <DD></DD>
	 * </DL>
	 * @return タイトル 
	 */
	public String getTitle() {
		if(issue.getTitle()==null) {
			return "";
		}
		return issue.getTitle();
	}


	/**
	 * <DL>
	 * <DT><B>詳細 を戻します。</B></B>
	 * <DD></DD>
	 * </DL>
	 * @return 詳細 
	 */
	public String getDescription() {
		if(issue.getDescription()==null) {
			return "";
		}

		return issue.getDescription();
	}


	/**
	 * <DL>
	 * <DT><B>優先度 を戻します。</B></B>
	 * <DD></DD>
	 * </DL>
	 * @return 優先度 
	 */
	public String getPriority() {
		if(issue.getPriority()==null) {
			return "";
		}
		return issue.getPriority();
	}



	/**
	 * <DL>
	 * <DT><B>ステータスが「起票」かどうかを戻します。</B></B>
	 * <DD></DD>
	 * </DL>
	 * @return		true--ステータス「起票」、false--ステータス「起票」以外 		 
	 */
	public boolean isStatusOpened() {
		
		// ステータスを取得し「起票」の場合はTrueを返します。
		if(issue.getStatus()==null) {
			// 新規課題票などステータスが設定されてない場合は「起票」を初期値とするためtrueを返す
			return true;
		}
		return issue.getStatus().equals(IssueStatus.OPENED_VALUE);
	}

	/**
	 * <DL>
	 * <DT><B>ステータスが「担当者割り当て」かどうかを戻します。</B></B>
	 * <DD></DD>
	 * </DL>
	 * @return		true--ステータス「担当者割り当て」、false--ステータス「担当者割り当て」以外 		 
	 */
	public boolean isStatusAssigned() {

		// ステータスを取得し「担当者割り当て」の場合はTrueを返します。
		if(issue.getStatus()==null) {
			return false;
		}
		return issue.getStatus().equals(IssueStatus.ASSIGNED_VALUE);
	}
	
	/**
	 * <DL>
	 * <DT><B>ステータスが「対応済み」かどうかを戻します。</B></B>
	 * <DD></DD>
	 * </DL>
	 * @return		true--ステータス「対応済み」、false--ステータス「対応済み」以外 		 
	 */
	public boolean isStatusResolved() {

		// ステータスを取得し「対応済み」の場合はTrueを返します。
		if(issue.getStatus()==null) {
			return false;
		}
		return issue.getStatus().equals(IssueStatus.RESOLVED_VALUE);
	}

	/**
	 * <DL>
	 * <DT><B>ステータスが「確認済み」かどうかを戻します。</B></B>
	 * <DD></DD>
	 * </DL>
	 * @return		true--ステータス「確認済み」、false--ステータス「確認済み」以外 		 
	 */
	public boolean isStatusVerified() {

		// ステータスを取得し「確認済み」の場合はTrueを返します。
		if(issue.getStatus()==null) {
			return false;
		}
		return issue.getStatus().equals(IssueStatus.VERIFIED_VALUE);
	}

	/**
	 * <DL>
	 * <DT><B>ステータスが「完了」かどうかを戻します。</B></B>
	 * <DD></DD>
	 * </DL>
	 * @return		true--ステータス「完了」、false--ステータス「完了」以外 		 
	 */
	public boolean isStatusClosed() {

		// ステータスを取得し「完了」の場合はTrueを返します。
		if(issue.getStatus()==null) {
			return false;
		}
		return issue.getStatus().equals(IssueStatus.CLOSED_VALUE);
	}

	/**
	 * <DL>
	 * <DT><B>担当者 を戻します。</B></B>
	 * <DD></DD>
	 * </DL>
	 * @return 担当者 
	 */
	public String getAssigned() {
		if(issue.getAssigned()==null) {
			return "";
		}
		return issue.getAssigned();
	}

	/**
	 * <DL>
	 *  <DT><B>完了期限を取得。</B></DL>
	 *  <DD></DD>
	 * </DL>
	 * @return 完了期限
	 */
	public String getDeadline(){
		if(issue.getDeadline() == null) {
			return "";
		}
		return BTSUtility.formatDate(issue.getDeadline(), "yyyy/MM/dd");
	}

	/**
	 * <DL>
	 * <DT><B>内緒のバグ を戻します。</B></B>
	 * <DD>内緒のバグをBooleanのプリミティブ型booleanで取得します。</DD>
	 * </DL>
	 * @return 内緒のバグ 
	 */
	public boolean isSecretBug() {
		if(issue.isSecretBug() == null) {
			return false;
		}
		return issue.isSecretBug().booleanValue();
	}

	/**
	 * <DL>
	 * <DT><B>保留 を戻します。</B></B>
	 * <DD>保留をBooleanのプリミティブ型booleanで取得します。</DD>
	 * </DL>
	 * @return 保留 
	 */
	public boolean isSuspend() {
		if(issue.isSuspend() == null) {
			return false;
		}
		return issue.isSuspend().booleanValue();
	}
	
	/**
	 * <DL>
	 *  <DT>{@link Attachment 添付ファイル}のリストをArrayListで返します。</DT>
	 *  <DD></DD>
	 * </DL>
	 * @return			添付ファイルのリスト
	 */
	public ArrayList getAttachmentList() {
		ArrayList list = issue.getAttachmentList();
		if(list==null) {
			return new ArrayList();
		}
		return list;
	}


	/**
	 * <DL>
	 *  <DT>{@link CommentHistory コメント履歴}を返します。</DT>
	 *  <DD></DD>
	 * </DL>
	 * @return			添付ファイルのリスト
	 */
	public CommentHistory[] getCommentHistories() {
		return issue.getCommentHistories();
	}

}