/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.ui.issuelist;

import jp.valtech.bts.data.CurrentProject;
import jp.valtech.bts.data.IssueStatus;
import jp.valtech.bts.facade.ModifyIssueFacade;
import jp.valtech.bts.ui.BtsPlugin;
import jp.valtech.bts.ui.action.OpenCalendarDlgAction;
import jp.valtech.bts.util.BTSUtility;
import jp.valtech.bts.util.Logging;

import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.viewers.CheckboxTableViewer;
import org.eclipse.jface.viewers.ColumnWeightData;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.TableLayout;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.TraverseEvent;
import org.eclipse.swt.events.TraverseListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.forms.events.HyperlinkAdapter;
import org.eclipse.ui.forms.events.HyperlinkEvent;
import org.eclipse.ui.forms.widgets.ImageHyperlink;

/**
 * フィルターパターン入力用のダイアログです。
 * {@link jp.valtech.bts.ui.issuelist.IssueListView 課題票一覧ビュー}から呼ばれるダイアログです。
 * 
 * @author		<A href="mailto:m_sugitou@valtech.jp">M.Sugito</A>
 * @version	Ver.0.8
 */
public class IssueFilterDialog extends Dialog implements Logging {

	/** 「課題票ID」用テキストボックス */
	private Text issueIDTxt;

    /** 「From日付」用テキストボックス */
    private Text frmDateTxt;
    /** 「To日付」用テキストボックス */
	private Text toDateTxt;

    /** 「タイトル」用コンボ */
    private Combo titleCmb;
    /** 「タイトル」用テキストボックス */
    private Text titleTxt;

    /** 「詳細」用コンボ */
    private Combo descCmb;
    /** 「詳細」用テキストボックス */
    private Text descTxt;

    /**「 担当者一覧テーブル」用テーブルビューア */
    private CheckboxTableViewer assignedViewer;
    /**「 優先度一覧テーブル」用テーブルビューア */
    private CheckboxTableViewer priorityViewer;

    /** 「ステータス：起票」用チェックボックス */
    private Button stOpenChk;
	/** 「ステータス：担当者割り当て済み」用チェックボックス */
    private Button stAssignedChk;
	/** 「ステータス：対応済み」用チェックボックス */
    private Button stResolvedChk;
	/** 「ステータス：確認済み」用チェックボックス */
    private Button stVerifiedChk;
	/** 「ステータス：完了」用チェックボックス */
    private Button stClosedChk;

	/** 「内緒のバグ」用チェックボックス */
    private Button secretChk;
	/** 「保留中の課題」用チェックボックス */
    private Button reserveChk;
	/** 「完了期限」用チェックボックス */
    private Button deadLineChk;

	/** 旧課題票フィルタ情報 */
	private IssueFilter oldConfig;
	
	/** 課題票ビュー */
	private IssueListView view;
	
	/** 担当者一覧 */
	private String[] assigned;
	
	/** 優先度一覧 */
	private String[] priority;

	
	
	/**
	 * フィルター設定のダイアログを生成します。
	 * 
	 * @param			parent		呼び出し元のビューオブジェ
	 */
	public IssueFilterDialog(IssueListView parent) {
		super(parent.getSite().getShell());
		
		// ビューを設定
		this.view = parent;
		
		// フィルターの設定情報取得
		IDialogSettings settings = 
			BtsPlugin.getInstance().getDialogSettings().getSection(IssueFilterManager.SECTION_NAME);
		if(settings != null) {
			this.oldConfig = IssueFilterManager.getIssueFilter();
		}
	}

	
    /**
     * 当ダイアログの設定します。タイトルを設定します。
     * 
     * @param			newShell			該ダイアログのシェル
     */
    protected void configureShell(Shell newShell) {
        super.configureShell(newShell);
        newShell.setText(Messages.getString("IssueFilterDialog.0"));  //$NON-NLS-1$
    }



    /**
     * フィルターパターン編集用のダイアログを生成します。
     * 
     * @param			parent					親コンポジット
     * @return			当ダイアログのコントロールインスタンス
     * @see				Dialog#createDialogArea(org.eclipse.swt.widgets.Composite)
     */
    protected Control createDialogArea(Composite parent) {
        Composite composite = (Composite) super.createDialogArea(parent);
        try {
        	// 「課題票ID」設定のUIを生成
        	createIssueIDArea(composite);

        	// 「起票日」設定のUIを生成
        	createDateArea(composite);
        	
        	// 「タイトル」「詳細」文字列指定のUIを生成
            createContainsArea(composite);
        	
            // 「優先度」「担当者」設定のUIを生成
            createTablesArea(composite);
        	
            // 「ステータス」設定のUIを生成
            createStatusArea(composite);

            // セパレータ生成
            createSeparatorLine(composite);
            
            // 設定をロード
            if(oldConfig != null) {
                loadFilterSettings();
            }
            
        }catch (Exception e) {
			String msg = Messages.getString("IssueFilterDialog.1"); //$NON-NLS-1$
			logger.fatal(msg, e);
			BtsPlugin.getInstance().error(msg, e);
		}
        return composite;
    }
    
    
    /**
     * 「ヘッダ設定」「フッタ設定」生成。
     * 親コンポジットに「ヘッダ設定」「フッタ設定」のUIを生成します。
     * 
     * @param			parent					親コンポジット
     * @throws			Exception
     */
    private void createIssueIDArea(Composite parent) throws Exception {
    	// レイアウト設定。グリッドレイアウトを使う
    	Composite thisArea = new Composite(parent, SWT.NONE);
        GridLayout gridLayout = new GridLayout();
        gridLayout.numColumns = 3;
        thisArea.setLayout(gridLayout);
    	
        // ラベル
        Label label = new Label(thisArea, SWT.NONE);
        label.setText(Messages.getString("IssueFilterDialog.2")); //$NON-NLS-1$

        issueIDTxt = new Text(thisArea,SWT.BORDER);
		GridData gridData = new GridData(GridData.FILL); 
        gridData.widthHint = 80;
        issueIDTxt.setLayoutData(gridData);
        
        // ラベル
        label = new Label(thisArea, SWT.NONE);
        label.setText(Messages.getString("IssueFilterDialog.3")); //$NON-NLS-1$
        
     }

    
    /**
     * 起票日時指定のUIを生成。
     * 親コンポジットに起票日時指定のUIを生成します。
     * 
     * @param			parent					親コンポジット
     * @throws			Exception
     */
    private void createDateArea(Composite parent) throws Exception {
    	
    	// この部品のコンポジット生成
    	Composite thisArea = new Composite(parent, SWT.NONE);

    	// レイアウト設定。７列表示のグリッドレイアウトを使う
        GridLayout gridLayout = new GridLayout();
        gridLayout.numColumns = 7;
        thisArea.setLayout(gridLayout);
    	
        // 「FromDate」ラベル
        Label label = new Label(thisArea, SWT.NONE);
        label.setText(Messages.getString("IssueFilterDialog.4")); //$NON-NLS-1$

        // 「FromDate」テキストボックス
        frmDateTxt = new Text(thisArea,SWT.BORDER);
		GridData gridData = new GridData(GridData.FILL); 
        gridData.widthHint = 80;
        frmDateTxt.setLayoutData(gridData);
        
        // 「FromDate」カレンダーボタン
		ImageHyperlink calFrom = new ImageHyperlink(thisArea, SWT.NULL);
		calFrom.setImage(BtsPlugin.getInstance().getImage("calendar.gif"));
		calFrom.setBackground(Display.getDefault().getSystemColor(SWT.COLOR_WIDGET_BACKGROUND));
		calFrom.setToolTipText(Messages.getString("IssueFilterDialog.5")); //$NON-NLS-1$
		calFrom.addHyperlinkListener(new HyperlinkAdapter() {
			public void linkActivated(HyperlinkEvent e) {
				// 日付入力用のカレンダーを開きます。
				OpenCalendarDlgAction action = new OpenCalendarDlgAction(frmDateTxt);
				action.run();
			}
		});


        // 「ToDate」ラベル
        label = new Label(thisArea, SWT.NONE);
        label.setText(Messages.getString("IssueFilterDialog.6")); //$NON-NLS-1$
        
        //「ToDate」テキストボックス
        toDateTxt = new Text(thisArea,SWT.BORDER);
		toDateTxt.setLayoutData(gridData);
		
        // 「ToDate」カレンダーボタン
		ImageHyperlink calTo = new ImageHyperlink(thisArea, SWT.NULL);
		calTo.setImage(BtsPlugin.getInstance().getImage("calendar.gif"));
		calTo.setBackground(Display.getDefault().getSystemColor(SWT.COLOR_WIDGET_BACKGROUND));
		calTo.setToolTipText(Messages.getString("IssueFilterDialog.7")); //$NON-NLS-1$
		calTo.addHyperlinkListener(new HyperlinkAdapter() {
			public void linkActivated(HyperlinkEvent e) {
				// 日付入力用のカレンダーを開きます。
				OpenCalendarDlgAction action = new OpenCalendarDlgAction(toDateTxt);
				action.run();
			}
		});		

        // 書式用ラベル
        label = new Label(thisArea, SWT.NONE);
        label.setText("(YYYY/MM/DD)");

    }
    
    
    /**
     * 検索用文字列指定のUIを生成。
     * 「タイトル」と「詳細」に含む（または含まない）文字列を設定するUIを生成します。
     * 
     * @param			parent					親コンポジット
     * @throws			Exception
     */
    private void createContainsArea(Composite parent) throws Exception {
        
    	// この部品のコンポジット生成
    	Composite thisArea = new Composite(parent, SWT.NONE);

    	// レイアウト設定。3列のグリッドレイアウトにする。
    	GridLayout gridLayout = new GridLayout();
        gridLayout.numColumns = 3;
        thisArea.setLayout(gridLayout);
        thisArea.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
    	
        // 「タイトル」ラベル
        Label label = new Label(thisArea, SWT.NONE);
        label.setText(Messages.getString("IssueFilterDialog.8")); //$NON-NLS-1$

        // 「タイトル」コンボ
        titleCmb = new Combo(thisArea, SWT.READ_ONLY);
        titleCmb.add(Messages.getString("IssueFilterDialog.9")); //$NON-NLS-1$
        titleCmb.add(Messages.getString("IssueFilterDialog.10")); //$NON-NLS-1$
//        combo.addSelectionListener(selectionListener);
        titleCmb.addTraverseListener(new TraverseListener() {
            public void keyTraversed(TraverseEvent e) {
                if (e.detail == SWT.TRAVERSE_ESCAPE
                        || e.detail == SWT.TRAVERSE_RETURN) {
                    e.doit = false;
                }
            }
        });
        // デフォルト設定
        titleCmb.select(titleCmb.indexOf(Messages.getString("IssueFilterDialog.11"))); //$NON-NLS-1$

        // 「タイトル」テキストボックス
        titleTxt = new Text(thisArea, SWT.SINGLE | SWT.BORDER);
        titleTxt.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
//        titleContains.addModifyListener(new ModifyListener() {
//            public void modifyText(ModifyEvent e) {
//                DialogBookmarkFilter.this.markDirty();
//            }
//        });

    
        // 「詳細」ラベル
        label = new Label(thisArea, SWT.NONE);
        label.setText(Messages.getString("IssueFilterDialog.12")); //$NON-NLS-1$

        // 「詳細」コンボ
        descCmb = new Combo(thisArea, SWT.READ_ONLY);
        descCmb.add(Messages.getString("IssueFilterDialog.13")); //$NON-NLS-1$
        descCmb.add(Messages.getString("IssueFilterDialog.14")); //$NON-NLS-1$
//        combo.addSelectionListener(selectionListener);
        descCmb.addTraverseListener(new TraverseListener() {
            public void keyTraversed(TraverseEvent e) {
                if (e.detail == SWT.TRAVERSE_ESCAPE
                        || e.detail == SWT.TRAVERSE_RETURN) {
                    e.doit = false;
                }
            }
        });

        // コンボのデフォルト設定
        descCmb.select(descCmb.indexOf(Messages.getString("IssueFilterDialog.15"))); //$NON-NLS-1$

        // 「詳細」テキストボックス
        descTxt = new Text(thisArea, SWT.SINGLE | SWT.BORDER);
        descTxt.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
//        titleContains.addModifyListener(new ModifyListener() {
//            public void modifyText(ModifyEvent e) {
//                DialogBookmarkFilter.this.markDirty();
//            }
//        });
    }


    /**
     * 「カテゴリ一覧テーブル」と「優先度一覧テーブル」を表示します。
     * 親コンポジットに対し「カテゴリ一覧テーブル」と「優先度一覧テーブル」のUIを生成します。
     * 
     * @param			parent					親コンポジット
     * @throws			Exception
     */
    private void createTablesArea(Composite parent) throws Exception {
      
    	// このパーツのコンポジットを生成
        Composite composite = new Composite(parent, SWT.NONE);

        // グリッドのマージン設定
        GridLayout gridLayout = new GridLayout();
        gridLayout.marginHeight = 0;
        gridLayout.marginWidth = 0;
        gridLayout.horizontalSpacing = 0;
        gridLayout.verticalSpacing = 0;

        // このグリッドにはテーブルを横に２つ並べる
        gridLayout.numColumns = 2;
        composite.setLayout(gridLayout);

        // 「優先度一覧テーブル」を表示
        createPriorityArea(composite);
        // 「担当者一覧テーブル」を表示
        createAssignedArea(composite);
    	
    }

    
    /**
     * 「優先度」テーブルの表示。
     * 共通設定画面で設定した優先度をテーブルに表示する。
     * 
     * @param		parent				親コンポジット
     */
    private void createPriorityArea(Composite parent) throws Exception {
        
        // このパーツのレイアウト設定。グリッドレイアウトを使う
        Composite thisArea = new Composite(parent, SWT.NONE);
        thisArea.setLayout(new GridLayout());

        // テーブルタイトルの設定
        Label label = new Label(thisArea, SWT.NONE);
        label.setText(Messages.getString("IssueFilterDialog.16")); //$NON-NLS-1$

        // テーブルの表示設定
        Table table = new Table(thisArea
        		, SWT.CHECK | SWT.H_SCROLL | SWT.V_SCROLL | SWT.MULTI | SWT.FULL_SELECTION | SWT.BORDER);
        table.setLinesVisible(true);
        table.setHeaderVisible(true);

        // テーブルレイアウトの設定
        TableLayout tableLayout = new TableLayout();
        table.setLayout(tableLayout);
        
        // 1カラム目（担当者名）の設定
        tableLayout.addColumnData(new ColumnWeightData(350, true));
        TableColumn tc = new TableColumn(table, SWT.NONE, 0);
        tc.setText(Messages.getString("IssueFilterDialog.17")); //$NON-NLS-1$

        // テーブルのビューアオブジェクト設定
        priorityViewer = new CheckboxTableViewer(table);

        // テーブルの横幅・高さの調整
        GridData gridData = new GridData(GridData.FILL_HORIZONTAL);
        gridData.heightHint = 45;
        gridData.widthHint = 180;
        priorityViewer.getControl().setLayoutData(gridData);
        
        // テーブルにプロバイダークラスを設定
        priorityViewer.setContentProvider(getPriorityTableContentProvider());
        priorityViewer.setLabelProvider(getPriorityTableLabelProvider());
        
        // 初期値を設定
		CurrentProject project = CurrentProject.getInsance();
		if(project.isOpen()) {
			ModifyIssueFacade issueModifyFacade = new ModifyIssueFacade(); 
			priority = issueModifyFacade.getPriority();
			priorityViewer.setInput(priority);
		}
        
        // デフォルト設定
        priorityViewer.setAllChecked(true);
    }
   

    /**
     * 「優先度一覧テーブル」のコンテントプロバイダを返します。
     * 
     * @return			「優先度一覧テーブル」のコンテントプロバイダ
     */
    private IStructuredContentProvider getPriorityTableContentProvider() {
        return new IStructuredContentProvider() {
            public Object[] getElements(Object inputElement) {
            	return priority;
            }

            public void dispose() {
            }

            public void inputChanged(Viewer viewer, Object oldInput,
                    Object newInput) {
            }
        };
    }

    /**
     * {@link PriorityTableLabelProvider 「優先度一覧テーブル」のラベルプロバイダ}を返します。
     * 
     * @return			「優先度一覧テーブル」のラベルプロバイダ
     */
    private ILabelProvider getPriorityTableLabelProvider() {
        return new PriorityTableLabelProvider();
    }

    
    /**
     * 「優先度一覧テーブル」のラベルプロバイダです。 
     */
    private class PriorityTableLabelProvider extends LabelProvider implements ITableLabelProvider {

		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.ITableLabelProvider#getColumnImage(java.lang.Object, int)
		 */
		public Image getColumnImage(Object element, int columnIndex) {
		    return null;
		}
		
		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.ITableLabelProvider#getColumnText(java.lang.Object, int)
		 */
		public String getColumnText(Object element, int columnIndex) {
			switch (columnIndex) {
				case 0:
					return element.toString();
			}
			return "";
		}
    }

    /**
     * 「担当者」テーブルの表示。
     * 当該プロジェクトにアクセスしたことあるユーザをテーブルに表示する。
     * 
     * @param		parent				親コンポジット
     */
    private void createAssignedArea(Composite parent) throws Exception {
        
        // このパーツのレイアウト設定。グリッドレイアウトを使う
        Composite thisArea = new Composite(parent, SWT.NONE);
        thisArea.setLayout(new GridLayout());

        // テーブルタイトルの設定
        Label label = new Label(thisArea, SWT.NONE);
        label.setText(Messages.getString("IssueFilterDialog.18")); //$NON-NLS-1$

        // テーブルの表示設定
        Table table = new Table(thisArea
        		, SWT.CHECK | SWT.H_SCROLL | SWT.V_SCROLL | SWT.MULTI | SWT.FULL_SELECTION | SWT.BORDER);
        table.setLinesVisible(true);
        table.setHeaderVisible(true);

        // テーブルレイアウトの設定
        TableLayout tableLayout = new TableLayout();
        table.setLayout(tableLayout);
        
        // 1カラム目（担当者名）の設定
        tableLayout.addColumnData(new ColumnWeightData(350, true));
        TableColumn tc = new TableColumn(table, SWT.NONE, 0);
        tc.setText(Messages.getString("IssueFilterDialog.19")); //$NON-NLS-1$

        // テーブルのビューアオブジェクト設定
        assignedViewer = new CheckboxTableViewer(table);

        // テーブルの横幅・高さの調整
        GridData gridData = new GridData(GridData.FILL_HORIZONTAL);
        gridData.heightHint = 45;
        gridData.widthHint = 180;
        assignedViewer.getControl().setLayoutData(gridData);
        
        // テーブルにプロバイダを設定
        assignedViewer.setContentProvider(getAssignedTableContentProvider());
        assignedViewer.setLabelProvider(getAssignedTableLabelProvider());
        
        
        // 初期値を設定
		CurrentProject project = CurrentProject.getInsance();
		if(project.isOpen()) {
			ModifyIssueFacade issueModifyFacade = new ModifyIssueFacade(); 
			assigned = issueModifyFacade.getAssigned();
			assignedViewer.setInput(assigned);
		}
        
        // デフォルト設定
        assignedViewer.setAllChecked(true);
    }
    
    
    /**
     * 「担当者一覧テーブル」のコンテントプロバイダを返します。
     * 
     * @return			「担当者一覧テーブル」のラベルプロバイダ
     * @throws			Exception
     */
    private IStructuredContentProvider getAssignedTableContentProvider() throws Exception {
        return new IStructuredContentProvider() {
        	/*
        	 *  (非 Javadoc)
        	 * @see org.eclipse.jface.viewers.IStructuredContentProvider#getElements(java.lang.Object)
        	 */
            public Object[] getElements(Object inputElement) {
            	return assigned;
            }

            /*
             *  (非 Javadoc)
             * @see org.eclipse.jface.viewers.IContentProvider#dispose()
             */
            public void dispose() {
            }

            /*
             *  (非 Javadoc)
             * @see org.eclipse.jface.viewers.IContentProvider#inputChanged(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
             */
            public void inputChanged(Viewer viewer, Object oldInput,
                    Object newInput) {
            }
        };
    }


    /**
     * {@link AssignedTableLabelProvider 「担当者一覧テーブル」のラベルプロバイダ}を返します。
     * 
     * @return			「担当者一覧テーブル」のラベルプロバイダ
     */
    private ILabelProvider getAssignedTableLabelProvider() {
        return new AssignedTableLabelProvider();
    }

    /**
     * 「担当者一覧テーブル」のラベルプロバイダです。 
     */
    private class AssignedTableLabelProvider extends LabelProvider implements ITableLabelProvider {

		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.ITableLabelProvider#getColumnImage(java.lang.Object, int)
		 */
		public Image getColumnImage(Object element, int columnIndex) {
		    return null;
		}
		
		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.ITableLabelProvider#getColumnText(java.lang.Object, int)
		 */
		public String getColumnText(Object element, int columnIndex) {
			switch (columnIndex) {
				case 0:
					return element.toString();
			}
			return "";
		}
    }

    
    /**
     * 「ステータス選択」のUIを生成します。
     * 親コンポジットに対して「ステータス選択」のUIを生成します。<br>
     * 「ステータス」と「その他表示設定」を２つ横に並べて表示します。
     * 
     * @param			parent						親コンポジット
     * @throws			Exception
     */
    private void createStatusArea(Composite parent) throws Exception {
        
        // このエリアのコンポジット生成。
        Composite thisArea = new Composite(parent, SWT.NONE);

        // グリッドレイアウトを使う。このエリアはグループを横に２つ並べます。
        GridLayout gridLayout = new GridLayout(); 
        gridLayout.numColumns = 2;
        thisArea.setLayout(gridLayout);
        thisArea.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));


        // 「ステータス」のグループを生成
    	Group statusGroup = new Group(thisArea, SWT.NONE);
        statusGroup.setText(Messages.getString("IssueFilterDialog.20")); //$NON-NLS-1$

        // レイアウト設定
        statusGroup.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
        statusGroup.setLayout(new GridLayout());

        // 「ステータス：起票」チェックボックス
        stOpenChk = new Button(statusGroup, SWT.CHECK);
        stOpenChk.setText(IssueStatus.OPENED.getDescription());
        stOpenChk.setSelection(true);

        // 「ステータス：担当者割り当て済み」チェックボックス
        stAssignedChk = new Button(statusGroup, SWT.CHECK);
        stAssignedChk.setText(IssueStatus.ASSIGNED.getDescription());
        stAssignedChk.setSelection(true);

        // 「ステータス：対応済み」チェックボックス
        stResolvedChk = new Button(statusGroup, SWT.CHECK);
        stResolvedChk.setText(IssueStatus.RESOLVED.getDescription());
        stResolvedChk.setSelection(true);

        // 「ステータス：確認済み」チェックボックス
        stVerifiedChk = new Button(statusGroup, SWT.CHECK);
        stVerifiedChk.setText(IssueStatus.VERIFIED.getDescription());
        stVerifiedChk.setSelection(true);

        // 「ステータス：完了」チェックボックス
        stClosedChk = new Button(statusGroup, SWT.CHECK);
        stClosedChk.setText(IssueStatus.CLOSED.getDescription());
        stClosedChk.setSelection(true);



        // 「その他表示設定」のグループを生成
        Group hiddenGroup = new Group(thisArea, SWT.NONE);
        hiddenGroup.setText(Messages.getString("IssueFilterDialog.21")); //$NON-NLS-1$
        
        // レイアウトを設定
        hiddenGroup.setLayoutData(new GridData(GridData.FILL_BOTH));
        hiddenGroup.setLayout(new GridLayout());

        // 「内緒のバグ」チェックボックス
        secretChk = new Button(hiddenGroup, SWT.CHECK);
        secretChk.setText(Messages.getString("IssueFilterDialog.22")); //$NON-NLS-1$
        secretChk.setSelection(true);

        // 「保留中の課題」チェックボックス
        reserveChk = new Button(hiddenGroup, SWT.CHECK);
        reserveChk.setText(Messages.getString("IssueFilterDialog.23")); //$NON-NLS-1$
        reserveChk.setSelection(true);

        // 「完了期限を過ぎた課題」チェックボックス
        deadLineChk = new Button(hiddenGroup, SWT.CHECK);
        deadLineChk.setText(Messages.getString("IssueFilterDialog.24")); //$NON-NLS-1$
        deadLineChk.setSelection(true);

    }
    

    /**
     * セパレータラインを生成します。
     * 親コンポジットに対してセパレータラインを生成します。
     * 
     * @param			parent						親コンポジット
     */
    protected void createSeparatorLine(Composite parent) {
        // 親コンポーネントにラベルを生成
        Label separator = new Label(parent, SWT.HORIZONTAL | SWT.SEPARATOR);
        
        // レイアウトを設定。グリッドレイアウトを使う。
        GridData gd = new GridData(GridData.FILL_HORIZONTAL);
        gd.horizontalSpan = 1;
        separator.setLayoutData(gd);
    }
    
    
	/* (非 Javadoc)
	 * @see org.eclipse.jface.dialogs.Dialog#okPressed()
	 */
	protected void okPressed() {
		
		// 入力された起票日取得
		String frmDateString = frmDateTxt.getText().trim();
		String toDateString = toDateTxt.getText().trim();
		
		// 起票日が入力されているか判定
		if(!"".equals(frmDateString) || !"".equals(toDateString)) {
			
			// エラーチェック
			if(!BTSUtility.dateTextError(frmDateString, toDateString)) {
				// エラーあり
				return;
			}
			
		}
		
		// エラーなし
		save();
		super.okPressed();
		
		// フィルタを実行
		IssueListViewFilter filter = new IssueListViewFilter(view);
		filter.addIssueFilters();
	}

	
	/**
	 * フィルタの設定をXMLファイルに保存します。
	 */
	private boolean save() {
		
		// メッセージフィルタ情報を生成
		IssueFilter newConfig = new IssueFilter();

		// 入力項目を課題票フィルタ情報にセット		
		
		newConfig.setIssueID(issueIDTxt.getText().trim());
		newConfig.setFrmDate(frmDateTxt.getText().trim());
		newConfig.setToDate(toDateTxt.getText().trim());
		newConfig.setTitleCmb(titleCmb.getText());
		newConfig.setTitle(titleTxt.getText().trim());
		newConfig.setDescCmb(descCmb.getText());
		newConfig.setDesc(descTxt.getText().trim());
		
		// チェックが付いていない担当者を「|」区切りで保存
		StringBuffer notAssignedChk = new StringBuffer();
		for(int i=0; i<((String[])assignedViewer.getInput()).length; i++) {
			// チェックが付いていない担当者名を文字列に追加
			if(assignedViewer.getChecked(assignedViewer.getElementAt(i)) == false) {
				notAssignedChk.append(assignedViewer.getElementAt(i) + "|");
			}
		}
		// 最後尾の「|」を削除
		if(notAssignedChk.length() != 0) {
			notAssignedChk.deleteCharAt(notAssignedChk.length()-1);
		}
		newConfig.setAssignedChk(new String(notAssignedChk));
		
		// チェックが付いていない優先度を「|」区切りで保存
		StringBuffer notPriorityChk = new StringBuffer();
		for(int i=0; i<((String[])priorityViewer.getInput()).length; i++) {
			// チェックが付いていないカテゴリ名を文字列に追加
			if(priorityViewer.getChecked(priorityViewer.getElementAt(i)) == false) {
				notPriorityChk.append(priorityViewer.getElementAt(i) + "|");
			}
		}
		// 最後尾の「|」を削除
		if(notPriorityChk.length() != 0) {
			notPriorityChk.deleteCharAt(notPriorityChk.length()-1);
		}
		newConfig.setPriorityChk(new String(notPriorityChk));
		
		newConfig.setStOpenChk(stOpenChk.getSelection());
		newConfig.setStAssignedChk(stAssignedChk.getSelection());
		newConfig.setStResolvedChk(stResolvedChk.getSelection());
		newConfig.setStVerifiedChk(stVerifiedChk.getSelection());
		newConfig.setStClosedChk(stClosedChk.getSelection());
		
		newConfig.setSecretChk(secretChk.getSelection());
		newConfig.setReserveChk(reserveChk.getSelection());
		newConfig.setDeadLineChk(deadLineChk.getSelection());
		
		
		// 旧課題票フィルタ情報の有無を判定
		if (oldConfig == null) {
			// 旧課題票フィルタ情報がなければ新規作成
			IssueFilterManager.save(newConfig);
		} else {
			// 旧課題票フィルタ情報があれば編集
			IssueFilterManager.modify(oldConfig, newConfig);
		}
		
		return true;

	}
	
	/**
	 * フィルタ設定をロードします。
	 *
	 */
	private void loadFilterSettings() {
		
		issueIDTxt.setText(oldConfig.getIssueID());
		frmDateTxt.setText(oldConfig.getFrmDate());
		toDateTxt.setText(oldConfig.getToDate());
		titleCmb.setText(oldConfig.getTitleCmb());
		titleTxt.setText(oldConfig.getTitle());
		descCmb.setText(oldConfig.getDescCmb());
		descTxt.setText(oldConfig.getDesc());
		
		// チェックが付いていない担当者の文字列を配列に変換
		String[] assignedArgs = oldConfig.getAssignedChk().split("\\|");
		for(int i=0; i<assignedArgs.length; i++) {
			assignedViewer.setChecked(assignedArgs[i], false);
		}
		
		// チェックが付いていない優先度の文字列を配列に変換
		String[] prioritydArgs = oldConfig.getPriorityChk().split("\\|");
		for(int i=0; i<prioritydArgs.length; i++) {
			priorityViewer.setChecked(prioritydArgs[i], false);
		}
		
		stOpenChk.setSelection(oldConfig.isStOpenChk());
		stAssignedChk.setSelection(oldConfig.isStAssignedChk());
		stResolvedChk.setSelection(oldConfig.isStResolvedChk());
		stVerifiedChk.setSelection(oldConfig.isStVerifiedChk());
		stClosedChk.setSelection(oldConfig.isStClosedChk());
		
		secretChk.setSelection(oldConfig.isSecretChk());
		reserveChk.setSelection(oldConfig.isReserveChk());
		deadLineChk.setSelection(oldConfig.isDeadLineChk());
	}
}
/*
 * ＜参考＞
 * ブックマークビュー
 * 　org.eclipse.ui.ide
 *  　├org.eclipse.ui.views.markers.internal.DialogBookmarkFilter
 *  　└org.eclipse.ui.views.markers.internal.DialogMarkerFilter
 * 
 */

