/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.ui.issuelist;

import java.util.List;

import jp.valtech.bts.data.Issue;
import jp.valtech.bts.data.MessagePacket;
import jp.valtech.bts.network.ClientInfo;
import jp.valtech.bts.ui.BtsPlugin;
import jp.valtech.bts.ui.IBtsListener;
import jp.valtech.bts.ui.IBtsViewPart;
import jp.valtech.bts.util.Logging;

import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Table;

/**
 * {@link IssueListView 課題票一覧ビュー}のプロバイダクラスです。
 * 
 * {@link jp.valtech.bts.ui.IBtsListener リスナーインタフェース}を実装することにより、BTS情報に更新があった際には更新通知を受けとることができます。<BR>
 * 更新通知を受けとると{@link IssueListView 課題票一覧ビュー}に表示を反映します。
 * 
 * @author		<A href="mailto:m_sugitou@valtech.jp">M.Sugito</A>
 * @version	Ver.0.8
 */
public class IssueListViewContentProvider implements IStructuredContentProvider,IBtsListener, Logging {

	/** 課題票一覧ビューのビューアオブジェクト */
	private TableViewer viewer;
	/** 呼び出し元テーブルオブジェクト */
	private IBtsViewPart viewPart;
	
	
	/**
	 * 当該リスナインスタンスををプラグインに登録します。
	 * 
	 */
	public IssueListViewContentProvider(IBtsViewPart viewPart) {
		super();
		
		// 呼び出し元ビューを保持します。
		this.viewPart = viewPart;
		// 当該リスナインスタンスをプラグインに登録します。
		BtsPlugin.getInstance().addListener(this);
		
	}


	/* (非 Javadoc)
	 * @see org.eclipse.jface.viewers.IStructuredContentProvider#getElements(java.lang.Object)
	 */
	public Object[] getElements(Object inputElement) {
		return ((List)inputElement).toArray();
	}

	
	/**
	 * 当該リスナインスタンスをプラグインから削除します。
	 */
	public void dispose() {
		BtsPlugin.getInstance().removeListener(this);
	}

	
	/* (非 Javadoc)
	 * @see org.eclipse.jface.viewers.IContentProvider#inputChanged(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
	 */
	public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
		this.viewer= (TableViewer)viewer;
	}


	/**
	 * {@link Issue 課題票}の新規登録処理です。
	 * 登録された課題票情報を{@link IssueListView 課題票一覧ビュー}に追加して再描画ます。
	 * 
	 * @param		issue				新規登録する課題票
	 */
	public void addIssue(final Issue newIssue) {
		
		// テーブルビューアがnullの時は何もしない
		if(viewer == null ) {
			return ;
		}
		// メインスレッドで実行されるようにRunnableをメインスレッドに渡します。
		Control ctrl= viewer.getControl();
		ctrl.getDisplay().syncExec(new Runnable() {
			public void run() {
				try {
					// 実データの更新
					List issueList = (List)viewer.getInput();
					issueList.add(newIssue);
					
					// 一覧表示の更新
					viewer.insert(newIssue, 0);
					viewer.refresh();
					
					// 表示件数更新
					Table table = viewer.getTable();
					viewPart.updateDescription(table.getItemCount());

				}catch (Exception e) {
					String msg = Messages.getString("IssueListViewContentProvider.0"); //$NON-NLS-1$
					logger.fatal(msg, e);
					BtsPlugin.getInstance().error(msg, e);
				}
			}
		});
		
	}

	/**
	 * {@link Issue 課題票}の更新処理です。
	 * 更新された課題票情報を{@link IssueListView 課題票一覧ビュー}に再描画します。
	 * 
	 * @param		issue				更新された課題票
	 */
	public void updateIssue(final Issue oldIssue, final Issue newIssue) {
		// テーブルビューアがnullの時は何もしない
		if(viewer == null ) {
			return ;
		}
		// メインスレッドで実行されるようにRunnableをメインスレッドに渡します。
		Control ctrl= viewer.getControl();
		ctrl.getDisplay().syncExec(new Runnable() {
			public void run() {
				try {
					// テーブルの実データを取得
					List issueList = (List)viewer.getInput();

					// 更新前の情報は一覧から削除する
					if(oldIssue != null) {
						issueList.remove(oldIssue);
						viewer.remove(oldIssue);
					}
					
					// 削除後、更新後の課題票を追加する
					if(!issueList.contains(newIssue)) {
						issueList.add(newIssue);
						viewer.insert(newIssue, 0);
					}

					viewer.refresh();
	
					// 表示件数更新
					Table table = viewer.getTable();
					viewPart.updateDescription(table.getItemCount());

				}catch (Exception e) {
					String msg = Messages.getString("IssueListViewContentProvider.1"); //$NON-NLS-1$
					logger.fatal(msg, e);
					BtsPlugin.getInstance().error(msg, e);
				}
			}
		});
	}


	/**
	 * {@link Issue 課題票}の削除処理です。
	 * 削除された課題票情報を{@link IssueListView 課題票一覧ビュー}に再描画します。
	 * 
	 * @param		deleteIssue				削除された課題票
	 */
	public void deleteIssue(final Issue deleteIssue) {
		// テーブルビューアがnullの時は何もしない
		if(viewer == null ) {
			return ;
		}

		// メインスレッドで実行されるようにRunnableをメインスレッドに渡します。
		Control ctrl= viewer.getControl();
		ctrl.getDisplay().syncExec(new Runnable() {
			public void run() {
				try {
					// テーブルの実データを削除
					List issueList = (List)viewer.getInput();
					issueList.remove(deleteIssue);

					// 一覧表示の更新
					viewer.remove(deleteIssue);
	
					// 表示件数更新
					Table table = viewer.getTable();
					viewPart.updateDescription(table.getItemCount());

				}catch (Exception e) {
					String msg = Messages.getString("IssueListViewContentProvider.2"); //$NON-NLS-1$
					logger.fatal(msg, e);
					BtsPlugin.getInstance().error(msg, e);
				}
			}
		});

	}
	
	/*
	 *  (非 Javadoc)
	 * @see jp.valtech.bts.ui.IBtsListener#joinMember(jp.valtech.bts.network.ClientInfo)
	 */
	public void joinMember(ClientInfo clientInfo) {
		// 何もしない
	}
	
	/*
	 *  (非 Javadoc)
	 * @see jp.valtech.bts.ui.IBtsListener#leaveMember(jp.valtech.bts.network.ClientInfo)
	 */
	public void leaveMember(ClientInfo clientInfo) {
		// 何もしない
	}

	/*
	 *  (非 Javadoc)
	 * @see jp.valtech.bts.ui.IBtsListener#addMessage(jp.valtech.bts.data.MessagePacket)
	 */
	public void addMessage(MessagePacket messagePacket) {
		// 何もしない
	}

	/*
	 *  (非 Javadoc)
	 * @see jp.valtech.bts.ui.IBtsListener#deleteMessage(jp.valtech.bts.data.MessagePacket)
	 */
	public void deleteMessage(MessagePacket messagePacket) {
		// 何もしない
	}
	
	/**
	 * 現在のローカルDBの情報で課題票一覧ビューを再表示します。
	 * 再表示する前に一旦一覧表示をクリアします。その後最新の情報を表示します。
	 * 
	 * @param		issueList				課題票を格納したリスト
	 */
	public void refresh(final List issueList) {

		// テーブルビューアがnullの時は何もしない
		if(viewer == null ) {
			return ;
		}
		
		// メインスレッドで実行されるようにRunnableをメインスレッドに渡します。
		Control ctrl= viewer.getControl();
		ctrl.getDisplay().syncExec(new Runnable() {
			public void run() {
				try {
					// 一旦データをクリアする
					List oldIssueList = (List)viewer.getInput();
					oldIssueList.clear();
					viewer.refresh();

					// DBから取得したデータを表示する
					viewer.setInput( issueList );

					// 表示件数更新
					viewPart.updateDescription(issueList.size());

				} catch (Exception e) {
					String msg = Messages.getString("IssueListViewContentProvider.3"); //$NON-NLS-1$
					logger.fatal(msg, e);
					BtsPlugin.getInstance().error(msg, e);
				}
			}
		});
	}
	
	
	/**
	 * 課題票一覧ビューをクリアします。
	 */
	public void clear() {
		// テーブルビューアがnullの時は何もしない
		if(viewer == null ) {
			return ;
		}
		
		// メインスレッドで実行されるようにRunnableをメインスレッドに渡します。
		Control ctrl= viewer.getControl();
		ctrl.getDisplay().syncExec(new Runnable() {
			public void run() {
				try {
					// 一旦データをクリアする
					List list = (List)viewer.getInput();
					list.clear();
					
					viewer.refresh();


				} catch (Exception e) {
					String msg = Messages.getString("IssueListViewContentProvider.4"); //$NON-NLS-1$
					logger.fatal(msg, e);
					BtsPlugin.getInstance().error(msg, e);
				}
			}
		});
	}
}
