/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.ui.navigator;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.swt.graphics.Image;


/**
 * ツリーノードの基本クラスです。
 * 
 * @author		<A href="mailto:m_sugitou@valtech.jp">M.Sugito</A>
 * @version	Ver.0.8
 */
public class BtsTreeNode implements Comparable {
    
	/** このノードの名前 */
	protected String textName = "";
	
	/** このノードのアイコン */
	protected Image image = null;

	/** 親ノード */
    protected BtsTreeNode parent = null;

    /** 子ノード */
    protected List children = new ArrayList();
    

    /**
	 * ツリーノードを生成します。
     */
    public BtsTreeNode() {
    }

    /**
	 * ツリーノードを生成します。
     * 親ノード・名前・イメージファイル名を設定します。
     * 
     * @param		parent				このノードの親
     * @param		textName			このノードの表示名
     * @param		image				このノードのアイコン
     */
    public BtsTreeNode(BtsTreeNode parent, String textName, Image image) {
        this.parent 	= parent;
        this.textName 	= textName;
        this.image	 	= image;
    }
    
    /**
     * 親ノードを取得します。
     * 
     * @return		このツリーノードの親
     */
	public BtsTreeNode getParent() {
        return this.parent;
	}
	
	/**
	 * 子ノードの有無を判定。
	 * 
	 * @return		true--子ノード有。false--子ノード無し
	 */
	public boolean hasChildren() {
		
		// 子ノードが無い場合はfalseを返す
		if(children.size() == 0) {
			return false;
		}
		// 子ノードがある場合はtrueを返す
		return true;
	}
	
	/**
	 * 子ノードを追加します。
	 * 
	 * @param		treeNode			このインスタンスに結びつく子ノード
	 */
	public void addChild(BtsTreeNode treeNode) {
		
		// 既に同一のノードがある場合は一旦削除します。
		if(children.contains(treeNode)) {
			children.remove(treeNode);
		}
		this.children.add(treeNode);
	}
	
	/**
	 * 既に保持している子ノードを更新します。
	 * 
	 * @param		treeNode			このインスタンスに結びつく子ノード
	 */
	public void updateChild(BtsTreeNode treeNode) {
		
		// 既に同一のノードがある場合は一旦削除します。
		if(children.contains(treeNode)) {
			children.remove(treeNode);
		}
		this.children.add(treeNode);
	}

	
	/**
	 * 指定のノードが子ノードにある場合に削除します。
	 * 指定のノードを辞書式に比較し、{@link #equals(Object)}でtrueが返ってきたものを「等しい」と判断し、そのノードを削除します。
	 * 
	 * @param		node		削除対象のノード
	 */
	public void removeChild(BtsTreeNode node) {
		children.remove(node);
	}

	/**
	 * 指定のノード子ノードの中に既にあるかどうか判別します。
	 * 指定のノードを辞書式に比較し、{@link #equals(Object)}でtrueが返ってきたものを「等しい」と判断します。
	 * 
	 * @param		node		削除対象のノード
	 * @return		true-- 指定のノード子ノードの中に既にある。
	 * 				false--指定のノード子ノードの中にない。
	 */
	public boolean contains(BtsTreeNode node) {
		return children.contains(node);
	}
	
	
	/**
	 * このノードが持つ子ノードを配列で返す。
	 * 
	 * @return		このノードが持つ子ノードの配列
	 */
	public List getChildren() {
		return this.children;
	}

	
	/**
	 * 子ノードを特定するためのキー文字列を用いて子ノードを取得します。
	 * 
	 * @param		key			子ノードを特定するためのキー文字列です
	 * @return		
	 */
	public BtsTreeNode getChild(String key) {
		for (int idx = 0; idx < children.size(); idx++) {
			BtsTreeNode child = (BtsTreeNode)children.get(idx);
			if(child.getNodeKey().equals(key)) {
				return child;
			}
		}
		return null;
	}
	
	
	/**
	 * このノードを表示する時のテキスト文字列を取得します。
	 * 
	 * @return		このノードのテキスト文字列
	 */
	public String getTextName() {;
		return this.textName;
	}

	/**
	 * このノードを表示する時のテキスト文字列を返します。
	 * 
	 * @param		textName			このノードのテキスト文字列
	 */
	public void setTextName(String textName) {
		this.textName = textName;
	}
	
	/**
	 * このノードを表示するときのイメージファイル名を取得します。
	 *  
	 * @return		このノードのイメージファイル
	 */
    public Image getImage(){
    	return image;
    }
    
    
    /**
     * 子ノードを特定するためのキー文字列を取得します。
     * 
     * @return		子ノードを特定するためのキー文字列
     */
    public String getNodeKey() {
    	return this.textName;
    }

    
	/**
	 * このノードを他のノードと比較します。
	 * 比較のキーには{@link #getTextName()}を使います。
	 * 
	 * @param		obj			比較対象のツリーノード
	 * @return		引数文字列がこの文字列と辞書式に等しい場合は「0」。<br>
	 * 				引数がこの文字列より辞書式に大きい文字列である場合は「0より小さい値」<br>
	 * 				引数がこの文字列より辞書式に小さい場合は「0 より大きい値」<br>		
	 */
	public int compareTo(Object obj) {
		if(obj==null){
			return 1;
		}
		BtsTreeNode that = (BtsTreeNode)obj;
		String thisKey = this.getTextName();
		String thatKey = that.getTextName();
		
		return thisKey.compareTo(thatKey);
	}

	
	/**
	 * 指定のノードが自分と同じでの場合はtrueを返します。
	 * 
	 * @param		obj		比較対照のノード
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	public boolean equals(Object obj) {
		if(obj==null) {
			return false;
		}
		if(this.compareTo(obj) == 0) { 
			return true;
		}
		return false;
	}
}
