/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.ui.navigator;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import jp.valtech.bts.data.CurrentProject;
import jp.valtech.bts.data.Issue;
import jp.valtech.bts.facade.ModifyIssueFacade;
import jp.valtech.bts.ui.BtsPlugin;
import jp.valtech.bts.ui.action.OpenCalendarDlgAction;
import jp.valtech.bts.util.BTSUtility;
import jp.valtech.bts.util.IssueChartRender;
import jp.valtech.bts.util.Logging;

import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.CheckboxTableViewer;
import org.eclipse.jface.viewers.ColumnWeightData;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.TableLayout;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.forms.events.HyperlinkAdapter;
import org.eclipse.ui.forms.events.HyperlinkEvent;
import org.eclipse.ui.forms.widgets.ImageHyperlink;

/**
 * バグ累計ダイアログを作成します。
 * 
 * @author		<A href="mailto:s_imai@valtech.jp">S.Imai</A>
 * @version	Ver.0.8
 */
public class IssueChartDialog extends Dialog implements Logging {
	
	/** グラフを表示するための課題票リスト */
	private List chartIssueList;
	
	/** グラフを表示するラベル */
	private Label chartLabel;

	/** カレントディレクトリを指定 */
	private String currentDir = BtsPlugin.getInstance().getStateLocation() + "/";
	
    /** 「From日付」用テキストボックス */
	private Text frmDateTxt;
	
    /** 「To日付」用テキストボックス */
	private Text toDateTxt;
	
    /**「 カテゴリ一覧テーブル」用テーブルビューア */
    private CheckboxTableViewer categoryViewer;

	/** 「内緒のバグを含めない」用チェックボックス */
	private Button secretChk;
	
	/** 「保留中の課題を含めない」用チェックボックス */
	private Button reserveChk;
	
	/** カテゴリ一覧 */
	private String[] category;
	
	/** 背景色 */
	Color color = new Color(Display.getCurrent(), 255, 255, 255);

	
	/**
	 * バグ累計グラフ生成用の課題票リストを設定します。
	 * 
	 * @param		parent		シェル
	 */
	public IssueChartDialog(Shell parent, List chartIssueList) {
		super(parent);
		this.chartIssueList = chartIssueList;
	}
	

	/* (非 Javadoc)
	 * @see org.eclipse.jface.window.Window#configureShell(org.eclipse.swt.widgets.Shell)
	 */
	protected void configureShell(Shell newShell) {
		super.configureShell(newShell);
		newShell.setText(Messages.getString("IssueChartDialog.0")); //$NON-NLS-1$
		PlatformUI.getWorkbench().getHelpSystem().setHelp(newShell, "jp.valtech.bts.help.chartContext");
	}


	/*
	 * (非 Javadoc)
	 * 
	 * @see org.eclipse.jface.dialogs.Dialog#createDialogArea(org.eclipse.swt.widgets.Composite)
	 */
	protected Control createDialogArea(Composite parent) {
		
		// コンポジット生成
		Composite composite = (Composite) super.createDialogArea(parent);
		
		try {
					
			// ダイアログの背景色の設定
			composite.setBackground(color);
			
			// バグ累計グラフを表示
			createChartArea(composite);
			
			// 表示条件を表示
			createConditionArea(composite);
			
			// 注釈を表示
			createNotes(composite);

		} catch (Exception e) {
			String msg = Messages.getString("IssueChartDialog.14"); //$NON-NLS-1$
			logger.fatal(msg, e);
			BtsPlugin.getInstance().error(msg, e);
		}
		
		return composite;
		
	}

	/**
	 * バグ累計グラフを表示します。
	 * 
	 * @param parent
	 */
	private void createChartArea(Composite parent) {
		
		// ラベル生成
		chartLabel = new Label(parent, SWT.NONE);

		// グラフイメージ取得
		Image image = new Image(Display.getCurrent(), currentDir + "bug.png");
		
		// グラフイメージをラベルにセット
		chartLabel.setImage(image);
		
	}
    
    /**
     * 「表示する条件」UIを生成します。
     * 
     * @param parent
     * @throws Exception
     */
    private void createConditionArea(Composite parent) throws Exception {

    	// このエリアのコンポジットを生成。
    	Composite thisArea = new Composite(parent, SWT.NONE);
    	
    	// このエリアの背景色設定
    	thisArea.setBackground(color);
    	
    	// レイアウトを設定。
        thisArea.setLayout(new GridLayout());
        GridData thisAreaGridData = new GridData(GridData.HORIZONTAL_ALIGN_CENTER);
        thisAreaGridData.widthHint = 400;
        thisArea.setLayoutData(thisAreaGridData);
        
        // 「出力する条件」グループの生成
    	Group group = new Group(thisArea, SWT.NONE);
        group.setText(Messages.getString("IssueChartDialog.1")); //$NON-NLS-1$
        group.setFont(new Font(Display.getCurrent(), "", 10, 1));
        group.setBackground(color);
        
        // 「出力する条件」のレイアウト設定
    	GridLayout layout = new GridLayout();
        layout.numColumns = 8;
        layout.verticalSpacing = 10;
    	group.setLayout(layout);
    	group.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
    	
        // 「FromDate」ラベル
        Label label = new Label(group, SWT.NONE);
        label.setText(Messages.getString("IssueChartDialog.2")); //$NON-NLS-1$
        label.setBackground(color);

        // 「FromDate」テキストボックス
        frmDateTxt = new Text(group,SWT.BORDER);
		GridData dateGridData = new GridData(GridData.FILL_HORIZONTAL); 
		dateGridData.widthHint = 80;
        frmDateTxt.setLayoutData(dateGridData);
        
        // 「FromDate」カレンダーボタン
		ImageHyperlink calFrom = new ImageHyperlink(group, SWT.NULL);
		calFrom.setImage(BtsPlugin.getInstance().getImage("calendar.gif"));
		calFrom.setBackground(Display.getDefault().getSystemColor(SWT.COLOR_WIDGET_BACKGROUND));
		calFrom.setToolTipText(Messages.getString("IssueChartDialog.3")); //$NON-NLS-1$
		calFrom.setBackground(color);
		calFrom.addHyperlinkListener(new HyperlinkAdapter() {
			public void linkActivated(HyperlinkEvent e) {
				// 日付入力用のカレンダーを開きます。
				OpenCalendarDlgAction action = new OpenCalendarDlgAction(frmDateTxt);
				action.run();
			}
		});

        // 「ToDate」ラベル
        label = new Label(group, SWT.NONE);
        label.setText(Messages.getString("IssueChartDialog.4")); //$NON-NLS-1$
        label.setBackground(color);
        
        //「ToDate」テキストボックス
        toDateTxt = new Text(group,SWT.BORDER);
		toDateTxt.setLayoutData(dateGridData);
		
        // 「ToDate」カレンダーボタン
		ImageHyperlink calTo = new ImageHyperlink(group, SWT.NULL);
		calTo.setImage(BtsPlugin.getInstance().getImage("calendar.gif"));
		calTo.setBackground(Display.getDefault().getSystemColor(SWT.COLOR_WIDGET_BACKGROUND));
		calTo.setToolTipText(Messages.getString("IssueChartDialog.5")); //$NON-NLS-1$
		calTo.setBackground(color);
		calTo.addHyperlinkListener(new HyperlinkAdapter() {
			public void linkActivated(HyperlinkEvent e) {
				// 日付入力用のカレンダーを開きます。
				OpenCalendarDlgAction action = new OpenCalendarDlgAction(toDateTxt);
				action.run();
			}
		});		

        // 書式用ラベル
        label = new Label(group, SWT.NONE);
        label.setText("(YYYY/MM/DD)");
        label.setBackground(color);
        GridData labelGridData = new GridData();
        labelGridData.horizontalSpan = 2;
        label.setLayoutData(labelGridData);

        // 「カテゴリ」テーブルの表示設定
        Table table = new Table(group
        		, SWT.CHECK | SWT.H_SCROLL | SWT.V_SCROLL | SWT.MULTI | SWT.FULL_SELECTION | SWT.BORDER);
        table.setLinesVisible(true);
        table.setHeaderVisible(true);

        // テーブルレイアウトの設定
        TableLayout tableLayout = new TableLayout();
        table.setLayout(tableLayout);
        
        // カラムの設定
        tableLayout.addColumnData(new ColumnWeightData(350, true));
        TableColumn tc = new TableColumn(table, SWT.COLOR_WHITE, 0);
        tc.setText(Messages.getString("IssueChartDialog.6")); //$NON-NLS-1$

        // テーブルの横幅・高さ・マージンの調整
        GridData tableGridData = new GridData();
        tableGridData.heightHint = 45;
        tableGridData.widthHint = 200;
        tableGridData.horizontalSpan = 8;

        // テーブルのビューアオブジェクト設定
        categoryViewer = new CheckboxTableViewer(table);
        categoryViewer.getControl().setLayoutData(tableGridData);
        
        // テーブルにプロバイダを設定
        categoryViewer.setContentProvider(getCategoryTableContentProvider());
        categoryViewer.setLabelProvider(getCategoryTableLabelProvider());

        // 初期値を設定
		CurrentProject project = CurrentProject.getInsance();
		if(project.isOpen()) {
			ModifyIssueFacade issueModifyFacade = new ModifyIssueFacade(); 
			category = issueModifyFacade.getCategoryTypeR();
			categoryViewer.setInput(category);
		}
        
        // デフォルト設定
        categoryViewer.setAllChecked(true);
        
        
        // 「内緒のバグを含めない」チェックボックス
        secretChk = new Button(group, SWT.CHECK);
        secretChk.setText(Messages.getString("IssueChartDialog.7")); //$NON-NLS-1$
        secretChk.setBackground(color);
        secretChk.setSelection(false);
        GridData data = new GridData(GridData.FILL_HORIZONTAL);
        data.horizontalSpan = 6;
        secretChk.setLayoutData(data);

        // 「保留中の課題を含めない」チェックボックス
        reserveChk = new Button(group, SWT.CHECK);
        reserveChk.setText(Messages.getString("IssueChartDialog.8")); //$NON-NLS-1$
        reserveChk.setBackground(color);
        reserveChk.setSelection(false);
        reserveChk.setLayoutData(data);
        
        // 「適用」ボタン設定
        Button submitBtn = new Button(group, SWT.PUSH | SWT.FLAT);
        submitBtn.setText(Messages.getString("IssueChartDialog.9")); //$NON-NLS-1$
        GridData submitGrid = new GridData(GridData.HORIZONTAL_ALIGN_END);
        submitGrid.widthHint = 60;
        submitGrid.heightHint = 30;
        submitGrid.verticalSpan = 2;
        submitBtn.setLayoutData(submitGrid);
        
        // 「適用」ボタンにセレクションリスナを追加
        submitBtn.addSelectionListener(new SelectionAdapter() {

        	/*
        	 *  (非 Javadoc)
        	 * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
        	 */
			public void widgetSelected(SelectionEvent e) {
				// グラフ再表示
				submitPressed();
			}
        	
        });
        
        // 「閉じる」ボタン設定
        Button closeBtn = new Button(group, SWT.PUSH | SWT.FLAT);
        closeBtn.setText(Messages.getString("IssueChartDialog.10")); //$NON-NLS-1$
        GridData closeGrid = new GridData();
        closeGrid.widthHint = 60;
        closeGrid.heightHint = 30;
        closeGrid.verticalSpan = 2;
        closeBtn.setLayoutData(closeGrid);
        
        // 「閉じる」ボタンにセレクションリスナを追加
        closeBtn.addSelectionListener(new SelectionAdapter() {

        	/*
        	 *  (非 Javadoc)
        	 * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
        	 */
			public void widgetSelected(SelectionEvent e) {
				// ダイアログを閉じる
				close();
			}
        	
        });
    	
    }
    
    /**
     * 注釈を表示します。
     * 
     * @param parent
     */
    private void createNotes(Composite parent) {
    	
    	// 注釈設定
    	Label notesLabel = new Label(parent, SWT.NONE);
    	notesLabel.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_END));
    	notesLabel.setText(Messages.getString("IssueChartDialog.11")); //$NON-NLS-1$
    	notesLabel.setBackground(color);
    	notesLabel.setForeground(new Color(Display.getCurrent(), 49, 106, 197));
    	
    }
    
    /**
     * 「カテゴリテーブル一覧」のコンテントプロバイダを返します。
     * 
     * @return			「カテゴリテーブル一覧」のコンテントプロバイダ
     */
    private IStructuredContentProvider getCategoryTableContentProvider() {
        return new IStructuredContentProvider() {
            public Object[] getElements(Object inputElement) {
            	return category;
            }
            public void dispose() {
            }
            public void inputChanged(Viewer viewer, Object oldInput,
                    Object newInput) {
            }
        };
    }

    /**
     * {@link CategoryTableLabelProvider 「カテゴリテーブル一覧」のコンテントプロバイダ}を返します。
     * 
     * @return			{@link CategoryTableLabelProvider}
     */
    private ILabelProvider getCategoryTableLabelProvider() {
        return new CategoryTableLabelProvider();
    }
    
    /**
     * カテゴリテーブル用のラベルプロバイダです。
     * 
     */
    private class CategoryTableLabelProvider extends LabelProvider implements ITableLabelProvider {

		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.ITableLabelProvider#getColumnImage(java.lang.Object, int)
		 */
		public Image getColumnImage(Object element, int columnIndex) {
		    return null;
		}
		
		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.ITableLabelProvider#getColumnText(java.lang.Object, int)
		 */
		public String getColumnText(Object element, int columnIndex) {
			if(columnIndex == 0) {
				return element.toString();
			}
			return "";
		}
    }
    
    /*
     *  (非 Javadoc)
     * @see org.eclipse.jface.window.Window#createContents(org.eclipse.swt.widgets.Composite)
     */
	protected Control createContents(Composite parent) {
		// create the top level composite for the dialog
		Composite composite = new Composite(parent, 0);
		GridLayout layout = new GridLayout();
		layout.marginHeight = 0;
		layout.marginWidth = 0;
		layout.verticalSpacing = 0;
		composite.setLayout(layout);
		composite.setLayoutData(new GridData(GridData.FILL_BOTH));
		applyDialogFont(composite);
		// initialize the dialog units
		initializeDialogUnits(composite);
		// create the dialog area and button bar
		dialogArea = createDialogArea(composite);
		// ボタンエリア非表示
		// buttonBar = createButtonBar(composite);
		return composite;
	}
	
	/**
	 * 「適用」ボタンを押下時の処理です。
	 * バグ累計グラフを再描画します。
	 *
	 */
	private void submitPressed() {
		
		// 入力された起票日取得
		String frmDateString = frmDateTxt.getText().trim();
		String toDateString = toDateTxt.getText().trim();
		
		// 起票日が入力されているか判定
		if(!"".equals(frmDateString) || !"".equals(toDateString)) {
			
			// エラーチェック
			if(!BTSUtility.dateTextError(frmDateString, toDateString)) {
				// エラーあり
				return;
			}
			
		}
		
		// グラフ生成用の課題票リストを更新
		List renewList = renewChartIssueList(new ArrayList(chartIssueList));
		
		// リストが空の場合
		if(renewList.size() == 0) {
			
			MessageDialog.openError(this.getShell(), Messages.getString("IssueChartDialog.12"), Messages.getString("IssueChartDialog.13")); //$NON-NLS-1$ //$NON-NLS-2$
			return;
			
		}
		
		// バグ累積数を再カウント、グラフ再出力
		IssueChartRender render = new IssueChartRender(renewList);
		render.createChart();

		// グラフ再表示
		Image image = new Image(Display.getCurrent(), currentDir + "bug.png");
		chartLabel.setImage(image);
	}
	
	
	/**
	 * グラフ生成用の課題票リストを更新します。
	 * 
	 * @param		更新前のリスト
	 * @return		更新後のリスト
	 */
	private List renewChartIssueList(List renewList) {
		
		for (int i = 0; i < chartIssueList.size(); i++) {
			
			// 課題票取得
			Issue issue = (Issue) chartIssueList.get(i);
			
			// 起票日テキストが両方とも空ではない場合
			if(!"".equals(frmDateTxt.getText().trim()) && !"".equals(toDateTxt.getText().trim())) {
				
				// 入力された起票日取得
				Date frmDate = BTSUtility.createDate(frmDateTxt.getText().trim());
				Date toDate = new Date((long)(BTSUtility.createDate(toDateTxt.getText().trim()).getTime() + 86400000L));

				// 範囲外の課題票をリストから削除
				if(issue.getCreateDate().before(frmDate) || issue.getCreateDate().after(toDate)) {
					renewList.remove(issue);
				}
				
			}
			
			// 起票日(from)のみ入力時
			else if(!"".equals(frmDateTxt.getText().trim()) && "".equals(toDateTxt.getText().trim())) {
				
				// 入力された起票日取得
				Date frmDate = BTSUtility.createDate(frmDateTxt.getText().trim());
				
				// 範囲外の課題票をリストから削除
				if(issue.getCreateDate().before(frmDate)) {
					renewList.remove(issue);
				}
				
			}
			
			// 起票日(to)のみ入力時
			else if("".equals(frmDateTxt.getText().trim()) && !"".equals(toDateTxt.getText().trim())) {
				
				// 入力された起票日取得
				Date toDate = new Date((long)(BTSUtility.createDate(toDateTxt.getText().trim()).getTime() + 86400000L));
				
				// 範囲外の課題票をリストから削除
				if(issue.getCreateDate().after(toDate)) {
					renewList.remove(issue);
				}
				
			}
			
			for(int j=0; j<category.length; j++) {
				
				// 課題票のカテゴリチェックが外れている場合はリストから削除
				if (!categoryViewer.getChecked(category[j]) && category[j].equals(issue.getCategory())) {
					renewList.remove(issue);
				}
				
			}
			
			// 「内緒のバグを含めない」がチェック、かつ内緒のバグの課題票はリストから削除
			if (secretChk.getSelection() && issue.getSecretBug().booleanValue()) {
				renewList.remove(issue);
			}
			
			// 「保留中の課題票を含めない」がチェック、かつ保留中の課題票はリストから削除
			if (reserveChk.getSelection() && issue.getSuspend().booleanValue()) {
				renewList.remove(issue);
			}
			
		}

		return renewList;

	}
	
}
