/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.ui.navigator;

import java.util.List;

import jp.valtech.bts.command.client.Join;
import jp.valtech.bts.data.CurrentProject;
import jp.valtech.bts.data.Issue;
import jp.valtech.bts.data.MessagePacket;
import jp.valtech.bts.data.NetworkConfig;
import jp.valtech.bts.network.ClientInfo;
import jp.valtech.bts.network.ClientList;
import jp.valtech.bts.ui.BtsPlugin;
import jp.valtech.bts.ui.IBtsListener;
import jp.valtech.bts.util.Logging;

import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.widgets.Control;

/**
 * {@link NavigatorView BTSナビゲータビュー}用のプロバイダクラスです。
 * 
 * {@link jp.valtech.bts.ui.IBtsListener リスナーインタフェース}を実装することにより、BTS情報に更新があった際には更新通知を受けとることができます。<BR>
 * 更新通知を受けとると{@link NavigatorView BTSナビゲータビュー}に表示を反映します。
 * 
 * @author		<A href="mailto:m_sugitou@valtech.jp">M.Sugito</A>
 * @version	Ver.0.8
 */
public class NavigatorViewContentProvider implements ITreeContentProvider, IBtsListener, Logging {

	/** 課題票一覧ビューのビューアオブジェクト */
	private TreeViewer viewer;


	/**
	 * 当該リスナインスタンスををプラグインに登録します。
	 */
	public NavigatorViewContentProvider() {
		// 当該リスナインスタンスをプラグインに登録します。
		BtsPlugin.getInstance().addListener(this);
	}
	
	/*
	 *  (非 Javadoc)
	 * @see org.eclipse.jface.viewers.IContentProvider#inputChanged(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
	 */
	public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
		this.viewer= (TreeViewer)viewer;
    }

	/*
	 *  (非 Javadoc)
	 * @see org.eclipse.jface.viewers.IStructuredContentProvider#getElements(java.lang.Object)
	 */
    public Object[] getElements(Object rootNode) {
        List project = ((BtsTreeNode) rootNode).getChildren();
        return project.toArray();
    }

    /*
     *  (非 Javadoc)
     * @see org.eclipse.jface.viewers.ITreeContentProvider#getChildren(java.lang.Object)
     */
    public Object[] getChildren(Object parent) {
        List children = ((BtsTreeNode) parent).getChildren();
        return children.toArray();
    }

    /*
     *  (非 Javadoc)
     * @see org.eclipse.jface.viewers.ITreeContentProvider#hasChildren(java.lang.Object)
     */
    public boolean hasChildren(Object object) {
        return ((BtsTreeNode) object).hasChildren();
    }

    /*
     *  (非 Javadoc)
     * @see org.eclipse.jface.viewers.ITreeContentProvider#getParent(java.lang.Object)
     */
    public Object getParent(Object child) {
        return ((BtsTreeNode) child).getParent();
    }

    

	/**
	 * 当該リスナインスタンスをプラグインから削除します。
	 */
    public void dispose() {
		BtsPlugin.getInstance().removeListener(this);
    }

    /**
	 * 追加された課題票情報を{@link NavigatorView ナビゲータビュー}のツリーに反映させます。
	 * 
	 * @param		issue			更新された課題票
     * @see			jp.valtech.bts.ui.IBtsListener#addIssue(jp.valtech.bts.data.Issue)
     */
    public void addIssue(final Issue issue) {
		// テーブルビューアがnullの時は何もしない
		if(viewer == null ) {
			return ;
		}

		// メインスレッドで実行されるようにRunnableをメインスレッドに渡します。
		Control ctrl= viewer.getControl();
		ctrl.getDisplay().syncExec(new Runnable() {	
			public void run() {
				try {
					// カレントプロジェクトのSSIDを取得します。
					CurrentProject current = CurrentProject.getInsance();
					String ssid = current.getSSID();
					
					// ビューアからツリーオブジェクトを取得
					BtsTreeNode rootNode = (BtsTreeNode)viewer.getInput();
					ProjectNode projectNode = (ProjectNode)rootNode.getChild(ssid);

					// ノードカウントのインクリメント処理
					TreeHelper.incrementIssueCount(issue, projectNode);
					
					// ビュー表示を更新
					viewer.refresh(projectNode);

				} catch (Exception e) {
					String msg = Messages.getString("NavigatorViewContentProvider.0"); //$NON-NLS-1$
					logger.fatal(msg, e);
					BtsPlugin.getInstance().error(msg, e);
				}
			}
		});
	}
    

	/**
	 * 更新された課題票情報を{@link NavigatorView ナビゲータビュー}のツリーに反映させます。
	 * 
	 * @param		issue			更新された課題票
	 * @see			jp.valtech.bts.ui.IBtsListener#updateIssue(jp.valtech.bts.data.Issue)
	 */
	public void updateIssue(final Issue oldIssue, final Issue newIssue) {
		// テーブルビューアがnullの時は何もしない
		if(viewer == null ) {
			return ;
		}

		// メインスレッドで実行されるようにRunnableをメインスレッドに渡します。
		Control ctrl= viewer.getControl();
		ctrl.getDisplay().syncExec(new Runnable() {	
			public void run() {
				try {
					// カレントプロジェクトのSSIDを取得します。
					CurrentProject current = CurrentProject.getInsance();
					String ssid = current.getSSID();
					
					// ビューアからツリーオブジェクトを取得
					BtsTreeNode rootNode = (BtsTreeNode)viewer.getInput();
					ProjectNode projectNode = (ProjectNode)rootNode.getChild(ssid);

					// ノードカウントのデクリメント処理
					if(oldIssue != null) {
						TreeHelper.decrementIssueCount(oldIssue, projectNode);
					}

					// ノードカウントのインクリメント処理
					TreeHelper.incrementIssueCount(newIssue, projectNode);
					
					// ビュー表示を更新
					viewer.refresh(projectNode);

			
				} catch (Exception e) {
					String msg = Messages.getString("NavigatorViewContentProvider.1"); //$NON-NLS-1$
					logger.fatal(msg, e);
					BtsPlugin.getInstance().error(msg, e);
				}
			}
		});
	}
	
	

	/**
	 * 削除された課題票情報を{@link NavigatorView ナビゲータビュー}のツリーに反映させます。
	 * 
	 * @param		issue			削除された課題票
	 * @see			jp.valtech.bts.ui.IBtsListener#deleteIssue(jp.valtech.bts.data.Issue)
	 */
	 public void deleteIssue(final Issue deleteIssue) {
		// テーブルビューアがnullの時は何もしない
		if(viewer == null ) {
			return ;
		}
		
		// メインスレッドで実行されるようにRunnableをメインスレッドに渡します。
		Control ctrl= viewer.getControl();
		ctrl.getDisplay().syncExec(new Runnable() {	
			public void run() {
				try {
					// カレントプロジェクトのSSIDを取得します。
					CurrentProject current = CurrentProject.getInsance();
					String ssid = current.getSSID();
					
					// ビューアからツリーオブジェクトを取得
					BtsTreeNode rootNode = (BtsTreeNode)viewer.getInput();
					ProjectNode projectNode = (ProjectNode)rootNode.getChild(ssid);

					// ノードカウントのデクリメント処理
					TreeHelper.decrementIssueCount(deleteIssue, projectNode);

					// ビュー表示を更新
					viewer.refresh(projectNode);

			
				} catch (Exception e) {
					String msg = Messages.getString("NavigatorViewContentProvider.2"); //$NON-NLS-1$
					logger.fatal(msg, e);
					BtsPlugin.getInstance().error(msg, e);
				}
			}
		});
	}


	
	/**
	 * プロジェクトにJoinしてきたユーザを{@link NavigatorView Btsナビゲータビュー}に表示します。
	 * 
	 * @param		clientInfo		プロジェクトにJoinしてきたユーザ
	 */
	public void joinMember(final ClientInfo clientInfo) {
		// テーブルビューアがnullの時は何もしない
		if(viewer == null ) {
			return ;
		}
		
		// メインスレッドで実行されるようにRunnableをメインスレッドに渡します。
		Control ctrl= viewer.getControl();
		ctrl.getDisplay().syncExec(new Runnable() {
			public void run() {
				try {
					// カレントプロジェクトのSSIDを取得します。
					CurrentProject current = CurrentProject.getInsance();
					String ssid = current.getSSID();
					
					// ビューアからツリーオブジェクトを取得
					BtsTreeNode rootNode = (BtsTreeNode)viewer.getInput();

					// 「現在の参加者」をあらわすノードを取得
					ProjectNode projectNode = (ProjectNode)rootNode.getChild(ssid);
					MemberListNode memberListNode = projectNode.getMemberListNode();
					
					// プロジェクトオープン直後の同期処理ではここを通ってしまうので、NullPointerException対策
					if(memberListNode ==null) {
						return;
					}
					// Joinしてきたユーザ情報をツリーに加える
					MemberNode memberNode = (MemberNode)memberListNode.getChild(clientInfo.getClientAddress());
					
					// 既に対象のノードがある場合と無い場合で処理を変える
					if(memberNode == null) {

						// ツリーを生成して加える
						memberNode = new MemberNode(memberListNode, clientInfo);
						memberListNode.addChild(memberNode);
						// ビューを更新する
						viewer.add(memberListNode, memberNode);
						viewer.refresh(memberListNode);

					} else {
						// 表示名を更新する
						memberNode.setTextName(clientInfo.getClientName());
						viewer.refresh(memberListNode);
					}
					
				
				} catch (Exception e) {
					String msg = Messages.getString("NavigatorViewContentProvider.3"); //$NON-NLS-1$
					logger.fatal(msg, e);
					BtsPlugin.getInstance().error(msg, e);
				}
			}
		});
	}

	
	/**
	 * プロジェクトからLeaveしたユーザを{@link NavigatorView Btsナビゲータビュー}から削除します。
	 * 
	 * @param		clientInfo		プロジェクトからLeaveしたユーザ
	 */
	public void leaveMember(final ClientInfo clientInfo) {
		// テーブルビューアがnullの時は何もしない
		if(viewer == null ) {
			return ;
		}

		// メインスレッドで実行されるようにRunnableをメインスレッドに渡します。
		Control ctrl= viewer.getControl();
		ctrl.getDisplay().syncExec(new Runnable() {
			public void run() {
				try {

					// カレントプロジェクトのSSIDを取得します。
					CurrentProject current = CurrentProject.getInsance();
					String ssid = current.getSSID();
					
					// ビューアからツリーオブジェクトを取得
					BtsTreeNode rootNode = (BtsTreeNode)viewer.getInput();

					// 「現在の参加者」をあらわすノードを取得
					ProjectNode projectNode = (ProjectNode)rootNode.getChild(ssid);
					MemberListNode memberListNode = projectNode.getMemberListNode();
					
					// タイミングが悪いとnullの場合があるのでNullPointerException対策
					if(memberListNode ==null) {
						return;
					}

					// IPアドレスをキーにして対象のノードを取得する
					BtsTreeNode removeNode = memberListNode.getChild(clientInfo.getClientAddress());

					// ノードを取得できたときだけ処理する
					if(removeNode != null) {
						// ツリーからノードを削除する
						memberListNode.removeChild(removeNode);
						
						// ビューを更新する
						viewer.remove(removeNode);
						viewer.refresh(memberListNode);
					}
				
				} catch (Exception e) {
					String msg = Messages.getString("NavigatorViewContentProvider.4"); //$NON-NLS-1$
					logger.fatal(msg, e);
					BtsPlugin.getInstance().error(msg, e);
				}
			}
		});
	}

	
	/*
	 *  (非 Javadoc)
	 * @see jp.valtech.bts.ui.IBtsListener#addMessage(jp.valtech.bts.data.MessagePacket)
	 */
	public void addMessage(MessagePacket messagePacket) {
		// 何もしない
	}
	
	
	/*
	 *  (非 Javadoc)
	 * @see jp.valtech.bts.ui.IBtsListener#deleteMessage(jp.valtech.bts.data.MessagePacket)
	 */
	public void deleteMessage(MessagePacket messagePacket) {
		// 何もしない
	}
	
	
	/**
	 * 現在のローカルDBの情報で課題票一覧ビューを再表示します。
	 * 再表示する前に一旦一覧表示をクリアします。その後最新の情報を表示します。
	 * 
	 * @param		issueList				課題票を格納したリスト
	 */
	public void refresh(final List issueList) {
		// テーブルビューアがnullの時は何もしない
		if(viewer == null ) {
			return ;
		}
		
		// 事前にクライアントリストをクリアする。
		ClientList.clear();
		ClientList.addMyInfo();
		
		// メインスレッドで実行されるようにRunnableをメインスレッドに渡します。
		Control ctrl= viewer.getControl();
		ctrl.getDisplay().syncExec(new Runnable() {
			public void run() {
				try {
					// 課題票のListを配列に変換する
					Issue[] issues = (Issue[])issueList.toArray(new Issue[issueList.size()]);

					// カレントプロジェクト情報取得
					CurrentProject current = CurrentProject.getInsance();

					// ツリー生成
					TreeBuilder treeBuilder = new TreeBuilder();
					ProjectConfig[] projectConfig = ProjectConfigManager.getProjectConfigs();
					for (int i = 0; i < projectConfig.length; i++) {
						
						// プロジェクトノード生成
						String ssid = projectConfig[i].getSsid();
						String name = projectConfig[i].getProjectName();
						ProjectNode projectNode = treeBuilder.addProjectNode(name, ssid);

						// ここはカレントプロジェクトの場合のみの処理
						if(current.isCurrentProject(ssid)) {
							// プロジェクトをオープンに設定する。
							projectNode.open();
							
							// 課題票ノード追加（配布済み・下書き保存・競合・ゴミ箱）
							treeBuilder.addIssueNode(ssid, issues);

							// 参加者ノード追加
							treeBuilder.addMemberListNode(ssid, ClientList.getClientInfoList());
						}
					}

					// ツリー表示更新
					viewer.setInput((Object)treeBuilder.getRootNode());

					// (以降参加者リストの最新表示)
					
					//　最新の接続ユーザを知るためにJoinコマンドを出す。
					Join join = new Join( new NetworkConfig());
					join.execute();

				} catch (Exception e) {
					String msg = Messages.getString("NavigatorViewContentProvider.5"); //$NON-NLS-1$
					logger.fatal(msg, e);
					BtsPlugin.getInstance().error(msg, e);
				}
			}
		});
	}
	
	/*
	 *  (非 Javadoc)
	 * @see jp.valtech.bts.ui.IBtsListener#clear()
	 */
	public void clear() {
		// 何もしない
		;
	}
}
