/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.ui.navigator;

import java.util.HashMap;
import java.util.Map;

import jp.valtech.bts.data.Issue;
import jp.valtech.bts.data.IssueType;
import jp.valtech.bts.network.ClientInfo;

/**
 * ツリー構造を生成します。
 * {@link jp.valtech.bts.ui.navigator.NavigatorView}に表示するツリーを生成します。
 * 
 * @author		<A href="mailto:m_sugitou@valtech.jp">M.Sugito</A>
 * @version	Ver.0.8
 */
public class TreeBuilder {

	/** ツリーのルートノード */
	private BtsTreeNode rootNode = new BtsTreeNode();
	
	/** プロジェクトノード */
	private Map project = new HashMap();
	
	/**
	 * 何もしません。
	 */
	public TreeBuilder() {
		;
	}

	/**
	 * プロジェクトノードをツリーに追加します。
	 * 
	 * @param		projectName			プロジェクト名
	 * @param		SSID				プロジェクトのSSID
	 */
	public ProjectNode addProjectNode(String projectName, String ssid) {
		// プロジェクトノードの生成
		ProjectNode projectNode = new ProjectNode(rootNode, projectName, ssid);

		// ルートノードにプロジェクトノードを追加します。
    	rootNode.addChild(projectNode);
    	project.put(ssid, projectNode);
    	
    	return projectNode;
	}
	
	
	/**
	 * 課題票ノードをツリーに追加します。
	 * 課題票は種別分類されます。<br>
	 * <br>
	 * ゴミ箱課題の場合にはその課題票をゴミ箱ノードに加えます。<br>
	 * 「ゴミ箱課題」とは{@link Issue#getType() 課題票種別}が{@link IssueType#GARBAGE_VALUE}のものです。<br>
	 * <br>
	 * 下書き保存課題の場合にはその課題票を下書きノードに加えます。<br>
	 * 「下書き保存課題」とは{@link Issue#getType() 課題票種別}が{@link IssueType#DRAFT_VALUE}のものです。<br>
	 * <br>
	 * 競合課題の場合にはその課題票を下書きノードに加えます。<br>
	 * 「競合課題」とは{@link Issue#getType() 課題票種別}が{@link IssueType#CONFLICT_VALUE}のものです。<br>
	 * <br>
	 * 上記以外の課題票は、通常の課題票としカテゴリノードに加えます。カテゴリノードが無い場合は新規に生成します。<br>
	 * また、カテゴリノードは「配布ずみノード」の下に生成されます。
	 * 「通常の課題」とは{@link Issue#getType() 課題票種別}が{@link IssueType#RELEASE_VALUE}のものです。<br>
	 * 
	 * @param		ssid				プロジェクトのSSID
	 * @param		issueList			課題票一覧
	 */
	public void addIssueNode(String ssid, Issue[] issueList) {

		// プロジェクトノードを取得
		ProjectNode projectNode = (ProjectNode)project.get(ssid);
		if(projectNode == null) {
			// 一致するプロジェクト名が無い場合は例外を投げる
			throw new IllegalArgumentException("Invalid Project SSID : " + ssid);
		}

		// 配布済みきノード生成
		ReleaseNode releaseNode = new ReleaseNode(projectNode);
		projectNode.addChild(releaseNode);
		
		// 下書きノード生成
		DraftNode draftNode = new DraftNode(projectNode);
		projectNode.addChild(draftNode);

		// コンフリクトノード追加
		ConflictNode confrictNode = new ConflictNode(projectNode);
		projectNode.addChild(confrictNode);
		
		// ゴミ箱ノード生成
		GarbageBoxNode garbageBoxNode = new GarbageBoxNode(projectNode); 
		projectNode.addChild(garbageBoxNode);

		// 課題票が無い場合は以降の処理はやらない 
		if(issueList == null) {
			return;
		}
		
		// 課題票ツリーを生成する。課題票はカテゴリごとに分類する。
		for (int idx = 0; idx < issueList.length; idx++) {
			
			// ゴミ箱の課題を追加
			if( IssueType.GARBAGE_VALUE.equals(issueList[idx].getType()) ){
				
				// 課題票ノードのをゴミ箱ノードに追加
				garbageBoxNode.incrementChildCount();
			}
			
			// 下書き保存の課題を追加
			else if( IssueType.DRAFT_VALUE.equals(issueList[idx].getType()) ){
				
				// 課題票ノードのを下書きノードに追加
				draftNode.incrementChildCount();
			}
			
			// 競合ノード追加
			else if(IssueType.CONFLICT_VALUE.equals(issueList[idx].getType())) {
				// 課題票ノードをコンフリクトノードに追加
				confrictNode.incrementChildCount();
			}

			// カテゴリノード追加
			else {
				String category = issueList[idx].getCategory(); 
	
				// 取得したカテゴリのノードが既に存在するかどうか調べる。
				CategoryNode ctgryNode = releaseNode.getCategoryNode(category);
				if(ctgryNode == null) {
					// 新しいカテゴリの場合はカテゴリノードを作成する。
					ctgryNode = new CategoryNode(releaseNode, category);
					releaseNode.addChild(ctgryNode);
				}
				
				// 課題票ノードをカテゴリノードに追加
				ctgryNode.incrementChildCount();
			}
		}
	}
	
	
	/**
	 * 参加者ノードをツリーに追加します。
	 * 
	 * @param		ssid				プロジェクトのSSID
	 * @param		memberList			参加者リスト
	 */
	public void addMemberListNode(String ssid, ClientInfo[] clientInfoList){

		// プロジェクトノードを取得
		ProjectNode projectNode = (ProjectNode)project.get(ssid);
		if(projectNode == null) {
			// 一致するプロジェクト名が無い場合は例外を投げる
			throw new IllegalArgumentException("Invalid Project SSID : " + ssid);
		}
		
		// メンバリストノード追加
		MemberListNode memberListNode = new MemberListNode(projectNode);
		projectNode.addChild(memberListNode);
		
		// メンバノード追加
		for (int idx = 0; idx < clientInfoList.length; idx++) {
			memberListNode.addChild(new MemberNode(memberListNode, clientInfoList[idx]));
		}
	}
	
	
	/**
	 * 構築したツリーのルートノードを取得します。
	 * 
	 * @return		ルートノード
	 */
	public BtsTreeNode getRootNode() {
		return rootNode;
	}
}
