/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.util;

import java.awt.Color;
import java.awt.Font;
import java.io.File;
import java.io.IOException;
import java.sql.Timestamp;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import jp.valtech.bts.data.CurrentProject;
import jp.valtech.bts.data.Issue;
import jp.valtech.bts.data.IssueAttribute;
import jp.valtech.bts.data.IssueHistory;
import jp.valtech.bts.data.IssueStatus;
import jp.valtech.bts.facade.IssueHistoryFacade;
import jp.valtech.bts.ui.BtsPlugin;

import org.jfree.chart.ChartFactory;
import org.jfree.chart.ChartUtilities;
import org.jfree.chart.JFreeChart;
import org.jfree.chart.axis.DateAxis;
import org.jfree.chart.axis.DateTickUnit;
import org.jfree.chart.axis.NumberAxis;
import org.jfree.chart.axis.TickUnits;
import org.jfree.chart.plot.XYPlot;
import org.jfree.chart.title.LegendTitle;
import org.jfree.chart.title.TextTitle;
import org.jfree.data.time.Day;
import org.jfree.data.time.TimeSeries;
import org.jfree.data.time.TimeSeriesCollection;
import org.jfree.ui.HorizontalAlignment;
import org.jfree.ui.RectangleEdge;

/**
 * 統計グラフダイアログを開くアクションです。
 * 
 * @author		<A href="mailto:s_imai@valtech.jp">S.Imai</A>
 * @version	Ver.0.8
 */
public class IssueChartRender {

	/** バグ累計グラフ生成用課題票リスト */
	private List chartIssueList;
	
	/** 最小日付タイムスタンプ */
	private Timestamp minTimestamp;
	
	/** 最大日付タイムスタンプ */
	private Timestamp maxTimestamp;
	
	/**  日毎の発生バグ累計カウント */
	private int[] updateOpenTotal;
	
	/**  日毎の解決バグ累計カウント */
	private int[] updateCloseTotal;
	
	
	/**
	 * バグ累計グラフ生成用の課題票リストを設定します。
	 * 
	 * @param chartIssueList		バグ累計グラフ生成用の課題票リスト
	 */
	public IssueChartRender(List chartIssueList) {
		this.chartIssueList = chartIssueList;
	}
	
	
	/**
	 * バグ累計グラフダイアログを開きます。
	 */
	public void createChart() {
		// 日毎の発生バグと解決バグをカウント
		countBug();
		
		// カウントをもとにグラフをPNGで出力
		createImage();
	}
	
	
	/**
	 * 日毎の発生バグと解決バグをカウントします。
	 *
	 */
	private void countBug() {
		
		// 最小日付の初期値を設定
		minTimestamp = new Timestamp(9999999999999L);
		
		// 最大日付の初期値を設定
		maxTimestamp = new Timestamp(0L);
		
		// グラフを生成するための更新履歴リスト取得
		List bugChartList = getIssueHistoryForChart();
		
		// 最小日付をカレンダにセット
    	Calendar calendarMin = Calendar.getInstance();
    	calendarMin.setTimeInMillis(minTimestamp.getTime());
    	
    	// 最大日付をカレンダにセット
    	Calendar calendarMax = Calendar.getInstance();
    	calendarMax.setTimeInMillis(maxTimestamp.getTime());
    	
    	// 最小日付の年を取得
    	int minYear = calendarMin.get(Calendar.YEAR);
    	// 最大日付の年を取得
    	int maxYear = calendarMax.get(Calendar.YEAR);
    	
		// 最小日付の日
    	int minDay = calendarMin.get(Calendar.DAY_OF_YEAR);
    	// 最大日付の日
    	int maxDay = calendarMax.get(Calendar.DAY_OF_YEAR);
    	
		// バグ累計カウント用配列の要素数
		int argsNumMax;
    	
    	// バグ累計カウント用の配列要素数を計算
    	argsNumMax = (maxYear - minYear) * 365 + maxDay - minDay + 1;
		
		// 日毎の発生バグ累計カウント用の配列生成
		updateOpenTotal = new int[argsNumMax];
		
		// 日毎の解決バグ累計カウント用の配列生成
		updateCloseTotal = new int[argsNumMax];
		
		// カレンダインスタンス生成
		Calendar calendar = Calendar.getInstance();
		
		if(bugChartList==null || bugChartList.isEmpty()) {
			return;
		}
		
		IssueHistory[] bugCharts = (IssueHistory[])bugChartList.toArray(new IssueHistory[0]);
		
		
		for(int i=0; i<bugCharts.length; i++) {
			
			// 更新タイムスタンプ（long）取得
			long updateDateLong = bugCharts[i].getUpdateDateTimestamp();
			
			// カレンダに更新タイムスタンプをセット
			calendar.setTimeInMillis(updateDateLong);
			
			// 更新日の年を取得
			int updateYear = calendar.get(Calendar.YEAR);
			
			// 配列の要素番号を計算
			int argsNum = (updateYear - minYear) * 365 + calendar.get(Calendar.DAY_OF_YEAR) - minDay;
			
			// 「起票」の場合
			if( IssueHistory.ISSUE_INITIAL_REGIST.equals( bugCharts[i].getUpdateAttribute() ) ){
				
				// 発生バグカウント＋１
				while(argsNum < updateOpenTotal.length) {
					updateOpenTotal[argsNum] += 1;
					argsNum++;
				}

			}
			// 「完了（閉じる）」の場合
			else {
				
				// 解決バグカウント＋１
				while(argsNum < updateOpenTotal.length) {
					updateCloseTotal[argsNum] += 1;
					argsNum++;
				}
			}
		}
	}
	
	
	/**
	 * カウントをもとにグラフをPNGで出力します。
	 */
	private void createImage() {
	
		// カテゴリ生成
		TimeSeries openTimeSeries = new TimeSeries(Messages.getString("IssueChartRender.6"), Day.class); //$NON-NLS-1$
		TimeSeries closeTimeSeries = new TimeSeries(Messages.getString("IssueChartRender.7"), Day.class); //$NON-NLS-1$
		TimeSeries remainderTimeSeries = new TimeSeries(Messages.getString("IssueChartRender.8"), Day.class); //$NON-NLS-1$
		
		// カレンダに最小日付をセット
    	Calendar calendar = Calendar.getInstance();
    	calendar.setTimeInMillis(minTimestamp.getTime());
		
		for(int i=0; i<updateOpenTotal.length; i++) {

			// 時系列用の日付を設定
			Day day = new Day(calendar.get(Calendar.DAY_OF_MONTH), calendar.get(Calendar.MONTH)+1, calendar.get(Calendar.YEAR));
			
			// 日数+1
			calendar.add(Calendar.DAY_OF_MONTH, 1);
			
			// 発生バグ累計に値をセット
			openTimeSeries.add(day, updateOpenTotal[i]);
			
			// 解決バグ累計に値をセット
			closeTimeSeries.add(day, updateCloseTotal[i]);
			
			// 残存バグ累計に値をセット
			remainderTimeSeries.add(day, updateOpenTotal[i] - updateCloseTotal[i]);
			
		}
		
		// データセットに値をセット
		TimeSeriesCollection dataset = new TimeSeriesCollection();
		dataset.addSeries(openTimeSeries);
		dataset.addSeries(closeTimeSeries);
		dataset.addSeries(remainderTimeSeries);

		// JFreeChartオブジェクトの生成
		JFreeChart chart = ChartFactory.createTimeSeriesChart(null, null, Messages.getString("IssueChartRender.9"), dataset, true, true, false); //$NON-NLS-1$
		
		// グラフ外枠の背景色を設定(#FFFFFF)
		chart.setBackgroundPaint(new Color(255, 255, 255));
		
		// プロジェクト名取得
		String projectName = CurrentProject.getInsance().getProjectConfig().getProjectName();
		
		// 作成日時取得
		String nowDate = BTSUtility.formatDate(new Date());
		
		// プロジェクト名を設定
		TextTitle projectNameTitle = new TextTitle(Messages.getString("IssueChartRender.10") + projectName); //$NON-NLS-1$
		projectNameTitle.setFont(new Font("SansSerif", 2, 11));
		projectNameTitle.setHorizontalAlignment(HorizontalAlignment.LEFT);
		chart.setTitle(projectNameTitle);
		
		// グラフタイトルを設定
		TextTitle textTitle = new TextTitle(Messages.getString("IssueChartRender.11")); //$NON-NLS-1$
		textTitle.setFont(new Font("SansSerif", 1, 16));
		textTitle.setHorizontalAlignment(HorizontalAlignment.CENTER);
		chart.addSubtitle(textTitle);
		
		// 作成日時を設定
		TextTitle nowDateTitle = new TextTitle(Messages.getString("IssueChartRender.12") + nowDate); //$NON-NLS-1$
		nowDateTitle.setFont(new Font("SansSerif", 2, 11));
		nowDateTitle.setHorizontalAlignment(HorizontalAlignment.RIGHT);
		chart.addSubtitle(nowDateTitle);
		
		// 横軸の目盛り線を非表示
		XYPlot xyplot = (XYPlot) chart.getPlot();
		xyplot.setDomainGridlinesVisible(false);
		
		// 横軸の日付フォーマット設定
		DateAxis dateaxis = (DateAxis) xyplot.getDomainAxis();
		
		TickUnits tickUnits = new TickUnits();
		
		// 横軸の表示設定を追加
		tickUnits.add(new DateTickUnit(DateTickUnit.DAY, 1, new SimpleDateFormat("M/d")));
		tickUnits.add(new DateTickUnit(DateTickUnit.DAY, 7, new SimpleDateFormat("M/d")));
		tickUnits.add(new DateTickUnit(DateTickUnit.MONTH, 1, new SimpleDateFormat("yyyy/M")));
		tickUnits.add(new DateTickUnit(DateTickUnit.MONTH, 3, new SimpleDateFormat("yyyy/M")));
		tickUnits.add(new DateTickUnit(DateTickUnit.MONTH, 6, new SimpleDateFormat("yyyy/M")));
		
		// 表示設定をセット
		dateaxis.setStandardTickUnits(tickUnits);
		
		// 表示設定を自動的に選択する
		dateaxis.setAutoTickUnitSelection(true);
		
		// 縦軸を整数単位で表示
		NumberAxis axis = (NumberAxis) xyplot.getRangeAxis();
		axis.setStandardTickUnits(NumberAxis.createIntegerTickUnits());
		
		// カテゴリを右端に設定
		LegendTitle legendtitle = (LegendTitle) chart.getSubtitle(0);
		legendtitle.setPosition(RectangleEdge.RIGHT);
		
		// カレントディレクトリを指定。「ワークスペースの.metadata/.plutins/jp.valtech.bts/」を設定する。
		String currentDir = BtsPlugin.getInstance().getStateLocation() + "/"; 
		// ファイル出力
		File outputFile = new File(currentDir + "bug.png");
		try {
			// PNGで出力
			ChartUtilities.saveChartAsPNG(outputFile, chart, 600, 300);
		} catch (IOException e) {
			BtsPlugin.getInstance().errorlog(e);
		}
		
	}
	
	
	/**
	 * 「起票」または「完了(閉じる)」の更新履歴リストを取得します。
	 * また、更新日の最大と最小を求めます。
	 * 
	 * @return		「起票」または「完了(閉じる)」の更新履歴リスト
	 */
	private List getIssueHistoryForChart() {	


		IssueHistoryFacade facade = new IssueHistoryFacade();

		// リスト生成
		List issueHistorys = new ArrayList();

		String targetAttribute = IssueAttribute.STATUS.getDescription();
		
		for(int i=0; i<chartIssueList.size(); i++) {
			
			// フィルタを通過した課題票を取得
			Issue issue = ((Issue)chartIssueList.get(i));
			
			// フィルタを通過した課題票の更新履歴を取得
			IssueHistory[] filterIssueHistories 
				= facade.getByFingerPrint(issue.getFingerPrint(), issue.getType());

			for(int j=0; j<filterIssueHistories.length; j++) {
				
				// 更新した属性名取得
				String updateAttribute = filterIssueHistories[j].getUpdateAttribute();

				// 更新後の属性取得
				String updateAfter = filterIssueHistories[j].getUpdateAfter();
				
				// 「起票」または「完了(閉じる)」の更新履歴を抽出
				if (IssueHistory.ISSUE_INITIAL_REGIST.equals(updateAttribute) 
						|| targetAttribute.equals(updateAttribute) 
								&& IssueStatus.CLOSED.getDescription().equals(updateAfter)) {

					// リストに追加
					issueHistorys.add(filterIssueHistories[j]);

					// 時間切捨て
					Timestamp updateDateTimestamp 
							= new Timestamp(filterIssueHistories[j].getUpdateDateTimestamp());

					// 最小日付タイムスタンプ更新
					if (minTimestamp.compareTo(updateDateTimestamp) > 0) {
						minTimestamp = updateDateTimestamp;
					}

					// 最大日付タイムスタンプ更新
					if (maxTimestamp.compareTo(updateDateTimestamp) < 0) {
						maxTimestamp = updateDateTimestamp;
					}

				}
				
			}
			
		}
		
		facade.close();
		return issueHistorys;
	}
	
}
