/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.ui.issuelist;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;

import jp.valtech.bts.data.Issue;
import jp.valtech.bts.data.IssueStatus;
import jp.valtech.bts.data.IssueType;
import jp.valtech.bts.ui.BtsPlugin;

import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerFilter;
import org.eclipse.swt.widgets.Table;

/**
 * {@link IssueListView 課題票一覧テーブル}のフィルタークラスです。
 * 
 * @author		<A href="mailto:s_imai@valtech.jp">S.Imai</A>
 * @version	Ver.0.8
 */
public class IssueListViewFilter {

	/** 課題票一覧ビュー */
	private IssueListView view;
	
	/** 課題票一覧テーブルのビューアインスタンス */
	private TableViewer viewer;
	
	/** 現在のフィルター情報 */
	private IDialogSettings section
		= BtsPlugin.getInstance().getDialogSettings().getSection(IssueFilterManager.SECTION_NAME);

	/** 
	 * {@link IssueListView 課題票一覧テーブル}のテーブルビューアを設定します
	 * 
	 * @parem		view			課題票一覧ビュー
	 * @param		viewer			課題票一覧テーブルのテーブルビューア
	 */
	public IssueListViewFilter(IssueListView view){
		this.view = view;
		this.viewer = view.getViewer();
	}
	
	
	/**
	 * 課題票フィルタを追加します。
	 */
	public void addIssueFilters() {
		
		// セクションが空の場合
		if(section == null) {
			return;
		}

		// 最初にフィルタをリセット
//		viewer.resetFilters();

		// TreeSelectionFilter以外のフィルタ設定を消す
		ViewerFilter[] filters = viewer.getFilters();
		if(filters != null) {
			for (int i = 0; i < filters.length; i++) {
				// TreeSelectionFilter以外のインスタンスを探す
				if((filters[i] instanceof TreeSelectionFilter) ==false) {
					viewer.removeFilter(filters[i]);
				}
			}
		}
		
		// 課題票IDテキストが空ではない場合
		if(!"".equals(section.get(IssueFilterManager.KEY_ISSUEID))) {
			
			// 課題票IDフィルタを実行
			viewer.addFilter(new IssueIdFilter());
		}
		
		// 起票日テキストが両方とも空ではない場合
		if(!"".equals(section.get(IssueFilterManager.KEY_FROM)) && !"".equals(section.get(IssueFilterManager.KEY_TO))) {
			
			// 課題票(from)フィルタ実行
			viewer.addFilter(new DateFromFilter());
			
			// 課題票(to)フィルタ実行
			viewer.addFilter(new DateToFilter());
			
		} else if(!"".equals(section.get(IssueFilterManager.KEY_FROM)) && "".equals(section.get(IssueFilterManager.KEY_TO))) {

			// 起票日(from)のみ入力時　課題票(from)フィルタ実行
			viewer.addFilter(new DateFromFilter());
			
		} else if("".equals(section.get(IssueFilterManager.KEY_FROM)) && !"".equals(section.get(IssueFilterManager.KEY_TO))) {
			
			// 起票日(to)のみ入力時　課題票(to)フィルタ実行
			viewer.addFilter(new DateToFilter());
		}
		
		// タイトルテキストが空ではない場合
		if(!"".equals(section.get(IssueFilterManager.KEY_TITLE))) {
			
			// タイトルフィルタを実行
			viewer.addFilter(new TitleFilter());
		}
		
		// 詳細テキストが空ではない場合
		if(!"".equals(section.get(IssueFilterManager.KEY_DESC))) {
			
			// 詳細フィルタを実行
			viewer.addFilter(new DescFilter());
		}
		
		// 担当者チェックが1つ以上外れている場合
		if(!"".equals(section.get(IssueFilterManager.KEY_ASSIGNED))) {
			
			// チェックが外れている担当者取得
			String[] assignedNames = section.get(IssueFilterManager.KEY_ASSIGNED).split("\\|");
			
			for(int i=0; i<assignedNames.length; i++) {
				
				// 担当者フィルタを実行
				viewer.addFilter(new AssignedFilter(assignedNames[i]));
			}
		}
		
		// 優先度チェックが1つ以上外れている場合
		if(!"".equals(section.get(IssueFilterManager.KEY_PRIORITY))) {
			
			// チェックが外れている優先度を取得
			String[] priorityNames = section.get(IssueFilterManager.KEY_PRIORITY).split("\\|");
			
			for(int i=0; i<priorityNames.length; i++) {
				
				// 優先度フィルタを実行
				viewer.addFilter(new PriorityFilter(priorityNames[i]));
			}
		}
		
		// 「起票」チェックが外れている場合
		if(!section.getBoolean(IssueFilterManager.KEY_STOPEN)) {
			
			// ステータスフィルタ実行
			viewer.addFilter(new StatusFilter(IssueStatus.OPENED_VALUE));
		}
		
		// 「担当者割り当て済み」チェックが外れている場合
		if(!section.getBoolean(IssueFilterManager.KEY_STASSIGNED)) {
			
			// ステータスフィルタ実行
			viewer.addFilter(new StatusFilter(IssueStatus.ASSIGNED_VALUE));
		}
		
		// 「対応済み」チェックが外れている場合
		if(!section.getBoolean(IssueFilterManager.KEY_STRESOLVED)) {
			
			// ステータスフィルタ実行
			viewer.addFilter(new StatusFilter(IssueStatus.RESOLVED_VALUE));
		}
		
		// 「確認済み」チェックが外れている場合
		if(!section.getBoolean(IssueFilterManager.KEY_STVERIFIED)) {
			
			// ステータスフィルタ実行
			viewer.addFilter(new StatusFilter(IssueStatus.VERIFIED_VALUE));
		}
		
		// 「完了」チェックが外れている場合
		if(!section.getBoolean(IssueFilterManager.KEY_STCLOSED)) {
			
			// ステータスフィルタ実行
			viewer.addFilter(new StatusFilter(IssueStatus.CLOSED_VALUE));
		}
		
		// 「内緒のバグを表示する」チェックが付いていない場合
		if(!section.getBoolean(IssueFilterManager.KEY_SECRET)) {
			
			// 内緒のバグフィルタを実行
			viewer.addFilter(new SecretBugFilter());
		}
		
		// 「保留中の課題を表示する」チェックが付いていない場合
		if(!section.getBoolean(IssueFilterManager.KEY_RESERVE)) {
			
			// 保留中の課題フィルタを実行
			viewer.addFilter(new ReserveIssueFilter());
		}
		
		// 「完了期限を過ぎた課題を表示する」チェックが付いていない場合
		if(!section.getBoolean(IssueFilterManager.KEY_DEADLINE)) {
			
			// 完了期限フィルタを実行
			viewer.addFilter(new DeadLineFilter());
		}
		
		// 表示件数更新
		Table table = viewer.getTable();
		view.updateDescription(table.getItemCount());
	}
	
	/**
	 * カテゴリフィルタを設定します。
	 * 指定のカテゴリのみを一覧表示の対象にします。
	 *<br> 
	 * カテゴリフィルタを設定する前に{@link TreeSelectionFilter}のフィルタ設定のみ一旦クリアします。
	 * 
	 * @param		category		カテゴリ名
	 */
	public void setCategoryFilter(String category) {
		
		// 一旦TreeSelectionFilterのみをクリア
		clearTreeSelectionFilter();
		
		// IssueTypeFilterフィルターで「配布済み」のみを表示するよう設定する
		viewer.addFilter(new IssueTypeFilter(IssueType.RELEASE_VALUE));
		
		// Categoryフィルターを設定する
		viewer.addFilter(new CategoryFilter(category));

		// 表示件数更新
		Table table = viewer.getTable();
		view.updateDescription(table.getItemCount());
	}


	/**
	 * 指定の課題票種別でフィルタを設定します。
	 *<br> 
	 * 課題票種別でフィルタを設定する前に{@link TreeSelectionFilter}のフィルタ設定のみ一旦クリアします。
	 * 
	 * @param		issueType		課題票種別
	 */
	public void setIssueTypeFilter(String issueType) {
		// 一旦TreeSelectionFilterのみをクリア
		clearTreeSelectionFilter();
		
		// IssueTypeFilterフィルタを設定する
		viewer.addFilter(new IssueTypeFilter(issueType));
		
		// 表示件数更新
		Table table = viewer.getTable();
		view.updateDescription(table.getItemCount());
	}

	
	/**
	 * 一覧表示を０件にします。
	 * フィルター機能を使って一覧表示を０件にします。
	 */
	public void setClearFilter() {
		// 一旦TreeSelectionFilterのみをクリア
		clearTreeSelectionFilter();
		
		// Clearフィルタを設定する
		viewer.addFilter(new ClearFilter());
		
		// 表示件数更新
		view.updateDescription(0);
	}
	
	
	/**
	 * 課題票IDフィルタを設定します。
	 * 
	 * @author		<A href="mailto:s_imai@valtech.jp">$Author$</A>
	 * @version		$Rev$
	 */
	private class IssueIdFilter extends ViewerFilter {

		/* (非 Javadoc)
		 * @see org.eclipse.jface.viewers.ViewerFilter#select(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
		 */
		public boolean select(Viewer viewer, Object parentElement, Object element) {
			// 指定された課題票IDのみ表示する
			return section.get(IssueFilterManager.KEY_ISSUEID).equals(((Issue)element).getDisplayIssueID());
		}
		
	}
	
	/**
	 * 起票日(from)フィルタを設定します。
	 * 起票日(from)に入力された日付より後に起票された課題票のみ表示します。
	 * 
	 * @author		<A href="mailto:s_imai@valtech.jp">$Author$</A>
	 * @version		$Rev$
	 */
	private class DateFromFilter extends ViewerFilter {
		
		/* (非 Javadoc)
		 * @see org.eclipse.jface.viewers.ViewerFilter#select(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
		 */
		public boolean select(Viewer viewer, Object parentElement, Object element) {
			
			// fromテキストが空の場合は無視
			if("".equals(section.get(IssueFilterManager.KEY_FROM))) {
				return true;
			}
			
			// フォーマット設定
			SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy/MM/dd");
			Date fromDate = null;
			
			try {
				// 入力された起票日(to)のタイムスタンプ生成
				fromDate = new Date(dateFormat.parse(section.get(IssueFilterManager.KEY_FROM)).getTime());
			} catch (ParseException e) {
				BtsPlugin.getInstance().errorlog(e);
			}
			
			// 課題票のタイムスタンプ取得
			Date timestamp = new Date(((Issue)element).getCreateDateTimestamp());
			
			// 起票日(from)に入力された日付より後に起票された課題票のみ表示
			return timestamp.after(fromDate);

		}
		
	}
	
	/**
	 * 起票日(to)フィルタを設定します。
	 * 起票日(from)に入力された日付より前に起票された課題票のみ表示します。
	 * 
	 * @author		<A href="mailto:s_imai@valtech.jp">$Author$</A>
	 * @version		$Rev$
	 */
	private class DateToFilter extends ViewerFilter {
		
		/* (非 Javadoc)
		 * @see org.eclipse.jface.viewers.ViewerFilter#select(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
		 */
		public boolean select(Viewer viewer, Object parentElement, Object element) {
			
			// toテキストが空の場合は無視
			if("".equals(section.get(IssueFilterManager.KEY_TO))) {
				return true;
			}
			
			// フォーマット設定
			SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy/MM/dd");
			Date toDate = null;
			
			try {
				// 入力された起票日(to)のタイムスタンプ生成（toDateは+1日）
				toDate = new Date(dateFormat.parse(section.get(IssueFilterManager.KEY_TO)).getTime() + 86400000);
			} catch (ParseException e) {
				BtsPlugin.getInstance().errorlog(e);
			}
			
			// 課題票のタイムスタンプ取得
			Date timestamp = new Date(((Issue)element).getCreateDateTimestamp());
			
			// 起票日(from)に入力された日付より前に起票された課題票のみ表示
			return timestamp.before(toDate);

		}
		
	}
	
	
	
	/**
	 * タイトル検索フィルタ設定します。
	 * 
	 * @author		<A href="mailto:s_imai@valtech.jp">$Author$</A>
	 * @version		$Rev$
	 */
	private class TitleFilter extends ViewerFilter {

		/* (非 Javadoc)
		 * @see org.eclipse.jface.viewers.ViewerFilter#select(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
		 */
		public boolean select(Viewer viewer, Object parentElement, Object element) {
			// 「タイトル」に指定された文字列が含まれている位置を調べる
			int index = ((Issue)element).getTitle().indexOf(section.get(IssueFilterManager.KEY_TITLE));
			// 「含む」が選択されている場合
			if(Messages.getString("IssueListViewFilter.0").equals(section.get(IssueFilterManager.KEY_TITLECOMBO))) { //$NON-NLS-1$
				return index != -1;
			} else {
				// 「含まない」が選択されている場合
				return index == -1;
			}
		}
		
	}
	
	/**
	 * 詳細検索フィルタを設定します。
	 * 
	 * @author		<A href="mailto:s_imai@valtech.jp">$Author$</A>
	 * @version		$Rev$
	 */
	private class DescFilter extends ViewerFilter {

		/* (非 Javadoc)
		 * @see org.eclipse.jface.viewers.ViewerFilter#select(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
		 */
		public boolean select(Viewer viewer, Object parentElement, Object element) {
			
			// 「詳細」文字列取得
			String desc = ((Issue)element).getDescription();
			
			// 「詳細」がnullの場合
			if(desc == null) {
				desc = "";
			}
			
			// 「詳細」に指定された文字列が含まれている位置を調べる
			int index = desc.indexOf(section.get(IssueFilterManager.KEY_DESC));
			
			// 「含む」が選択されている場合
			if(Messages.getString("IssueListViewFilter.1").equals(section.get(IssueFilterManager.KEY_DESCCOMBO))) { //$NON-NLS-1$
				return index != -1;
			} else {
				// 「含まない」が選択されている場合
				return index == -1;
			}
			
		}
		
	}
	
	/**
	 * 担当者フィルタを設定します。
	 * 
	 * @author		<A href="mailto:s_imai@valtech.jp">$Author$</A>
	 * @version		$Rev$
	 */
	private class AssignedFilter extends ViewerFilter {
		
		/** チェックが外れている担当者 */
		private String assignedName;
		
		/**
		 * チェックが外れている担当者を更新
		 * @param assignedName
		 */
		private AssignedFilter(String assignedName) {
			this.assignedName = assignedName;
		}

		/* (非 Javadoc)
		 * @see org.eclipse.jface.viewers.ViewerFilter#select(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
		 */
		public boolean select(Viewer viewer, Object parentElement, Object element) {
			// チェックが外れている担当者は表示しない
			return !assignedName.equals(((Issue)element).getAssigned());
		}
		
	}
	
	
	/**
	 * 優先度フィルタを設定します。
	 * 
	 * @author		<A href="mailto:s_imai@valtech.jp">$Author$</A>
	 * @version		$Rev$
	 */
	private class PriorityFilter extends ViewerFilter {
		
		/** チェックが外れている優先度 */
		private String priorityName;
		
		/**
		 * チェックが外れている優先度を更新
		 * @param priorityName
		 */
		private PriorityFilter(String priorityName) {
			this.priorityName = priorityName;
		}

		/* (非 Javadoc)
		 * @see org.eclipse.jface.viewers.ViewerFilter#select(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
		 */
		public boolean select(Viewer viewer, Object parentElement, Object element) {
			// チェックが外れている優先度は表示しない
			return !priorityName.equals(((Issue)element).getPriority());
		}
		
	}
	
	/**
	 * ステータスフィルタを設定します。
	 * 
	 * @author		<A href="mailto:s_imai@valtech.jp">$Author$</A>
	 * @version		$Rev$
	 */
	private class StatusFilter extends ViewerFilter {
		
		/** ステータス番号 */
		private String value;
		
		/**
		 * ステータス番号を更新
		 * @param value
		 */
		private StatusFilter(String value) {
			this.value = value;
		}

		/* (非 Javadoc)
		 * @see org.eclipse.jface.viewers.ViewerFilter#select(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
		 */
		public boolean select(Viewer viewer, Object parentElement, Object element) {
			// チェックが外れているステータスを表示しない
			return !value.equals(((Issue)element).getStatus());
		}
		
	}
	
	/**
	 * 内緒のバグフィルタを設定します。
	 * 
	 * @author		<A href="mailto:s_imai@valtech.jp">$Author$</A>
	 * @version		$Rev$
	 */
	private class SecretBugFilter extends ViewerFilter {

		/* (非 Javadoc)
		 * @see org.eclipse.jface.viewers.ViewerFilter#select(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
		 */
		public boolean select(Viewer viewer, Object parentElement, Object element) {
			// 内緒のバグは表示しない
			return !((Issue)element).isSecretBug().booleanValue();
		}
		
	}
	
	/**
	 * 保留中の課題フィルタを設定します。
	 * 
	 * @author		<A href="mailto:s_imai@valtech.jp">$Author$</A>
	 * @version		$Rev$
	 */
	private class ReserveIssueFilter extends ViewerFilter {

		/* (非 Javadoc)
		 * @see org.eclipse.jface.viewers.ViewerFilter#select(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
		 */
		public boolean select(Viewer viewer, Object parentElement, Object element) {
			// 保留中の課題は表示しない
			return !((Issue)element).isSuspend().booleanValue();
		}
		
	}
	
	/**
	 * 完了期限フィルタ
	 * 
	 * @author		<A href="mailto:s_imai@valtech.jp">$Author$</A>
	 * @version		$Rev$
	 */
	private class DeadLineFilter extends ViewerFilter {

		/* (非 Javadoc)
		 * @see org.eclipse.jface.viewers.ViewerFilter#select(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
		 */
		public boolean select(Viewer viewer, Object parentElement, Object element) {
			// 現在の日時のタイムスタンプ取得
			Date nowStamp = new Date();
			
			// 完了期限が設定されている場合
			if(((Issue)element).getDeadline() != null) {
				
				// 完了期限日時のタイムスタンプ取得（完了期限日+1日）
				Date deadLineStamp = new Date(((Issue)element).getDeadlineTimestamp() + 86400000);
				
				// 完了期限を過ぎた課題は表示しない
				return !nowStamp.after(deadLineStamp);
				
			}
			// 完了期限が設定されていない場合
			else {
				return true;
			}
		}
	}
	

	/**
	 * カテゴリフィルタを設定します。
	 * 
	 * @author		<A href="mailto:s_imai@valtech.jp">$Author$</A>
	 * @version		$Rev$
	 */
	private class CategoryFilter extends ViewerFilter implements TreeSelectionFilter  {
		
		/** チェックが外れているカテゴリ */
		private String categoryName;
		
		/**
		 * チェックが外れているカテゴリを更新
		 * @param categotyName
		 */
		private CategoryFilter(String categotyName) {
			this.categoryName = categotyName;
		}

		
		/* (非 Javadoc)
		 * @see org.eclipse.jface.viewers.ViewerFilter#select(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
		 */
		public boolean select(Viewer viewer, Object parentElement, Object element) {
			// 指定のカテゴリのみを表示する。
			return categoryName.equals(((Issue)element).getCategory());
		}
		
	}

	/**
	 * 課題票種別フィルタを設定します。
	 * 
	 * @author		<A href="mailto:s_imai@valtech.jp">$Author$</A>
	 * @version		$Rev$
	 */
	private class IssueTypeFilter extends ViewerFilter implements TreeSelectionFilter  {
		
		private String issueType;
		
		private IssueTypeFilter(String issueType) { 
			this.issueType = issueType;
		}

		/* (非 Javadoc)
		 * @see org.eclipse.jface.viewers.ViewerFilter#select(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
		 */
		public boolean select(Viewer viewer, Object parentElement, Object element) {
			
			// 課題票種別が「R」のみ表示する
			return issueType.equals(((Issue)element).getType());
			
		}
		
	}

	/**
	 * 表示を０件にするフィルタ。
	 *  
	 * @author		<A href="mailto:m_sugitou@valtech.jp">$Author$</A>
	 * @version		$Rev$
	 */
	private class ClearFilter extends ViewerFilter implements TreeSelectionFilter {

		/* (非 Javadoc)
		 * @see org.eclipse.jface.viewers.ViewerFilter#select(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
		 */
		public boolean select(Viewer viewer, Object parentElement, Object element) {
			// 無条件にfalseを返す。
			return false;
		}
		
	}
	
	/**
	 * 設定中のフィルタから{@link TreeSelectionFilter}のみクリアします。
	 *
	 */
	private void clearTreeSelectionFilter() {
		
		// 設定されているフィルターを取得
		ViewerFilter[] filters = viewer.getFilters();
		if(filters==null) {
			return;
		}
		
		// 取得した数だけ回してTreeSelectionFilterを探す
		for (int i = 0; i < filters.length; i++) {
			if(filters[i] instanceof TreeSelectionFilter) {
				// TreeSelectionFilterが見つかったら削除する
				viewer.removeFilter(filters[i]);
			}
		}
	}
	

	/**
	 * {@link jp.valtech.bts.ui.navigator.NavigatorView BTSナビゲータビュー}のノードクリックにて設定されるフィルターです。
	 * メソッドは持ちません。。 
	 * 
	 * @author		<A href="mailto:m_sugitou@valtech.jp">$Author$</A>
	 * @version		$Rev$
	 */
	private interface TreeSelectionFilter {
		
	}
}
