#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "snp_CalcScoreParallel.h"
#include "mpi.h"

#define NUM_TM   100
double tm[NUM_TM];  /* 時間計測用 */

#define TM_START       0
#define TM_B_BCAST     1
#define TM_A_BCAST     2
#define TM_B_CALC      3
#define TM_A_CALC      4
#define TM_B_GATHER    5
#define TM_A_GATHER    6
#define TM_END         7

#define TM_B_READ      8
#define TM_A_READ      9
#define TM_B_WRITE    10
#define TM_A_WRITE    11
#define TM_B_F_BCAST  12
#define TM_A_F_BCAST  13
#define TM_B_F_GATHER 14
#define TM_A_F_GATHER 15

/* プロトタイプ宣言 */
int Define_MPI_InputStruct(MPI_Datatype *mpi_inputStruct);


int main(int argc, char* argv[])
{
    InputScore inputScore={"", "", "", 0, "", 0, 0};

    int nProc = 0;   /* 確保したプロセッサの個数（並列度） */
    int rank = 0;    /* ランク番号 */
    int i = 0;

    /* 時間計測用変数の初期化 */
    for (i = 0; i < NUM_TM; i++){
        tm[i] = 0.0;
    }

    MPI_Init(&argc, &argv);
    MPI_Comm_size(MPI_COMM_WORLD, &nProc);
    MPI_Comm_rank(MPI_COMM_WORLD, &rank);

    /* 時間計測 */
    tm[TM_START] = MPI_Wtime();

    if(argc != 8){
        printf("[usage]main.exe [InputFile1] [InputFile2] [OutputFile] [AreaFileType] [BlockAreaFile] [Score] [dataType] \n");
        return 255;
    }
    strcpy(inputScore.inputFile1, argv[1]);
    strcpy(inputScore.inputFile2, argv[2]);
    strcpy(inputScore.outputFile1, argv[3]);
    inputScore.areaFileType = atoi(argv[4]);
    strcpy(inputScore.blockAreaFile, argv[5]);
    inputScore.score = atoi(argv[6]);
    inputScore.dataType = atoi(argv[7]);

    if (rank == 0) {
        printf("inputFile1: %s\n", inputScore.inputFile1);
        printf("inputFile2: %s\n", inputScore.inputFile2);
        printf("outputFile1: %s\n", inputScore.outputFile1);
        printf("blockAreaFile: %s\n", inputScore.blockAreaFile);
        printf("score: %d\n", inputScore.score);
        printf("dataType: %d\n", inputScore.dataType);
        printf("nProc: %d\n", nProc);
    }

    /* スコア計算方法の指定 */
    iWay = inputScore.score;

    /* SNPのScore計算を行う */
    MainProgramScore(&inputScore, nProc, rank);

    /* 時間計測 */
    tm[TM_END] = MPI_Wtime();

    MPI_Finalize();    /* MPI終了処理 */

    printf("%4d  1 time Init-READ      %lf (sec)\n", rank, tm[TM_B_READ]     - tm[TM_START]);
    printf("%4d  2 time READ           %lf (sec)\n", rank, tm[TM_A_READ]     - tm[TM_B_READ]);
    printf("%4d  3 time READ-FBcast    %lf (sec)\n", rank, tm[TM_B_F_BCAST]  - tm[TM_A_READ]);
    printf("%4d  4 time FBcast         %lf (sec)\n", rank, tm[TM_A_F_BCAST]  - tm[TM_B_F_BCAST]);
    printf("%4d  5 time FBcast-Bcast   %lf (sec)\n", rank, tm[TM_B_BCAST]    - tm[TM_A_F_BCAST]);
    printf("%4d  6 time Bcast          %lf (sec)\n", rank, tm[TM_A_BCAST]    - tm[TM_B_BCAST]);
    printf("%4d  7 time Bcast-Calc     %lf (sec)\n", rank, tm[TM_B_CALC]     - tm[TM_A_BCAST]);
    printf("%4d  8 time Calculation    %lf (sec)\n", rank, tm[TM_A_CALC]     - tm[TM_B_CALC]);
    printf("%4d  9 time Calc-FGather   %lf (sec)\n", rank, tm[TM_B_F_GATHER] - tm[TM_A_CALC]);
    printf("%4d 10 time FGather        %lf (sec)\n", rank, tm[TM_A_F_GATHER] - tm[TM_B_F_GATHER]);
    printf("%4d 11 time FGather-Gather %lf (sec)\n", rank, tm[TM_B_GATHER]   - tm[TM_A_F_GATHER]);
    printf("%4d 12 time Gather         %lf (sec)\n", rank, tm[TM_A_GATHER]   - tm[TM_B_GATHER]);
    printf("%4d 13 time Gather-Write   %lf (sec)\n", rank, tm[TM_B_WRITE]    - tm[TM_A_GATHER]);
    printf("%4d 14 time Write          %lf (sec)\n", rank, tm[TM_A_WRITE]    - tm[TM_B_WRITE]);
    printf("%4d 15 time Write-End      %lf (sec)\n", rank, tm[TM_END]        - tm[TM_A_WRITE]);

    printf("%4d 20 time Init           %lf (sec)\n", rank, tm[TM_B_F_BCAST]  - tm[TM_START]);
    printf("%4d 21 time FBcast         %lf (sec)\n", rank, tm[TM_B_BCAST]    - tm[TM_B_F_BCAST]);
    printf("%4d 22 time Bcast          %lf (sec)\n", rank, tm[TM_B_CALC]     - tm[TM_B_BCAST]);
    printf("%4d 23 time Calculation    %lf (sec)\n", rank, tm[TM_A_CALC]     - tm[TM_B_CALC]);
    printf("%4d 24 time Gather         %lf (sec)\n", rank, tm[TM_A_GATHER]   - tm[TM_A_CALC]);
    printf("%4d 25 time End            %lf (sec)\n", rank, tm[TM_END]        - tm[TM_A_GATHER]);

    printf("%4d 30 time Calculation    %lf (sec)\n", rank, tm[TM_A_CALC]     - tm[TM_B_CALC]);
    printf("%4d 31 time F_Bcast-Gather %lf (sec)\n", rank, tm[TM_A_GATHER]   - tm[TM_B_F_BCAST]);
    printf("%4d 32 time A_Read-B_Write %lf (sec)\n", rank, tm[TM_A_WRITE]    - tm[TM_B_READ]);
    printf("%4d 33 time Total          %lf (sec)\n", rank, tm[TM_END]        - tm[TM_START]);

    return 0;
}


/* ScoreのScore計算を行う */
void MainProgramScore(InputScore *inputScore, int nProc, int rank)
{
    int retval = 0;
    int flag = 0;
    int numOutput = 0;
    int outDataOffset = 0;
    long i = 0;
    long j = 0;
    long k = 0;
    long index = 0;
    long fileLine1 = 0; /* 入力ファイル（case）のライン数 */
    long fileLine2 = 0; /* 入力ファイル（contorl）のライン数 */
    long areaFileLine = 0; /* haplotypeブロック領域指定ファイルのライン数 */
    long a = 0;         /* number of haplotype copies (=sequences) in case */
    long b = 0;         /* number of haplotype copies (=sequences) in control */
    long n = 0;
    long dataNum;       /* case、controlデータ整合後の総SNP数 */
    long jStart = 0;    /* haplotypeブロックの最初のSNPを示す座位 */
    long jEnd = 0;      /* haplotypeブロックの最後のSNPを示す座位 */
    long blockNum = 0;  /* haplotypeブロック数 */
    long startPos = 0;
    long endPos = 0;
    long partNum = 0;
    long blockPartNum = 0;
    long blockPartStart = 0;
    double S = 0;

    /* Gather用 */
    int *numOutputs = NULL;
    int *outDataOffsets = NULL;

    int **T = NULL;                 /* 偶現表 */
    int *populationType = NULL;
    char *caseData = NULL;          /* サンプルデータ（case）格納用 */
    char *controlData = NULL;       /* サンプルデータ（control）格納用 */
    long *blockArea = NULL;         /* 各haplotypeブロック領域格納 */
    long *linkSNPNum = NULL;        /* 各haplotypeブロックのSNP数 */
    long *linkSNPStart = NULL;      /* 各haplotypeブロックの最初のSNPを示す座位 */
    double *Sobs = NULL;            /* 各haplotypeブロック毎のスコア最大値 */
    double *SobsAll = NULL;         /* Haplotypeブロック毎のスコア最大値 */

    FILE *fpCase = NULL;            /* 入力（case）ファイルポインタ */
    FILE *fpCntl = NULL;            /* 入力（control）ファイルポインタ */
    FILE *fpOut = NULL;             /* 出力ファイルポインタ */
    FILE *fpArea = NULL;            /* haplotypeブロック領域指定ファイルポインタ */

    SnpData *snpTmpData1 = NULL;
    SnpData *snpTmpData2 = NULL;
    SnpData *snpData1 = NULL;
    SnpData *snpData2 = NULL;

    /* MPI用構造体設定 */
    MPI_Datatype MPI_INPUT_STRUCT;

/****************************************************************/
/* データ入力                                                   */
/****************************************************************/

    /* 時間計測 */
    tm[TM_B_READ] = MPI_Wtime();

    /* rank0のみ実行 */
    if (rank == 0) {
        /* ファイルオープン */
        retval = InputFileOpen(&fpCase, inputScore->inputFile1);
        if (retval != 0){
            goto finalize;
        }
        retval = InputFileOpen(&fpCntl, inputScore->inputFile2);
        if (retval != 0){
            goto finalize;
        }
        retval = OutputFileOpen(&fpOut, inputScore->outputFile1);
        if (retval != 0){
            goto finalize;
        }
        retval = InputFileOpen(&fpArea, inputScore->blockAreaFile);
        if (retval != 0){
            goto finalize;
        }

        /* 入力ファイルのライン数を取得 */
        fileLine1 = DataReaderCountFileLine(fpCase);
        fileLine2 = DataReaderCountFileLine(fpCntl);

        /* ファイルポインタを先頭に戻す */
        fseek(fpCase, 0L, SEEK_SET);
        fseek(fpCntl, 0L, SEEK_SET);

        /* データ一時格納用構造体のメモリ確保 */
        snpTmpData1 = (SnpData*)malloc1Dim(sizeof(SnpData), fileLine1);
        if (NULL == snpTmpData1){ goto finalize; }
        snpTmpData2 = (SnpData*)malloc1Dim(sizeof(SnpData), fileLine2);
        if (NULL == snpTmpData2){ goto finalize; }

        /* データをファイルから読み込み構造体に収める */
        DataReaderSetAllData(fpCase, snpTmpData1, fileLine1, inputScore->dataType);
        DataReaderSetAllData(fpCntl, snpTmpData2, fileLine2, inputScore->dataType);

        /* 入力データの整合性をチェックして並列化用の入力データを作成する */
        /* MPI_Bcastの回数を減らすためにサンプルデータは別配列（caseData, controlData）で保持する */
        dataNum = DataReaderMakeParallelData(snpTmpData1, snpTmpData2, fileLine1, fileLine2, &snpData1, &snpData2, &caseData, &controlData);

        /* 使用しない配列のメモリ開放 */
        /* 構造体SnpDataメンバのメモリを開放する */
        DataReaderSnpDataMemoryFree(snpTmpData1, fileLine1);
        DataReaderSnpDataMemoryFree(snpTmpData2, fileLine2);
        snpTmpData1 = NULL;
        snpTmpData2 = NULL;

        /* haplotypeブロック領域指定ファイルのライン数を取得 */
        areaFileLine = DataReaderCountFileLine(fpArea);
        /* haplotypeブロック領域格納用配列のメモリ確保 */
        blockArea = (long*)malloc1Dim(sizeof(long), areaFileLine);
        if (NULL == blockArea){ goto finalize; }
        /* ファイルポインタを先頭に戻す */
        fseek(fpArea, 0L, SEEK_SET);
        /* haplotypeブロック領域を配列に収める */
        DataReaderSetHaplotypeBlockArea(fpArea, blockArea);
        /* haplotypeブロック数 */
        if (inputScore->areaFileType == 0){
            blockNum = areaFileLine - 1;
        }
        else {
            /* haplotypeブロックの個数を計算する */
            if (blockArea[0] > dataNum){
                blockNum = 1;
                blockArea[1] = dataNum; /* linkSNPNumの値がデータ数だけになる */
            }
            else{
                blockNum = (dataNum - (blockArea[0] - blockArea[1]) ) / blockArea[1];
            }
        }
    }/* rank0のみ実行ここまで */

    /* 時間計測 */
    tm[TM_A_READ] = MPI_Wtime();
    /* 時間計測 */
    tm[TM_B_F_BCAST] = MPI_Wtime();

    /* 全データ数とブロック数の送信 */
    MPI_Bcast(&dataNum, 1, MPI_LONG, 0, MPI_COMM_WORLD);
    MPI_Bcast(&blockNum, 1, MPI_LONG, 0, MPI_COMM_WORLD);
    MPI_Bcast(&areaFileLine, 1, MPI_LONG, 0, MPI_COMM_WORLD);

    /* 時間計測 */
    tm[TM_A_F_BCAST] = MPI_Wtime();

    if (nProc > blockNum) {
        /* rank0のみ実行 */
        if (rank == 0) {
            fprintf(stderr, "プロセッサ数(%d)がデータブロック数(%ld)よりも多く設定されています.\n", nProc, blockNum);
        }
        MPI_Abort(MPI_COMM_WORLD, 1);
        goto finalize;
    }
    if (nProc > dataNum) {
        /* rank0のみ実行 */
        if (rank == 0) {
            fprintf(stderr, "プロセッサ数(%d)がデータ数(%ld)よりも多く設定されています.\n", nProc, dataNum);
        }
        MPI_Abort(MPI_COMM_WORLD, 1);
        goto finalize;
    }

    /* ランク0以外の入力データのメモリ確保 */
    if (rank != 0) {
        /* snpData1（case）のメモリ確保 */
        snpData1 = (SnpData *)malloc1Dim(sizeof(SnpData), dataNum);
        if (snpData1 == NULL) {
            fprintf(stderr, "rank:%d メモリを確保できませんでした.\n", rank);
            MPI_Abort(MPI_COMM_WORLD, 1);
            goto finalize;
        }
        /* snpData2（control）のメモリ確保 */
        snpData2 = (SnpData *)malloc1Dim(sizeof(SnpData), dataNum);
        if (snpData2 == NULL) {
            fprintf(stderr, "rank:%d メモリを確保できませんでした.\n", rank);
            MPI_Abort(MPI_COMM_WORLD, 1);
            goto finalize;
        }
        /* haplotypeブロック領域のメモリ確保 */
        blockArea = (long*)malloc1Dim(sizeof(long), areaFileLine);
        if (blockArea == NULL) {
            fprintf(stderr, "rank:%d メモリを確保できませんでした.\n", rank);
            MPI_Abort(MPI_COMM_WORLD, 1);
            goto finalize;
        }
    }

    /* MPI用型の定義 */
    Define_MPI_InputStruct(&MPI_INPUT_STRUCT);

    /* 時間計測 */
    tm[TM_B_BCAST] = MPI_Wtime();

    /* 入力データ(構造体)のブロードキャスト */
    MPI_Bcast(snpData1, dataNum, MPI_INPUT_STRUCT, 0, MPI_COMM_WORLD);
    MPI_Bcast(snpData2, dataNum, MPI_INPUT_STRUCT, 0, MPI_COMM_WORLD);
    MPI_Bcast(blockArea, areaFileLine, MPI_LONG, 0, MPI_COMM_WORLD);

    /* 入力データのサンプル数取得 */
    a = snpData1[0].dataNum;
    b = snpData2[0].dataNum;
    n = a + b;

    /* ランク0以外の入力データのメモリ確保 */
    if (rank != 0) {
        /* サンプルデータ（case）格納用のメモリ確保 */
        caseData = (char *)malloc1Dim(sizeof(char), dataNum * a);
        if (caseData == NULL) {
            fprintf(stderr, "rank:%d メモリを確保できませんでした.\n", rank);
            MPI_Abort(MPI_COMM_WORLD, 1);
            goto finalize;
        }
        /* サンプルデータ（control）格納用のメモリ確保 */
        controlData = (char *)malloc1Dim(sizeof(char), dataNum * b);
        if (controlData == NULL) {
            fprintf(stderr, "rank:%d メモリを確保できませんでした.\n", rank);
            MPI_Abort(MPI_COMM_WORLD, 1);
            goto finalize;
        }
    }

    /* 入力データ(データ部)のブロードキャスト */
    MPI_Bcast(caseData, dataNum * a, MPI_CHAR, 0, MPI_COMM_WORLD);
    MPI_Bcast(controlData, dataNum * b, MPI_CHAR, 0, MPI_COMM_WORLD);

    /* 時間計測 */
    tm[TM_A_BCAST] = MPI_Wtime();

    /* 構造体SnpDataにサンプルデータをコピーする */
    DataReaderDataCopyToSnpData(snpData1, caseData, dataNum, a);
    DataReaderDataCopyToSnpData(snpData2, controlData, dataNum, b);

/****************************************************************/
/* メモリ確保                                                   */
/****************************************************************/

    /* プロセッサごとのブロック数計算 */
    blockPartNum = blockNum / nProc;
    if (rank < blockNum % nProc) {
        blockPartNum++;
    }
    /* 各プロセッサにおけるデータのオフセット値の計算 */
    if (rank < blockNum % nProc) {
        blockPartStart = blockPartNum * rank;
    }
    else{
        blockPartStart = (blockPartNum + 1) * (blockNum % nProc)
                       + blockPartNum * (rank - blockNum % nProc);
    }

    /* 各haplotypeブロック領域内のSNP数格納用配列のメモリ確保 */
    linkSNPNum = (long*)malloc1Dim(sizeof(long), blockNum);
    if (NULL == linkSNPNum){ goto finalize; }
    /* 各haplotypeブロック領域内の最初のSNPを示す座位格納用配列のメモリ確保 */
    linkSNPStart = (long*)malloc1Dim(sizeof(long), blockNum);
    if (NULL == linkSNPStart){ goto finalize; }

    /* 各haplotypeブロック領域に該当するSNPデータを決定 */
    if (inputScore->areaFileType == 0){
        for (i = 0; i < blockNum; i++){ /* 将来、領域の重複を許す場合も考える */
            startPos = blockArea[i];
            endPos = blockArea[i+1];
            flag = 0;
            /* 入力データはポジションでソートされていると仮定 */
            for (j = 0; j < dataNum; j++){
                if (startPos <= snpData1[j].pos){
                    if (snpData1[j].pos < endPos){
                        /* 領域内のSNP数カウント */
                        linkSNPNum[i]++;
                        /* 領域内の最初のSNPを示す座位を保持 */
                        if (0 == flag ){
                            linkSNPStart[i] = j;
                            flag = 1;
                        }
                    }
                    /* これ以降、領域に該当するデータは出現しないので次のブロックを調べる */
                    else {
                        break;
                    }
                }
            }
        }
    }
    else {
        for (i = 0; i < blockNum; i++){
            linkSNPNum[i] = blockArea[0];
            linkSNPStart[i] = i * blockArea[1];
        }
    }

    /* 各ランクでSNP数をカウント */
    for (i = 0, k = blockPartStart; i < blockPartNum; i++, k++){
        partNum += linkSNPNum[k];
    }
    /* データオフセット値計算 */
    for (i = 0; i < blockPartStart; i++){
        outDataOffset += linkSNPNum[i];
    }

    /* 偶現表Tのメモリ確保 */
    T = (int**)mallocInt2Dim(ROW, COLUMN);
    if (NULL == T){ goto finalize; }
    /* populationTypeのメモリ確保 */
    populationType = (int*)malloc1Dim(sizeof(int), a + b);
    if (NULL == populationType) { goto finalize; }
    /* Haplotypeブロック毎のスコア最大値のメモリ確保 */
    Sobs = (double*)malloc1Dim(sizeof(double), partNum);
    if (NULL == Sobs) { goto finalize; }

/****************************************************************/
/* Score計算                                                    */
/****************************************************************/

    /* 時間計測 */
    tm[TM_B_CALC] = MPI_Wtime();

    /* haplotypeブロック単位でループ */
    jStart = 0;
    index = 0;
    for (i = 0, k = blockPartStart; i < blockPartNum; i++, k++){
        jStart = linkSNPStart[k];
        jEnd = jStart + linkSNPNum[k];
        /* 各haplotypeブロック内でループ */
        for (j = jStart; j < jEnd; j++){
            /* 観測値から偶現表を作成する */
            DataReaderPopulationType(&snpData1[j], &snpData2[j], populationType);
            DataReaderMakeTableDi(&snpData1[j], &snpData2[j], populationType, T);
            /* スコアを計算　スコア計算中0割になってしまう場合は-1を返す */
            Sobs[index] = TableCalcScore(T);
            index++;
        }
    }
    /* 時間計測 */
    tm[TM_A_CALC] = MPI_Wtime();

    if (rank == 0) {
        /* 各ランクのブロック数を格納する配列のメモリを確保 */
        numOutputs = (int *)malloc1Dim(sizeof(int), nProc);
        if (numOutputs == NULL) {
            MPI_Abort(MPI_COMM_WORLD, 1);
            goto finalize;
        }
        /* 各ランクデータのオフセット値を格納する配列のメモリを確保 */
        outDataOffsets = (int *)malloc1Dim(sizeof(int), nProc);
        if (outDataOffsets == NULL) {
            MPI_Abort(MPI_COMM_WORLD, 1);
            goto finalize;
        }
    }
    /* 時間計測 */
    tm[TM_B_F_GATHER] = MPI_Wtime();

    //numOutput = blockPartNum;
    numOutput = index;

    /* 各ランクが計算した出力値の個数 */
    MPI_Gather(&numOutput, 1, MPI_INT, numOutputs, 1,MPI_INT, 0, MPI_COMM_WORLD);
    MPI_Gather(&outDataOffset, 1, MPI_INT, outDataOffsets, 1,MPI_INT, 0, MPI_COMM_WORLD);

    /* 時間計測 */
    tm[TM_A_F_GATHER] = MPI_Wtime();

    if (rank == 0) {
        /* 出力用データを格納する構造体配列のメモリ確保 */
        SobsAll = (double *)malloc1Dim(sizeof(double), dataNum);
    }

    /* 時間計測 */
    tm[TM_B_GATHER] = MPI_Wtime();
    /* 各プロセッサからのデータの収集 */
    MPI_Gatherv(Sobs, numOutput, MPI_DOUBLE, SobsAll,
               numOutputs, outDataOffsets, MPI_DOUBLE, 0, MPI_COMM_WORLD);
    /* 時間計測 */
    tm[TM_A_GATHER] = MPI_Wtime();

/****************************************************************/
/* 検定結果出力                                                 */
/****************************************************************/

    /* 時間計測 */
    tm[TM_B_WRITE] = MPI_Wtime();

    if (rank == 0) {
        index = 0;
        fprintf(fpOut, "CaseData    = %s\n", inputScore->inputFile1);
        fprintf(fpOut, "ControlData = %s\n", inputScore->inputFile2);
        fprintf(fpOut, "BlockArea\tSNPNum\trsNumber\tPosition\tScore\tP\n");
        if (inputScore->areaFileType == 0){
            for (i = 0; i < blockNum; i++){
                jStart = linkSNPStart[i];
                jEnd = jStart + linkSNPNum[i];
                /* 各haplotypeブロック内でループ */
                for (j = jStart; j < jEnd; j++){
                    fprintf(fpOut, "%ld-%ld\t%ld\t%s\t%ld\t%.10lf\t \n", 
                        blockArea[i],
                        blockArea[i+1],
                        linkSNPNum[i],
                        snpData1[j].rsNumber,
                        snpData1[j].pos,
                        SobsAll[index]);
                    index++;
                }
            }
        }
        else {
            for (i = 0; i < blockNum; i++){
                jStart = linkSNPStart[i];
                jEnd = jStart + linkSNPNum[i];
                /* 各haplotypeブロック内でループ */
                for (j = jStart; j < jEnd; j++){
                    fprintf(fpOut, "%ld-%ld\t%ld\t%s\t%ld\t%.10lf\t \n", 
                        snpData1[ linkSNPStart[i] ].pos,
                        snpData1[ linkSNPStart[i] + linkSNPNum[i] - 1 ].pos,
                        linkSNPNum[i],
                        snpData1[j].rsNumber,
                        snpData1[j].pos,
                        SobsAll[j]);
                }
            }
        }
    }

    /* 時間計測 */
    tm[TM_A_WRITE] = MPI_Wtime();

/****************************************************************/
/* 終了処理                                                     */
/****************************************************************/

finalize:;
    /* ファイルクローズ */
    FileClose(fpCase);
    FileClose(fpCntl);
    FileClose(fpOut);
    FileClose(fpArea);
    /* 確保したメモリを開放する */
    free1Dim(numOutputs);
    free1Dim(outDataOffsets);
    freeInt2Dim(T, ROW);
    free1Dim(populationType);
    free1Dim(blockArea);
    free1Dim(linkSNPNum);
    free1Dim(linkSNPStart);
    free1Dim(Sobs);
    free1Dim(SobsAll);
    DataReaderSnpDataMemoryFree(snpTmpData1, fileLine1);
    DataReaderSnpDataMemoryFree(snpTmpData2, fileLine2);
    free1Dim(caseData);
    free1Dim(controlData);
    free1Dim(snpData1);
    free1Dim(snpData2);

    return;
}
/**********************************************************************/
/* 入力データのMPIに対応した型の定義                                  */
/* mpi_InputStruct  O   MPI対応の型                                   */
/* 返り値： 0: 正常終了                                               */
/**********************************************************************/
int Define_MPI_InputStruct(MPI_Datatype *mpi_inputStruct)
{
    int i = 0;

/* MPI関係 */
#define MAX_BLOCK_NUM 10    /* MPI用構造体を作成するときの型のブロック数 */

    int blockcount[MAX_BLOCK_NUM];
    MPI_Aint address[MAX_BLOCK_NUM];
    MPI_Datatype type[MAX_BLOCK_NUM];

    SnpData sti;  // 元となる型（構造体）

    /* MPI用の構造体を作るパラメータの初期化 */
    for (i = 0; i < MAX_BLOCK_NUM; i++) {
        blockcount[i] = 0;
        address[i] = (MPI_Aint)NULL;
        type[i] = MPI_INT;
    }

    /* 入力データ用のMPI用の構造体定義 */
    blockcount[0] = RS_NUM_LEN;
    blockcount[1] = SNP_NUM;
    blockcount[2] = CHROM_LEN;
    blockcount[3] = 1;
    blockcount[4] = 3 + SNP_COUNT;
    MPI_Address(sti.rsNumber, &address[0]);
    MPI_Address(sti.SNPalleles, &address[1]);
    MPI_Address(sti.chrom, &address[2]);
    MPI_Address(&sti.pos, &address[3]);
    MPI_Address(&sti.allelesNum, &address[4]);
    for (i = 4; i >= 0; i--) {
        address[i] -= address[0];
    }
    /* 型指定 */
    type[0] = MPI_CHAR;
    type[1] = MPI_CHAR;
    type[2] = MPI_CHAR;
    type[3] = MPI_LONG;
    type[4] = MPI_INT;
    MPI_Type_struct(5, blockcount, address, type, mpi_inputStruct);
    MPI_Type_commit(mpi_inputStruct);

#undef MAX_BLOCK_NUM

	return 0;
}
