#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef MPI_USE
#include <mpi.h>
#endif

#define _MAIN_DEF

#include "HaplotypePhasing.h"


#ifdef MPI_USE
/** MPIの初期化処理
 *  MPI用の引数が処理されて、削除される。
 *  @param argc [IN/OUT] 引数の数
 *  @param argv [IN/OUT] 引数
 */
void setupMPI(int *argc, char ***argv)
{
    MPI_Init(argc, argv);
    MPI_Comm_size(MPI_COMM_WORLD, &MyMpiSize);
    MPI_Comm_rank(MPI_COMM_WORLD, &MyMpiRank);
}
#endif /* MPI_USE */

/* メイン関数 */
int main(int argc, char* argv[])
{
    int i = 0;
    InputPhasing inputPhasing={"", "", "", 0};

#ifdef MPI_USE
    /* MPIによる初期化処理 */
    setupMPI(&argc, &argv);

    /* 時間計測用変数の初期化 */
    for (i = 0; i < NUM_TM; i++){
        tm[i] = 0.0;
    }

    /* 時間計測 */
    tm[TM_START] = MPI_Wtime();
#endif /* MPI_USE */

    if(argc != 6){
        printf("[usage]main.exe [InputFile1] [LDblockFile] [OutputFile] [PhasingType] [SNPNumLimit]\n");
        return 255;
    }
    strcpy(inputPhasing.inputFile, argv[1]);
    strcpy(inputPhasing.ldBlockFile, argv[2]);
    strcpy(inputPhasing.outputFile, argv[3]);
    inputPhasing.phasingType = atoi(argv[4]);
    inputPhasing.snpNumLimit = atoi(argv[5]);


    /* ハプロタイプフェージングを実行 */
    haplotypePhasing(&inputPhasing);

#ifdef MPI_USE
    /* 時間計測 */
    tm[TM_END] = MPI_Wtime();

    /* MPI終了処理 */
    MPI_Finalize();

    printf("%4d\t%lf\t%lf\t%lf\t%lf\t%lf\t%lf\t%lf \n",
           MyMpiRank,
           tm[TM_SPLIT]    - tm[TM_START],
           tm[TM_BCAST]    - tm[TM_SPLIT],
           tm[TM_PHASE]    - tm[TM_BCAST],
           tm[TM_Barrier]  - tm[TM_PHASE],
           tm[TM_MERGE]    - tm[TM_Barrier],
           tm[TM_END]      - tm[TM_MERGE],
           tm[TM_END]      - tm[TM_START]);
/*
    printf("%4d  1 time Init        %lf (sec)\n", MyMpiRank, tm[TM_READ]     - tm[TM_START]);
    printf("%4d  2 time Read        %lf (sec)\n", MyMpiRank, tm[TM_SPLIT]    - tm[TM_READ]);
    printf("%4d  3 time Split       %lf (sec)\n", MyMpiRank, tm[TM_BCAST]    - tm[TM_SPLIT]);
    printf("%4d  4 time Bcast       %lf (sec)\n", MyMpiRank, tm[TM_PHASE]    - tm[TM_BCAST]);
    printf("%4d  5 time Phase       %lf (sec)\n", MyMpiRank, tm[TM_Barrier]  - tm[TM_PHASE]);
    printf("%4d  6 time Barrier2    %lf (sec)\n", MyMpiRank, tm[TM_MERGE]    - tm[TM_Barrier]);
    printf("%4d  7 time Merge       %lf (sec)\n", MyMpiRank, tm[TM_END]      - tm[TM_MERGE]);
    printf("%4d  8 time Total       %lf (sec)\n", MyMpiRank, tm[TM_END]      - tm[TM_START]);
*/
#endif /* MPI_USE */

    return 0;
}


/* ハプロタイプフェージングを実行 */
void haplotypePhasing(InputPhasing *inputPhasing)
{
    int retval = 0;
    int flag = 0;
    long i = 0;
    long j = 0;
    long k = 0;
    long index = 0;
    long fileLine = 0;          /* 入力ファイルのライン数 */
    long ldBlockFileLine = 0;   /* LDブロックファイルのライン数 */
    long jStart = 0;            /* LDブロックの最初のSNPを示す座位格納変数 */
    long jEnd = 0;              /* LDブロックの最後のSNPを示す座位格納変数 */
    long blockNum = 0;          /* LDブロック数 */
    long tmpBlockNum = 0;       /* LDブロック数 */
    long startPos = 0;
    long endPos = 0;
    long count = 0;

    long *ldBlock = NULL;       /* 各LDブロック格納用 */
    long *linkSNPNum = NULL;    /* 各LDブロックのSNP数格納用 */
    long *linkSNPStart = NULL;  /* 各LDブロックの最初のSNPを示す座位格納用 */

    FILE *fpIn = NULL;      /* 入力ファイルポインタ */
    FILE *fpOut = NULL;     /* 出力ファイルポインタ */
    FILE *fpLD = NULL;      /* LDブロックファイルポインタ */


    SnpData *snpData = NULL;

#ifdef MPI_USE
    /* 時間計測 */
    tm[TM_READ] = MPI_Wtime();

    /* rank0のみ実行 */
    if (MyMpiRank == 0) {
#endif /* MPI_USE */

        /* ファイルオープン */
        retval = InputFileOpen(&fpIn, inputPhasing->inputFile);
        if (retval != 0){
            goto finalize;
        }
        retval = InputFileOpen(&fpLD, inputPhasing->ldBlockFile);
        if (retval != 0){
            goto finalize;
        }

/****************************************************************/
/* データ入力                                                   */
/****************************************************************/
 
        /* LDブロックファイルのライン数を取得 */
        ldBlockFileLine = DataReaderCountFileLine(fpLD);
        /* LDブロック格納用配列のメモリ確保 */
        ldBlock = (long*)malloc1Dim(sizeof(long), ldBlockFileLine);
        if (NULL == ldBlock){ goto finalize; }
        /* ファイルポインタを先頭に戻す */
        fseek(fpLD, 0L, SEEK_SET);
        /* LDブロックを配列に収める */
        DataReaderSetLDBlock(fpLD, ldBlock);
        /* 入力ファイルのライン数を取得 */
        fileLine = DataReaderCountFileLine(fpIn);
        /* ファイルポインタを先頭に戻す */
        fseek(fpIn, 0L, SEEK_SET);
        /* データ格納用構造体のメモリ確保 */
        snpData = (SnpData*)malloc1Dim(sizeof(SnpData), fileLine);
        if (NULL == snpData){ goto finalize; }
        /* データをファイルから読み込み構造体に収める */
        DataReaderSetAllHapmapData(fpIn, snpData, fileLine, inputPhasing->phasingType);

/****************************************************************/
/* LDブロック作成準備                                                   */
/****************************************************************/

        /* 入力LDブロック数 */
        blockNum = ldBlockFileLine - 1;
        /* 入力LDブロックに該当するSNPデータを決定 */
        for (i = 0; i < blockNum; i++){ /* 将来、領域の重複を許す場合も考える */
            startPos = ldBlock[i];
            endPos = ldBlock[i+1];
            count = 0;
            /* 入力データはポジションでソートされていると仮定 */
            for (j = 0; j < fileLine; j++){
                if (startPos <= snpData[j].pos){
                    if (snpData[j].pos < endPos){
                        /* 領域内のSNP数カウント */
                        count++;
                    }
                    /* これ以降、領域に該当するデータは出現しないので次のブロックを調べる */
                    else {
                        break;
                    }
                }
            }
            /* ブロック内のSNP数が上限値を超えている場合、さらに分割する */
            tmpBlockNum += (count - 1) / inputPhasing->snpNumLimit;
        }


        blockNum += tmpBlockNum;
        /* 各LDブロックのSNP数格納用配列のメモリ確保 */
        linkSNPNum = (long*)malloc1Dim(sizeof(long), blockNum);
        if (NULL == linkSNPNum){ goto finalize; }
        /* 各LDブロックの最初のSNPを示す座位格納用配列のメモリ確保 */
        linkSNPStart = (long*)malloc1Dim(sizeof(long), blockNum);
        if (NULL == linkSNPStart){ goto finalize; }


        /* 各LDブロックに該当するSNPデータを決定 SNP */
        for (i = 0, k = 0; i < blockNum; i++){ /* 将来、領域の重複を許す場合も考える */
            startPos = ldBlock[k];
            endPos = ldBlock[k+1];
            flag = 0;
            /* 入力データはポジションでソートされていると仮定 */
            for (j = 0; j < fileLine; j++){
                if (startPos <= snpData[j].pos){
                    if (snpData[j].pos < endPos){
                        /* LDブロック内のSNP数が上限値を超える場合 */
                        if (linkSNPNum[i] == inputPhasing->snpNumLimit){
                            i++;
                            flag = 0;
                        }
                        /* 領域内のSNP数カウント */
                        linkSNPNum[i]++;
                        /* 領域内の最初のSNPを示す座位を保持 */
                        if (0 == flag ){
                            linkSNPStart[i] = j;
                            flag = 1;
                        }
                    }
                    /* これ以降、領域に該当するデータは出現しないので次のブロックを調べる */
                    else {
                        break;
                    }
                }
            }
            k++;
        }

#ifdef MPI_USE
    /* rank0のみ実行ここまで */
    }
#endif /* MPI_USE */

/****************************************************************/
/* Hapmapデータの分割変換処理                                   */
/****************************************************************/

#ifdef MPI_USE
    /* 時間計測 */
    tm[TM_SPLIT] = MPI_Wtime();

    /* rank0のみ実行 */
    if (MyMpiRank == 0) {
#endif /* MPI_USE */

        /* 入力Hapmapデータをフェージング処理入力形式に変換 */
        transHapmapForPhasing(inputPhasing, snpData, blockNum, linkSNPNum, linkSNPStart);

#ifdef MPI_USE
    /* rank0のみ実行ここまで */
    }

    /* 時間計測 */
    tm[TM_BCAST] = MPI_Wtime();

    /* ブロック数の送信 */
    MPI_Bcast(&blockNum, 1, MPI_LONG, 0, MPI_COMM_WORLD);

    if (MyMpiSize > blockNum) {
        /* rank0のみ実行 */
        if (MyMpiRank == 0) {
            fprintf(stderr, "プロセッサ数(%d)がデータブロック数(%ld)よりも多く設定されています.\n", MyMpiSize, blockNum);
        }
        MPI_Abort(MPI_COMM_WORLD, 1);
        goto finalize;
    }

    /* 時間計測 */
    tm[TM_PHASE] = MPI_Wtime();
#endif /* MPI_USE */

/****************************************************************/
/* フェージング処理                                             */
/****************************************************************/

    /* フェージング処理を実行する */
    executePhasing(inputPhasing, blockNum);

/****************************************************************/
/* フェージング結果をPhasingHapmapデータへ変換統合処理処理      */
/****************************************************************/

#ifdef MPI_USE
    /* 時間計測 */
    tm[TM_Barrier] = MPI_Wtime();

    /* 全rankのフェージング処理が終了するまで待機 */
    MPI_Barrier(MPI_COMM_WORLD);

    /* 時間計測 */
    tm[TM_MERGE] = MPI_Wtime();

    /* rank0のみ実行 */
    if (MyMpiRank == 0) {
#endif /* MPI_USE */

        /* フェージング出力結果をHapmap形式に変換 */
        transPhasedDataFotHapmap(inputPhasing, snpData, blockNum, linkSNPNum, linkSNPStart);

#ifdef MPI_USE
    /* rank0のみ実行ここまで */
    }
#endif /* MPI_USE */

/****************************************************************/
/* 終了処理                                                     */
/****************************************************************/

finalize:;

#ifdef MPI_USE
    /* rank0のみ実行 */
    if (MyMpiRank == 0) {
#endif /* MPI_USE */

        /* ファイルクローズ */
        FileClose(fpIn);
        FileClose(fpOut);
        FileClose(fpLD);
        /* 確保したメモリを開放する */
        DataReaderSnpDataMemoryFree(snpData, fileLine);
        free1Dim(ldBlock);
        free1Dim(linkSNPNum);
        free1Dim(linkSNPStart);

#ifdef MPI_USE
    /* rank0のみ実行ここまで */
    }
#endif /* MPI_USE */

    return;
}

/* 入力Hapmapデータをフェージング処理入力形式に変換 */
int transHapmapForPhasing(InputPhasing *inputPhasing, SnpData *snpData, long blockNum, long *linkSNPNum, long *linkSNPStart)
{
    int retval = 0;
    char tmpFileName[MAX_LEN];
    long i = 0;
    long j = 0;
    long k = 0;
    long jStart = 0;            /* LDブロックの最初のSNPを示す座位格納変数 */
    long jEnd = 0;              /* LDブロックの最後のSNPを示す座位格納変数 */

    FILE *fpOut = NULL;     /* 出力ファイルポインタ */

    /* LDブロック単位でループ */
    jStart = 0;
    for (i = 0; i < blockNum; i++){
        /* 中間データ出力用のファイルオープン */
        sprintf(tmpFileName, "%s_input%ld", inputPhasing->outputFile, i);
        retval = OutputFileOpen(&fpOut, tmpFileName);
        if (retval != 0){
            goto finalize;
        }

        jStart = linkSNPStart[i];
        jEnd = jStart + linkSNPNum[i];

        /* フェージング処理への入力ファイル作成 */
        if (inputPhasing->phasingType == 1){
            /* PHASEでのフェージング処理の場合 */
            fprintf(fpOut, "%d\n", snpData[1].sampleNum);
            fprintf(fpOut, "%ld\n", linkSNPNum[i]);
            for (j = jStart; j < jEnd; j++){
                fprintf(fpOut, "S");
            }
            fprintf(fpOut, "\n");
        }
        for (k = 0; k < snpData[1].sampleNum; k++){
            if (inputPhasing->phasingType == 0){
            /* snphapでのフェージング処理の場合 */
                fprintf(fpOut, "%ld ", k+1);
            }
            for (j = jStart; j < jEnd; j++){
                fprintf(fpOut, "%c %c ", (snpData+j)->SNPdata[2*k], (snpData+j)->SNPdata[2*k+1]);
            }
            fprintf(fpOut, "\n");
        }

        /* 中間データ出力用のファイルクローズ */
        FileClose(fpOut);
        fpOut = NULL;
    }

/****************************************************************/
/* 終了処理                                                     */
/****************************************************************/

finalize:;
    /* ファイルクローズ */
    FileClose(fpOut);

    return retval;
}

/* フェージング処理を実行する */
int executePhasing(InputPhasing *inputPhasing, long blockNum)
{
    int retval = 0;
    char tmpFileName[MAX_LEN];
    char inputFileName[MAX_LEN];
    char outputFileName[MAX_LEN];
    char cmd[MAX_LEN];
    long i = 0;
    long blockPartNum = 0;
    long blockPartStart = 0;

#ifdef MPI_USE
    /* プロセッサごとのブロック数計算 */
    blockPartNum = blockNum / MyMpiSize;
    if (MyMpiRank < blockNum % MyMpiSize) {
        blockPartNum++;
    }
    /* 各プロセッサにおけるデータのオフセット値の計算 */
    if (MyMpiRank < blockNum % MyMpiSize) {
        blockPartStart = blockPartNum * MyMpiRank;
    }
    else{
        blockPartStart = (blockPartNum + 1) * (blockNum % MyMpiSize)
                       + blockPartNum * (MyMpiRank - blockNum % MyMpiSize);
    }
#else
    blockPartStart = 0;
    blockPartNum = blockNum;
#endif /* MPI_USE */

    /* LDブロック単位でループ */
    for (i = blockPartStart; i < blockPartStart + blockPartNum; i++){
        /* フェージング処理の入力ファイル名セット */
        sprintf(inputFileName, "%s_input%ld", inputPhasing->outputFile, i);
        /* フェージング処理の出力ファイル名セット */
        sprintf(outputFileName, "%s_output%ld", inputPhasing->outputFile, i);
        /* フェージング処理のテンポラリ出力ファイル名セット */
        sprintf(tmpFileName, "%s_output_tmp%ld", inputPhasing->outputFile, i);

        /* フェージング処理実行 */
        if (inputPhasing->phasingType == 0) {
            /* snphap実行 */
            //spawnlp(P_WAIT, "snphap", "snphap", inputFileName, tmpFileName, outputFileName, NULL);
            //sprintf(cmd, "snphap -mb 1000 %s %s %s > /dev/null", inputFileName, tmpFileName, outputFileName);
            sprintf(cmd, "snphap -mb 1000 %s %s %s > /dev/null", inputFileName, tmpFileName, outputFileName);
            system(cmd);
            /* 不要な中間データの削除 */
            remove(inputFileName);
            remove(tmpFileName);
        }
        else {
            /* PHASE実行 */
            //sprintf(cmd, "PHASE -n -f1 %s %s >& /dev/null", inputFileName, outputFileName);
            sprintf(cmd, "PHASE -n -f1 %s %s 1> /dev/null 2> /dev/null", inputFileName, outputFileName);
            system(cmd);
            /* 不要な中間データの削除 */
            remove(inputFileName);
            sprintf(tmpFileName, "%s_freqs", outputFileName);
            remove(tmpFileName);
            sprintf(tmpFileName, "%s_hbg", outputFileName);
            remove(tmpFileName);
            sprintf(tmpFileName, "%s_monitor", outputFileName);
            remove(tmpFileName);
            sprintf(tmpFileName, "%s_pairs", outputFileName);
            remove(tmpFileName);
            sprintf(tmpFileName, "%s_probs", outputFileName);
            remove(tmpFileName);
            sprintf(tmpFileName, "%s_recom", outputFileName);
            remove(tmpFileName);
        }
    }

    return retval;
}

/* フェージング出力結果をHapmap形式に変換 */
int transPhasedDataFotHapmap(InputPhasing *inputPhasing, SnpData *snpData, long blockNum, long *linkSNPNum, long *linkSNPStart)
{
    int retval = 0;
    char tmpFileName[MAX_LEN];
    long i = 0;
    long j = 0;
    long k = 0;
    long jStart = 0;            /* LDブロックの最初のSNPを示す座位格納変数 */

    char** haplotype = NULL;    /* ハプロタイプ格納用配列 */

    FILE *fpIn = NULL;      /* 入力ファイルポインタ */
    FILE *fpOut = NULL;     /* 出力ファイルポインタ */

    /* 出力ファイルオープン */
    retval = OutputFileOpen(&fpOut, inputPhasing->outputFile);
    if (retval != 0){
        goto finalize;
    }
    /* 出力ファイルのヘッダ部分書込み */
    fprintf(fpOut, "rsID\tphys_position\t");
    for (k = 0; k < snpData[1].sampleNum; k++){
        // 個人の識別ID？はもとの入力ファイルから取得するようにする
        fprintf(fpOut, "NA%ld_A NA%ld_B\t", k, k);
    }
    fprintf(fpOut, "\n");

    /* LDブロック単位でループ */
    jStart = 0;
    for (i = 0; i < blockNum; i++){
        /* 中間データのファイルオープン */
        sprintf(tmpFileName, "%s_output%ld", inputPhasing->outputFile, i);
        retval = InputFileOpen(&fpIn, tmpFileName);
        if (retval != 0){
            goto finalize;
        }

        jStart = linkSNPStart[i];

        /* ハプロタイプ格納用配列のメモリ確保 */
        haplotype = (char**)mallocChar2Dim(snpData[1].sampleNum*2, linkSNPNum[i]);
        if (NULL == haplotype) { goto finalize; }

        /* 中間データのハプロタイプ読み込み */
        if (inputPhasing->phasingType == 0) {
            /* snphap出力結果読み込み */
            DataReaderSetSnphapData(fpIn, haplotype, linkSNPNum[i], snpData[1].sampleNum);
        }
        else {
            /* PHASE出力結果読み込み */
            DataReaderSetPHASEData(fpIn, haplotype);
        }

        /* フェージング処理結果ファイル出力 */
        for (j = 0; j < linkSNPNum[i]; j++){
            fprintf(fpOut, "%s\t%ld\t", snpData[jStart].rsNumber, snpData[jStart].pos);
            jStart++;
            for (k = 0; k < snpData[1].sampleNum; k++){
                fprintf(fpOut, "%c %c\t", haplotype[2*k][j], haplotype[2*k+1][j]);
            }
            fprintf(fpOut, "\n");
        }

        /* 中間データのファイルクローズ */
        FileClose(fpIn);
        fpIn = NULL;
        /* 確保したメモリを開放する */
        freeChar2Dim(haplotype, snpData[1].sampleNum*2);
        haplotype = NULL;
        /* 不要な中間データの削除 */
        remove(tmpFileName);
    }

/****************************************************************/
/* 終了処理                                                     */
/****************************************************************/

finalize:;
    /* ファイルクローズ */
    FileClose(fpIn);
    FileClose(fpOut);
    /* 確保したメモリを開放する */
    freeChar2Dim(haplotype, snpData[1].sampleNum*2);

    return retval;
}

