#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <math.h>

#include "snp_DataReader.h"
#include "snp_Config.h"
#include "snp_MemoryControl.h"

/**********************/
/* 入力ファイルを開く */
/**********************/
int InputFileOpen(FILE **fp, char *file)
{
    int retval = 0;

    /* ファイルオープン */
    *fp = fopen(file, "r");
    if (NULL == *fp){
        printf("input file open error! FileName:%s\n", file);
        retval = 1;
    }

    return retval;
}

/**********************/
/* 出力ファイルを開く */
/**********************/
int OutputFileOpen(FILE **fp, char *file)
{
    int retval = 0;

    /* ファイルオープン */
    *fp = fopen(file, "w");
    if (NULL == *fp){
        printf("output file open error! FileName:%s\n", file);
        retval = 1;
    }

    return retval;
}

/********************/
/* ファイルを閉じる */
/********************/
void FileClose(FILE *fp)
{
    if (fp != NULL){
        fclose(fp);
    }
}


/********************************/
/* ファイルの行数をカウントする */
/********************************/
long DataReaderCountFileLine(FILE *fp)
{
    long index = 0;
    long count = 0;
    char buf[MAX_LEN];

    /* 特定文字数ずつ読み込む */
    while ( fgets(buf, MAX_LEN, fp) != NULL ){
        index = strlen(buf);
        if ('\n' == buf[index-1]){
            count++;
        }
    }

    return count;
}

/********************************/
/* 入力データを構造体に収める */
/********************************/
int DataReaderSetAllData(FILE *fp, SnpData *snpData, long line, int dataType)
{
    int retval = 0;

    switch(dataType){
        case 0:
            /* 入力データがHapmap */
            retval = DataReaderSetAllHapmapData(fp, snpData, line);
            break;
        case 1:
            /* 入力データがHaplotypeData */
            retval = DataReaderSetAllHaplotypeData(fp, snpData, line);
            break;
        case 2:
            /* 入力データがPhasingHapmap */
            retval = DataReaderSetAllPhasingHapmapData(fp, snpData, line);
            break;
        default:
            break;
    }
    return retval;
}

/********************************/
/* 入力データ一行を構造体に収める */
/********************************/
int DataReaderSetData(FILE *fp, SnpData *snpData, long line, int dataType)
{
    int retval = 0;

    switch(dataType){
        case 0:
            /* 入力データがHapmap */
            retval = DataReaderSetHapmapData(fp, snpData, line);
            break;
        case 1:
            /* 入力データがHaplotypeData */
            retval = DataReaderSetHaplotypeData(fp, snpData, line);
            break;
        case 2:
            /* 入力データがPhasingHapmap */
            retval = DataReaderSetPhasingHapmapData(fp, snpData, line);
            break;
        default:
            break;
    }
    return retval;
}

/********************************/
/* Hapmapデータを構造体に収める */
/********************************/
int DataReaderSetAllHapmapData(FILE *fp, SnpData *snpData, long line)
{
    int retval = 0;
    long i = 0;
    long sampleNum = 0;

    /* HapmapDataのサンプル数を取得する */
    sampleNum = DataReaderGetHapmapSampleNum(fp);
    /* ファイルから読み込み */
    for (i = 1; i < line; i++){
        /* SNPデータ格納用配列のメモリ確保 */
        snpData[i].SNPdata = (char*)malloc1Dim(sizeof(char), sampleNum);
        retval = DataReaderSetHapmapData(fp, &snpData[i], i);
    }

    return retval;
}

/************************************/
/* Hapmapデータ一行を構造体に収める */
/************************************/
int DataReaderSetHapmapData(FILE *fp, SnpData *snpData, long line)
{
    int i = 0;
    int retval = -1;    /* ファイル最後尾まで来ている場合は-1を返す */
    int character = 0;
    int column = 1;     /* Hapmapデータの列 */
    int refNum = 0;     /* SNPallelesのパターン数 */
    int index = 0;
    int dataIndex = 0;
    char tmp[20];

    /* 構造体の初期化 */
    snpData->missingDataCount = 0;
    snpData->SNPallelesCount[0] = 0;
    snpData->SNPallelesCount[1] = 0;
    snpData->SNPallelesCount[2] = 0;

    /* １文字ずつ読み込む */
    while ( feof(fp) == 0 ){
        character = fgetc(fp);
        /* １行読み込んだら値を保持して終了 */
        if ('\n' == character){
            snpData->dataNum = dataIndex;
            snpData->allelesNum = refNum;
            retval = 0;
            dataIndex = 0;
            break;
        }
        /* ファイルの２行目以降から処理を行う */
        else if (line > 0){
            /* ファイル内の列をカウント */
            if (isspace(character) != 0){
                if (HAPMAP_RS_NUM == column){
                    snpData->rsNumber[index] = '\0';
                }
                else if (HAPMAP_POS == column){
                    tmp[index] = '\0';
                    snpData->pos = atol(tmp);
                }
                else if (HAPMAP_CHROM == column){
                    snpData->chrom[index] = '\0';
                }
                column++;
                index = 0;
            }
            /* rs#を取得する */
            else if (HAPMAP_RS_NUM == column){
                if (isalnum(character) != 0){
                    snpData->rsNumber[index] = character;
                    index++;
                }
            }
            /* SNPallelsを取得する */
            else if (HAPMAP_ALLELES == column){
                if (isalpha(character) != 0){
                    snpData->SNPalleles[refNum] = character;
                    refNum++;
                }
            }
            /* chromを取得する */
            else if (HAPMAP_CHROM == column){
                if (isalnum(character) != 0){
                    snpData->chrom[index] = character;
                    index++;
                }
            }
            /* posを取得する */
            else if (HAPMAP_POS == column){
                tmp[index] = character;
                index++;
            }
            /* 12列目移行のデータを参照する */
            else if (column >= HAPMAP_DATA){
                for (i = 0; i < refNum; i++){
                    /* SNPallelesと一致するデータをカウント */
                    if (snpData->SNPalleles[i] == character){
                        snpData->SNPallelesCount[i]++;
                    }
                }
                /* Nと一致するデータをカウント */
                if ('N' == character){
                    snpData->missingDataCount++;
                }
                /* 遺伝子データを格納 */
                snpData->SNPdata[dataIndex] = character;
                dataIndex++;
            }
        }
    }
    return retval;
}

/*********************************/
/* HaplotypeDataを構造体に収める */
/*********************************/
int DataReaderSetAllHaplotypeData(FILE *fp, SnpData *snpData, long line)
{
    int retval = 0;
    long i = 0;
    long sampleNum = 0;


    /* Haplotypeデータのサンプル数を取得する */
    sampleNum = DataReaderGetHaplotypeSampleNum(fp);
    for (i = 1; i < line; i++){
        /* SNPデータ格納用配列のメモリ確保 */
        snpData[i].SNPdata = (char*)malloc1Dim(sizeof(char), sampleNum);
        retval = DataReaderSetHaplotypeData(fp, &snpData[i], i);
    }

    return retval;
}

/*************************************/
/* HaplotypeData一行を構造体に収める */
/*************************************/
int DataReaderSetHaplotypeData(FILE *fp, SnpData *snpData, long line)
{
    int i = 0;
    int retval = -1;    /* ファイル最後尾まで来ている場合は-1を返す */
    int character = 0;
    int column = 1;     /* Haplotypeデータの列 */
    int refNum = 0;     /* SNPallelesのパターン数 */
    int index = 0;
    int dataIndex = 0;
    char tmp[20];

    /* 構造体の初期化 */
    snpData->missingDataCount = 0;
    snpData->SNPallelesCount[0] = 0;
    snpData->SNPallelesCount[1] = 0;
    snpData->SNPallelesCount[2] = 0;

    /* １文字ずつ読み込む */
    while ( feof(fp) == 0 ){
        character = fgetc(fp);
        /* １行読み込んだら値を保持して終了 */
        if ('\n' == character){
            snpData->dataNum = dataIndex;
            snpData->allelesNum = refNum;
            retval = 0;
            dataIndex = 0;
            break;
        }
        /* ファイルの２行目以降から処理を行う */
        else if (line > 0){
            /* ファイル内の列をカウント */
            if (isspace(character) != 0){
                if (HAPLOTYPE_RS_NUM == column){
                    snpData->rsNumber[index] = '\0';
                }
                else if (HAPLOTYPE_POS == column){
                    tmp[index] = '\0';
                    snpData->pos = atol(tmp);
                }
                else if (HAPLOTYPE_CHROM == column){
                    snpData->chrom[index] = '\0';
                }
                column++;
                index = 0;
            }
            /* rs#を取得する */
            else if (HAPLOTYPE_RS_NUM == column){
                if (isalnum(character) != 0){
                    snpData->rsNumber[index] = character;
                    index++;
                }
            }
            /* SNPallelsを取得する */
            else if (HAPLOTYPE_ALLELES == column){
                if (isalpha(character) != 0){
                    snpData->SNPalleles[refNum] = character;
                    refNum++;
                }
            }
            /* chromを取得する */
            else if (HAPLOTYPE_CHROM == column){
                if (isalnum(character) != 0){
                    snpData->chrom[index] = character;
                    index++;
                }
            }
            /* posを取得する */
            else if (HAPLOTYPE_POS == column){
                tmp[index] = character;
                index++;
            }
            /* 7列目移行のデータを参照する */
            else if (column >= HAPLOTYPE_DATA){
                for (i = 0; i < refNum; i++){
                    /* SNPallelesと一致するデータをカウント */
                    if (snpData->SNPalleles[i] == character){
                        snpData->SNPallelesCount[i]++;
                    }
                }
                /* Nと一致するデータをカウント */
                if ('N' == character){
                    snpData->missingDataCount++;
                }
                /* 遺伝子データを格納 */
                snpData->SNPdata[dataIndex] = character;
                dataIndex++;
            }
        }
    }
    return retval;
}

/*********************************************/
/* phasingされたHapmapデータを構造体に収める */
/*********************************************/
int DataReaderSetAllPhasingHapmapData(FILE *fp, SnpData *snpData, long line)
{
    int retval = 0;
    long i = 0;
    long sampleNum = 0;

    /* phasingされたHapmapデータのサンプル数を取得する */
    sampleNum = DataReaderGetPhasingHapmapSampleNum(fp);
    /* ファイルから読み込み */
    for (i = 1; i < line; i++){
        /* SNPデータ格納用配列のメモリ確保 */
        snpData[i].SNPdata = (char*)malloc1Dim(sizeof(char), sampleNum);
        retval = DataReaderSetPhasingHapmapData(fp, &snpData[i], i);
    }

    return retval;
}

/*************************************************/
/* phasingされたHapmapデータ一行を構造体に収める */
/*************************************************/
int DataReaderSetPhasingHapmapData(FILE *fp, SnpData *snpData, long line)
{
    int i = 0;
    int retval = -1;    /* ファイル最後尾まで来ている場合は-1を返す */
    int character = 0;
    int column = 1;     /* データの列 */
    int refNum = 0;     /* SNPallelesのパターン数 */
    int index = 0;
    int dataIndex = 0;
    int flag = 0;
    char tmp[20];

    /* 構造体の初期化 */
    snpData->missingDataCount = 0;
    snpData->SNPallelesCount[0] = 0;
    snpData->SNPallelesCount[1] = 0;
    snpData->SNPallelesCount[2] = 0;

    /* chromを仮設定 */
    strcpy(snpData->chrom, "----");

    /* １文字ずつ読み込む */
    while ( feof(fp) == 0 ){
        character = fgetc(fp);
        /* １行読み込んだら値を保持して終了 */
        if ('\n' == character){
            snpData->dataNum = dataIndex;
            snpData->allelesNum = refNum;
            retval = 0;
            dataIndex = 0;
            break;
        }
        /* ファイルの２行目以降から処理を行う */
        else if (line > 0){
            /* ファイル内の列をカウント */
            if (isspace(character) != 0){
                if (PHASE_RS_NUM == column){
                    snpData->rsNumber[index] = '\0';
                }
                else if (PHASE_POS == column){
                    tmp[index] = '\0';
                    snpData->pos = atol(tmp);
                }
                column++;
                index = 0;
            }
            /* rs#を取得する */
            else if (PHASE_RS_NUM == column){
                if (isalnum(character) != 0){
                    snpData->rsNumber[index] = character;
                    index++;
                }
            }
            /* posを取得する */
            else if (PHASE_POS == column){
                tmp[index] = character;
                index++;
            }
            /* 3列目以降のデータを参照する */
            else if (column >= PHASE_DATA){
                /* Nと一致するデータをカウント */
                if ('N' == character){
                    snpData->missingDataCount++;
                } 
                else{
                    for (i = 0; i < refNum; i++){
                        /* SNPallelesと一致するデータをカウント */
                        if (snpData->SNPalleles[i] == character){
                            snpData->SNPallelesCount[i]++;
                            flag = 1;
                        }
                    } 
                    /* フラグがたっていない場合、２つ目（３つ目）のNPallelesを取得 */
                    if (flag == 0){
                        /* SNPallelesを取得 */
                        snpData->SNPalleles[refNum] = character;
                        /* SNPallelesと一致するデータをカウント */
                        snpData->SNPallelesCount[refNum]++;
                        refNum++;
                    }
                    /* フラグのクリア */
                    flag = 0;
                }
                /* 遺伝子データを格納 */
                snpData->SNPdata[dataIndex] = character;
                dataIndex++;
            }
        }
    }
    return retval;
}

/*********************************************/
/* 構造体SnpDataにサンプルデータをコピーする */
/*********************************************/
int DataReaderDataCopyToSnpData(SnpData *snpData, char *sampleData, long dataNum, long sampleNum)
{
    long i = 0;

    for (i = 0; i < dataNum; i++){
        /* サンプルデータのアドレス代入 */
        snpData[i].SNPdata = &sampleData[i * sampleNum];
    }

    return 0;
}

/*****************************************/
/* 構造体SnpDataのメモリを確保する */
/*****************************************/
int DataReaderSnpDataMemoryAllocate(SnpData *snpData, long dataNum, long sampleNum)
{
    long i = 0;

    /* 最初に初期化しておく */
    for (i = 0; i < dataNum; i++){
        snpData[i].SNPdata = NULL;
    }

    /* メモリを確保する */
    for (i = 0; i < dataNum; i++){
        /* SNPデータ格納用配列のメモリ確保 */
        snpData[i].SNPdata = (char*)malloc1Dim(sizeof(char), sampleNum);
        if (NULL == snpData[i].SNPdata){
            return -1;
        }
    }

    return 0;
}

/*****************************************/
/* 構造体SnpDataメンバのメモリを開放する */
/*****************************************/
int DataReaderSnpDataMemoryFree(SnpData *snpData, long dataNum)
{
    long i = 0;

    if (NULL != snpData){
        for (i = 0; i < dataNum; i++){
            /* SNPデータ格納用配列のメモリ開放 */
            free1Dim(snpData[i].SNPdata);
        }
        free1Dim(snpData);
    }

    return 0;
}

/*****************************************/
/* haplotypeブロック領域を配列に格納する */
/*****************************************/
int DataReaderSetHaplotypeBlockArea(FILE *fp, long *blockArea)
{
    long index = 0;
    long strIndex = 0;
    long valIndex = 0;
    char string[MAX_LEN];
    char value[MAX_LEN];

    /* 一行ずつ読み込む */
    while ( (fgets(string, MAX_LEN, fp)) != NULL){
        strIndex = 0;
        valIndex = 0;
        while (string[strIndex] != '\n'){
            value[valIndex] = string[strIndex];
            strIndex++;
            valIndex++;
        }
        value[valIndex] = '\0';
        blockArea[index] = atol(value);
        index++;
    }
    return 0;
}

/********************************************/
/* すべての構造体データをファイルに出力する */
/********************************************/
//void DataReaderOutputAllData(FILE *fp, SnpData *snpData, long line)
//{
//    long i = 0;
//
//    for (i = 0; i < line; i++){
//        DataReaderOutputData(fp, &snpData[i]);
//
//    }
//}

/************************************/
/* 構造体データをファイルに出力する */
/************************************/
//void DataReaderOutputData(FILE *fp, SnpData *snpData)
//{
//    int j = 0;
//
//    fprintf(fp, "%10s, %10ld, ", snpData->rsNumber, snpData->pos);
//    for (j = 0; j < snpData->allelesNum; j++){
//        fprintf(fp, "%c=>%3d, ", snpData->SNPalleles[j], snpData->SNPallelesCount[j]);
//    }
//    fprintf(fp, "N=>%3d, DataNum=>%3d, ", snpData->missingDataCount, snpData->dataNum);
//    /* 各人が持つ遺伝子のデータは保持しない */
//    //for (j = 0; j < snpData->dataNum*2; j++){
//    //    fprintf(fp, "%c", snpData->SNPdata[j]);
//    //}
//    fprintf(fp, "\n");
//}

/******************************/
/* 入力データを構造体に収める */
/******************************/
//int DataReaderSetInputData(char *filename, InputData *inputData)
//{
//    int retval = 0;
//    FILE *fp = NULL;
//    char tmp[MAX_LEN];
//
//    /* ファイルオープン */
//    retval = InputFileOpen(&fp, filename);
//    if (retval != 0){
//        return -1;
//    }
//
//    retval = DataReaderGetValue(fp, "InputFile1", tmp);
//    if(retval == 0){
//        strcpy(inputData->inputFile1, tmp);
//    }
//    retval = DataReaderGetValue(fp, "InputFile2", tmp);
//    if(retval == 0){
//        strcpy(inputData->inputFile2, tmp);
//    }
//    retval = DataReaderGetValue(fp, "OutputFile1", tmp);
//    if(retval == 0){
//        strcpy(inputData->outputFile1, tmp);
//    }
//    retval = DataReaderGetValue(fp, "OutputLevel", tmp);
//    if(retval == 0){
//        inputData->outputLevel = atoi(tmp);
//    }
//    retval = DataReaderGetValue(fp, "Score", tmp);
//    if(retval == 0){
//        inputData->score = atoi(tmp);
//    }
//    retval = DataReaderGetValue(fp, "Repeat", tmp);
//    if(retval == 0){
//        inputData->repeat = atol(tmp);
//    }
//    retval = DataReaderGetValue(fp, "Analysis", tmp);
//    if(retval == 0){
//        inputData->analysis = atoi(tmp);
//    }
//
//    /* ファイルクローズ */
//    FileClose(fp);
//
//    return retval;
//}

/******************************************************/
/* 入力設定ファイルから指定したキーに対応する値を返す */
/******************************************************/
//int DataReaderGetValue(FILE *fp, char *key, char *value)
//{
//    int len = 0;
//    int strIndex = 0;
//    int valIndex = 0;
//    int retval = -1;
//    char string[MAX_LEN];
//
//    len = strlen(key);
//
//    /* ファイルポインタを先頭に戻す */
//    fseek(fp, 0L, SEEK_SET);
//
//    /* 一行ずつ読み込む */
//    while ( (fgets(string, MAX_LEN, fp)) != NULL){
//        /* 指定したキーが見つかった場合、値を取得する */
//        if (strncmp(key, string, len) == 0){
//            strIndex = len + 1;
//            while (string[strIndex] != '\n'){
//                value[valIndex] = string[strIndex];
//                strIndex++;
//                valIndex++;
//            }
//            value[valIndex] = '\0';
//            retval = 0;
//        }
//    }
//    return retval;
//}

/************************************/
/* ２つの入力データから偶現表を作成 */
/************************************/
int DataReaderMakeTable(SnpData *snpData1, SnpData *snpData2, int **T)
{
    /* SNPallelesが２つの場合 */
    if ( (2 == snpData1->allelesNum) && (2 == snpData2->allelesNum) ){
        /* 偶現表の作成 */
        T[0][0] = snpData1->SNPallelesCount[0];
        T[0][1] = snpData1->SNPallelesCount[1];
        T[1][0] = snpData2->SNPallelesCount[0];
        T[1][1] = snpData2->SNPallelesCount[1];
    }
    /* SNPallelesが３つの場合 */
    else if ( (3 == snpData1->allelesNum) && (3 == snpData2->allelesNum) ){
        /* 偶現表の作成 */
        T[0][0] = snpData1->SNPallelesCount[0];
        T[0][1] = snpData1->SNPallelesCount[1] + snpData1->SNPallelesCount[2];
        T[1][0] = snpData2->SNPallelesCount[0];
        T[1][1] = snpData2->SNPallelesCount[1] + snpData2->SNPallelesCount[2];
    }
    return 0;
}

/***********************************/
/* ２つのSNPalleleの順番入れ替える */
/***********************************/
int DataReaderShiftSNPalleles(SnpData *snpData, int allele1, int allele2)
{
    int tmpCount = 0;
    char tmpSNP = 0;

    /* 元のデータを一時退避 */
    tmpSNP = snpData->SNPalleles[allele1];
    tmpCount = snpData->SNPallelesCount[allele1];
    /* 順番入れ替え */
    snpData->SNPalleles[allele1] = snpData->SNPalleles[allele2];
    snpData->SNPalleles[allele2] = tmpSNP;
    snpData->SNPallelesCount[allele1] = snpData->SNPallelesCount[allele2];
    snpData->SNPallelesCount[allele2] = tmpCount;

    return 0;
}

/********************************************/
/* ２つのデータのSNPallelesの順番をチェック */
/********************************************/
int DataReaderCheckSNPalleles(SnpData *snpData1, SnpData *snpData2)
{
    int i = 0;
    int j = 0;
    int index = 0;
    int count = 0;

    /* SNPallelesが２つの場合 */
    if ( (2 == snpData1->allelesNum) && (2 == snpData2->allelesNum) ){
        /* ２つのデータでSNPallelesの順番が同じ場合、何もしない */
        if ( (snpData1->SNPalleles[0] == snpData2->SNPalleles[0]) && (snpData1->SNPalleles[1] == snpData2->SNPalleles[1]) ){
            return 0;
        }
        /* ２つのデータでSNPallelesの順番が逆転している場合、順番を入れ替える */
        else if ( (snpData1->SNPalleles[0] == snpData2->SNPalleles[1]) && (snpData1->SNPalleles[1] == snpData2->SNPalleles[0]) ){
            /* ２つのSNPalleleの順番入れ替える */
            DataReaderShiftSNPalleles(snpData2, 0, 1);
            return 0;
        }
        /* ２つのデータでSNPallelesが異なっている場合 */
        else{
            return -1;
        }
    }
    /* SNPallelesが３つの場合 */
    else if ( (3 == snpData1->allelesNum) && (3 == snpData2->allelesNum) ){
        /* １つのデータでgenotyoe１とするSNPを決定する */
        for (i = 1; i < 3; i++){
            if(snpData1->SNPallelesCount[i] > snpData1->SNPallelesCount[index]){
                index = i;
            }
        }
        if (index != 0){
            /* genotyoe１とするSNPデータを配列の先頭にする */
            /* ２つのSNPalleleの順番入れ替える */
            DataReaderShiftSNPalleles(snpData1, 0, index);
        }
        /* snpData1とsnpData2のSNPデータの並びをそろえる */
        for (i = 0; i < 3; i++){
            for (j = 0; j < 3; j++){
                if (snpData1->SNPalleles[i] == snpData2->SNPalleles[j]){
                    count++;
                    if (i == j){
                        break;
                    }
                    else{
                        /* snpData2のSNPデータの並び順を変更する */
                        /* ２つのSNPalleleの順番入れ替える */
                        DataReaderShiftSNPalleles(snpData2, i, j);
                        break;
                    }
                }
            }
        }
        /* ２つのデータが同じSNPallelesでなかった場合 */
        if (count != 3){
            return -1;
        }
        return 0;
    }
    /* caseのSNPallelesが２つ、controlのSNPallelesが１つの場合 */
    /* PhasingHapmapでcaseのデータにA/Tが存在し、controlのデータにAしか存在しないような場合 */
    else if ( (2 == snpData1->allelesNum) && (1 == snpData2->allelesNum) ){
        /* １つ目のSNPalleleが同じ場合 */
        if (snpData1->SNPalleles[0] == snpData2->SNPalleles[0]){
            /* controlに２つ目のSNPalleleをコピー */
            snpData2->SNPalleles[1] = snpData1->SNPalleles[1];
            snpData2->allelesNum++;
            return 0;
        }
        /* １つ目のSNPalleleが違う場合 */
        else if (snpData1->SNPalleles[1] == snpData2->SNPalleles[0]){
            /* controlに２つ目のSNPalleleをコピー */
            snpData2->SNPalleles[0] = snpData1->SNPalleles[1];
            snpData2->allelesNum++;
            /* ２つのSNPalleleの順番入れ替える */
            DataReaderShiftSNPalleles(snpData2, 0, 1);
            return 0;
        }
    }
    /* caseのSNPallelesが１つ、controlのSNPallelesが２つの場合 */
    /* PhasingHapmapでcaseのデータにAが存在し、controlのデータにA/Tが存在するような場合 */
    else if ( (1 == snpData1->allelesNum) && (2 == snpData2->allelesNum) ){
        /* １つ目のSNPalleleが同じ場合 */
        if (snpData1->SNPalleles[0] == snpData2->SNPalleles[0]){
            /* controlに２つ目のSNPalleleをコピー */
            snpData1->SNPalleles[1] = snpData2->SNPalleles[1];
            snpData1->allelesNum++;
            return 0;
        }
        /* １つ目のSNPalleleが違う場合 */
        else if (snpData1->SNPalleles[0] == snpData2->SNPalleles[1]){
            /* caseに２つ目のSNPalleleをコピー */
            snpData1->SNPalleles[1] = snpData2->SNPalleles[0];
            snpData1->allelesNum++;
            /* ２つのSNPalleleの順番入れ替える */
            DataReaderShiftSNPalleles(snpData2, 0, 1);
            return 0;
        }
    }
    /* caseのSNPallelesが１つ、controlのSNPallelesが１つの場合 */
    /* PhasingHapmapでcase、controlのデータ共にAしか存在しないような場合 */
    else if ( (1 == snpData1->allelesNum) && (1 == snpData2->allelesNum) ){
        /* １つ目のSNPalleleが同じ場合 */
        if (snpData1->SNPalleles[0] == snpData2->SNPalleles[0]){
            /* ２つ目のSNPalleleを仮代入 */
            snpData1->SNPalleles[1] = '-';
            snpData2->SNPalleles[1] = '-';
            snpData1->allelesNum++;
            snpData2->allelesNum++;
            return 0;
        }
        /* １つ目のSNPalleleが違う場合 */
        else{
            /* caseに２つ目のSNPalleleをコピー */
            snpData1->SNPalleles[1] = snpData2->SNPalleles[0];
            snpData1->allelesNum++;
            /* controlに２つ目のSNPalleleをコピー */
            snpData2->SNPalleles[1] = snpData1->SNPalleles[0];
            snpData2->allelesNum++;
            /* ２つのSNPalleleの順番入れ替える */
            DataReaderShiftSNPalleles(snpData2, 0, 1);
            return 0;
        }
    }
    else{
        return -1;
    }
    return 0;
}

/******************************************************************/    
/* 入力データの整合性をチェックして並列化用の入力データを作成する */
/******************************************************************/
long DataReaderMakeParallelData(SnpData *snpTmpData1, SnpData *snpTmpData2, long line1, long line2, SnpData **snpData1, SnpData **snpData2, char **caseData, char **controlData)
{
    int retval = 0;
    long a = 0;
    long b = 0;
    long index = 0;
    long dataCount = 0;
    long caseIndex = 1;
    long controlIndex = 1;

    a = snpTmpData1[1].dataNum;
    b = snpTmpData2[1].dataNum;

    while ( caseIndex < line1 || controlIndex < line2 ){
        /* データのposを比較し、同じ箇所の遺伝子であることを確認 */
        if (snpTmpData1[caseIndex].pos == snpTmpData2[controlIndex].pos){
            /* ２つのデータのSNPallelesの順番をチェック */
            retval = DataReaderCheckSNPalleles(&snpTmpData1[caseIndex], &snpTmpData2[controlIndex]);
            if(0 == retval){
                /* Data数をカウント */
                dataCount++;
            }
            /* snpTmpData1(case)の次のデータを参照する */
            caseIndex++;
            /* snpTmpData2の次のデータを参照する */
            controlIndex++;
        }
        else if (snpTmpData1[caseIndex].pos < snpTmpData2[controlIndex].pos){
            /* snpData1(case)の次のデータを参照する */
            caseIndex++;
        }
        else if (snpTmpData1[caseIndex].pos > snpTmpData2[controlIndex].pos){
            /* snpData2(control)の次のデータを参照する */
            controlIndex++;
        }
    }

    /* データ格納用構造体のメモリ確保 */
    *snpData1 = (SnpData*)malloc1Dim(sizeof(SnpData), dataCount);
    *snpData2 = (SnpData*)malloc1Dim(sizeof(SnpData), dataCount);
    /* サンプルデータ格納用char型配列のメモリ確保 */
    *caseData = (char*)malloc1Dim(sizeof(char), dataCount * a);
    *controlData = (char*)malloc1Dim(sizeof(char), dataCount * b);

    caseIndex = 1;
    controlIndex = 1;
    while ( caseIndex < line1 || controlIndex < line2 ){
        /* データのposを比較し、同じ箇所の遺伝子であることを確認 */
        if (snpTmpData1[caseIndex].pos == snpTmpData2[controlIndex].pos){
            /* ２つのデータのSNPallelesの順番をチェック */
            retval = DataReaderCheckSNPalleles(&snpTmpData1[caseIndex], &snpTmpData2[controlIndex]);
            if(0 == retval){
                /* 並列化データの作成 */
                //(*snpData1)[index] = snpTmpData1[caseIndex];
                strcpy( (*snpData1)[index].rsNumber, snpTmpData1[caseIndex].rsNumber );
                strcpy( (*snpData1)[index].SNPalleles, snpTmpData1[caseIndex].SNPalleles );
                strcpy( (*snpData1)[index].chrom, snpTmpData1[caseIndex].chrom );
                (*snpData1)[index].pos = snpTmpData1[caseIndex].pos;
                strncpy( &(*caseData)[index * a], snpTmpData1[caseIndex].SNPdata, a );
                (*snpData1)[index].allelesNum = snpTmpData1[caseIndex].allelesNum;
                (*snpData1)[index].dataNum = snpTmpData1[caseIndex].dataNum;
                (*snpData1)[index].SNPallelesCount[0] = snpTmpData1[caseIndex].SNPallelesCount[0];
                (*snpData1)[index].SNPallelesCount[1] = snpTmpData1[caseIndex].SNPallelesCount[1];
                (*snpData1)[index].SNPallelesCount[2] = snpTmpData1[caseIndex].SNPallelesCount[2];
                (*snpData1)[index].missingDataCount = snpTmpData1[caseIndex].missingDataCount;
                //(*snpData2)[index] = snpTmpData2[controlIndex];
                strcpy( (*snpData2)[index].rsNumber, snpTmpData2[controlIndex].rsNumber );
                strcpy( (*snpData2)[index].SNPalleles, snpTmpData2[controlIndex].SNPalleles );
                strcpy( (*snpData2)[index].chrom, snpTmpData2[controlIndex].chrom );
                (*snpData2)[index].pos = snpTmpData2[controlIndex].pos;
                strncpy( &(*controlData)[index * b], snpTmpData2[controlIndex].SNPdata, b );
                (*snpData2)[index].allelesNum = snpTmpData2[controlIndex].allelesNum;
                (*snpData2)[index].dataNum = snpTmpData2[controlIndex].dataNum;
                (*snpData2)[index].SNPallelesCount[0] = snpTmpData2[controlIndex].SNPallelesCount[0];
                (*snpData2)[index].SNPallelesCount[1] = snpTmpData2[controlIndex].SNPallelesCount[1];
                (*snpData2)[index].SNPallelesCount[2] = snpTmpData2[controlIndex].SNPallelesCount[2];
                (*snpData2)[index].missingDataCount = snpTmpData2[controlIndex].missingDataCount;

                index++;
            }
            /* snpTmpData1(case)の次のデータを参照する */
            caseIndex++;
            /* snpTmpData2の次のデータを参照する */
            controlIndex++;
        }
        else if (snpTmpData1[caseIndex].pos < snpTmpData2[controlIndex].pos){
            /* snpData1(case)の次のデータを参照する */
            caseIndex++;
        }
        else if (snpTmpData1[caseIndex].pos > snpTmpData2[controlIndex].pos){
            /* snpData2(control)の次のデータを参照する */
            controlIndex++;
        }
    }
    
    return dataCount;
}


/******************************************/
/* 入力ファイルの情報をファイルに出力する */
/******************************************/
//void DataReaderOutputHeader(FILE *fp, InputData *inputData)
//{
//    fprintf(fp, "case Data    => %s\n", inputData->inputFile1);
//    fprintf(fp, "control Data => %s\n\n", inputData->inputFile2);
//    if (0 == iWay){
//        fprintf(fp, "スコア計算：PearsonScore\n");
//    }
//    else{
//        fprintf(fp, "スコア計算：Fst\n");
//    }
//
//    switch(inputData->outputLevel){
//        case 0:
//            /* 検定結果ファイル出力（level0） */
//            break;
//        case 1:
//            /* 検定結果ファイル出力（level1） ヘッダー出力*/
//            fprintf(fp, "\nrs# chrom pos alleles p score a1 a2 b1 b2\n");
//            break;
//        default:
//            /* 検定結果ファイル出力（level0） */
//            break;
//    }
//}

/********************************/
/* 検定結果をファイルに出力する */
/********************************/
//void DataReaderOutputAllResult(int level, FILE *fp, SnpData *snpData1, SnpData *snpData2, OutputData *outputData, InputData *inputData, long dataNum)
//{
//    long i = 0;
//
//    /* 入力ファイル情報をファイル出力 */
//    DataReaderOutputHeader(fp, inputData);
//
//    for (i = 0; i < dataNum; i++){
//        DataReaderOutputResult(level, fp, &snpData1[i], &snpData2[i], &outputData[i], inputData->repeat);
//
//    }
//}

/********************************************/
/* １遺伝子分の検定結果をファイルに出力する */
/********************************************/
//void DataReaderOutputResult(int level, FILE *fp, SnpData *snpData1, SnpData *snpData2, OutputData *outputData, long repeat)
//{
//    switch(level){
//        case 0:
//            /* 検定結果ファイル出力（level0） */
//            DataReaderOutputResultLevel0(fp, snpData1, snpData2, outputData, repeat);
//            break;
//        case 1:
//            /* 検定結果ファイル出力（level1） */
//            DataReaderOutputResultLevel1(fp, snpData1, snpData2, outputData);
//            break;
//        default:
//            /* 検定結果ファイル出力（level0） */
//            DataReaderOutputResultLevel0(fp, snpData1, snpData2, outputData, repeat);
//            break;
//    }
//}

/**********************************/
/* 検定結果ファイル出力（level0） */
/**********************************/
//void DataReaderOutputResultLevel0(FILE *fp, SnpData *snpData1, SnpData *snpData2, OutputData *outputData, long repeat)
//{
//    int a1 = 0;
//    int a2 = 0;
//    int b1 = 0;
//    int b2 = 0;
//
//    /* ファイル出力 */
//    fprintf(fp, "\n");
//    fprintf(fp, "*** SnpData Infomation ***\n");
//    fprintf(fp, "rs# => %10s : chrom => %5s : pos => %10ld : ", snpData1->rsNumber, snpData1->chrom, snpData1->pos);
//    if (2 == snpData1->allelesNum){
//        fprintf(fp, "SNPalleles => %c/%c \n", snpData1->SNPalleles[0], snpData1->SNPalleles[1]);
//        a1 = snpData1->SNPallelesCount[0];
//        a2 = snpData1->SNPallelesCount[1];
//        b1 = snpData2->SNPallelesCount[0];
//        b2 = snpData2->SNPallelesCount[1];
//    }
//    else if (3 == snpData1->allelesNum){
//        fprintf(fp, "SNPalleles => %c/%c%c \n", snpData1->SNPalleles[0], snpData1->SNPalleles[1], snpData1->SNPalleles[2]);
//        a1 = snpData1->SNPallelesCount[0];
//        a2 = snpData1->SNPallelesCount[1] + snpData1->SNPallelesCount[2];
//        b1 = snpData2->SNPallelesCount[0];
//        b2 = snpData2->SNPallelesCount[1] + snpData2->SNPallelesCount[2];
//    }
//    fprintf(fp, "  ** Observer **\n");
//    fprintf(fp, "    Table   | genotype1 | genotype2 |  sum  || NNデータ数 | サンプル人数\n");
//    fprintf(fp, "    case    | %9d | %9d | %4d  || %10d | %12d\n", a1, a2, a1+a2, snpData1->missingDataCount, snpData1->dataNum);
//    fprintf(fp, "    control | %9d | %9d | %4d  || %10d | %12d\n", b1, b2, b1+b2, snpData2->missingDataCount, snpData2->dataNum);
//    fprintf(fp, "    sum     | %9d | %9d | %4d\n", a1+b1, a2+b2, a1+a2+b1+b2);
//    fprintf(fp, "    Sobs => %.10lf\n", outputData->Sobs);
//    fprintf(fp, "    Pr => %.10e\n", exp(outputData->Pr));
//    fprintf(fp, "  ** Permutation（ランダム関数） **\n");
//    fprintf(fp, "    count(S>=Sobs) => %d : repeat(roop) => %ld\n", outputData->count, repeat);
//    fprintf(fp, "    count / repeat => %.10lf\n", (double)outputData->count/repeat);
//    fprintf(fp, "  ** 正確法 p = Hj/F **\n");
//    //fprintf(fp, "    Hj => %.10e\n", outputData->Hj);
//    //fprintf(fp, "    F  => %.10e\n", outputData->F);
//    fprintf(fp, "    p  => %.10e\n", outputData->p);
//}

/**********************************/
/* 検定結果ファイル出力（level1） */
/**********************************/
//void DataReaderOutputResultLevel1(FILE *fp, SnpData *snpData1, SnpData *snpData2, OutputData *outputData)
//{
//    int a1 = 0;
//    int a2 = 0;
//    int b1 = 0;
//    int b2 = 0;
//
//    /* ファイル出力 */
//    //fprintf(fp, "\nrs# chrom pos alleles p score a1 a2 b1 b2\n");
//    fprintf(fp, "%s %s %ld ", snpData1->rsNumber, snpData1->chrom, snpData1->pos);
//    if (2 == snpData1->allelesNum){
//        fprintf(fp, "%c/%c ", snpData1->SNPalleles[0], snpData1->SNPalleles[1]);
//        a1 = snpData1->SNPallelesCount[0];
//        a2 = snpData1->SNPallelesCount[1];
//        b1 = snpData2->SNPallelesCount[0];
//        b2 = snpData2->SNPallelesCount[1];
//    }
//    else if (3 == snpData1->allelesNum){
//        fprintf(fp, "%c/%c%c ", snpData1->SNPalleles[0], snpData1->SNPalleles[1], snpData1->SNPalleles[2]);
//        a1 = snpData1->SNPallelesCount[0];
//        a2 = snpData1->SNPallelesCount[1] + snpData1->SNPallelesCount[2];
//        b1 = snpData2->SNPallelesCount[0];
//        b2 = snpData2->SNPallelesCount[1] + snpData2->SNPallelesCount[2];
//    }
//    fprintf(fp, "%.10e %.10lf %d %d %d %d\n", outputData->p, outputData->Sobs, a1, a2, b1, b2);
//}

/**************************************/
/* 入力データのサンプル数を取得する */
/**************************************/
long DataReaderGetSampleNum(FILE *fp, int dataType)
{
    long retval = 0;

    switch(dataType){
        case 0:
            /* 入力データがHapmap */
            retval = DataReaderGetHapmapSampleNum(fp);
            break;
        case 1:
            /* 入力データがHaplotypeData */
            retval = DataReaderGetHaplotypeSampleNum(fp);
            break;
        case 2:
            /* 入力データがPhasingHapmap */
            retval = DataReaderGetPhasingHapmapSampleNum(fp);
            break;
        default:
            break;
    }

    return retval;
}

/*****************************************/
/* HapmapDataのサンプル数を取得する */
/*****************************************/
long DataReaderGetHapmapSampleNum(FILE *fp)
{
    int character = 0;
    long column = 1;     /* Haplotypeデータの列 */
    long dataNum = 0;    /* サンプル人数 */

    /* １文字ずつ読み込む */
    while ( feof(fp) == 0 ){
        character = fgetc(fp);
        /* １行読み込んだら値を返して終了 */
        if ('\n' == character){
            return dataNum * 2;
        }
        /* ファイル内の列をカウント */
        else if (isspace(character) != 0){
            column++;
            /* 12列目移行のデータ数をカウント */
            if (column >= HAPMAP_DATA){
                dataNum++;
            }
        }
    }

    /* HapmapDataは一人（１列）で２つの遺伝子データを持っているので２倍する */
    return dataNum * 2;
}

/***************************************/
/* HaplotypeDataのサンプル数を取得する */
/***************************************/
long DataReaderGetHaplotypeSampleNum(FILE *fp)
{
    int character = 0;
    long column = 1;     /* Haplotypeデータの列 */
    long dataNum = 0;    /* サンプル数 */

    /* １文字ずつ読み込む */
    while ( feof(fp) == 0 ){
        character = fgetc(fp);
        /* １行読み込んだら値を返して終了 */
        if ('\n' == character){
            return dataNum;
        }
        /* ファイル内の列をカウント */
        else if (isspace(character) != 0){
            column++;
            /* 7列目移行のデータ数をカウント */
            if (column >= HAPLOTYPE_DATA){
                dataNum++;
            }
        }
    }
    return dataNum;
}

/***************************************************/
/* phasingされたHapmapデータのサンプル数を取得する */
/***************************************************/
long DataReaderGetPhasingHapmapSampleNum(FILE *fp)
{
    int character = 0;
    long column = 1;     /* phasingされたHapmapデータの列 */
    long dataNum = 0;    /* サンプル数 */

    /* １文字ずつ読み込む */
    while ( feof(fp) == 0 ){
        character = fgetc(fp);
        /* １行読み込んだら値を返して終了 */
        if ('\n' == character){
            return dataNum;
        }
        /* ファイル内の列をカウント */
        else if (isspace(character) != 0){
            column++;
            /* 3列目移行のデータ数をカウント */
            if (column >= PHASE_DATA){
                dataNum++;
            }
        }
    }
    return dataNum;
}

/***********************************************/
/* HaplotypeData一行の遺伝子データを配列に格納 */
/***********************************************/
int DataReaderSetHaplotypeSequence(FILE *fp, char **sequence, int index)
{
    int retval = -1;    /* ファイル最後尾まで来ている場合は-1を返す */
    int character = 0;
    int column = 1;     /* Haplotypeデータの列 */
    int dataIndex = 0;

    /* 構造体の初期化 */

    /* １文字ずつ読み込む */
    while ( feof(fp) == 0 ){
        character = fgetc(fp);
        /* １行読み込んだら値を保持して終了 */
        if ('\n' == character){
            retval = 0;
            break;
        }
        /* ファイル内の列をカウント */
        else if (isspace(character) != 0){
            column++;
        }
        /* 12列目移行のデータを参照する */
        else if (column >= HAPLOTYPE_DATA){
            /* 遺伝子データを格納 */
            sequence[dataIndex][index] = character;
            dataIndex++;
        }
    }
    return retval;
}

/***************************/
/* 各Haplotypeの頻度を計算 */
/***************************/
int DataReaderCalcHaplotypeFrequency(char **sequence, long dataNum, int L, double **frequency)
{
    int i = 0;
    int total = 0;
	int patern = 0;
	int flag = 0;
	int index = 0;
	int nextIndex = 0;
    int retval = -1;

    int *reference = NULL;
	int *count = NULL;

    /* メモリ確保 */
	reference = (int*)malloc1Dim(sizeof(int), dataNum);
    if (NULL == reference) { goto finalize; }
	count = (int*)malloc1Dim(sizeof(int), dataNum);
    if (NULL == count) { goto finalize; }

    while(total < dataNum){
        count[patern] = 0;
		flag = 0;
		nextIndex = 0;

		for(i = index; i < dataNum; i++){
            /* すでに検出されたパターンの二重カウント防止 */
			if(reference[i] == 0){
                /* 参照パターンと同じパターンであればカウントする */
				if(strncmp(sequence[index], sequence[i], L)==0){
					reference[i] = 1;
					count[patern]++;
					total++;
				}
				else{
                    /* 次の参照パターンを記憶 */
					if(nextIndex == 0){
						nextIndex = i;
					}
				}
			}
		}
		patern++;
		index = nextIndex;
	}

    /* 頻度格納用メモリ確保 */
	*frequency = (double*)malloc1Dim(sizeof(double), patern);
    if (NULL == *frequency) { goto finalize; }
    /* 頻度を計算 */
    for (i = 0; i < patern; i++){
        (*frequency)[i] = (double)count[i] / total;
    }
    retval = patern;

finalize:;
    /* 確保したメモリを開放する */
    free1Dim(reference);
    free1Dim(count);

    return retval;
}

/************************/
/* populationTypeを計算 */
/************************/
int DataReaderPopulationType(SnpData *snpData1, SnpData *snpData2, int *populationType)
{
    long a = 0;
    long b = 0;
    long n = 0;
    long h = 0;

    a = snpData1->dataNum;
    b = snpData2->dataNum;
    n = a+b;

    for (h = 0; h < a; h++){
        populationType[h] = 0;
    }
    for (h = a; h < n; h++){
        populationType[h] = 1;
    }

    return 0;
}

/****************************************/
/* ２つの入力データとDiから偶現表を作成 */
/****************************************/
int DataReaderMakeTableDi(SnpData *snpData1, SnpData *snpData2, int *d, int **T)
{
    int i = 0;
    int j = 0;
    long h = 0;
    long a = 0;
    long b = 0;
    long n = 0;
    long len = 0;
    char reference = 0;

    int *data = NULL;
    int *data1 = NULL;
    int *data2 = NULL;

    a = snpData1->dataNum;
    b = snpData2->dataNum;
    n = a + b;
   
    /* dataのメモリ確保 */
    data = (int*)malloc1Dim(sizeof(int), n);
    if (NULL == data) { goto finalize; }
    /* data1のメモリ確保 */
    data1 = (int*)malloc1Dim(sizeof(int), a);
    if (NULL == data1) { goto finalize; }
    /* data2のメモリ確保 */
    data2 = (int*)malloc1Dim(sizeof(int), b);
    if (NULL == data2) { goto finalize; }

    /* 参照文字を決定 */
    reference = DataReaderMaximumNuc(snpData1);

    /* genotypeを計算 */
    DataReaderCalcGenotype(snpData1, reference, data1);
    DataReaderCalcGenotype(snpData2, reference, data2);
    for (h = 0; h < a; h++){
        data[h] = data1[h];
    }
    for (h = 0; h < b; h++){
        data[h + a] = data2[h];
    }

    /* 偶現表を0で初期化する */
    for (i = 0; i < ROW; i++){
        for (j = 0; j < COLUMN; j++){
            T[i][j] = 0;
        }
    }
    /* 偶現表を作成 */
    for (h = 0; h < n; h++){
        T[ d[h] ][ data[h] ]++;
    }

finalize:;
    /* 確保したメモリを開放する */
	free1Dim(data);
    free1Dim(data1);
    free1Dim(data2);

    return 0;
}

/************************************************************************/
/* ２つの入力データとDiから偶現表を作成（呼び出し側で必要メモリ確保版） */
/* 2010.11.10 高速化のため、呼び出し側で確保したメモリを使いまわす      */
/************************************************************************/
int DataReaderMakeTableDiMemoryGiven(
    SnpData *snpData1,
    SnpData *snpData2,
    int *d,
    int **T,
    int *data1,
    int *data2,
    int *data)
{
    int i = 0;
    int j = 0;
    long h = 0;
    long a = 0;
    long b = 0;
    long n = 0;
    long len = 0;
    char reference = 0;

    a = snpData1->dataNum;
    b = snpData2->dataNum;
    n = a + b;
   
    /* 参照文字を決定 */
    reference = DataReaderMaximumNuc(snpData1);

    /* genotypeを計算 */
    DataReaderCalcGenotype(snpData1, reference, data1);
    DataReaderCalcGenotype(snpData2, reference, data2);
    for (h = 0; h < a; h++){
        data[h] = data1[h];
    }
    for (h = 0; h < b; h++){
        data[h + a] = data2[h];
    }

    /* 偶現表を0で初期化する */
    for (i = 0; i < ROW; i++){
        for (j = 0; j < COLUMN; j++){
            T[i][j] = 0;
        }
    }
    /* 偶現表を作成 */
    for (h = 0; h < n; h++){
        T[ d[h] ][ data[h] ]++;
    }

    return 0;
}

/******************/
/* 参照文字を返す */
/******************/
char DataReaderMaximumNuc(SnpData *snpData)
{
    int tmp = 0;
    int max = 0;
    int maxi = 0;
    int count[5] = {0, 0, 0, 0, 0};
    char nuc[5] = {'N', 'T', 'C', 'A', 'G'};
    long h = 0;
    long maxIndex = 0;

    maxIndex = snpData->dataNum;

    for (h = 0; h < maxIndex; h++){
        tmp = 0;
        switch (snpData->SNPdata[h]){
            case 'U': tmp = 1;
                    break;
            case 'u': tmp = 1;
                    break;
            case 'T': tmp = 1;
                    break;
            case 't': tmp = 1;
                    break;
            case 'C': tmp = 2;
                    break;
            case 'c': tmp = 2;
                    break;
            case 'A': tmp = 3;
                    break;
            case 'a': tmp = 3;
                    break;
            case 'G': tmp = 4;
                    break;
            case 'g': tmp = 4;
                    break;
            default:  break;
        }
        count[tmp]++;
    }

    for (tmp = 1; tmp < 5; tmp++){
        if (max < count[tmp]){
            max = count[tmp];
            maxi = tmp;
        }
    }

    return nuc[maxi];
}

/******************/
/* genotypeを計算 */
/******************/
int DataReaderCalcGenotype(SnpData *snpData, char reference, int *result)
{
    long h = 0;
    long maxIndex = 0;

    maxIndex = snpData->dataNum;

    for (h = 0; h < maxIndex; h++){
        if (snpData->SNPdata[h] == reference){
            result[h] = 0;
        }
        else{
            result[h] = 1;
        }
    }

    return 0;
}

/***************************/
/* 各Haplotypeの頻度を計算 */
/***************************/
long DataReaderHaplotypeFrequency(SnpData *snpData1, SnpData *snpData2, long jStart, long jEnd, char ***haplotype, double **freq)
{
    int retval = -1;
    long i = 0;
    long j = 0;
    long h = 0;
    long a = 0;
    long b = 0;
    long m = 0;
    long maxHaplotypeNumber = 0;
    long total = 0;
	long patern = 0;
	long index = 0;
	long nextIndex = 0;

    char **allHaplotype = NULL; /* すべてのhaplotypeデータを格納用 */
    int *referenceFlag = NULL;
	int *count = NULL;
	int *tmp = NULL;

    m = jEnd - jStart;

    /* サンプル人数 */
    a = snpData1[0].dataNum;
    b = snpData2[0].dataNum;
    maxHaplotypeNumber = a + b;

    /* allHaplotypeのメモリ確保 */
    allHaplotype = (char**)mallocChar2Dim(maxHaplotypeNumber, m);
    if (NULL == allHaplotype) { goto finalize; }
    /* メモリ確保 */
	referenceFlag = (int*)malloc1Dim(sizeof(int), maxHaplotypeNumber);
    if (NULL == referenceFlag) { goto finalize; }
	count = (int*)malloc1Dim(sizeof(int), maxHaplotypeNumber);
    if (NULL == count) { goto finalize; }
	tmp = (int*)malloc1Dim(sizeof(int), maxHaplotypeNumber);
    if (NULL == tmp) { goto finalize; }

    /* すべてのhaplotypeデータを格納 */
    for (j = jStart; j < jEnd; j++){
        for (h = 0; h < a; h++){
            allHaplotype[h][index] = snpData1[j].SNPdata[h];
        }
        for (h = 0; h < b; h++){
            allHaplotype[h + a][index] = snpData2[j].SNPdata[h];
        }
        index++;
    }

    index = 0;
    while(total < maxHaplotypeNumber){
        count[patern] = 0;
		nextIndex = 0;
        tmp[patern] = index;

		for(i = index; i < maxHaplotypeNumber; i++){
            /* すでに検出されたパターンの二重カウント防止 */
			if(referenceFlag[i] == 0){
                /* 参照パターンと同じパターンであればカウントする */
				if(strncmp(allHaplotype[index], allHaplotype[i], m)==0){
					referenceFlag[i] = 1;
					count[patern]++;
					total++;
				}
				else{
                    /* 次の参照パターンを記憶 */
					if(nextIndex == 0){
						nextIndex = i;
					}
				}
			}
		}
		patern++;
		index = nextIndex;
	}

    /* haplotype格納用メモリ確保 */
    *haplotype = (char**)mallocChar2Dim(patern, m);
    if (NULL == haplotype) { goto finalize; }
    /* 頻度格納用メモリ確保 */
	*freq = (double*)malloc1Dim(sizeof(double), patern);
    if (NULL == freq) { goto finalize; }

    for (i = 0; i < patern; i++){
        /* 出現したhaplotypeパターンを格納 */
        strncpy((*haplotype)[i], allHaplotype[ tmp[i] ], m);
        /* 頻度を計算 */
        (*freq)[i] = (double)count[i] / total;
    }
    retval = patern;

finalize:;
    /* 確保したメモリを開放する */
    freeChar2Dim(allHaplotype, maxHaplotypeNumber);
    free1Dim(referenceFlag);
    free1Dim(count);
    free1Dim(tmp);

    return retval;
}

/*****************************/
/* SnpDataをシーケンスに変換 */
/*****************************/
int DataReaderSequences(SnpData *snpData, long jStart, long jEnd, char **sequence)
{
    long j = 0;
    long h = 0;
    long m = 0;
	long index = 0;
    long dataNum = 0;

    /* サンプル数 */
    dataNum = snpData[0].dataNum;

    /* すべてのhaplotypeデータを格納 */
    for (j = jStart; j < jEnd; j++){
        for (h = 0; h < dataNum; h++){
            sequence[h][index] = snpData[j].SNPdata[h];
        }
        index++;
    }

    return 0;
}

/**************************************/
/* DNA配列データをSnpDataの配列に変換 */
/**************************************/
int DataReaderSequenceToSnpData(char **sequences, SnpData *snpData, long dataNum, long size)
{
    /* dataNum：SNP数 */
    /* size：サンプル人数（a or b） */
    long j = 0;
    long h = 0;

    /* 縦と横を変換 */
    for (j = 0; j < dataNum; j++){
        snpData[j].dataNum = size;
        snpData[j].pos = j;
        for (h = 0; h < size; h++){
            snpData[j].SNPdata[h] = sequences[h][j];
        }
    }

    return 0;
}
