/*
 * Paraselene
 * Copyright (c) 2009  Akira Terasaki
 * このファイルは同梱されているLicense.txtに定めた条件に同意できる場合にのみ
 * 利用可能です。
 */
package paraselene.tag.table;


import java.util.*;
import paraselene.*;
import paraselene.tag.*;

/**
 * 行を表す。trタグ。
 */
public class Line extends Tag {
	private static final long serialVersionUID = 2L;
	/**
	 * コンストラクタ。
	 */
	public Line() {
		super( Table.TR, false );
	}

	protected Tag newReplica() {
		return new Line();
	}

	void setDirty() {
		Tag	tag = getOuterTag();
		if ( !(tag instanceof Table) )	return;
		Table	table = (Table)tag;
		if ( table != null )	table.setDirty();
	}

	void removeMe() {
		Table	table = (Table)getOuterTag();
		if ( table != null )	table.removeLine( this );
	}

	/**
	 * 列の列挙。td(th)タグを検出し配列にして返します。
	 * @return 列。
	 */
	public Column[] getColumnArray() {
		Tag[]	tag = getTagArray();
		ArrayList<Column>	clm = new ArrayList<Column>();
		for ( int i = 0; i < tag.length; i++ ) {
			if ( tag[i] instanceof Column ) {
				clm.add( (Column)tag[i] );
			}
		}
		return clm.toArray( new Column[0] );
	}

	/**
	 * 列の取得。
	 * 指定位置のtd(th)タグを返します。
	 * @param no 列番号。
	 * @return 列。
	 */
	public Column getColumn( int no ) {
		if ( no < 0 )	return null;
		Column[]	tag = getColumnArray();
		if ( no >= tag.length )	return null;
		return tag[no];
	}

	/**
	 * 列の追加。末尾に追加されます。
	 * @param column 追加列。
	 */
	public void addColumn( Column column ) {
		setDirty();
		addHTMLPart( column );
	}

	/**
	 * 列の追加。指定位置に追加します。
	 * 現在の最大列よりも大きい位置を指定すると末尾に追加します。
	 * @param idx 列番号。
	 * @param column 追加列。
	 */
	public void addColumn( int idx, Column column ) {
		setDirty();
		Column[]	clm = getColumnArray();
		if ( clm.length <= idx ) {
			addHTMLPart( column );
			return;
		}
		addHTMLPart( indexOf( clm[idx] ), column );
	}

	/**
	 * 列への一括値設定。startから順にstrを設定します。
	 * strの個数が列数を上回ると無視されます。
	 * @param start 設定開始列。
	 * @param str 設定値。
	 */
	public void setValueString( int start, String ... str ) {
		if ( str == null )	return;
		Column[]	clm = getColumnArray();
		for ( int i = 0; (i + start) < clm.length && i < str.length; i++ ) {
			clm[i + start].setValueString( str[i] );
		}
	}

	/**
	 * 列番号を返す。最初に登場した指定インスタンスの位置を返します。
	 * @param column 検索列。
	 * @return 列番号。無ければ-1を返す。
	 */
	public int indexOfColumn( Column column ) {
		Column[]	clm = getColumnArray();
		for ( int i = 0; i < clm.length; i++ ) {
			if ( clm[i] == column )	return i;
		}
		return -1;
	}

	/**
	 * 列の削除。指定列を削除します。
	 * 現在の最大列よりも大きい位置を指定すると何もしません。
	 * @param idx 列番号。
	 */
	public void removeColumn( int idx ) {
		setDirty();
		Column[]	clm = getColumnArray();
		if ( clm.length <= idx )	return;
		removeHTMLPart( clm[idx] );
		if ( getColumnArray().length == 0 )	removeMe();
	}

	/**
	 * 列の削除。最初に登場した指定インスタンスを削除します。
	 * @param column 削除列。
	 */
	public void removeColumn( Column column ) {
		setDirty();
		removeHTMLPart( column );
		if ( getColumnArray().length == 0 )	removeMe();
	}

	/**
	 * 列の内容のクリア。全ての列のタグ内部を消去します。
	 */
	public void clearColumnInner() {
		clearColumnInner( 0, Integer.MAX_VALUE );
	}

	/**
	 * 列の内容のクリア。列のタグ内部を消去します。<br>
	 * start ～ (start + count - 1)の列がクリア対象となります。
	 * @param start クリア対象の開始列番号。列の範囲外を指定すると何もしません。
	 * @param count クリア個数。列数を越えるとそこまでを対象とします。
	 */
	public void clearColumnInner( int start, int count ) {
		Column[]	clm = getColumnArray();
		count += start;
		if ( count > clm.length )	count = clm.length;
		if ( start < 0 )	start = 0;
		for ( ; start < count; start++ ) {
			clm[start].removeHTMLPart();
		}
	}

	int getColumnRowCount() {
		int	cnt = 0;
		Column[]	clm = getColumnArray();
		for ( int i = 0; i < clm.length; i++ ) {
			cnt += clm[i].getColspan();
		}
		return cnt;
	}
}



