/*
 * Paraselene
 * Copyright (c) 2009-2012  Akira Terasaki
 * このファイルは同梱されているLicense.txtに定めた条件に同意できる場合にのみ
 * 利用可能です。
 */
package paraselene.dyna;


import paraselene.*;
import paraselene.supervisor.*;
import java.io.*;
import java.util.*;

/**
 * HTMLテンプレート。DynamicPage による、実行時の外観変更処理をサポートします。
 * 以下の条件を満たす場合、このクラスは正しく機能します。
 * <ul>
 <li>サーバーのローカルディスクにあるHTMLをテンプレートとして読み込む。
 <li>テンプレートHTMLファイル名と、スケルトンソース生成時のHTMLファイル名が一致する。
 <li>テンプレートHTMLのディレクトリ階層構造が、スケルトンソース生成時のそれと
 一致する。
 * </ul><br>
 * このクラスは、{@link Dominion}と{@link DynamicPageReloader}を利用して動作します。
 * そのため、以下の動作となります。
 * <ul>
 <li>テンプレートHTML中に登場する相対パスURLで別のテンプレートHTMLを指すものは、
 正しい遷移先となるようURLを書き換えます。
 <li>テンプレートHTML中に登場する相対パスURLで静的コンテンツを指すものは、
 正しい参照先となるようURLを書き換えます。
 <li>テンプレートHTML中に登場する絶対パスURIは変更しません。
 <li>ParamタグのURI判定は、与えられた名称群と突き合わせて判定します。
 <li>テンプレートHTMLはバックグラウンドで、HTTPリクエストと非同期に
 ファイル読み込みを行います。
 その読み込み周期等の説明は{@link DynamicPageReloader}を参照して下さい。
 * </ul>
 */
public class Skin {
	private Dominion	pr;
	private String	dec;
	private GrantTagProvider[]	gtp;
	private HashMap<String, String>	name = new HashMap<String, String>();
	/**
	 * コンストラクタ。
	 * @param dominion 相対パス対応表。
	 * @param enc URIデコード時に使用する文字コード。
 	 * @param provider 派生クラス置換用。
 	 {@link paraselene.ui.UI}は自動的に含まれます。
	 * @param param_name &lt;param&gt;タグでURIと判断する name 属性。
	 */
	public Skin( Dominion dominion, String enc, GrantTagProvider[] provider, String ... param_name ) {
		pr = dominion;
		dec = enc;
		gtp = provider;
		for ( String n: param_name )	name.put( n.toLowerCase( Locale.ENGLISH ), n );
	}

	private class Resolver implements URIResolver {
		private PageID	id;
		private String	other;

		private Resolver( PageID p, String other_path ) {
			id = p;
			other = other_path;
		}
		public String resolve( String src ) throws Exception {
			Dominion.Path	path = pr.resolve( id, src, dec );
			String	file = path.getPath();
			if ( path.getType() == Dominion.Path.Type.RELATIVE ) {
				return Dominion.fix( other, file, true );
			}
			return file;
		}
		public boolean isParamURIName( String n ) {
			return name.get( n.toLowerCase( Locale.ENGLISH ) ) != null;
		}
	}

	private class Key {
		private File	file;
		private PageID	id;
		private int	hash;
		private Key( File f, PageID pid ) {
			file = f;
			id = pid;
			hash = (~file.hashCode()) ^ pid.getID();
		}
		public int hashCode() { return hash; }
		public boolean equals( Object obj ) {
			if ( !(obj instanceof Key) )	return false;
			Key	key = (Key)obj;
			return key.id == id && key.file.equals( file );
		}
	}
	private HashMap<Key, DynamicPageReloader.Magazine>	magazin
		= new HashMap<Key, DynamicPageReloader.Magazine>();
	/**
	 * ページの検証。
	 */
	public interface PageInspecter {
		/**
		 * ページ検証。テンプレートとして適用不可ならば例外をスローして下さい。
		 * {@link Skin#apply(File,String,Page,PageInspecter)}は例外をスローして
		 * テンプレート適用を中断します。
		 * @param page 検証するテンプレートを読み込んだページ。
		 * @exception Exception テンプレート適用処理の中断。
		 */
		void inspect( DynamicPage page ) throws Exception;
	}
	/**
	 * テンプレートHTML適用。
	 * @param root テンプレートHTMLの格納ディレクトリ。
	 ここが、スケルトン生成時の -html 引数で指定したディレクトリに相当します。
	 * @param other_path 静的コンテンツに付与する/から始まるパス。
	 スケルトン生成時の -other 引数で指定したディレクトリに相当します。
	 * @param destination テンプレート適用先のインスタンス。
	 * @param inspecter ページ検証。検証不要ならnull。
	 * @exception DynamicPageException テンプレートHTMLに誤りがある場合などに
	 * スローします。
	 */
	public void apply( File root, String other_path, Page destination, PageInspecter inspecter ) throws DynamicPageException {
		if ( other_path.charAt( 0 ) != '/' )	throw new DynamicPageException( "bad directory[" + other_path + "]" );
		DynamicPage	page = null;
		PageID	id = destination.getID();
		Key	key = new Key( root, id );
		synchronized( magazin ) {
			DynamicPageReloader.Magazine	mz = magazin.get( key );
			if ( mz != null )	page = mz.getPage();
			else {
				page = new DynamicPage( new Resolver( id, other_path ), destination, gtp );
				page.create( new File( pr.getHTMLPath( root.getPath(), id ) ), null );
				mz = DynamicPageReloader.issue( page );
				magazin.put( key, mz );
			}
			DynamicPageException	e = mz.getLastError();
			if ( e != null )	throw e;
		}
		try {
			if ( inspecter != null )	inspecter.inspect( page );
		}
		catch( Exception e ) {
			throw new DynamicPageException( e );
		}
		destination.moveMainTag( page );
	}
}

