#ifndef SYNC_H
#define SYNC_H

#include "libpards.h"

#ifdef PARDS_USE_EXCEPTION
#include "PardsException.h"
#endif

template <class T> class Sync {
 private:
  int read_sem_id;
  int write_sem_id;
  int *is_written; // 0: value is not written, 1: written
  T *value;
 public:
  static void* operator new(size_t size);
  static void operator delete(void* obj);
  Sync();
  virtual ~Sync() {}; // Do nothing: (need to add delete to derived classes)
  T read();
  T* readptr();
  void write(T val);
  void free();
  void free_sem();
  void free_shm();
};

template <class T> void* Sync<T>::operator new(size_t size)
{
  void* ptr = pards_shmalloc(size);

#ifdef PARDS_USE_EXCEPTION  
  if(ptr == 0) throw MemoryException();
#endif
  else return ptr;
}

template <class T> void Sync<T>::operator delete(void* obj)
{
  ((Sync<T>*)obj)->free();
  pards_shmfree(obj);
  return;
}

template <class T> Sync<T>::Sync()
{
  read_sem_id = -1;
  write_sem_id = -1;
  is_written = 0;
  value = 0;
  read_sem_id = pards_semget();
  if(read_sem_id == -1){
#ifdef PARDS_USE_EXCEPTION
    throw SemaphoreException();
#endif
    return;
  }
  write_sem_id = pards_semget();
  if(write_sem_id == -1){
    pards_semfree(read_sem_id);
    read_sem_id = -1; // avoid multiple free at delete 
#ifdef PARDS_USE_EXCEPTION
    throw SemaphoreException();
#endif
    return;
  }

  is_written = (int*)pards_shmalloc(sizeof(int));
  if(is_written == 0){
    pards_semfree(read_sem_id);
    pards_semfree(write_sem_id);
    read_sem_id = -1; // avoid multiple free at delete 
    write_sem_id = -1; 
#ifdef PARDS_USE_EXCEPTION  
    throw MemoryException();
#endif
    return;
  }
  *is_written = 0;

  value = (T*)pards_shmalloc(sizeof(T));
  if(value == 0){
    pards_semfree(read_sem_id);
    pards_semfree(write_sem_id);
    pards_shmfree(is_written);
    read_sem_id = -1; // avoid multiple free at delete 
    write_sem_id = -1;
    is_written = 0;
#ifdef PARDS_USE_EXCEPTION  
    throw MemoryException();
#endif
    return;
  }

  pards_lock(read_sem_id);
}

template <class T> T Sync<T>::read()
{
#if 0  // deletion after read might overtake writer's unlock
  if(*is_written){
#ifndef NO_EXTEND_SHM
    attach_shm();
#endif
    return *value; // already set
  } else
#endif
    pards_lock(read_sem_id); // wait for write

  pards_unlock(read_sem_id); // for other waiting processes

#ifndef NO_EXTEND_SHM
  attach_shm();
#endif
  return *value;  
}

template <class T> T* Sync<T>::readptr() // almost same as read()
{
#if 0  // deletion after read might overtake writer's unlock
  if(*is_written){
#ifndef NO_EXTEND_SHM
    attach_shm();
#endif
    return value; // already set
  }
  else
#endif
    pards_lock(read_sem_id); // wait for write

  pards_unlock(read_sem_id); // for other waiting processes
  
#ifndef NO_EXTEND_SHM
  attach_shm();
#endif

  return value;  
}

template <class T> void Sync<T>::write(T val)
{
  pards_lock(write_sem_id); // lock for changing *is_written
  
  if(*is_written){
    pards_error("Sync::write: Value is already written",INFO);
    pards_unlock(write_sem_id);
    return;
  } else {
    *value = val; // This is "copy"
    *is_written = 1;
    pards_unlock(write_sem_id);
  }

  pards_unlock(read_sem_id); // wake up readers
}

template <class T> void Sync<T>::free_sem()
{
  if(read_sem_id != -1) pards_semfree(read_sem_id);
  if(write_sem_id != -1) pards_semfree(write_sem_id);
}

template <class T> void Sync<T>::free_shm()
{
  if(is_written) pards_shmfree(is_written);
  if(value) pards_shmfree(value);
}

template <class T> void Sync<T>::free()
{
  free_sem();
  free_shm();
}
#endif
