#ifndef SYNCLIST_H
#define SYNCLIST_H

#include "Sync.h"

template <class T> class SyncList : public Sync<T> {
 private:
  int readcdr_sem_id;
  int writecdr_sem_id;
  void init();
 protected:
  int *cdr_is_written; // 0: not written, 1: written
  SyncList<T> **cdr;
 public:
  static void operator delete(void* obj);
  SyncList();
  SyncList(SyncList<T>*);
  SyncList<T> *readcdr();
  void writecdr(SyncList<T>* cdr);
  void free();
  void free_sem();
  void free_shm();
  SyncList<T> *release();
};

template <class T> void SyncList<T>::operator delete(void* obj)
{
  ((SyncList<T>*)obj)->free();
  pards_shmfree(obj);
  return;
}

template <class T> void SyncList<T>::init()
{
  readcdr_sem_id = -1;
  writecdr_sem_id = -1;
  cdr_is_written = 0;
  cdr = 0;
  readcdr_sem_id = pards_semget();
  if(readcdr_sem_id == -1){
#ifdef PARDS_USE_EXCEPTION
    throw SemaphoreException();
#endif
    return;
  }
  writecdr_sem_id = pards_semget();
  if(writecdr_sem_id == -1){
    pards_semfree(readcdr_sem_id);
    readcdr_sem_id = -1; // avoid multiple free at delete 
#ifdef PARDS_USE_EXCEPTION
    throw SemaphoreException();
#endif
    return;
  }

  cdr_is_written = (int*)pards_shmalloc(sizeof(int));
  if(cdr_is_written == 0){
    pards_semfree(readcdr_sem_id);
    pards_semfree(writecdr_sem_id);
    readcdr_sem_id = -1; // avoid multiple free at delete 
    writecdr_sem_id = -1;
#ifdef PARDS_USE_EXCEPTION
    throw MemoryException();
#endif
  }
  *cdr_is_written = 0;

  cdr = (SyncList<T>**)pards_shmalloc(sizeof(SyncList<T>*));
  if(cdr == 0){
    pards_semfree(readcdr_sem_id);
    pards_semfree(writecdr_sem_id);
    pards_shmfree(cdr_is_written);
    readcdr_sem_id = -1; // avoid multiple free at delete
    writecdr_sem_id = -1;
    cdr_is_written = 0;
#ifdef PARDS_USE_EXCEPTION
    throw MemoryException();
#endif
  }

  pards_lock(readcdr_sem_id);
}

template <class T> SyncList<T>::SyncList()
{
  init();
}

template <class T> SyncList<T>::SyncList(SyncList<T>* prev)
{
  init();
  prev->writecdr(this);
}

template <class T> SyncList<T>* SyncList<T>::readcdr()
{
#if 0  // deletion after read might overtake writer's unlock
  if(*cdr_is_written){
#ifndef NO_EXTEND_SHM
    attach_shm();
#endif
    return *cdr; // already set
  } else 
#endif
    pards_lock(readcdr_sem_id); // wait for write

  pards_unlock(readcdr_sem_id); // for other waiting processes

#ifndef NO_EXTEND_SHM
  attach_shm();
#endif

  return *cdr;
}

template <class T> void SyncList<T>::writecdr(SyncList<T>* c)
{
  pards_lock(writecdr_sem_id); // lock for *cdr_is_written
  
  if(*cdr_is_written){
    pards_error("SyncList::writecdr: cdr is already written",INFO);
    pards_unlock(writecdr_sem_id);
    return;
  } else {
    *cdr = c;
    *cdr_is_written = 1;
    pards_unlock(writecdr_sem_id);
  }

  pards_unlock(readcdr_sem_id); // wake up readers
}

template <class T> void SyncList<T>::free_sem()
{
  if(readcdr_sem_id != -1) pards_semfree(readcdr_sem_id);
  if(writecdr_sem_id != -1) pards_semfree(writecdr_sem_id);

  Sync<T>::free_sem();
}

template <class T> void SyncList<T>::free_shm()
{
  if(cdr_is_written) pards_shmfree(cdr_is_written);
  if(cdr) pards_shmfree(cdr);

  Sync<T>::free_shm();
}

template <class T> void SyncList<T>::free()
{
  free_sem();
  free_shm();
}

template <class T> SyncList<T>* SyncList<T>::release()
{
  SyncList<T>* crntcdr;
  crntcdr = readcdr();
  delete this;
  return crntcdr;
}

#endif
