#include "libpards.h"
#ifdef PARDS_USE_EXCEPTION
#include "PardsException.h"
#endif

static int pards_error_level = INFO;

//
// Semaphore List, Shmem ID List, Num of available proc
// Pointer to shmem
//

static int* max_seminfo;

// if == -1, this seminfo[i] is freed and this entry can be reused
static int* seminfo;

static int* current_shminfo;

typedef struct pards_shminfo{
  int shm_id;
  void *addr;
} Shminfo;

static int shm_attached = 1; // local for each process

static Shminfo *shminfo;

static int* avail_procs;

static int* is_finalized;

// Semaphore ID: created by init, shared by all processes
static int sem_proc_id, sem_shm_id;
static int sem_sem_id;

// For alloc-free

static unsigned nalloc; 

typedef long Align;
union header {
  struct {
  union header *ptr;
  unsigned size;
  } s;
  Align x;
};

typedef union header Header;

static Header *basep;
static Header **freepp;

static int signals_to_catch[] = {
  1, // SIGHUP
  2, // SIGINT
  3, // SIGQUIT
  4, // SIGILL
  6, // SIGABRT
  7, // SIGBUS
  8, // SIGFPE
  10,// SIGBUS or SIGUSR1
  11,// SIGSEGV
  //  13,// SIGPIPE
  15,// SIGTERM
  0  // for termination
};

typedef void (*my_sighandler_t)(int); // sighander_t might not be defined

static my_sighandler_t prevsighandler[32];
static void sighandler(int);

void pards_init(int ap, unsigned bytes){
  int shm_id;
  int* shmaddr;
  union semun arg;

  nalloc = (bytes+sizeof(Header)-1)/sizeof(Header); // unit of allocation

  // Allocate system shmem
  int initalloc = (1 + NUM_SEM_ID + // for semaphore table
                   1 +              // for current_shminfo
                   1 +              // for avail_procs
                   1                // for is_finalized
                   ) * sizeof(int) +
    NUM_SHM_ID * sizeof(Shminfo) + // for shminfo
    1 * sizeof(Header) + // for basep
    1 * sizeof(Header*); // for freepp
  
  if((shm_id = shmget(IPC_PRIVATE,initalloc,IPC_CREAT|IPC_PERM)) == -1)
    pards_perror("error creating shm_key", FATAL);

  if((shmaddr = (int*)shmat(shm_id, 0, IPC_CREAT|IPC_PERM)) == (int*)-1)
    pards_perror("error attaching shmaddr",FATAL);

  max_seminfo = shmaddr++;
  *max_seminfo = 0;
  seminfo = shmaddr;
  shmaddr += NUM_SEM_ID;

  current_shminfo = shmaddr++;
  *current_shminfo = 0;
  shminfo = (Shminfo*)shmaddr;
  shmaddr = (int*)(shminfo + NUM_SHM_ID);

  shminfo[(*current_shminfo)].shm_id = shm_id;
  shminfo[(*current_shminfo)].addr = shmaddr;
  (*current_shminfo)++;

  avail_procs = shmaddr++;
  *avail_procs = ap;

  is_finalized = shmaddr++;
  *is_finalized = 0;

  basep = (Header*) shmaddr;

  freepp = (Header**) (basep + 1);
  *freepp = NULL;

  // Create System semaphore
  if((sem_proc_id = semget(IPC_PRIVATE,1,IPC_CREAT|IPC_PERM)) == -1){
    pards_finalize(NO_WAIT);
    pards_perror("error creating sem_proc_id",FATAL);
  }
  seminfo[(*max_seminfo)++] = sem_proc_id;

  arg.val = 0;
  if((semctl(sem_proc_id,0,SETVAL,arg)) == -1){
    pards_finalize(NO_WAIT);
    pards_perror("error initializing sem_proc_id",FATAL);
  }

  if((sem_sem_id = semget(IPC_PRIVATE,1,IPC_CREAT|IPC_PERM)) == -1){
    pards_finalize(NO_WAIT);
    pards_perror("error creating sem_sem_id",FATAL);
  }
  seminfo[(*max_seminfo)++] = sem_sem_id;

  arg.val = 0;
  if((semctl(sem_sem_id,0,SETVAL,arg)) == -1){
    pards_finalize(NO_WAIT);
    pards_perror("error initializing sem_sem_id",FATAL);
  }

  if((sem_shm_id = semget(IPC_PRIVATE,1,IPC_CREAT|IPC_PERM)) == -1){
    pards_finalize(NO_WAIT);
    pards_perror("error creating sem_shm_id",FATAL);
  }
  seminfo[(*max_seminfo)++] = sem_shm_id;

  arg.val = 0;
  if((semctl(sem_shm_id,0,SETVAL,arg)) == -1){
    pards_finalize(NO_WAIT);
    pards_perror("error initializing sem_shm_id",FATAL);
  }

  for(int i = 0; signals_to_catch[i] != 0; i++){
    my_sighandler_t prevhandler;
    if((prevhandler = signal(signals_to_catch[i],sighandler)) == SIG_ERR){
      pards_finalize(NO_WAIT);
      pards_perror("error setting signal handler",FATAL);
    }
    prevsighandler[signals_to_catch[i]] = prevhandler;
  }

  /* disable zombie process */
  struct sigaction sa;
  sa.sa_handler = SIG_IGN;
#ifdef SA_NOCLDWAIT
  sa.sa_flags = SA_NOCLDWAIT;
#else
  sa.sa_flags = 0;
#endif
  sigemptyset(&sa.sa_mask);
  sigaction(SIGCHLD, &sa, NULL);
}

void sighandler(int sig)
{
#ifndef NO_EXTEND_SHM
  int attach_shm();

  if(sig == 11){ // Segmentation Fault
    if(attach_shm()){ // because of non-attached shmem created by other procs?
      pards_error("WARNING: Segmentation Fault caused by non-attached shmem. Use of shmem without system sync?",INFO);
      return;  // try again
    }
  }
#endif
  char str[100];
  sprintf(str,"signal %d is caught", sig);
  pards_error(str,INFO);
  pards_finalize(NO_WAIT);
  if(prevsighandler[sig])
    prevsighandler[sig](sig);
  exit(0);
}

int attach_shm()
{
  int attach_shm_nolock();

  if(*is_finalized)
    pards_error("attach_shm: already finalized",FATAL);

  pards_lock(sem_shm_id);
  int val = attach_shm_nolock();
  pards_unlock(sem_shm_id);
  return val;
}

int attach_shm_nolock()
{
  int attach_done = 0;
  void* shmaddr;
  int i;

  for(i = shm_attached; i < *current_shminfo; i++){
    if((shmaddr =
        (void*)shmat(shminfo[i].shm_id, shminfo[i].addr, 0)) == (void*)-1 ||
       shmaddr != shminfo[i].addr){
      pards_perror("error attaching shmaddr",FATAL);
      return attach_done; // not reached
    }
    attach_done = 1;
  }
  shm_attached = i;
  return attach_done;
}

void pards_finalize(int w){
  int i;
  
  if(w == DO_WAIT){
    int tmp;
    while(wait(&tmp) != -1);
  }

  // can't be guarded by semaphore, since this might be called
  // when semaphores are already removed
  if(*is_finalized){ 
    pards_error("Already finalized",INFO);
    return;
  } else 
    *is_finalized = 1;

  for(i = 0; i < *max_seminfo; i++){
    if(seminfo[i] != -1)
      if(semctl(seminfo[i],0,IPC_RMID) == -1)
        pards_perror("error removing semaphore",CRITICAL);
  }    
  
  // shminfo[0] includes system data, remove last
  for(i = *current_shminfo - 1; i >= 0; i--){
    struct shmid_ds tmp;
    if(shmctl(shminfo[i].shm_id,IPC_RMID,&tmp) == -1)
      pards_perror("error removing shmem",CRITICAL);
  }
}

int pards_semget(int v){
  int sem_id;
  union semun arg;
  int i;


  if(*is_finalized)
    pards_error("pards_semget: already finalized",FATAL);

  if((sem_id = semget(IPC_PRIVATE,1,IPC_CREAT|IPC_PERM)) == -1){
    pards_perror("error creating sem_id",CRITICAL);
    return -1;
  }

  arg.val = v;
  if((semctl(sem_id,0,SETVAL,arg)) == -1)
    pards_perror("error initializing sem_id",CRITICAL);

  pards_lock(sem_sem_id);

  for(i = 0; i < *max_seminfo; i++){
    if(seminfo[i] == -1){
      seminfo[i] = sem_id;
      break;
    }
  }
  if(i == *max_seminfo){
    if(*max_seminfo >= NUM_SEM_ID){
      pards_error("exceed max number of sem_id",CRITICAL);
      if(semctl(sem_id,0,IPC_RMID) == -1)
        pards_perror("error removing semaphore in pards_semget",CRITICAL);
      return -1;
    }
    else
      seminfo[(*max_seminfo)++] = sem_id;
  }

  pards_unlock(sem_sem_id);

  return sem_id;
}

void pards_semfree(int sem_id)
{
  int i;

  if(*is_finalized)
    pards_error("pards_semfree: already finalized",FATAL);

  pards_lock(sem_sem_id);

  if(semctl(sem_id,0,IPC_RMID) == -1)
    pards_perror("error removing semaphore in pards_semfree",CRITICAL);

  for(i = 0; i < *max_seminfo; i++){
    if(seminfo[i] == sem_id){
      seminfo[i] = -1;
      break;
    }
  }

  if(i == *max_seminfo){
    char tmp[200];
    sprintf(tmp,"System inconsistency: removing semaphore not registered? id = %d", sem_id);
    pards_error(tmp,CRITICAL);
  }

  pards_unlock(sem_sem_id);

}

void pards_lock(int sem_id)
{
  struct sembuf op_lock[2];

  op_lock[0].sem_num = 0;
  op_lock[0].sem_op = 0;
  op_lock[0].sem_flg = 0;

  op_lock[1].sem_num = 0;
  op_lock[1].sem_op = 1;
  op_lock[1].sem_flg = 0;

  if(*is_finalized)
    pards_error("pards_lock: already finalized",FATAL);

  if(semop(sem_id,op_lock,2) == -1) {
    pards_perror("error in locking", CRITICAL);
  }
}

void pards_unlock(int sem_id)
{
  struct sembuf op_unlock[1];

  op_unlock[0].sem_num = 0;
  op_unlock[0].sem_op = -1;
  op_unlock[0].sem_flg = 0;


  if(*is_finalized)
    pards_error("pards_unlock: already finalized",FATAL);

  if(semop(sem_id,op_unlock,1) == -1)
    pards_perror("error in unlocking",CRITICAL);
}

void pards_sem_rel(int sem_id)
{
  struct sembuf op_rel[1];

  op_rel[0].sem_num = 0;
  op_rel[0].sem_op = 1;
  op_rel[0].sem_flg = 0;

  if(*is_finalized)
    pards_error("pards_sem_rel: already finalized",FATAL);

  if(semop(sem_id,op_rel,1) == -1) {
    pards_perror("error in pards_sem_rel", CRITICAL);
  }
}

void pards_sem_use(int sem_id)
{
  struct sembuf op_use[1];

  op_use[0].sem_num = 0;
  op_use[0].sem_op = -1;
  op_use[0].sem_flg = 0;

  if(*is_finalized)
    pards_error("pards_sem_use: already finalized",FATAL);

  if(semop(sem_id,op_use,1) == -1) {
    pards_perror("error in pards_sem_use", CRITICAL);
  }
}

void pards_sem_use_val(int sem_id, int val)
{
  struct sembuf op_use_val[1];

  op_use_val[0].sem_num = 0;
  op_use_val[0].sem_op = -val;
  op_use_val[0].sem_flg = 0;

  if(*is_finalized)
    pards_error("pards_sem_use: already finalized",FATAL);
  /*
  int v = semctl(sem_id,0,GETVAL);
  char tmp[100];
  sprintf(tmp,"pards_sem_use_val: current value = %d",v);
  pards_error(tmp,DBG);
  */
  if(semop(sem_id,op_use_val,1) == -1) {
    pards_perror("error in pards_sem_use", CRITICAL);
  }
  /*
  v = semctl(sem_id,0,GETVAL);
  sprintf(tmp,"pards_sem_use_val: after value = %d",v);
  pards_error(tmp,DBG);
  */
}

pid_t pards_fork(int mode){
  pid_t pid;

  if(*is_finalized){
    pards_error("pards_fork: already finalized",FATAL);
    return -1; // not reached
  }

  pards_lock(sem_proc_id);

  if(*avail_procs > 0){ 
    (*avail_procs)--;
    pards_unlock(sem_proc_id);

    if((pid = fork()) == -1){
      pards_perror("error in fork",CRITICAL);
#ifdef PARDS_USE_EXCEPTION      
      throw ForkException();
#endif
      return -1; // not reached
    }
    return pid;
  } else {
    pards_unlock(sem_proc_id);
    if(mode == ERROR_IF_NOT_AVAIL){
      pards_error("too many processes",CRITICAL); // CRITICAL? FATAL?
#ifdef PARDS_USE_EXCEPTION      
      throw ForkException();
#endif
      return -1; // not reached
    } else if (mode == FUNC_IF_NOT_AVAIL){
      pards_error("too many processes, call as function",DBG);
      return -1;
    } else
      return -1; // other mode; not reached...
  }
}

void pards_exit(){

  if(*is_finalized)
    pards_error("pards_exit: already finalized",FATAL);

  pards_lock(sem_proc_id);

  (*avail_procs)++; 

  pards_unlock(sem_proc_id);

  int tmp;
  while(wait(&tmp) != -1);
  _exit(0);
}

// mod might be positive or negative
void pards_modify_avail_procs(int mod){

  if(*is_finalized)
    pards_error("pards_modify_avail_procs: already finalized",FATAL);

  pards_lock(sem_proc_id);

  (*avail_procs)+=mod;

  pards_unlock(sem_proc_id);
}
  

//
// K & R style alloc-free
//

static Header *morecore(unsigned);
static void shm_free_nolock(void*);

void* pards_shmalloc(unsigned nbytes){
  Header *p, *prevp;
  unsigned nunits;

  if(*is_finalized)
    pards_error("pards_shmalloc: already finalized",FATAL);

  pards_lock(sem_shm_id);
#ifndef NO_EXTEND_SHM
  attach_shm_nolock();
#endif

  nunits = (nbytes+sizeof(Header)-1)/sizeof(Header) + 1;
  if((prevp = *freepp) == NULL){
    basep->s.ptr = *freepp = prevp = basep;
    basep->s.size = 0;
  }

  for(p = prevp->s.ptr; ; prevp = p, p = p->s.ptr){
    if(p->s.size >= nunits){
      if(p->s.size == nunits)
        prevp->s.ptr = p->s.ptr;
      else {
        p->s.size -= nunits;
        p += p->s.size;
        p->s.size = nunits;
      }
      *freepp = prevp;

      pards_unlock(sem_shm_id);

      return (void *) (p+1);
    }
    if (p == *freepp)
      if ((p = morecore(nunits)) == NULL){
        pards_unlock(sem_shm_id);
        return NULL;
      }
  }
}
      
static Header *morecore(unsigned nu){
  void *shmaddr;
  Header *up;
  int shm_id;

  if(nu < nalloc) // nalloc is set by pards_init
    nu = nalloc;

  if(*current_shminfo >= NUM_SHM_ID){
    pards_error("exceed max shm id",CRITICAL);
    return NULL;
  }

  if((shm_id = shmget(IPC_PRIVATE,nu * sizeof(Header),
                      IPC_CREAT|IPC_PERM)) == -1){
    pards_perror("error creating shm_key", CRITICAL);
    return NULL;
  }

  shminfo[*current_shminfo].shm_id = shm_id;

  if((shmaddr = (void*)shmat(shm_id, 0, IPC_CREAT|IPC_PERM)) == (void*)-1){
    pards_perror("error attaching shmaddr",CRITICAL);
    return NULL;
  }

  shminfo[*current_shminfo].addr = shmaddr;
  (*current_shminfo)++;
  shm_attached = (*current_shminfo);

  up = (Header*) shmaddr;
  up->s.size = nu;
  shm_free_nolock((void*)(up+1));
  return *freepp;
}

void pards_shmfree(void* ap){
  void shm_free_nolock(void*);

  if(*is_finalized)
    pards_error("shm_free: already finalized",FATAL);

  pards_lock(sem_shm_id);

#ifndef NO_EXTEND_SHM
  attach_shm_nolock();
#endif

  shm_free_nolock(ap);

  pards_unlock(sem_shm_id);
}


static void shm_free_nolock(void* ap){
  Header *bp,*p;

  if(ap == 0) return;
  bp = (Header *)ap - 1;
  for(p = *freepp; !(bp > p && bp < p->s.ptr); p = p->s.ptr)
    if(p >= p->s.ptr && (bp > p || bp < p->s.ptr))
      break;

  if(bp + bp->s.size == p->s.ptr){
    bp->s.size += p->s.ptr->s.size;
    bp->s.ptr = p->s.ptr->s.ptr;
  } else
    bp->s.ptr = p->s.ptr;
  if(p + p->s.size == bp){
    p->s.size += bp->s.size;
    p->s.ptr = bp->s.ptr;
  } else
    p->s.ptr = bp;
  *freepp = p;
}

void pards_error(char* str, int level){
  if(level >= pards_error_level){
      fprintf(stderr,"%s\n",str);
      fflush(stderr);
  }
  if(level == FATAL){
    pards_finalize(); // child process should not do this?
    exit(0); 
  }
}

void pards_perror(char* str, int level){
  if(level >= pards_error_level){
    perror(str); 
  }
  if(level == FATAL){
    pards_finalize(); // child process should not do this?
    exit(errno); 
  }
}

int pards_set_error_level(int level){
  int prev;
  
  prev = pards_error_level;
  pards_error_level = level;
  return prev;
}

int pards_get_error_level(){
  return pards_error_level;
}


int pards_get_nprocs()
{
  int ncpus;
#ifdef _SC_NPROCESSORS_ONLN
  if((ncpus = sysconf(_SC_NPROCESSORS_ONLN)) > 0)
    return ncpus;
  else
    return 1;
#elif defined(__hpux__)
  if((ncpus = mpctl(MPC_GETNUMSPUS,NULL,NULL)) > 0)
    return ncpus;
  else 
    return 1;
#elif defined(__FreeBSD__)
  size_t len = sizeof(ncpus);
  int mib[2];
  mib[0] = CTL_HW;
  mib[1] = HW_NCPU;
  if (sysctl(&mib[0], 2, &ncpus, &len, NULL, 0) == 0 && ncpus > 0)
    return ncpus;
  else
    return 1;
#else
  return 1;
#endif
}

int pards_avail_processes()
{
  return *avail_procs;
}
