#include <stdio.h>

#define MAXCHARLEN 1024 // for pards_errorf

#include <map>
#include <set>
using namespace std;

#include "libpards.h"
#ifdef PARDS_USE_EXCEPTION
#include "PardsException.h"
#endif

static pards_error_level current_error_level = DBG;

static int* current_shminfo;

typedef struct pards_shminfo{
  HANDLE shm_id;
  void *addr;
  int size;
} Shminfo;

// static int shm_attached = 1; // local for each process

static Shminfo *shminf; // name "shminfo" is already used by OS (Linux)

static int shm_attached = 1; // local for each process... not actually used on Windows

static int* avail_procs;
static int* numprocs_towait;
static int* finalize_waiting;

static int* is_finalized;

// Current Sem ID: incremented from maxSemIdInit;
static int* maxSemId;
static int maxSemIdInit = 0;

typedef enum semmsg_type {HOLD, RELEASE, EXIT} semmsg_type;

typedef struct semmsg {
	semmsg_type msg;
	int sem_id;
} Semmsg;

static Semmsg* sem_msg;

// Semaphore ID: created by init, shared by all processes
static int sem_proc_id, sem_shm_id;
static int sem_sem_id, sem_semmgrblk_id, sem_semprocblk_id;
static int sem_finalizewait_id;

// For alloc-free

static unsigned nalloc; 

typedef long Align;
union header {
  struct {
  union header *ptr;
  unsigned size;
  } s;
  Align x;
};

typedef union header Header;

static Header *basep;
static Header **freepp;


// For name of semaphore used in CreateSemaphore

TCHAR my_pid_str[SEMNAMELEN];
void create_semname(TCHAR* dst, int sem_num){
  TCHAR num[SEMNAMELEN];
  _stprintf(num, _TEXT("%d"), sem_num);
  _tcscpy(dst, SEMPFX);
  _tcscat(dst, my_pid_str); 
  _tcscat(dst, num);
}

// For remembering system handles
static set <HANDLE> syssem;
static HANDLE shm_id;

void pards_init(int ap, unsigned bytes){
  int* shmaddr;

  init_nativeapi();
  _stprintf(my_pid_str,_TEXT("%d_"), GetCurrentProcessId());

  nalloc = (bytes+sizeof(Header)-1)/sizeof(Header); // unit of allocation

  // Allocate system shmem
  int initalloc = (1 +              // for maxSemId;
                   1 +              // for current_shminfo
                   1 +              // for avail_procs
				   1 +				// for numprocs_towait
				   1 +				// for finalize_waiting
                   1                // for is_finalized
                   ) * sizeof(int) +
    NUM_SHM_ID * sizeof(Shminfo) + // for shminfo
	sizeof (Semmsg) + 
	1 * sizeof(Header) + // for basep
    1 * sizeof(Header*); // for freepp

  SECURITY_ATTRIBUTES sa;
  sa.nLength = sizeof (sa);
  sa.bInheritHandle = TRUE;
  sa.lpSecurityDescriptor = NULL;
  if((shm_id = CreateFileMapping(INVALID_HANDLE_VALUE, &sa ,PAGE_READWRITE, 0, initalloc, NULL)) == NULL){
	  pards_perror("Error in CreateFileMapping",FATAL);
	  pards_finalize(NO_WAIT);
  }
  if((shmaddr = (int*)MapViewOfFile(shm_id, FILE_MAP_ALL_ACCESS, 0,0,initalloc)) == NULL){
	  pards_perror("Error in MapViewOfFile",FATAL);
      pards_finalize(NO_WAIT);
  }
  maxSemId = shmaddr++;
  *maxSemId = maxSemIdInit;

  current_shminfo = shmaddr++;
  *current_shminfo = 0;
  shminf = (Shminfo*)shmaddr;
  shmaddr = (int*)(shminf + NUM_SHM_ID);

  shminf[(*current_shminfo)].shm_id = shm_id;
  shminf[(*current_shminfo)].addr = shmaddr;
  shminf[(*current_shminfo)].size = initalloc;
  (*current_shminfo)++;

  avail_procs = shmaddr++;
  *avail_procs = ap;

  numprocs_towait = shmaddr++;
  *numprocs_towait = 0;

  finalize_waiting = shmaddr++;
  *finalize_waiting = 0;

  is_finalized = shmaddr++;
  *is_finalized = 0;

  sem_msg = (Semmsg*) shmaddr;
  shmaddr = (int*)(sem_msg + 1);

  basep = (Header*) shmaddr;

  freepp = (Header**) (basep + 1);
  *freepp = NULL;

  int create_system_semaphore(int);
  sem_proc_id = create_system_semaphore(1);
  sem_shm_id = create_system_semaphore(1);
  sem_sem_id = create_system_semaphore(1);
  sem_semmgrblk_id = create_system_semaphore(0);
  sem_semprocblk_id = create_system_semaphore(0);
  sem_finalizewait_id = create_system_semaphore(0);

  if(win32_fork() == 0){
	  void semaphore_keeper();
	  semaphore_keeper();
	  win32_exit(0); // fail safe
  } 

}

int create_system_semaphore(int sem_val)
{
	TCHAR semname[SEMNAMELEN];
	HANDLE h;

	int sem_id = (*maxSemId)++;
	create_semname(semname, sem_id);
	if((h = CreateSemaphore(NULL, sem_val, MAXSEMVAL, semname)) == NULL){
		pards_finalize(NO_WAIT);
		pards_perror("error creating system semaphore",FATAL);
	}
	syssem.insert(h);
	return sem_id;
}

HANDLE pards_opensemaphore(int sem_id)
{
	TCHAR semname[SEMNAMELEN];
	create_semname(semname, sem_id);
	HANDLE h = OpenSemaphore(SEMAPHORE_ALL_ACCESS, TRUE, semname);
//	pards_error("pards_opensemaphore",DBG);
	if (h == NULL) {
		pards_errorf(CRITICAL, "error in pards_opensemaphore: sem_id = %d", sem_id); 
		return NULL;
	} else 
		return h;
}

// For readability
static void pards_wakeup_sem(int sem_id){
	pards_unlock(sem_id);
}

static void pards_wait_sem(int sem_id){
	pards_lock(sem_id);
}

void semaphore_keeper()
{
	map <int, HANDLE> m;
	map <int, HANDLE>::iterator p;

	while(1){
//		pards_error("semaphore_keeper: before wait",DBG);
		pards_wait_sem(sem_semmgrblk_id);
//		pards_error("semaphore_keeper: end wait",DBG);
		// waked up by the requesting process
		if(sem_msg->msg == HOLD) {
			HANDLE h = pards_opensemaphore(sem_msg->sem_id);
			m.insert(pair<int,HANDLE>(sem_msg->sem_id, h));
//			pards_errorf(DBG, "semaphore_keeper: insert id:%d, handle:%d",sem_msg->sem_id, h);
		} else if (sem_msg->msg == RELEASE) {
			p = m.find(sem_msg->sem_id);
			if(p != m.end()){ 
//				pards_errorf(DBG,"semaphore_keeper: close id:%d, handle:%d",sem_msg->sem_id, p->second);
				CloseHandle(p->second);
			}
			else{
				pards_errorf(CRITICAL, "HANDLE %d not found in semaphore_keeper",sem_msg->sem_id);
			}
		} else if (sem_msg->msg == EXIT){
			pards_wakeup_sem(sem_semprocblk_id);
			win32_exit(0);
		}
		pards_wakeup_sem(sem_semprocblk_id);
	}
}

int pards_is_interrupted(){
	pards_error("Interrupt is not supported on Windows", CRITICAL);
	return 0;
}

void pards_clear_interruption(){
	pards_error("Interrupt is not supported on Windows", CRITICAL);
}

// not supported yet
void pards_send_interrupt(pards_pid pid){
	pards_error("Interrupt is not supported on Windows", CRITICAL);
}

void pards_finalize(int w){

  if(w == DO_WAIT){ 
	pards_lock(sem_proc_id);
	if((*numprocs_towait) != 0){
		*finalize_waiting = 1;
		pards_unlock(sem_proc_id);

		pards_lock(sem_finalizewait_id);
		if((*numprocs_towait) != 0)
			pards_error("pards_finalize: remaining running processes",CRITICAL);
	} else
		pards_unlock(sem_proc_id);
  }

  // can't be guarded by semaphore, since this might be called
  // when semaphores are already removed
  if(*is_finalized){ 
    pards_error("Already finalized",INFO);
    return;
  } else 
    *is_finalized = 1;

  pards_lock(sem_sem_id);
  sem_msg->msg = EXIT;
  pards_wakeup_sem(sem_semmgrblk_id);
  pards_wait_sem(sem_semprocblk_id);
  pards_unlock(sem_sem_id);

  set <HANDLE>::iterator p = syssem.begin();
  while (p != syssem.end()){
	  CloseHandle(*p);
	  p++;
  }
  CloseHandle(shm_id);
  maxSemIdInit = *maxSemId++;
}

int pards_semget(int v){
  int sem_id;
  TCHAR name[SEMNAMELEN];
  HANDLE sem_handle;

  if(*is_finalized)
    pards_error("pards_semget: already finalized",FATAL);

  pards_lock(sem_sem_id);

  sem_id = (*maxSemId)++;
  create_semname(name,sem_id);
  if((sem_handle = CreateSemaphore(NULL, v, MAXSEMVAL, name)) == NULL){
    pards_perror("error creating sem_id",CRITICAL);
    return -1;
  }

//  pards_error(DBG, "creating semaphore: %d",sem_id);
  sem_msg->msg = HOLD;
  sem_msg->sem_id = sem_id;
  pards_wakeup_sem(sem_semmgrblk_id);
  pards_wait_sem(sem_semprocblk_id);

  pards_unlock(sem_sem_id);

  return sem_id;
}

void pards_semfree(int sem_id)
{
  if(*is_finalized)
    pards_error("pards_semfree: already finalized",FATAL);

  pards_lock(sem_sem_id);

  sem_msg->msg = RELEASE;
  sem_msg->sem_id = sem_id;
  pards_wakeup_sem(sem_semmgrblk_id);
  pards_wait_sem(sem_semprocblk_id);

  pards_unlock(sem_sem_id);
}

void pards_lock(int sem_id)
{
//	pards_error("pards_lock: before opensemaphore",DBG);
	HANDLE sem = pards_opensemaphore(sem_id);
//	pards_error("pards_lock: after opensemaphore",DBG);
	DWORD status = WaitForSingleObject(sem, INFINITE);
	if(status == WAIT_FAILED) pards_perror("error in pards_lock\n",CRITICAL);
	CloseHandle(sem);
}

pards_status pards_timedlock(int sem_id, struct timeval *tv)
{
	HANDLE sem = pards_opensemaphore(sem_id);
	DWORD status = WaitForSingleObject(sem, tv->tv_sec * 1000 + tv->tv_usec / 1000);
	CloseHandle(sem);
	if (status == WAIT_TIMEOUT) return TIMEOUT;
	else if (status == WAIT_FAILED) {
		pards_perror("error in pards_timedlock",CRITICAL);
		return SUCCESS;
	} else 
		return SUCCESS;
}

void pards_unlock(int sem_id)
{
	HANDLE sem = pards_opensemaphore(sem_id);
	if(ReleaseSemaphore(sem, 1, NULL) == 0) pards_perror("error in pards_unlock", CRITICAL);
	CloseHandle(sem);
}

void pards_sem_rel(int sem_id)
{
	pards_unlock(sem_id);
}

void pards_sem_use(int sem_id)
{
	pards_lock(sem_id);
}

pards_status pards_timed_sem_use(int sem_id, struct timeval *tv)
{
	return pards_timedlock(sem_id, tv);
}

void pards_sem_use_val(int sem_id, int val)
{
	HANDLE sem = pards_opensemaphore(sem_id);
	for(int i = 0; i < val; i++)
		WaitForSingleObject(sem, INFINITE);
	CloseHandle(sem);
}

pid_t pards_fork(int mode){
  pid_t pid;

  if(*is_finalized){
    pards_error("pards_fork: already finalized",FATAL);
    return -1; // not reached
  }

  pards_lock(sem_proc_id);

  if(*avail_procs > 0){ 
    (*avail_procs)--;
	(*numprocs_towait)++;
    
	pards_unlock(sem_proc_id);

    if((pid = win32_fork()) == -1){
      pards_perror("error in fork",CRITICAL);
#ifdef PARDS_USE_EXCEPTION      
      throw ForkException();
#endif
      return -1; // not reached
    }
    return pid;
  } else {
    pards_unlock(sem_proc_id);
    if(mode == ERROR_IF_NOT_AVAIL){
      pards_error("too many processes",CRITICAL); // CRITICAL? FATAL?
#ifdef PARDS_USE_EXCEPTION      
      throw ForkException();
#endif
      return -1; // not reached
    } else if (mode == FUNC_IF_NOT_AVAIL){
      pards_error("too many processes, call as function",DBG);
      return -1;
    } else
      return -1; // other mode; not reached...
  }
}

void pards_exit(){

  if(*is_finalized)
    pards_error("pards_exit: already finalized",FATAL);

  pards_lock(sem_proc_id);

  (*avail_procs)++; 

  (*numprocs_towait)--;

  if((*numprocs_towait) == 0 && (*finalize_waiting) == 1)
	  pards_unlock(sem_finalizewait_id);

  pards_unlock(sem_proc_id);

  /* // can't wait on Windows
  int tmp;
  while(wait(&tmp) != -1);
  */
  win32_exit(0);
}

// mod might be positive or negative
void pards_modify_avail_procs(int mod){

  if(*is_finalized)
    pards_error("pards_modify_avail_procs: already finalized",FATAL);

  pards_lock(sem_proc_id);

  (*avail_procs)+=mod;

  pards_unlock(sem_proc_id);
}
  

//
// K & R style alloc-free
//

static Header *morecore(unsigned);
static void shm_free_nolock(void*);

void* pards_shmalloc(unsigned nbytes){
  Header *p, *prevp;
  unsigned nunits;

  if(*is_finalized)
    pards_error("pards_shmalloc: already finalized",FATAL);

  pards_lock(sem_shm_id);
#ifndef NO_EXTEND_SHM
  attach_shm_nolock();
#endif

  nunits = (nbytes+sizeof(Header)-1)/sizeof(Header) + 1;
  if((prevp = *freepp) == NULL){
    basep->s.ptr = *freepp = prevp = basep;
    basep->s.size = 0;
  }

  for(p = prevp->s.ptr; ; prevp = p, p = p->s.ptr){
    if(p->s.size >= nunits){
      if(p->s.size == nunits)
        prevp->s.ptr = p->s.ptr;
      else {
        p->s.size -= nunits;
        p += p->s.size;
        p->s.size = nunits;
      }
      *freepp = prevp;

      pards_unlock(sem_shm_id);

      return (void *) (p+1);
    }
    if (p == *freepp)
      if ((p = morecore(nunits)) == NULL){
        pards_unlock(sem_shm_id);
        return NULL;
      }
  }
}
      
static Header *morecore(unsigned nu){
  void *shmaddr;
  Header *up;
  HANDLE shm_id;

  if(nu < nalloc) // nalloc is set by pards_init
    nu = nalloc;

  if(*current_shminfo >= NUM_SHM_ID){
    pards_error("out of shared memory (exceed max shm id)",CRITICAL);
    return NULL;
  }
  
  SECURITY_ATTRIBUTES sa;
  sa.nLength = sizeof (sa);
  sa.bInheritHandle = TRUE;
  sa.lpSecurityDescriptor = NULL;
  if((shm_id = CreateFileMapping(INVALID_HANDLE_VALUE, &sa ,PAGE_READWRITE, 0, nu * sizeof(Header), NULL)) == NULL){
	  pards_perror("Error in CreateFileMapping",FATAL);
	  pards_finalize(NO_WAIT);
  }
  if((shmaddr = (int*)MapViewOfFile(shm_id, FILE_MAP_ALL_ACCESS, 0,0, nu * sizeof(Header))) == NULL){
	  pards_perror("Error in MapViewOfFile",FATAL);
      pards_finalize(NO_WAIT);
  }
  shminf[*current_shminfo].addr = shmaddr;
  shminf[*current_shminfo].size = nu * sizeof(Header);
  (*current_shminfo)++;
  shm_attached = (*current_shminfo);

  up = (Header*) shmaddr;
  up->s.size = nu;
  shm_free_nolock((void*)(up+1));
  return *freepp;
}

void pards_shmfree(void* ap){
  void shm_free_nolock(void*);

  if(*is_finalized)
    pards_error("shm_free: already finalized",FATAL);

  pards_lock(sem_shm_id);

#ifndef NO_EXTEND_SHM
  attach_shm_nolock();
#endif

  shm_free_nolock(ap);

  pards_unlock(sem_shm_id);
}


static void shm_free_nolock(void* ap){
  Header *bp,*p;

  if(ap == 0) return;
  bp = (Header *)ap - 1;
  for(p = *freepp; !(bp > p && bp < p->s.ptr); p = p->s.ptr)
    if(p >= p->s.ptr && (bp > p || bp < p->s.ptr))
      break;

  if(bp + bp->s.size == p->s.ptr){
    bp->s.size += p->s.ptr->s.size;
    bp->s.ptr = p->s.ptr->s.ptr;
  } else
    bp->s.ptr = p->s.ptr;
  if(p + p->s.size == bp){
    p->s.size += bp->s.size;
    p->s.ptr = bp->s.ptr;
  } else
    p->s.ptr = bp;
  *freepp = p;
}

int pards_is_in_shm(char* ptr)
{
  for(int i = 1; i < (*current_shminfo); i++){ // 0 is system shmem
    if((char*)shminf[i].addr <= ptr &&
       ptr < (char*)shminf[i].addr + shminf[i].size/sizeof(char)){
      return 1;
    }
  }
  return 0;
}

HANDLE logfile = INVALID_HANDLE_VALUE;

// Only for Windows
void pards_set_logfile(TCHAR* filename)
{
	SECURITY_ATTRIBUTES sa;
	sa.nLength = sizeof (sa);
	sa.bInheritHandle = TRUE;
	sa.lpSecurityDescriptor = NULL;
	if((logfile = CreateFile(filename, GENERIC_ALL, FILE_SHARE_DELETE, &sa, 
		CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL)) == INVALID_HANDLE_VALUE)
		pards_error("Failed to open logfile", CRITICAL);
}

void pards_error(char* str, pards_error_level level){
  if(level >= current_error_level){
      fprintf(stderr,"%s\n",str);
      fflush(stderr);
	  if(logfile != INVALID_HANDLE_VALUE){
		  DWORD nb;
		  WriteFile(logfile, str, strlen(str), &nb, NULL);
		  WriteFile(logfile, "\r\n", 2, &nb, NULL);
	  }
  }
  if(level == FATAL){
    if(!*is_finalized)
      pards_finalize(); // child process should not do this?
    win32_exit(0); 
  }
}

void pards_errorf(pards_error_level level, char* format, ...){
	char buffer[MAXCHARLEN];

	va_list ap;
	va_start(ap, format);
	vsprintf(buffer, format, ap);
	va_end(ap);
	pards_error(buffer, level);
}

void pards_perror(char* str, pards_error_level level){
  if(level >= current_error_level){
	LPTSTR lpMsgBuf;
	FormatMessage(
		FORMAT_MESSAGE_ALLOCATE_BUFFER |
		FORMAT_MESSAGE_FROM_SYSTEM |
		FORMAT_MESSAGE_IGNORE_INSERTS,
		NULL,
		GetLastError(),
//		MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), // ̌
		MAKELANGID(LANG_ENGLISH, SUBLANG_ENGLISH_US),
		(LPTSTR) &lpMsgBuf,
		0,
		NULL
	);
	_tprintf(_TEXT("%s"),lpMsgBuf);
	printf("%s\n",str);
	if(logfile != INVALID_HANDLE_VALUE){
		DWORD nb;
		WriteFile(logfile, lpMsgBuf, _tcslen(lpMsgBuf), &nb, NULL);
		WriteFile(logfile, "\r\n", 2, &nb, NULL);
		WriteFile(logfile, str, strlen(str), &nb, NULL);
		WriteFile(logfile, "\r\n", 2, &nb, NULL);
	}
	LocalFree(lpMsgBuf);

  }
  if(level == FATAL){
    pards_finalize(); // child process should not do this?
    win32_exit(errno); 
  }
}

void pards_perrorf(pards_error_level level, char* format, ...){
	char buffer[MAXCHARLEN];

	va_list ap;
	va_start(ap, format);
	vsprintf(buffer, format, ap);
	va_end(ap);
	pards_perror(buffer, level);
}

void pards_ntsys_perror(char* str, DWORD syserror, pards_error_level level){
  if(level >= current_error_level){
	HMODULE Hand = LoadLibrary(_TEXT("NTDLL.DLL"));
	LPTSTR lpMsgBuf;
	FormatMessage(
		FORMAT_MESSAGE_ALLOCATE_BUFFER |
		FORMAT_MESSAGE_FROM_SYSTEM |
		FORMAT_MESSAGE_FROM_HMODULE,
		Hand,
		syserror,
//		MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), // ̌
		MAKELANGID(LANG_ENGLISH, SUBLANG_ENGLISH_US),
		(LPTSTR) &lpMsgBuf,
		0,
		NULL
	);
	_tprintf(_TEXT("%s\n"),lpMsgBuf);
	printf("%s\n",str);
	if(logfile != INVALID_HANDLE_VALUE){
		DWORD nb;
		WriteFile(logfile, lpMsgBuf, _tcslen(lpMsgBuf), &nb, NULL);
		WriteFile(logfile, "\r\n", 2, &nb, NULL);
		WriteFile(logfile, str, strlen(str), &nb, NULL);
		WriteFile(logfile, "\r\n", 2, &nb, NULL);
	}
	LocalFree(lpMsgBuf);

  }
  if(level == FATAL){
    pards_finalize(); // child process should not do this?
    win32_exit(errno); 
  }
}

pards_error_level pards_set_error_level(pards_error_level level){
  pards_error_level prev;
  
  prev = current_error_level;
  current_error_level = level;
  return prev;
}

pards_error_level pards_get_error_level(){
  return current_error_level;
}


int pards_get_nprocs()
{
	SYSTEM_INFO si;
    GetSystemInfo(&si);
	return si.dwNumberOfProcessors;
}

int pards_avail_processes()
{
  return *avail_procs;
}
