﻿using System;
using System.Collections.Generic;
using System.Text;

using iTextSharp.text.pdf;

namespace PDFAnnotationList
{
    /// <summary>
    /// 注釈クラスです。
    /// </summary>
    class Annotation
    {
        IList<Annotation> _children = new List<Annotation>(); // 返信

        /// <summary>
        /// 作者を取得します。
        /// </summary>
        public string Author { get; private set; }

        /// <summary>
        /// コンテンツを取得します。
        /// </summary>
        public string Contents { get; private set; }

        /// <summary>
        /// 返信のリストを取得します。
        /// </summary>
        public IList<Annotation> Children { get { return _children; } }

        /// <summary>
        /// 作成日時を取得します。
        /// </summary>
        public DateTime CreationDate { get; private set; }

        /// <summary>
        /// 更新日時を取得します。
        /// </summary>
        public DateTime ModifiedDate { get; private set; }

        /// <summary>
        /// ページ番号を取得します。
        /// </summary>
        public int PageNumber { get; private set; }

        /// <summary>
        /// 注釈のPdfDictionaryとページ番号で初期化します。
        /// </summary>
        /// <param name="annotDic">注釈のPdfDictionary。</param>
        /// <param name="pageNumber">ページ番号。</param>
        public Annotation(PdfDictionary annotDic, int pageNumber)
        {
            this.Author = GetString(annotDic, PdfName.T);
            this.Contents = GetString(annotDic, PdfName.CONTENTS);
            this.CreationDate = GetDateTime(annotDic, PdfName.CREATIONDATE);
            this.ModifiedDate = GetDateTime(annotDic, PdfName.M);

            this.PageNumber = pageNumber;
        }

        /// <summary>
        /// PdfDictionaryから日時を取得します。
        /// </summary>
        /// <param name="pdfDic">取得する日時を持つPdfDictionary。</param>
        /// <param name="pdfName">日時のキー。</param>
        /// <returns>pdfNameの日時。該当する日時が無ければDateTime(0)。</returns>
        private static DateTime GetDateTime(PdfDictionary pdfDic, PdfName pdfName)
        {
            try
            {
                return PdfDate.Decode(GetString(pdfDic, pdfName));
            }
            catch
            {
                return new DateTime(0);
            }

        }

        /// <summary>
        /// PdfDictionaryから文字列を取得します。
        /// </summary>
        /// <param name="pdfDic">取得する文字列を持つPdfDictionary。</param>
        /// <param name="pdfName">文字列のキー。</param>
        /// <returns>pdfNameの文字列。該当する文字列が無ければ空文字列。</returns>
        private static string GetString(PdfDictionary pdfDic, PdfName pdfName)
        {
            try
            {
                PdfString result = (PdfString)PdfReader.GetPdfObject(pdfDic.Get(pdfName));
                return result.ToUnicodeString();
            }
            catch
            {
                return "";
            }
        }
    }
}
