/*
 *  acm-FOMA.c
 *    2007.3.31 modified by IZAMUKARERA <pepolinux@users.sourceforge.jp>
 *    cdc-acm ->acm-FOMA ...for docomo FOMA Cellular phone
 *    v0.25-FOMA.02 - patch for kernel-2.6.20                 2007.4.7
 *    v0.25-FOMA.03 - delete acm_unset_modem_mode function    2007.4.14
 *    v0.25-FOMA.04 - for cdc-acm.c update of kernel-2.6.21   2007.4.30
 *    v0.25-FOMA.05 - add 904 Series.                         2007.9.2
 *    v0.25-FOMA.06 - remove productId.                       2007.11.17
 *    v0.26-FOMA.06 - for v0.26                               2009.1.15
 *    v0.26-FOMA.07 - Vend_ID LG Electronics add              2009.2.1
 *    v0.26-FOMA.08 - for kernel-2.6.29                       2009.2.21
 *    v0.26-FOMA.09 - for kernel-2.6.35 later                 2010.11.18
 *    v0.26-FOMA.10 - OBEX & AT command supported, clean up   2011.5.28
 * cdc-acm.c
 * Copyright (c) 1999 Armin Fuerst    <fuerst@in.tum.de>
 * Copyright (c) 1999 Pavel Machek    <pavel@suse.cz>
 * Copyright (c) 1999 Johannes Erdfelt    <johannes@erdfelt.com>
 * Copyright (c) 2000 Vojtech Pavlik    <vojtech@suse.cz>
 * Copyright (c) 2004 Oliver Neukum    <oliver@neukum.name>
 * Copyright (c) 2005 David Kubicek    <dave@awk.cz>
 *
 * USB Abstract Control Model driver for USB modems and ISDN adapters
 *
 * Sponsored by SuSE
 *
 * ChangeLog:
 *    v0.9  - thorough cleaning, URBification, almost a rewrite
 *    v0.10 - some more cleanups
 *    v0.11 - fixed flow control, read error doesn't stop reads
 *    v0.12 - added TIOCM ioctls, added break handling, made struct acm
 *        kmalloced
 *    v0.13 - added termios, added hangup
 *    v0.14 - sized down struct acm
 *    v0.15 - fixed flow control again - characters could be lost
 *    v0.16 - added code for modems with swapped data and control interfaces
 *    v0.17 - added new style probing
 *    v0.18 - fixed new style probing for devices with more configurations
 *    v0.19 - fixed CLOCAL handling (thanks to Richard Shih-Ping Chan)
 *    v0.20 - switched to probing on interface (rather than device) class
 *    v0.21 - revert to probing on device for devices with multiple configs
 *    v0.22 - probe only the control interface. if usbcore doesn't choose the
 *        config we want, sysadmin changes bConfigurationValue in sysfs.
 *    v0.23 - use softirq for rx processing, as needed by tty layer
 *    v0.24 - change probe method to evaluate CDC union descriptor
 *    v0.25 - downstream tasks paralelized to maximize throughput
 *    v0.26 - multiple write urbs, writesize increased
 */

/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#undef DEBUG
#undef VERBOSE_DEBUG
// #define DEBUG
// #define VERBOSE_DEBUG
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/tty.h>
#include <linux/serial.h>
#include <linux/tty_driver.h>
#include <linux/tty_flip.h>
#include <linux/module.h>
#include <linux/mutex.h>
#include <linux/uaccess.h>
#include <linux/usb.h>
#include <linux/usb/cdc.h>
#include <asm/byteorder.h>
#include <asm/unaligned.h>
#include <linux/list.h>
#include <linux/device.h>
#include <linux/mod_devicetable.h>

// #include "cdc-acm.h" is following
/*
 *
 * Includes for cdc-acm.c
 *
 * Mainly take from usbnet's cdc-ether part
 *
 */

/*
 * CMSPAR, some architectures can't have space and mark parity.
 */

#ifndef CMSPAR
#define CMSPAR            0
#endif

/*
 * Major and minor numbers.
 */

#define ACM_TTY_MAJOR        166
#define ACM_TTY_MINORS        32

/*
 * Requests.
 */

#define USB_RT_ACM        (USB_TYPE_CLASS | USB_RECIP_INTERFACE)

/*
 * Output control lines.
 */

#define ACM_CTRL_DTR        0x01
#define ACM_CTRL_RTS        0x02

/*
 * Input control lines and line errors.
 */

#define ACM_CTRL_DCD        0x01
#define ACM_CTRL_DSR        0x02
#define ACM_CTRL_BRK        0x04
#define ACM_CTRL_RI         0x08

#define ACM_CTRL_FRAMING    0x10
#define ACM_CTRL_PARITY     0x20
#define ACM_CTRL_OVERRUN    0x40

/*
 * Internal driver structures.
 */

/*
 * The only reason to have several buffers is to accomodate assumptions
 * in line disciplines. They ask for empty space amount, receive our URB size,
 * and proceed to issue several 1-character writes, assuming they will fit.
 * The very first write takes a complete URB. Fortunately, this only happens
 * when processing onlcr, so we only need 2 buffers. These values must be
 * powers of 2.
 */
#define ACM_NW  16
#define ACM_NR  16

struct acm_wb {
    unsigned char     *buf;
    dma_addr_t dmah;
    int len;
    int use;
    struct urb        *urb;
    struct acm        *instance;
};

struct acm_rb {
    struct list_head    list;
    int            size;
    unsigned char     *base;
    dma_addr_t        dma;
};

struct acm_ru {
    struct list_head    list;
    struct acm_rb     *buffer;
    struct urb        *urb;
    struct acm        *instance;
};

struct acm {
    struct usb_device    *dev;             /* the corresponding usb device */
    struct usb_interface *control;         /* control interface */
    struct usb_interface *data;            /* data interface */
    struct tty_port port;                  /* our tty port data */
    struct urb *ctrlurb;                   /* urbs */
    u8 *ctrl_buffer;                       /* buffers of urbs */
    dma_addr_t ctrl_dma;                   /* dma handles of buffers */
    u8 *country_codes;                     /* country codes from device */
    unsigned int country_code_size;        /* size of this buffer */
    unsigned int country_rel_date;         /* release date of version */
    struct acm_wb wb[ACM_NW];
    struct acm_ru ru[ACM_NR];
    struct acm_rb rb[ACM_NR];
    int rx_buflimit;
    int rx_endpoint;
    spinlock_t read_lock;
    struct list_head spare_read_urbs;
    struct list_head spare_read_bufs;
    struct list_head filled_read_bufs;
    int write_used;                         /* number of non-empty write buffers */
    int processing;
    int transmitting;
    spinlock_t write_lock;
    struct mutex mutex;
    struct usb_cdc_line_coding line;        /* bits, stop, parity */
    struct work_struct work;                /* work queue entry for line discipline waking up */
    wait_queue_head_t drain_wait;           /* close processing */
    struct tasklet_struct urb_task;         /* rx processing */
    spinlock_t throttle_lock;               /* synchronize throtteling and read callback */
    unsigned int ctrlin;                    /* input control lines (DCD, DSR, RI, break, overruns) */
    unsigned int ctrlout;                   /* output control lines (DTR, RTS) */
    unsigned int writesize;                 /* max packet size for the output bulk endpoint */
    unsigned int readsize,ctrlsize;         /* buffer sizes for freeing */
    unsigned int minor;                     /* acm minor number */
    unsigned char throttle;                 /* throttled by tty layer */
    unsigned char clocal;                   /* termios CLOCAL */
    unsigned int ctrl_caps;                 /* control capabilities from the class specific header */
    unsigned int susp_count;                /* number of suspended interfaces */
    unsigned int combined_interfaces:1;     /* control and data collapsed */
    unsigned int is_int_ep:1;               /* interrupt endpoints contrary to spec used */
    u8 bInterval;
    struct acm_wb *delayed_wb;              /* write queued for a device about to be woken */
    unsigned char check_mode;               /* ..pepo FOMA interface bMode_N */
};

#define CDC_DATA_INTERFACE_TYPE    0x0a

/* constants describing various quirks and errors */
#define NO_UNION_NORMAL        1
#define SINGLE_RX_URB          2
#define NO_CAP_LINE            4
#define NOT_A_MODEM            8
// # include "cdc-acm.h" is far

#define ACM_CLOSE_TIMEOUT    15    /* seconds to let writes drain */

/*
 * Version Information
 */
#define DRIVER_VERSION "v0.26-FOMA.10" /* cdc-acm + FOMA driver version 2011.5.28 */
#define DRIVER_AUTHOR "Armin Fuerst, Pavel Machek, Johannes Erdfelt, Vojtech Pavlik, David Kubicek, acm-FOMA patch by IZAMUKARERA"
#define DRIVER_DESC "USB driver for docomo FOMA Cellular phone"

#ifdef VERBOSE_DEBUG
#define verbose    1
#else
#define verbose    0
#endif

/* for docomo FOMA specify define ..pepo */
#define USB_CLASS_COMM_FOMA        (0x02)
#define USB_CDC_SUBCLASS_FOMA      (0x88)
#define USB_CDC_PROTO_FOMA         (0x01)
#define USB_CDC_GL_005_FOMA        (0x44)
#define USB_CDC_ACM_FOMA           (0x11)
#define USB_RT_VENDORSPEC          (0x41)
#define USB_REQ_ACTIVATE_MODE      (0x60)
#define USB_REQ_DEACTIVATE_MODE    (0x00)
#define USB_REQ_SET_LINK           (0x62)
#define USB_REQ_CLEAR_LINK         (0x63)
#define ACM_FOMA_WAIT              (5000)
#define FOMA_MODEM_MODE            (0x01)
#define FOMA_AT_MODE               (0x02)
#define FOMA_OBEX_MODE             (0x60)

#define USB_RR_ACM    USB_RT_ACM | USB_DIR_IN    // Response Recive
#define	ATCOMMAND_SEND_RETRY_CNT_MAX    100      // 10Count
#define FOMA_ATCOMMAND_MAX_LEN    256
#define ATCOMMAND_SEND_INTERVAL     5            // 5msec
#define ATCOMMAND_EXIST_CR          1            // Exist CR Code
/* USB_MATCH_CODE_FOMA from include/linux/mod_devicetable.h
#define USB_DEVICE_ID_MATCH_VENDOR          0x0001
#define USB_DEVICE_ID_MATCH_PRODUCT         0x0002
#define USB_DEVICE_ID_MATCH_DEV_LO          0x0004
#define USB_DEVICE_ID_MATCH_DEV_HI          0x0008
#define USB_DEVICE_ID_MATCH_DEV_CLASS       0x0010
#define USB_DEVICE_ID_MATCH_DEV_SUBCLASS    0x0020
#define USB_DEVICE_ID_MATCH_DEV_PROTOCOL    0x0040
#define USB_DEVICE_ID_MATCH_INT_CLASS       0x0080
#define USB_DEVICE_ID_MATCH_INT_SUBCLASS    0x0100
#define USB_DEVICE_ID_MATCH_INT_PROTOCOL    0x0200
* more detail modules.usbmap
* usb module match_flags list ->, idVendor idProduct bcdDevice_lo bcdDevice_hi bDeviceClass bDeviceSubClass bDeviceProtocol bInterfaceClass bInterfaceSubClass bInterfaceProtocol
*/
#define USB_MATCH_CODE_FOMA    (USB_DEVICE_ID_MATCH_VENDOR | USB_DEVICE_ID_MATCH_INT_CLASS | \
                                USB_DEVICE_ID_MATCH_INT_SUBCLASS | USB_DEVICE_ID_MATCH_INT_PROTOCOL)
#define USB_DEVICE_FOMA(vend,cl,sc,pr) \
    .match_flags = USB_MATCH_CODE_FOMA,\
    .idVendor = (vend),\
    .bInterfaceClass = (cl), \
    .bInterfaceSubClass = (sc),\
    .bInterfaceProtocol = (pr)
#define FOMA_VENDID_LEGACY        (0x0a3c)
#define FOMA_VENDID_PANASONIC     (0x04da)
#define FOMA_VENDID_NEC           (0x0409)
#define FOMA_VENDID_MOTOROLA      (0x22b8)
#define FOMA_VENDID_MITSUBISHI    (0x06d3)
#define FOMA_VENDID_FUJITSU       (0x04c5)
#define FOMA_VENDID_SHARP         (0x04dd)
#define FOMA_VENDID_SANYO         (0x0474)
#define FOMA_VENDID_SONY          (0x0fce)
#define FOMA_VENDID_LG            (0x1004)

static const struct usb_device_id acm_ids[] = {
    /* control interfaces with values FOMA's vender..pepo */
    { USB_DEVICE_FOMA(FOMA_VENDID_LEGACY,USB_CLASS_COMM_FOMA,USB_CDC_SUBCLASS_FOMA,USB_CDC_PROTO_FOMA) },    /* Legacy        */
    { USB_DEVICE_FOMA(FOMA_VENDID_PANASONIC,USB_CLASS_COMM_FOMA,USB_CDC_SUBCLASS_FOMA,USB_CDC_PROTO_FOMA) }, /* Panasonic    */
    { USB_DEVICE_FOMA(FOMA_VENDID_NEC,USB_CLASS_COMM_FOMA,USB_CDC_SUBCLASS_FOMA,USB_CDC_PROTO_FOMA) },       /* NEC            */
    { USB_DEVICE_FOMA(FOMA_VENDID_MOTOROLA,USB_CLASS_COMM_FOMA,USB_CDC_SUBCLASS_FOMA,USB_CDC_PROTO_FOMA) },  /* Motorola     */
    { USB_DEVICE_FOMA(FOMA_VENDID_MITSUBISHI,USB_CLASS_COMM_FOMA,USB_CDC_SUBCLASS_FOMA,USB_CDC_PROTO_FOMA) },/* Mitsubishi    */
    { USB_DEVICE_FOMA(FOMA_VENDID_FUJITSU,USB_CLASS_COMM_FOMA,USB_CDC_SUBCLASS_FOMA,USB_CDC_PROTO_FOMA) },   /* Fujitsu        */
    { USB_DEVICE_FOMA(FOMA_VENDID_SHARP,USB_CLASS_COMM_FOMA,USB_CDC_SUBCLASS_FOMA,USB_CDC_PROTO_FOMA) },     /* SHARP        */
    { USB_DEVICE_FOMA(FOMA_VENDID_SANYO,USB_CLASS_COMM_FOMA,USB_CDC_SUBCLASS_FOMA,USB_CDC_PROTO_FOMA) },     /* SANYO        */
    { USB_DEVICE_FOMA(FOMA_VENDID_SONY,USB_CLASS_COMM_FOMA,USB_CDC_SUBCLASS_FOMA,USB_CDC_PROTO_FOMA) },      /* SONY        */
    { USB_DEVICE_FOMA(FOMA_VENDID_LG,USB_CLASS_COMM_FOMA,USB_CDC_SUBCLASS_FOMA,USB_CDC_PROTO_FOMA) },        /* LG            */
    { }
};

MODULE_DEVICE_TABLE(usb, acm_ids);

static struct usb_driver acm_driver;
static struct tty_driver *acm_tty_driver;
static struct acm *acm_table[ACM_TTY_MINORS];
static DEFINE_MUTEX(open_mutex);

#define ACM_READY(acm)    (acm && acm->dev && acm->port.count)

static const struct tty_port_operations acm_port_ops = {
};

/*
 * Functions for set only FOMA ..pepo
 */
static int foma_set_init(struct acm *acm)
{
    int retval;
    int data_length;
    unsigned char modeTable[3];
    unsigned char active_mode = 0x00;
    modeTable[2] = 0x00;
    /* SET_LINK */
    switch (acm->check_mode) {    /* Mode Chek */
        case FOMA_MODEM_MODE:
            modeTable[0] = 0x03;  /* Length */
            modeTable[1] = FOMA_MODEM_MODE;
            active_mode = FOMA_MODEM_MODE;
            modeTable[2] = 0xc0;  /* Vender Specific */
            data_length = 3;
            break;
        case FOMA_AT_MODE:
            modeTable[0]=0x02;    /* Length */
            modeTable[1] = FOMA_AT_MODE;
            data_length = 2;
            break;
        case FOMA_OBEX_MODE:
            modeTable[0]=0x03;    /* Length */
            modeTable[1] = FOMA_OBEX_MODE;
            modeTable[2] = 0xc0;  /* Vender Specific */
            data_length = 3;
            break;
        default:
            return -EINVAL;
            break;
    }
    retval = usb_control_msg(acm->dev,
        usb_sndctrlpipe(acm->dev, 0),
        USB_REQ_SET_LINK,    /* Set Link */
        USB_RT_VENDORSPEC,   /* VendorSpecificRequest */
        0x0000,              /* ACM */
        acm->control->altsetting[0].desc.bInterfaceNumber,
        modeTable,           /* Data mode table      */
        data_length,         /* Data Length */
        ACM_FOMA_WAIT);
    return retval < 0 ? retval : 0;
}

static int foma_set_active(struct acm *acm)
{
    int retval;
    unsigned char active_mode = 0x00;
    switch (acm->check_mode) {    /* Mode Chek */
        case FOMA_MODEM_MODE:
             active_mode = FOMA_MODEM_MODE;
            break;
        case FOMA_AT_MODE:
            active_mode = FOMA_AT_MODE;
            break;
        case FOMA_OBEX_MODE:
            active_mode = FOMA_OBEX_MODE;
            break;
        default:
            return -EINVAL;
            break;
    }
    dbg("pepo_enter foma_set_active,active_mode:%02x\n",active_mode);
    /* ACTIVATE_MODE */
    retval = usb_control_msg(acm->dev,
        usb_sndctrlpipe(acm->dev, 0),
        USB_REQ_ACTIVATE_MODE,    /* Activate Mode */
        USB_RT_VENDORSPEC,        /* VendorSpecificRequest */
        active_mode,              /* Mode */
        acm->control->altsetting[0].desc.bInterfaceNumber,
        NULL,                     /* Data none */
        0,                        /* Data Length 0 */
        ACM_FOMA_WAIT);
    return retval < 0 ? retval : 0;
}

/*
 * Functions for ACM control messages.
 */
static int acm_ctrl_msg(struct acm *acm, int request, int value,
    void *buf, int len)
{
    int retval = usb_control_msg(acm->dev, usb_sndctrlpipe(acm->dev, 0),
        request, USB_RT_ACM, value,
        acm->control->altsetting[0].desc.bInterfaceNumber,
        buf, len, ACM_FOMA_WAIT);
    dbg("acm_control_msg: rq: 0x%02x val: %#x len: %#x result: %d",
        request, value, len, retval);
    return retval < 0 ? retval : 0;
}

/* devices aren't required to support these requests.
 * the cdc acm descriptor tells whether they do...
 */
#define acm_set_control(acm, control) \
    acm_ctrl_msg(acm, USB_CDC_REQ_SET_CONTROL_LINE_STATE, control, NULL, 0)
#define acm_set_line(acm, line) \
    acm_ctrl_msg(acm, USB_CDC_REQ_SET_LINE_CODING, 0, line, sizeof *(line));
#define acm_send_break(acm, ms) \
    acm_ctrl_msg(acm, USB_CDC_REQ_SEND_BREAK, ms, NULL, 0)

/*
 * Write buffer management.
 * All of these assume proper locks taken by the caller.
 */

static int acm_wb_alloc(struct acm *acm)
{
    int i, wbn;
    struct acm_wb *wb;

    wbn = 0;
    i = 0;
    for (;;) {
        wb = &acm->wb[wbn];
        if (!wb->use) {
            wb->use = 1;
            return wbn;
        }
        wbn = (wbn + 1) % ACM_NW;
        if (++i >= ACM_NW)
            return -1;
    }
}

static int acm_wb_is_avail(struct acm *acm)
{
    int i, n;
    unsigned long flags;

    n = ACM_NW;
    spin_lock_irqsave(&acm->write_lock, flags);
    for (i = 0; i < ACM_NW; i++)
        n -= acm->wb[i].use;
    spin_unlock_irqrestore(&acm->write_lock, flags);
    return n;
}

/*
 * Finish write. Caller must hold acm->write_lock
 */
static void acm_write_done(struct acm *acm, struct acm_wb *wb)
{
    wb->use = 0;
    acm->transmitting--;
    usb_autopm_put_interface_async(acm->control);
}

/*
 * Poke write.
 *
 * the caller is responsible for locking
 */

static int acm_start_wb(struct acm *acm, struct acm_wb *wb)
{
    int rc;
    if (acm->check_mode == FOMA_AT_MODE) {
        return rc = 0; // ..pepo
    }
    acm->transmitting++;

    wb->urb->transfer_buffer = wb->buf;
    wb->urb->transfer_dma = wb->dmah;
    wb->urb->transfer_buffer_length = wb->len;
    wb->urb->dev = acm->dev;

    rc = usb_submit_urb(wb->urb, GFP_ATOMIC);
    if (rc < 0) {
        dbg("usb_submit_urb(write bulk) failed: %d", rc);
        acm_write_done(acm, wb);
    }
    return rc;
}

static int acm_write_start(struct acm *acm, int wbn)
{
    unsigned long flags;
    struct acm_wb *wb = &acm->wb[wbn];
    int rc;

    spin_lock_irqsave(&acm->write_lock, flags);
    if (!acm->dev) {
        wb->use = 0;
        spin_unlock_irqrestore(&acm->write_lock, flags);
        return -ENODEV;
    }

    dbg("%s susp_count: %d", __func__, acm->susp_count);
    usb_autopm_get_interface_async(acm->control);
    if (acm->susp_count) {
        if (!acm->delayed_wb)
            acm->delayed_wb = wb;
        else
            usb_autopm_put_interface_async(acm->control);
        spin_unlock_irqrestore(&acm->write_lock, flags);
        return 0;    /* A white lie */
    }
    usb_mark_last_busy(acm->dev);

    rc = acm_start_wb(acm, wb);
    spin_unlock_irqrestore(&acm->write_lock, flags);

    return rc;

}
/*
 * attributes exported through sysfs
 */
static ssize_t show_caps
(struct device *dev, struct device_attribute *attr, char *buf)
{
    struct usb_interface *intf = to_usb_interface(dev);
    struct acm *acm = usb_get_intfdata(intf);

    return sprintf(buf, "%d", acm->ctrl_caps);
}
static DEVICE_ATTR(bmCapabilities, S_IRUGO, show_caps, NULL);

static ssize_t show_country_codes
(struct device *dev, struct device_attribute *attr, char *buf)
{
    struct usb_interface *intf = to_usb_interface(dev);
    struct acm *acm = usb_get_intfdata(intf);

    memcpy(buf, acm->country_codes, acm->country_code_size);
    return acm->country_code_size;
}

static DEVICE_ATTR(wCountryCodes, S_IRUGO, show_country_codes, NULL);

static ssize_t show_country_rel_date
(struct device *dev, struct device_attribute *attr, char *buf)
{
    struct usb_interface *intf = to_usb_interface(dev);
    struct acm *acm = usb_get_intfdata(intf);

    return sprintf(buf, "%d", acm->country_rel_date);
}

static DEVICE_ATTR(iCountryCodeRelDate, S_IRUGO, show_country_rel_date, NULL);
/*
 * Interrupt handlers for various ACM device responses
 */

/* control interface reports status changes with "interrupt" transfers */
static void acm_ctrl_irq(struct urb *urb)
{
    struct acm *acm = urb->context;
    struct usb_cdc_notification *dr = urb->transfer_buffer;
    struct tty_struct *tty;
    unsigned char *data;
    int newctrl;
    int retval;
    int status = urb->status;

    switch (status) {
    case 0:
        /* success */
        break;
    case -ECONNRESET:
    case -ENOENT:
    case -ESHUTDOWN:
        /* this urb is terminated, clean up */
        dbg("%s - urb shutting down with status: %d", __func__, status);
        return;
    default:
        dbg("%s - nonzero urb status received: %d", __func__, status);
        goto exit;
    }

    if (!ACM_READY(acm))
        goto exit;

    data = (unsigned char *)(dr + 1);
    switch (dr->bNotificationType) {
    case USB_CDC_NOTIFY_NETWORK_CONNECTION:
        dbg("%s network", dr->wValue ?
                    "connected to" : "disconnected from");
        break;

    case USB_CDC_NOTIFY_SERIAL_STATE:
        tty = tty_port_tty_get(&acm->port);
        newctrl = get_unaligned_le16(data);

        if (tty) {
            if (!acm->clocal &&
                (acm->ctrlin & ~newctrl & ACM_CTRL_DCD)) {
                dbg("calling hangup");
                tty_hangup(tty);
            }
            tty_kref_put(tty);
        }

        acm->ctrlin = newctrl;

        dbg("input control lines: dcd%c dsr%c break%c ring%c framing%c parity%c overrun%c",
            acm->ctrlin & ACM_CTRL_DCD ? '+' : '-',
            acm->ctrlin & ACM_CTRL_DSR ? '+' : '-',
            acm->ctrlin & ACM_CTRL_BRK ? '+' : '-',
            acm->ctrlin & ACM_CTRL_RI  ? '+' : '-',
            acm->ctrlin & ACM_CTRL_FRAMING ? '+' : '-',
            acm->ctrlin & ACM_CTRL_PARITY ? '+' : '-',
            acm->ctrlin & ACM_CTRL_OVERRUN ? '+' : '-');
            break;

    default:
        dbg("unknown notification %d received: index %d len %d data0 %d data1 %d",
            dr->bNotificationType, dr->wIndex,
            dr->wLength, data[0], data[1]);
        break;
    }
exit:
    usb_mark_last_busy(acm->dev);
    retval = usb_submit_urb(urb, GFP_ATOMIC);
    if (retval)
        dev_err(&urb->dev->dev, "%s - usb_submit_urb failed with "
            "result %d", __func__, retval);
}

/* data interface returns incoming bytes, or we got unthrottled */
static void acm_read_bulk(struct urb *urb)
{
    struct acm_rb *buf;
    struct acm_ru *rcv = urb->context;
    struct acm *acm = rcv->instance;
    int status = urb->status;

    dbg("Entering acm_read_bulk with status %d", status);

    if (!ACM_READY(acm)) {
        dev_dbg(&acm->data->dev, "Aborting, acm not ready");
        return;
    }
    usb_mark_last_busy(acm->dev);

    if (status)
        dev_dbg(&acm->data->dev, "bulk rx status %d\n", status);

    buf = rcv->buffer;
    buf->size = urb->actual_length;

    if (likely(status == 0)) {
        spin_lock(&acm->read_lock);
        acm->processing++;
        list_add_tail(&rcv->list, &acm->spare_read_urbs);
        list_add_tail(&buf->list, &acm->filled_read_bufs);
        spin_unlock(&acm->read_lock);
    } else {
        /* we drop the buffer due to an error */
        spin_lock(&acm->read_lock);
        list_add_tail(&rcv->list, &acm->spare_read_urbs);
        list_add(&buf->list, &acm->spare_read_bufs);
        spin_unlock(&acm->read_lock);
        /* nevertheless the tasklet must be kicked unconditionally
        so the queue cannot dry up */
    }
    if (likely(!acm->susp_count))
        tasklet_schedule(&acm->urb_task);
}

static void acm_rx_tasklet(unsigned long _acm)
{
    struct acm *acm = (void *)_acm;
    struct acm_rb *buf;
    struct tty_struct *tty;
    struct acm_ru *rcv;
    unsigned long flags;
    unsigned char throttled;

    dbg("Entering acm_rx_tasklet");

    if (!ACM_READY(acm)) {
        dbg("acm_rx_tasklet: ACM not ready");
        return;
    }

    spin_lock_irqsave(&acm->throttle_lock, flags);
    throttled = acm->throttle;
    spin_unlock_irqrestore(&acm->throttle_lock, flags);
    if (throttled) {
        dbg("acm_rx_tasklet: throttled");
        return;
    }

    tty = tty_port_tty_get(&acm->port);

next_buffer:
    spin_lock_irqsave(&acm->read_lock, flags);
    if (list_empty(&acm->filled_read_bufs)) {
        spin_unlock_irqrestore(&acm->read_lock, flags);
        goto urbs;
    }
    buf = list_entry(acm->filled_read_bufs.next,
             struct acm_rb, list);
    list_del(&buf->list);
    spin_unlock_irqrestore(&acm->read_lock, flags);

    dbg("acm_rx_tasklet: procesing buf 0x%p, size = %d", buf, buf->size);

    if (tty) {
        spin_lock_irqsave(&acm->throttle_lock, flags);
        throttled = acm->throttle;
        spin_unlock_irqrestore(&acm->throttle_lock, flags);
        if (!throttled) {
            tty_insert_flip_string(tty, buf->base, buf->size);
            tty_flip_buffer_push(tty);
        } else {
            tty_kref_put(tty);
            dbg("Throttling noticed");
            spin_lock_irqsave(&acm->read_lock, flags);
            list_add(&buf->list, &acm->filled_read_bufs);
            spin_unlock_irqrestore(&acm->read_lock, flags);
            return;
        }
    }

    spin_lock_irqsave(&acm->read_lock, flags);
    list_add(&buf->list, &acm->spare_read_bufs);
    spin_unlock_irqrestore(&acm->read_lock, flags);
    goto next_buffer;

urbs:
    tty_kref_put(tty);

    while (!list_empty(&acm->spare_read_bufs)) {
        spin_lock_irqsave(&acm->read_lock, flags);
        if (list_empty(&acm->spare_read_urbs)) {
            acm->processing = 0;
            spin_unlock_irqrestore(&acm->read_lock, flags);
            return;
        }
        rcv = list_entry(acm->spare_read_urbs.next,
                 struct acm_ru, list);
        list_del(&rcv->list);
        spin_unlock_irqrestore(&acm->read_lock, flags);

        buf = list_entry(acm->spare_read_bufs.next,
                 struct acm_rb, list);
        list_del(&buf->list);

        rcv->buffer = buf;

        if (acm->is_int_ep)
            usb_fill_int_urb(rcv->urb, acm->dev,
                     acm->rx_endpoint,
                     buf->base,
                     acm->readsize,
                     acm_read_bulk, rcv, acm->bInterval);
        else
            usb_fill_bulk_urb(rcv->urb, acm->dev,
                      acm->rx_endpoint,
                      buf->base,
                      acm->readsize,
                      acm_read_bulk, rcv);
        rcv->urb->transfer_dma = buf->dma;
        rcv->urb->transfer_flags |= URB_NO_TRANSFER_DMA_MAP;

        /* This shouldn't kill the driver as unsuccessful URBs are
           returned to the free-urbs-pool and resubmited ASAP */
        spin_lock_irqsave(&acm->read_lock, flags);
        if (acm->susp_count ||
                usb_submit_urb(rcv->urb, GFP_ATOMIC) < 0) {
            list_add(&buf->list, &acm->spare_read_bufs);
            list_add(&rcv->list, &acm->spare_read_urbs);
            acm->processing = 0;
            spin_unlock_irqrestore(&acm->read_lock, flags);
            return;
        } else {
            spin_unlock_irqrestore(&acm->read_lock, flags);
            dbg("acm_rx_tasklet: sending urb 0x%p, rcv 0x%p, buf 0x%p", rcv->urb, rcv, buf);
        }
    }
    spin_lock_irqsave(&acm->read_lock, flags);
    acm->processing = 0;
    spin_unlock_irqrestore(&acm->read_lock, flags);
}

/* data interface wrote those outgoing bytes */
static void acm_write_bulk(struct urb *urb)
{
    struct acm_wb *wb = urb->context;
    struct acm *acm = wb->instance;
    unsigned long flags;

    if (verbose || urb->status
            || (urb->actual_length != urb->transfer_buffer_length))
        dev_dbg(&acm->data->dev, "tx %d/%d bytes -- > %d\n",
            urb->actual_length,
            urb->transfer_buffer_length,
            urb->status);

    spin_lock_irqsave(&acm->write_lock, flags);
    acm_write_done(acm, wb);
    spin_unlock_irqrestore(&acm->write_lock, flags);
    if (ACM_READY(acm))
        schedule_work(&acm->work);
    else
        wake_up_interruptible(&acm->drain_wait);
}

static void acm_softint(struct work_struct *work)
{
    struct acm *acm = container_of(work, struct acm, work);
    struct tty_struct *tty;

    dev_vdbg(&acm->data->dev, "tx work\n");
    if (!ACM_READY(acm))
        return;
    tty = tty_port_tty_get(&acm->port);
    tty_wakeup(tty);
    tty_kref_put(tty);
}

/*
 * TTY handlers
 */

static int acm_tty_open(struct tty_struct *tty, struct file *filp)
{
    struct acm *acm;
    int rv = -ENODEV;
    int i;
    dbg("Entering acm_tty_open.");
    mutex_lock(&open_mutex);

    acm = acm_table[tty->index];
    if (!acm || !acm->dev)
        goto out;
    else
        rv = 0;


    set_bit(TTY_NO_WRITE_SPLIT, &tty->flags);
    tty->driver_data = acm;
    tty_port_tty_set(&acm->port, tty);

    if (usb_autopm_get_interface(acm->control) < 0)
        goto early_bail;
    else
        acm->control->needs_remote_wakeup = 1;

    mutex_lock(&acm->mutex);
    if (acm->port.count++) {
        mutex_unlock(&acm->mutex);
        usb_autopm_put_interface(acm->control);
        goto out;
    }

    acm->ctrlurb->dev = acm->dev;
    if (usb_submit_urb(acm->ctrlurb, GFP_KERNEL)) {
        dbg("usb_submit_urb(ctrl irq) failed");
        goto bail_out;
    }
    foma_set_active(acm);  // active mode set ..pepo
    if (acm->check_mode == FOMA_MODEM_MODE) {
        if (0 > acm_set_control(acm, acm->ctrlout = ACM_CTRL_DTR | ACM_CTRL_RTS) &&
            (acm->ctrl_caps & USB_CDC_CAP_LINE))
            goto full_bailout;
    }
    usb_autopm_put_interface(acm->control);

    INIT_LIST_HEAD(&acm->spare_read_urbs);
    INIT_LIST_HEAD(&acm->spare_read_bufs);
    INIT_LIST_HEAD(&acm->filled_read_bufs);

    for (i = 0; i < acm->rx_buflimit; i++)
        list_add(&(acm->ru[i].list), &acm->spare_read_urbs);
    for (i = 0; i < acm->rx_buflimit; i++)
        list_add(&(acm->rb[i].list), &acm->spare_read_bufs);

    acm->throttle = 0;

    set_bit(ASYNCB_INITIALIZED, &acm->port.flags);
    rv = tty_port_block_til_ready(&acm->port, tty, filp);
    if (acm->check_mode != FOMA_AT_MODE) tasklet_schedule(&acm->urb_task); // ..pepo

    mutex_unlock(&acm->mutex);
out:
    mutex_unlock(&open_mutex);
    return rv;

full_bailout:
    usb_kill_urb(acm->ctrlurb);
bail_out:
    acm->port.count--;
    mutex_unlock(&acm->mutex);
    usb_autopm_put_interface(acm->control);
early_bail:
    mutex_unlock(&open_mutex);
    tty_port_tty_set(&acm->port, NULL);
    return -EIO;
}

static void acm_tty_unregister(struct acm *acm)
{
    int i, nr;

    nr = acm->rx_buflimit;
    tty_unregister_device(acm_tty_driver, acm->minor);
    usb_put_intf(acm->control);
    acm_table[acm->minor] = NULL;
    usb_free_urb(acm->ctrlurb);
    for (i = 0; i < ACM_NW; i++)
        usb_free_urb(acm->wb[i].urb);
    for (i = 0; i < nr; i++)
        usb_free_urb(acm->ru[i].urb);
    kfree(acm->country_codes);
    kfree(acm);
}

static int acm_tty_chars_in_buffer(struct tty_struct *tty);

static void acm_port_down(struct acm *acm, int drain)
{
    int i, nr = acm->rx_buflimit;
    mutex_lock(&open_mutex);
    if (acm->dev) {
        usb_autopm_get_interface(acm->control);
        acm_set_control(acm, acm->ctrlout = 0);
        /* try letting the last writes drain naturally */
        if (drain) {
            wait_event_interruptible_timeout(acm->drain_wait,
                (ACM_NW == acm_wb_is_avail(acm)) || !acm->dev,
                    ACM_CLOSE_TIMEOUT * HZ);
        }
        usb_kill_urb(acm->ctrlurb);
        for (i = 0; i < ACM_NW; i++)
            usb_kill_urb(acm->wb[i].urb);
        for (i = 0; i < nr; i++)
            usb_kill_urb(acm->ru[i].urb);
        acm->control->needs_remote_wakeup = 0;
        usb_autopm_put_interface(acm->control);
    }
    mutex_unlock(&open_mutex);
}

static void acm_tty_hangup(struct tty_struct *tty)
{
    struct acm *acm = tty->driver_data;
    tty_port_hangup(&acm->port);
    acm_port_down(acm, 0);
}

static void acm_tty_close(struct tty_struct *tty, struct file *filp)
{
    struct acm *acm = tty->driver_data;

    /* Perform the closing process and see if we need to do the hardware
       shutdown */
    if (!acm)
        return;
    if (tty_port_close_start(&acm->port, tty, filp) == 0) {
        mutex_lock(&open_mutex);
        if (!acm->dev) {
            tty_port_tty_set(&acm->port, NULL);
            acm_tty_unregister(acm);
            tty->driver_data = NULL;
        }
        mutex_unlock(&open_mutex);
        return;
    }
    acm_port_down(acm, 0);
    tty_port_close_end(&acm->port, tty);
    tty_port_tty_set(&acm->port, NULL);
}

static int acm_tty_write(struct tty_struct *tty,
                    const unsigned char *buf, int count)
{
    struct acm *acm = tty->driver_data;
    int stat;
    unsigned long flags;
    int wbn;
    struct acm_wb *wb;
	// ..pepo
    int retval;
	int bResult;
	int i;
    unsigned char *snd_buff;
    unsigned char *rcv_buff;

    dbg("Entering acm_tty_write to write %d bytes,", count);

    if (!ACM_READY(acm))
        return -EINVAL;
    if (!count)
        return 0;
    // ..pepo
    if (acm->check_mode != FOMA_AT_MODE) {
        spin_lock_irqsave(&acm->write_lock, flags);
        wbn = acm_wb_alloc(acm);
        if (wbn < 0) {
            spin_unlock_irqrestore(&acm->write_lock, flags);
            return 0;
        }
        wb = &acm->wb[wbn];

        count = (count > acm->writesize) ? acm->writesize : count;
        dbg("Get %d bytes...", count);
        memcpy(wb->buf, buf, count);
        wb->len = count;
        spin_unlock_irqrestore(&acm->write_lock, flags);

        stat = acm_write_start(acm, wbn);
        if (stat < 0)
            return stat;
    }
    // "FOMA_AT_Command" code is from "Yuichiro Nakada". Thanks
     else {
        // Check CR Code
        bResult = 0;
        for(i = 0; i < count; i++) {
            if(buf[i] == '\r') {
                bResult = ATCOMMAND_EXIST_CR;
                break;
            }
        }

        snd_buff = kmalloc(count, GFP_KERNEL);

        if (!snd_buff)
            return -EINVAL;

        memcpy(snd_buff, buf, count);

        mutex_lock(&open_mutex);

        if (!ACM_READY(acm)) {
            mutex_unlock(&open_mutex);
            return -EINVAL;
        }

        // AT Command Send
        retval = usb_control_msg(acm->dev,
                    usb_sndctrlpipe(acm->dev, 0),
                    USB_CDC_SEND_ENCAPSULATED_COMMAND,
                     USB_RT_ACM,
                    0,
                    acm->control->cur_altsetting[0].desc.bInterfaceNumber,
                    snd_buff,
                    count,
                    ACM_FOMA_WAIT);

        mutex_unlock(&open_mutex);

        kfree(snd_buff);

        rcv_buff = kmalloc(FOMA_ATCOMMAND_MAX_LEN, GFP_KERNEL);

        if (rcv_buff) {
            // Recv Response Echo Polling
            for (i = 0; i < ATCOMMAND_SEND_RETRY_CNT_MAX; i++) {
                mutex_lock(&open_mutex);

                if (!ACM_READY(acm)) {
                    mutex_unlock(&open_mutex);
                    return -EINVAL;
                }

                // Encapsulated Command
                retval = usb_control_msg(acm->dev,
                            usb_rcvctrlpipe(acm->dev, 0),
                            USB_CDC_GET_ENCAPSULATED_RESPONSE,
                            USB_RR_ACM,
                            0,
                            acm->control->cur_altsetting[0].desc.bInterfaceNumber,
                            rcv_buff,
                            FOMA_ATCOMMAND_MAX_LEN,
                            ACM_FOMA_WAIT);

                if (retval > 0) {
                    tty_insert_flip_string(tty, rcv_buff, retval);
                    tty_flip_buffer_push(tty);

                    mutex_unlock(&open_mutex);
                    break;
                }
                else if (retval == -EPIPE){
                    usb_clear_halt(acm->dev, usb_rcvctrlpipe(acm->dev, 0));
                }

                mutex_unlock(&open_mutex);

                msleep(ATCOMMAND_SEND_INTERVAL);
            }

            if (bResult == ATCOMMAND_EXIST_CR) {
                for (i = 0; i < ATCOMMAND_SEND_RETRY_CNT_MAX; i++) {
                    mutex_lock(&open_mutex);

                    if (!ACM_READY(acm)) {
                        mutex_unlock(&open_mutex);
                        return -EINVAL;
                    }
                    // Encapsulated Command
                    retval = usb_control_msg(acm->dev,
                                usb_rcvctrlpipe(acm->dev, 0),
                                USB_CDC_GET_ENCAPSULATED_RESPONSE,
                                USB_RR_ACM,
                                0,
                                acm->control->cur_altsetting[0].desc.bInterfaceNumber,
                                rcv_buff,
                                FOMA_ATCOMMAND_MAX_LEN,
                                ACM_FOMA_WAIT);

                    if (retval > 0) {
                        tty_insert_flip_string(tty, rcv_buff, retval);
                        tty_flip_buffer_push(tty);

                        if (strstr(rcv_buff, "OK") != NULL) {
                            mutex_unlock(&open_mutex);
                            break;
                        }
                        else if (strstr(rcv_buff, "ERROR") != NULL) {
                            mutex_unlock(&open_mutex);
                            break;
                        }
                    }
                    else if (retval == -EPIPE){
                        usb_clear_halt(acm->dev, usb_rcvctrlpipe(acm->dev, 0));
                    }

                    mutex_unlock(&open_mutex);

                    msleep(ATCOMMAND_SEND_INTERVAL);
                }
            }

            kfree(rcv_buff);
        }
    }

        return count;
}

static int acm_tty_write_room(struct tty_struct *tty)
{
    struct acm *acm = tty->driver_data;
    if (!ACM_READY(acm))
        return -EINVAL;
    /*
     * Do not let the line discipline to know that we have a reserve,
     * or it might get too enthusiastic.
     */
    return acm_wb_is_avail(acm) ? acm->writesize : 0;
}

static int acm_tty_chars_in_buffer(struct tty_struct *tty)
{
    struct acm *acm = tty->driver_data;
    if (!ACM_READY(acm))
        return 0;
    /*
     * This is inaccurate (overcounts), but it works.
     */
    return (ACM_NW - acm_wb_is_avail(acm)) * acm->writesize;
}

static void acm_tty_throttle(struct tty_struct *tty)
{
    struct acm *acm = tty->driver_data;
    if (!ACM_READY(acm))
        return;
    spin_lock_bh(&acm->throttle_lock);
    acm->throttle = 1;
    spin_unlock_bh(&acm->throttle_lock);
}

static void acm_tty_unthrottle(struct tty_struct *tty)
{
    struct acm *acm = tty->driver_data;
    if (!ACM_READY(acm))
        return;
    spin_lock_bh(&acm->throttle_lock);
    acm->throttle = 0;
    spin_unlock_bh(&acm->throttle_lock);
    tasklet_schedule(&acm->urb_task);
}

static int acm_tty_break_ctl(struct tty_struct *tty, int state)
{
    struct acm *acm = tty->driver_data;
    int retval;
    if (!ACM_READY(acm))
        return -EINVAL;
    retval = acm_send_break(acm, state ? 0xffff : 0);
    if (retval < 0)
        dbg("send break failed");
    return retval;
}

static int acm_tty_tiocmget(struct tty_struct *tty, struct file *file)
{
    struct acm *acm = tty->driver_data;

    if (!ACM_READY(acm))
        return -EINVAL;

    return (acm->ctrlout & ACM_CTRL_DTR ? TIOCM_DTR : 0) |
           (acm->ctrlout & ACM_CTRL_RTS ? TIOCM_RTS : 0) |
           (acm->ctrlin  & ACM_CTRL_DSR ? TIOCM_DSR : 0) |
           (acm->ctrlin  & ACM_CTRL_RI  ? TIOCM_RI  : 0) |
           (acm->ctrlin  & ACM_CTRL_DCD ? TIOCM_CD  : 0) |
           TIOCM_CTS;
}

static int acm_tty_tiocmset(struct tty_struct *tty, struct file *file,
                unsigned int set, unsigned int clear)
{
    struct acm *acm = tty->driver_data;
    unsigned int newctrl;

    if (!ACM_READY(acm))
        return -EINVAL;

    newctrl = acm->ctrlout;
    set = (set & TIOCM_DTR ? ACM_CTRL_DTR : 0) |
                    (set & TIOCM_RTS ? ACM_CTRL_RTS : 0);
    clear = (clear & TIOCM_DTR ? ACM_CTRL_DTR : 0) |
                    (clear & TIOCM_RTS ? ACM_CTRL_RTS : 0);

    newctrl = (newctrl & ~clear) | set;

    if (acm->ctrlout == newctrl)
        return 0;
    return acm_set_control(acm, acm->ctrlout = newctrl);
}

static int acm_tty_ioctl(struct tty_struct *tty, struct file *file,
                    unsigned int cmd, unsigned long arg)
{
    struct acm *acm = tty->driver_data;

    if (!ACM_READY(acm))
        return -EINVAL;

    return -ENOIOCTLCMD;
}

static const __u32 acm_tty_speed[] = {
    0, 50, 75, 110, 134, 150, 200, 300, 600,
    1200, 1800, 2400, 4800, 9600, 19200, 38400,
    57600, 115200, 230400, 460800, 500000, 576000,
    921600, 1000000, 1152000, 1500000, 2000000,
    2500000, 3000000, 3500000, 4000000
};

static const __u8 acm_tty_size[] = {
    5, 6, 7, 8
};

static void acm_tty_set_termios(struct tty_struct *tty,
                        struct ktermios *termios_old)
{
    struct acm *acm = tty->driver_data;
    struct ktermios *termios = tty->termios;
    struct usb_cdc_line_coding newline;
    int newctrl = acm->ctrlout;
    if (!ACM_READY(acm))
        return;
    if (acm->check_mode != FOMA_MODEM_MODE) return; // ..pepo

    newline.dwDTERate = cpu_to_le32(tty_get_baud_rate(tty));
    newline.bCharFormat = termios->c_cflag & CSTOPB ? 2 : 0;
    newline.bParityType = termios->c_cflag & PARENB ?
                (termios->c_cflag & PARODD ? 1 : 2) +
                (termios->c_cflag & CMSPAR ? 2 : 0) : 0;
    newline.bDataBits = acm_tty_size[(termios->c_cflag & CSIZE) >> 4];
    /* FIXME: Needs to clear unsupported bits in the termios */
    acm->clocal = ((termios->c_cflag & CLOCAL) != 0);

    if (!newline.dwDTERate) {
        newline.dwDTERate = acm->line.dwDTERate;
        newctrl &= ~ACM_CTRL_DTR;
    } else
        newctrl |=  ACM_CTRL_DTR;

    if (newctrl != acm->ctrlout)
        acm_set_control(acm, acm->ctrlout = newctrl);

    if (memcmp(&acm->line, &newline, sizeof newline)) {
        memcpy(&acm->line, &newline, sizeof newline);
        dbg("set line: %d %d %d %d", le32_to_cpu(newline.dwDTERate),
            newline.bCharFormat, newline.bParityType,
            newline.bDataBits);
        acm_set_line(acm, &acm->line);
    }
}

/*
 * USB probe and disconnect routines.
 */

/* Little helpers: write/read buffers free */
static void acm_write_buffers_free(struct acm *acm)
{
    int i;
    struct acm_wb *wb;
    struct usb_device *usb_dev = interface_to_usbdev(acm->control);

    for (wb = &acm->wb[0], i = 0; i < ACM_NW; i++, wb++)
        usb_free_coherent(usb_dev, acm->writesize, wb->buf, wb->dmah);
}

static void acm_read_buffers_free(struct acm *acm)
{
    struct usb_device *usb_dev = interface_to_usbdev(acm->control);
    int i, n = acm->rx_buflimit;

    for (i = 0; i < n; i++)
        usb_free_coherent(usb_dev, acm->readsize,
                  acm->rb[i].base, acm->rb[i].dma);
}

/* Little helper: write buffers allocate */
static int acm_write_buffers_alloc(struct acm *acm)
{
    int i;
    struct acm_wb *wb;

    for (wb = &acm->wb[0], i = 0; i < ACM_NW; i++, wb++) {
        wb->buf = usb_alloc_coherent(acm->dev, acm->writesize, GFP_KERNEL,
            &wb->dmah);
        if (!wb->buf) {
            while (i != 0) {
                --i;
                --wb;
                usb_free_coherent(acm->dev, acm->writesize,
                    wb->buf, wb->dmah);
            }
            return -ENOMEM;
        }
    }
    return 0;
}

static int acm_probe(struct usb_interface *intf,
             const struct usb_device_id *id)
{
    struct usb_cdc_union_desc *union_header = NULL;
    struct usb_cdc_country_functional_desc *cfd = NULL;
    unsigned char *buffer = intf->altsetting->extra;
    int buflen = intf->altsetting->extralen;
    struct usb_interface *control_interface;
    struct usb_interface *data_interface;
    struct usb_endpoint_descriptor *epctrl = NULL;
    struct usb_endpoint_descriptor *epread = NULL;
    struct usb_endpoint_descriptor *epwrite = NULL;
    struct usb_device *usb_dev = interface_to_usbdev(intf);
    struct acm *acm;
    int minor;
    int ctrlsize, readsize;
    u8 *buf;
    u8 ac_management_function = 0;
    u8 call_management_function = 0;
    int call_interface_num = -1;
    int data_interface_num = -1;
    unsigned long quirks;
    int num_rx_buf;
    int i;
    int combined_interfaces = 0;
    /* foma specail by pepo */
    int pepo_f1 = 0 ;
    int pepo_f2 = 0 ;
    unsigned char check_interface_mode = 0x00;
    /* normal quirks */
    quirks = (unsigned long)id->driver_info;
    num_rx_buf = (quirks == SINGLE_RX_URB) ? 1 : ACM_NR;

    /* handle quirks deadly to normal probing*/
    if (quirks == NO_UNION_NORMAL) {
        data_interface = usb_ifnum_to_if(usb_dev, 1);
        control_interface = usb_ifnum_to_if(usb_dev, 0);
        goto skip_normal_probe;
    }

    /* normal probing*/
    if (!buffer) {
        dev_err(&intf->dev, "Weird descriptor references\n");
        return -EINVAL;
    }

    if (!buflen) {
        if (intf->cur_altsetting->endpoint &&
                intf->cur_altsetting->endpoint->extralen &&
                intf->cur_altsetting->endpoint->extra) {
            dev_dbg(&intf->dev,
                "Seeking extra descriptors on endpoint\n");
            buflen = intf->cur_altsetting->endpoint->extralen;
            buffer = intf->cur_altsetting->endpoint->extra;
        } else {
            dev_err(&intf->dev,
                "Zero length descriptor references\n");
            return -EINVAL;
        }
    }

    while (buflen > 0) {
//        for (i = 0 ;buflen > i;i++) dbg("buffer[%i]:%02x:",i,buffer[i]); //  ..pepo
        if (buffer[1] == USB_CDC_GL_005_FOMA && buffer[2] == USB_CDC_ACM_FOMA) {
            if (union_header){
                pepo_f1 = 1;
                check_interface_mode = buffer[4];
                goto next_desc;
            }
        }    
        if (buffer[1] != USB_DT_CS_INTERFACE) {
            dev_err(&intf->dev, "skipping garbage\n");
            goto next_desc;
        }
        switch (buffer[2]) {
        case USB_CDC_UNION_TYPE: /* we've found it */
            if (union_header) {
                dev_err(&intf->dev, "More than one "
                    "union descriptor, skipping ...\n");
                goto next_desc;
            }
            union_header = (struct usb_cdc_union_desc *)buffer;
            break;
        case USB_CDC_COUNTRY_TYPE: /* export through sysfs*/
            cfd = (struct usb_cdc_country_functional_desc *)buffer;
            break;
        case USB_CDC_HEADER_TYPE: /* maybe check version */
            break; /* for now we ignore it */
        case USB_CDC_ACM_TYPE:
            ac_management_function = buffer[3];
            break;
        case USB_CDC_CALL_MANAGEMENT_TYPE:
            call_management_function = buffer[3];
            call_interface_num = buffer[4];
            if ( (quirks & NOT_A_MODEM) == 0 && (call_management_function & 3) != 3)
                dev_err(&intf->dev, "This device cannot do calls on its own. It is not a modem.\n");
            break;
        default:
            /* there are LOTS more CDC descriptors that
             * could legitimately be found here.
             */
            dev_dbg(&intf->dev, "Ignoring descriptor: "
                    "type %02x, length %d\n",
                    buffer[2], buffer[0]);
            break;
        }
next_desc:
        buflen -= buffer[0];
        buffer += buffer[0];
    }

    if (!union_header) {
        if (call_interface_num > 0) {
            dev_dbg(&intf->dev, "No union descriptor, using call management descriptor\n");
            data_interface = usb_ifnum_to_if(usb_dev, (data_interface_num = call_interface_num));
            control_interface = intf;
        } else {
            if (intf->cur_altsetting->desc.bNumEndpoints != 3) {
                dev_dbg(&intf->dev,"No union descriptor, giving up\n");
                return -ENODEV;
            } else {
                dev_warn(&intf->dev,"No union descriptor, testing for castrated device\n");
                combined_interfaces = 1;
                control_interface = data_interface = intf;
                goto look_for_collapsed_interface;
            }
        }
    } else {
        control_interface = usb_ifnum_to_if(usb_dev, union_header->bMasterInterface0);
        data_interface = usb_ifnum_to_if(usb_dev, (data_interface_num =union_header->bSlaveInterface0));

        if (!control_interface || !data_interface) {
                        if (control_interface && !data_interface) {/* pepo AT command control mode */
                             data_interface = control_interface;
                             pepo_f2 = 1;
                             goto pepo_1;
                        }
            dev_dbg(&intf->dev, "no control_interfaces\n");
            return -ENODEV;
        }
    }
pepo_1:
    if (data_interface_num != call_interface_num)
        dev_dbg(&intf->dev, "Separate call control interface. That is not fully supported.\n");

    if (control_interface == data_interface) {
        /* some broken devices designed for windows work this way */
        dev_warn(&intf->dev,"Control and data interfaces are not separated!\n");
        combined_interfaces = 1;
        /* a popular other OS doesn't use it */
        quirks |= NO_CAP_LINE;
        if (data_interface->cur_altsetting->desc.bNumEndpoints != 3) {
            dev_err(&intf->dev, "This needs exactly 3 endpoints\n");
           if (check_interface_mode == FOMA_AT_MODE) goto pepo_2;
            return -EINVAL;
        }

look_for_collapsed_interface:
        for (i = 0; i < 3; i++) {
            struct usb_endpoint_descriptor *ep;
            ep = &data_interface->cur_altsetting->endpoint[i].desc;

            if (usb_endpoint_is_int_in(ep))
                epctrl = ep;
            else if (usb_endpoint_is_bulk_out(ep))
                epwrite = ep;
            else if (usb_endpoint_is_bulk_in(ep))
                epread = ep;
            else
                return -EINVAL;
        }
        if (!epctrl || !epread || !epwrite)
            return -ENODEV;
        else
            goto made_compressed_probe;
    }

skip_normal_probe:

    /*workaround for switched interfaces */
    if (data_interface->cur_altsetting->desc.bInterfaceClass
                        != CDC_DATA_INTERFACE_TYPE) {
        if (control_interface->cur_altsetting->desc.bInterfaceClass
                        == CDC_DATA_INTERFACE_TYPE) {
            struct usb_interface *t;
            dev_dbg(&intf->dev,
                "Your device has switched interfaces.\n");
            t = control_interface;
            control_interface = data_interface;
            data_interface = t;
        } else {
            return -EINVAL;
        }
    }
pepo_2:    
    /* Accept probe requests only for the control interface */
    if (!combined_interfaces && intf != control_interface)
        return -ENODEV;
    
    if (!combined_interfaces && usb_interface_claimed(data_interface)) {
        /* valid in this context */
        dev_dbg(&intf->dev, "The data interface isn't available\n");
        return -EBUSY;
    }

    if (check_interface_mode != FOMA_AT_MODE) {

    if (data_interface->cur_altsetting->desc.bNumEndpoints < 2)
        return -EINVAL;

    epctrl = &control_interface->cur_altsetting->endpoint[0].desc;
    epread = &data_interface->cur_altsetting->endpoint[0].desc;
    epwrite = &data_interface->cur_altsetting->endpoint[1].desc;
    }
    else 
    { //  ..pepo AT_MODE fake 
        epctrl = &control_interface->cur_altsetting->endpoint[0].desc;
        epread = &control_interface->cur_altsetting->endpoint[0].desc;
        epwrite = &control_interface->cur_altsetting->endpoint[0].desc;
        goto made_compressed_probe;
    }
    
    /* workaround for switched endpoints */
    if (!usb_endpoint_dir_in(epread)) {
        /* descriptors are swapped */
        struct usb_endpoint_descriptor *t;
        dev_dbg(&intf->dev,
            "The data interface has switched endpoints\n");
        t = epread;
        epread = epwrite;
        epwrite = t;
    }

made_compressed_probe:

    dbg("interfaces are valid");
    for (minor = 0; minor < ACM_TTY_MINORS && acm_table[minor]; minor++);

    if (minor == ACM_TTY_MINORS) {
        dev_err(&intf->dev, "no more free acm devices\n");
        return -ENODEV;
    }

    acm = kzalloc(sizeof(struct acm), GFP_KERNEL);
    if (acm == NULL) {
        dev_dbg(&intf->dev, "out of memory (acm kzalloc)\n");
        goto alloc_fail;
    }

    ctrlsize = le16_to_cpu(epctrl->wMaxPacketSize);
    readsize = le16_to_cpu(epread->wMaxPacketSize) *
                (quirks == SINGLE_RX_URB ? 1 : 2);
    acm->combined_interfaces = combined_interfaces;
    acm->writesize = le16_to_cpu(epwrite->wMaxPacketSize) * 20;
    acm->control = control_interface;
    acm->data = data_interface;
    acm->minor = minor;
    acm->dev = usb_dev;
    acm->ctrl_caps = ac_management_function;
    if (quirks & NO_CAP_LINE)
        acm->ctrl_caps &= ~USB_CDC_CAP_LINE;
    acm->ctrlsize = ctrlsize;
    acm->readsize = readsize;
    acm->rx_buflimit = num_rx_buf;
    acm->urb_task.func = acm_rx_tasklet;
    acm->urb_task.data = (unsigned long) acm;
    INIT_WORK(&acm->work, acm_softint);
    init_waitqueue_head(&acm->drain_wait);
    spin_lock_init(&acm->throttle_lock);
    spin_lock_init(&acm->write_lock);
    spin_lock_init(&acm->read_lock);
    mutex_init(&acm->mutex);
    acm->rx_endpoint = usb_rcvbulkpipe(usb_dev, epread->bEndpointAddress);
    acm->is_int_ep = usb_endpoint_xfer_int(epread);
    acm->check_mode = check_interface_mode; // pepo
    if (acm->is_int_ep)
        acm->bInterval = epread->bInterval;
    tty_port_init(&acm->port);
    acm->port.ops = &acm_port_ops;

    buf = usb_alloc_coherent(usb_dev, ctrlsize, GFP_KERNEL, &acm->ctrl_dma);
    if (!buf) {
        dev_dbg(&intf->dev, "out of memory (ctrl buffer alloc)\n");
        goto alloc_fail2;
    }
    acm->ctrl_buffer = buf;

    if (acm_write_buffers_alloc(acm) < 0) {
        dev_dbg(&intf->dev, "out of memory (write buffer alloc)\n");
        goto alloc_fail4;
    }

    acm->ctrlurb = usb_alloc_urb(0, GFP_KERNEL);
    if (!acm->ctrlurb) {
        dev_dbg(&intf->dev, "out of memory (ctrlurb kmalloc)\n");
        goto alloc_fail5;
    }
    for (i = 0; i < num_rx_buf; i++) {
        struct acm_ru *rcv = &(acm->ru[i]);

        rcv->urb = usb_alloc_urb(0, GFP_KERNEL);
        if (rcv->urb == NULL) {
            dev_dbg(&intf->dev,
                "out of memory (read urbs usb_alloc_urb)\n");
            goto alloc_fail6;
        }

        rcv->urb->transfer_flags |= URB_NO_TRANSFER_DMA_MAP;
        rcv->instance = acm;
    }
    for (i = 0; i < num_rx_buf; i++) {
        struct acm_rb *rb = &(acm->rb[i]);

        rb->base = usb_alloc_coherent(acm->dev, readsize,
                GFP_KERNEL, &rb->dma);
        if (!rb->base) {
            dev_dbg(&intf->dev,
                "out of memory (read bufs usb_alloc_coherent)\n");
            goto alloc_fail7;
        }
    }
    for (i = 0; i < ACM_NW; i++) {
        struct acm_wb *snd = &(acm->wb[i]);

        snd->urb = usb_alloc_urb(0, GFP_KERNEL);
        if (snd->urb == NULL) {
            dev_dbg(&intf->dev,
                "out of memory (write urbs usb_alloc_urb)");
            goto alloc_fail8;
        }

        if (usb_endpoint_xfer_int(epwrite))
            usb_fill_int_urb(snd->urb, usb_dev,
                usb_sndbulkpipe(usb_dev, epwrite->bEndpointAddress),
                NULL, acm->writesize, acm_write_bulk, snd, epwrite->bInterval);
        else
            usb_fill_bulk_urb(snd->urb, usb_dev,
                usb_sndbulkpipe(usb_dev, epwrite->bEndpointAddress),
                NULL, acm->writesize, acm_write_bulk, snd);
        snd->urb->transfer_flags |= URB_NO_TRANSFER_DMA_MAP;
        snd->instance = acm;
    }
    usb_set_intfdata(intf, acm);

    i = device_create_file(&intf->dev, &dev_attr_bmCapabilities);
    if (i < 0)
        goto alloc_fail8;

    if (cfd) { /* export the country data */
        acm->country_codes = kmalloc(cfd->bLength - 4, GFP_KERNEL);
        if (!acm->country_codes)
            goto skip_countries;
        acm->country_code_size = cfd->bLength - 4;
        memcpy(acm->country_codes, (u8 *)&cfd->wCountyCode0,
                            cfd->bLength - 4);
        acm->country_rel_date = cfd->iCountryCodeRelDate;

        i = device_create_file(&intf->dev, &dev_attr_wCountryCodes);
        if (i < 0) {
            kfree(acm->country_codes);
            goto skip_countries;
        }

        i = device_create_file(&intf->dev,
                        &dev_attr_iCountryCodeRelDate);
        if (i < 0) {
            device_remove_file(&intf->dev, &dev_attr_wCountryCodes);
            kfree(acm->country_codes);
            goto skip_countries;
        }
    }

skip_countries:
    usb_fill_int_urb(acm->ctrlurb, usb_dev,
             usb_rcvintpipe(usb_dev, epctrl->bEndpointAddress),
             acm->ctrl_buffer, ctrlsize, acm_ctrl_irq, acm,
             /* works around buggy devices */
             epctrl->bInterval ? epctrl->bInterval : 0xff);
    acm->ctrlurb->transfer_flags |= URB_NO_TRANSFER_DMA_MAP;
    acm->ctrlurb->transfer_dma = acm->ctrl_dma;

    dev_info(&intf->dev, "ttyACM%d: USB ACM device\n", minor);

    if (acm->check_mode == FOMA_MODEM_MODE) { // pepo
        acm_set_control(acm, acm->ctrlout);
        acm->line.dwDTERate = cpu_to_le32(9600);
        acm->line.bDataBits = 8;
        acm_set_line(acm, &acm->line); 
    }
    
    foma_set_init(acm); // pepo
 
   if (acm->check_mode != FOMA_AT_MODE) { // pepo
        usb_driver_claim_interface(&acm_driver, data_interface, acm);
        usb_set_intfdata(data_interface, acm);
    }
    else {
        usb_driver_claim_interface(&acm_driver, control_interface,acm);
        usb_set_intfdata(control_interface, acm);
    }
    

    usb_get_intf(control_interface);
    tty_register_device(acm_tty_driver, minor, &control_interface->dev);

    acm_table[minor] = acm;

    return 0;
alloc_fail8:
    for (i = 0; i < ACM_NW; i++)
        usb_free_urb(acm->wb[i].urb);
alloc_fail7:
    acm_read_buffers_free(acm);
alloc_fail6:
    for (i = 0; i < num_rx_buf; i++)
        usb_free_urb(acm->ru[i].urb);
    usb_free_urb(acm->ctrlurb);
alloc_fail5:
    acm_write_buffers_free(acm);
alloc_fail4:
    usb_free_coherent(usb_dev, ctrlsize, acm->ctrl_buffer, acm->ctrl_dma);
alloc_fail2:
    kfree(acm);
alloc_fail:
    return -ENOMEM;
}

static void stop_data_traffic(struct acm *acm)
{
    int i;
    dbg("Entering stop_data_traffic");

    tasklet_disable(&acm->urb_task);

    usb_kill_urb(acm->ctrlurb);
    for (i = 0; i < ACM_NW; i++)
        usb_kill_urb(acm->wb[i].urb);
    for (i = 0; i < acm->rx_buflimit; i++)
        usb_kill_urb(acm->ru[i].urb);

    tasklet_enable(&acm->urb_task);

    cancel_work_sync(&acm->work);
}

static void acm_disconnect(struct usb_interface *intf)
{
    struct acm *acm = usb_get_intfdata(intf);
    struct usb_device *usb_dev = interface_to_usbdev(intf);
    struct tty_struct *tty;

    /* sibling interface is already cleaning up */
    if (!acm)
        return;

    mutex_lock(&open_mutex);
    if (acm->country_codes) {
        device_remove_file(&acm->control->dev,
                &dev_attr_wCountryCodes);
        device_remove_file(&acm->control->dev,
                &dev_attr_iCountryCodeRelDate);
    }
    device_remove_file(&acm->control->dev, &dev_attr_bmCapabilities);
    acm->dev = NULL;
    usb_set_intfdata(acm->control, NULL);
    usb_set_intfdata(acm->data, NULL);

    stop_data_traffic(acm);

    acm_write_buffers_free(acm);
    usb_free_coherent(usb_dev, acm->ctrlsize, acm->ctrl_buffer,
              acm->ctrl_dma);
    acm_read_buffers_free(acm);

    if (!acm->combined_interfaces)
        usb_driver_release_interface(&acm_driver, intf == acm->control ?
                    acm->data : acm->control);

    if (acm->port.count == 0) {
        acm_tty_unregister(acm);
        mutex_unlock(&open_mutex);
        return;
    }

    mutex_unlock(&open_mutex);
    tty = tty_port_tty_get(&acm->port);
    if (tty) {
        tty_hangup(tty);
        tty_kref_put(tty);
    }
}

#ifdef CONFIG_PM
static int acm_suspend(struct usb_interface *intf, pm_message_t message)
{
    struct acm *acm = usb_get_intfdata(intf);
    int cnt;

    if (message.event & PM_EVENT_AUTO) {
        int b;

        spin_lock_irq(&acm->read_lock);
        spin_lock(&acm->write_lock);
        b = acm->processing + acm->transmitting;
        spin_unlock(&acm->write_lock);
        spin_unlock_irq(&acm->read_lock);
        if (b)
            return -EBUSY;
    }

    spin_lock_irq(&acm->read_lock);
    spin_lock(&acm->write_lock);
    cnt = acm->susp_count++;
    spin_unlock(&acm->write_lock);
    spin_unlock_irq(&acm->read_lock);

    if (cnt)
        return 0;
    /*
    we treat opened interfaces differently,
    we must guard against open
    */
    mutex_lock(&acm->mutex);

    if (acm->port.count)
        stop_data_traffic(acm);

    mutex_unlock(&acm->mutex);
    return 0;
}

static int acm_resume(struct usb_interface *intf)
{
    struct acm *acm = usb_get_intfdata(intf);
    struct acm_wb *wb;
    int rv = 0;
    int cnt;

    spin_lock_irq(&acm->read_lock);
    acm->susp_count -= 1;
    cnt = acm->susp_count;
    spin_unlock_irq(&acm->read_lock);

    if (cnt)
        return 0;

    mutex_lock(&acm->mutex);
    if (acm->port.count) {
        rv = usb_submit_urb(acm->ctrlurb, GFP_NOIO);

        spin_lock_irq(&acm->write_lock);
        if (acm->delayed_wb) {
            wb = acm->delayed_wb;
            acm->delayed_wb = NULL;
            spin_unlock_irq(&acm->write_lock);
            acm_start_wb(acm, wb);
        } else {
            spin_unlock_irq(&acm->write_lock);
        }

        /*
         * delayed error checking because we must
         * do the write path at all cost
         */
        if (rv < 0)
            goto err_out;

        tasklet_schedule(&acm->urb_task);
    }

err_out:
    mutex_unlock(&acm->mutex);
    return rv;
}

static int acm_reset_resume(struct usb_interface *intf)
{
    struct acm *acm = usb_get_intfdata(intf);
    struct tty_struct *tty;

    mutex_lock(&acm->mutex);
    if (acm->port.count) {
        tty = tty_port_tty_get(&acm->port);
        if (tty) {
            tty_hangup(tty);
            tty_kref_put(tty);
        }
    }
    mutex_unlock(&acm->mutex);
    return acm_resume(intf);
}

#endif /* CONFIG_PM */
static struct usb_driver acm_driver = {
// name change, acm -> acm_FOMA ..pepo
    .name =            "acm_FOMA",
    .probe =           acm_probe,
    .disconnect =      acm_disconnect,
#ifdef CONFIG_PM
    .suspend =         acm_suspend,
    .resume =          acm_resume,
    .reset_resume =    acm_reset_resume,
#endif
    .id_table =        acm_ids,
#ifdef CONFIG_PM
    .supports_autosuspend = 1,
#endif
};

/*
 * TTY driver structures.
 */

static const struct tty_operations acm_ops = {
    .open =            acm_tty_open,
    .close =           acm_tty_close,
    .hangup =          acm_tty_hangup,
    .write =           acm_tty_write,
    .write_room =      acm_tty_write_room,
    .ioctl =           acm_tty_ioctl,
    .throttle =        acm_tty_throttle,
    .unthrottle =      acm_tty_unthrottle,
    .chars_in_buffer = acm_tty_chars_in_buffer,
    .break_ctl =       acm_tty_break_ctl,
    .set_termios =     acm_tty_set_termios,
    .tiocmget =        acm_tty_tiocmget,
    .tiocmset =        acm_tty_tiocmset,
};

/*
 * Init / exit.
 */

static int __init acm_init(void)
{
    int retval;
    acm_tty_driver = alloc_tty_driver(ACM_TTY_MINORS);
    if (!acm_tty_driver)
        return -ENOMEM;
    acm_tty_driver->owner = THIS_MODULE,
// name change, acm -> acm-FOMA ..pepo
    acm_tty_driver->driver_name = "acm-FOMA",
    acm_tty_driver->name = "ttyACM",
    acm_tty_driver->major = ACM_TTY_MAJOR,
    acm_tty_driver->minor_start = 0,
    acm_tty_driver->type = TTY_DRIVER_TYPE_SERIAL,
    acm_tty_driver->subtype = SERIAL_TYPE_NORMAL,
    acm_tty_driver->flags = TTY_DRIVER_REAL_RAW | TTY_DRIVER_DYNAMIC_DEV;
    acm_tty_driver->init_termios = tty_std_termios;
    acm_tty_driver->init_termios.c_cflag = B9600 | CS8 | CREAD |
                                HUPCL | CLOCAL;
    tty_set_operations(acm_tty_driver, &acm_ops);

    retval = tty_register_driver(acm_tty_driver);
    if (retval) {
        put_tty_driver(acm_tty_driver);
        return retval;
    }

    retval = usb_register(&acm_driver);
    if (retval) {
        tty_unregister_driver(acm_tty_driver);
        put_tty_driver(acm_tty_driver);
        return retval;
    }

    printk(KERN_INFO KBUILD_MODNAME ": " DRIVER_VERSION ":"
           DRIVER_DESC "\n");

    return 0;
}

static void __exit acm_exit(void)
{
    usb_deregister(&acm_driver);
    tty_unregister_driver(acm_tty_driver);
    put_tty_driver(acm_tty_driver);
}

module_init(acm_init);
module_exit(acm_exit);

MODULE_AUTHOR(DRIVER_AUTHOR);
MODULE_DESCRIPTION(DRIVER_DESC);
MODULE_LICENSE("GPL");
MODULE_ALIAS_CHARDEV_MAJOR(ACM_TTY_MAJOR);
