#
# Fuse::Filesys::Virtual::FSWrapper
#
#

package Fuse::Filesys::Virtual::FSWrapper;

use warnings;
use strict;

=head1 NAME

Fuse::Filesys::Virtual::FSWrapper - Filesys::Virtual wrapper

=head1 VERSION

Version 0.01

=cut

our $VERSION = '0.01';

=head1 SYNOPSIS

Internal module for Fuse::Filesys::Virtual, provides some functions
to Filesys::Virtual object.

    use Fuse::Filesys::Virtual::FSWrapper;

    my $fs = Filesys::Virtual::Foo->new();
    my $wfs = Fuse::Filesys::Virtual::FSWrapper->new($fs);

    ...

    $wfs->rename("/path/to/oldname", "path/to/newname");

=head1 EXPORT

Nothing.

=cut

use Carp;
use Filesys::Virtual;

our $AUTOLOAD;

#
# get entries
# (\@dirs, \@files);
#
sub _list_recurse {
    my $self = shift;
    my ($cur) = @_;

    my @dirs;
    my @files;

    if ($self->test('d', $cur)) {
	my @entries =
	    grep { $_ ne '.' && $_ ne '..' } $self->list($cur);

	for my $e (@entries) {
	    my $path = ($cur =~ /\/$/) ? "$cur$e" : "$cur/$e";

	    if ($self->test('d', $path)) {
		push(@dirs, $path);

		my ($d, $f) = $self->_list_recurse($path);
		push(@dirs, @{$d});
		push(@files, @{$f});
	    }
	    else {
		push(@files, $path);
	    }
	}
    }
    else {
	push(@files, $cur);
    }

    return (\@dirs, \@files);
}

=head1 FUNCTIONS

=head2 new

Wrap Filesys::Virtual object.
wrapped object is returned.

=cut

sub new {
    my $class = shift;
    my ($fs) = @_;

    my $self = {
	filesys => $fs,
    };

    bless $self, $class;
}

sub _copy_file {
    my $self = shift;
    my ($srcname, $destname) = @_;

    my ($dev,$ino,$mode,$nlink,$uid,$gid,$rdev,$size,
	$atime,$mtime,$ctime,$blksize,$blocks)
	= $self->stat($srcname);

    my $in = $self->open_read($srcname)
	or die "$srcname: cannot open: $!";

    my $out = $self->open_write($destname, undef)
	or die "$destname: cannot create: $!";

    my $buf;
    my $buflen = 4096;
    while(sysread($in, $buf, $buflen)) {
	syswrite($out, $buf);
    }

    $self->close_write($out);
    $self->close_read($in);

    eval { $self->utime($atime, $mtime, $destname); };
    eval { $self->chmod($mode, $destname); };
}

#
# rename a file (not a directory)
#
sub _rename_file {
    my $self = shift;
    my ($oldname, $newname) = @_;

    eval {
	$self->_copy_file($oldname, $newname);

	unless ($self->delete($oldname)) {
	    die "cannot delete $oldname: $!";
	}
    };
    if ($@) {
	print STDERR "$@";
	my $err = $! || 1;

	$self->delete($newname);
	$! = $err;

	return; # undef
    }

    $self->delete($oldname);

    return 1;
}


#
# rename a directory (recursive copy)
#
sub _rename_dir {
    my $self = shift;
    my ($oldname, $newname) = @_;

    my ($srcdirs, $srcfiles) = $self->_list_recurse($oldname);
    unshift(@{$srcdirs}, $oldname);

    my %dirstats;

    for my $dir (@{$srcdirs}) {
	my $destdir = $dir;
	$destdir =~ s/^\Q$oldname\E/$newname/;
	
        my ($dev,$ino,$mode,$nlink,$uid,$gid,$rdev,$size,
	    $atime,$mtime,$ctime,$blksize,$blocks)
	    = $self->stat($dir);

	unless ($self->test('d', $destdir)) {
	    $self->mkdir($destdir, $mode);
	}
	$dirstats{$destdir} = { atime => $atime, mtime => $mtime };
    }

    for my $file (@{$srcfiles}) {
	my $destfile = $file;
	$destfile =~ s/^\Q$oldname\E/$newname/;

	$self->_copy_file($file, $destfile);
    }

    for my $d (keys %dirstats) {
	eval { $self->utime($dirstats{$d}->{atime},
			    $dirstats{$d}->{mtime},
			    $d); };
    }

    #
    # TODO:
    # If error is found while deleting original files, 
    # should I rollback filesys?
    #
    for my $file (@{$srcfiles}) {
	$self->delete($file);
    }
    for my $dir (reverse @{$srcdirs}) {
	$self->rmdir($dir);
    }

    return 1;
}

=head2 rename (OLDNAME, NEWNAME)

rename oldname to newname

=cut

sub rename {
    my $self = shift;
    my ($oldname, $newname) = @_;

    my $dest = $newname;

    if ($self->test('d', $newname)) {
	$dest .= '/' unless ($dest =~ /\/$/);
	my @segs = split(/\//, $oldname);
	$dest .= $segs[$#segs];
    }

    if ($self->test('d', $oldname)) {
	return $self->_rename_dir($oldname, $dest);
    }
    else {
	return $self->_rename_file($oldname, $dest);
    }
}

#
# other method is same as original...
#
sub AUTOLOAD {
    my $self = shift;

    my $method = $AUTOLOAD;
    $method =~ s/.*:://;

    {
	no warnings "redefine";
	local *Filesys::Virtual::carp = sub {
	    my $msg = shift;
	    Carp::croak($msg) if ($msg =~ / Unimplemented/);
	    Carp::carp($msg);
	};

	return $self->{filesys}->$method(@_);
    }
}

sub DESTROY {
}

1;
