/*
 * BDF2BDB - BDF font data conveter
 * Release version 0.0.0
 * Copyright (c)2006 satoshi akabane
 * http://www.logical-paradox.org/
 *
 * The MIT License
 * Copyright (c) 2006 satoshi akabane(akabane@logical-paradox.org)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights 
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell 
 * copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 * The above copyright notice and this permission notice shall be included in 
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE 
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#include <stdio.h>
#include <stdlib.h>
#include "common.h"
#include "bdb.h"
#include "bdf2bdb.h"
#include "mappings.h"
#include "apcommon.h"

/*
 * startup main function.
 * in:
 *		argc - number of argument.
 *		argv - pointers of argument.
 * out:
 *		program exit code.
 */
int main(int argc, char **argv)
{
	int exitcode = 0;

	if(argc != 3) {
		print_usage();
		return EXIT_FAILURE;
	}

	/* init runtime environment. */
	init();

	/* start .bdf --> .bdb file format conversion. */
	exitcode = execute(argv[1], argv[2]);

	if(exitcode == EXIT_SUCCESS) {
		printf("done.\n");
	} else {
		printf("terminate with exitcode [%d]\n", exitcode);
	}
	return exitcode;
}

/*
 * start conversion .bdf --> .bdb
 * in:
 *		bdf_filename - .bdf filename
 *		bdb_filename - .bdb filename
 * out:
 *		exit code.
 */
int execute(const char *bdf_filename, const char *bdb_filename)
{
	FILE *fp;
	int bdf_filesize;
	int readblocks;
	BDB *bdb;
	char *filebuf;
	char *nextptr;
	char *linebuf = (char *)malloc(LINEBUFLEN);
	int lineno = 0, error_count = 0;

	if((fp = fopen(bdf_filename, "rb")) == NULL) {
		printf("couldn't read %s\n", bdf_filename);
		return EXIT_FAILURE;
	}

	/* calc filesize */
	bdf_filesize = get_filesize(fp);

	/* read file into memory */
	filebuf = (char *)malloc(bdf_filesize +1);		// +1 = '\0'
	readblocks = fread(filebuf, 1, bdf_filesize, fp);
	fclose(fp);
	*(filebuf + readblocks) = '\0';					// store an end mark of string.

	/*
	 * create BDB data structure instance.
	 */
	bdb = new_bdb();

	/*
	 * parse lines.
	 */
	nextptr = filebuf;
	log(LOG_INFO, "phase 1: parsing BDF file...", -1);
	while((nextptr = readline(nextptr, LINEBUFLEN, linebuf)) != NULL) {
		int error_code = parser(linebuf, bdb);
		if(error_code != ERR_NO_ERROR) {
			log(LOG_ERROR, get_error_msg(error_code), lineno);
			error_count++;
		}
		lineno++;
	}

	if(error_count == 0) {
		/* if no error, output BDB data block as a file. */
		log(LOG_INFO, "phase 2: writing BDB file...", -1);
		output_bdb(bdb_filename, bdb);
	}

	log(LOG_INFO, "phase 3: release all buffers...", -1);

	/* release file and line buffer */
	free(filebuf);
	free(linebuf);

	/* release BDB data structure instance. */
	delete_bdb(bdb);

	return EXIT_SUCCESS;
}

/*
 * print usage of the bdf2bdb.
 * in:
 *		nothing.
 * out:
 *		nothing.
 */
void print_usage() {
	printf("BDF font file converter release 1.0.0\n");
	printf("Copyright (c)2006 satoshi akabane(http://www.logical-paradox.org/\n\n");
	printf("usage: bdf2bdb <bdf-filename> <output-filename>\n");
}


/*
 * initialize function of bdf2bdb
 * in:
 *		nothing.
 * out:
 *		nothing.
 */
void init()
{
	int i = 0;
	KEYWORD* keyword_ptr = gs_keyword_mappings;
	char **ptr;

	/* caluculate total keywords */
	while(keyword_ptr != NULL && keyword_ptr[i].keyword != NULL && keyword_ptr[i].func != NULL) {
		i++;
	}
	gs_number_of_keywords = i;

	/* count error messages */
	ptr = gs_error_msg_tbl;
	i = 0;
	while(ptr != NULL && ptr[i] != NULL) {
		i++;
	}
	gs_number_of_error_msgs = i;
}
