/*
 * BDF2BDB - BDF font data conveter
 * Release version 0.0.0
 * Copyright (c)2006 satoshi akabane
 * http://www.logical-paradox.org/
 *
 * The MIT License
 * Copyright (c) 2006 satoshi akabane(akabane@logical-paradox.org)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights 
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell 
 * copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 * The above copyright notice and this permission notice shall be included in 
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE 
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "common.h"
#include "bdb.h"
#include "apcommon.h"
#include "mappings.h"
#include "debug.h"


/*
 * allocate new BDB data structure.
 * in:
 *		nothing.
 * out:
 *		pointer of BDB data structure.
 */
BDB *new_bdb()
{
	BDB *bdb;

	/* allocate BDB buffer */
	bdb = (BDB *)malloc(sizeof(BDB));

	bdb->font_count = 0;
	bdb->scanning_bitmap = 0;

	/* setup magic number and major/minor version number. */
	strncpy(bdb->magic_number, "BDBF", 4);

	return bdb;
}


/*
 * delete BDB data structure.
 * in:
 *		bdb - BDB data structure.
 * out:
 *		nothing.
 */
void delete_bdb(BDB *bdb)
{
	int i;
	FONT *font;

	if(bdb == NULL) {
		return;
	}

	if(bdb->font != NULL) {
		/* release font data */
		for(i = 0; i < bdb->font_count; i++) {
			font = &bdb->font[i];
			if(font->bitmap_count > 0) {
				free(font->bitmap);
			}
		}
		free(bdb->font);
	}

	free(bdb);
}


/*
 * parse a line and store BDB file structure.
 * in:
 *		*ptr - a line of BDF file.
 *		*bdb - BDB file structure.
 * out:
 *		error code(0: no error).
 */
int parser(char *ptr, BDB *bdb)
{
	int token_length = 0;
	int num_of_tokens = 0;
	int in_string_literal = 0;
	int i;
	int guardian = 1;
	int error_code = ERR_NO_ERROR;
	char **tokens = (char **)calloc(MAX_TOKEN_NUM, sizeof(char *));
	char *token_bgn_ptr = ptr;

	/*
	 * token separation.
	 */
	while(guardian == 1 && num_of_tokens < MAX_TOKEN_NUM) {
		switch(*ptr) {
			case '"':
				in_string_literal = 1 - in_string_literal;
				break;
			case ' ':
			case '\0':
			case '\r':
			case '\n':
				if(!in_string_literal) {
					if(token_length > 0) {
						tokens[num_of_tokens] = (char *)calloc(token_length +1, sizeof(char));
						strncpy(tokens[num_of_tokens], token_bgn_ptr, token_length);
						*(tokens[num_of_tokens] + token_length) = '\0';
						num_of_tokens++;
					}
					if(*ptr == '\0' || *ptr == '\r' || *ptr == '\n') {
						guardian = 0;
						continue;
					}

					token_length = 0;
					ptr++;
					token_bgn_ptr = ptr;

					continue;
				} else {
					/* white space found in string literal. */
					token_length++;
				}
				break;
			default:
				token_length++;
				break;
		}
		ptr++;
	}

#ifdef DEBUG
	/* display tokens if debug mode is enabled. */
	print_tokens(tokens, num_of_tokens);
#endif

	/* evaluate expression - if one or more tokens has been found. */
	if(num_of_tokens > 0) {
		error_code = evaluate_expression(num_of_tokens, tokens, bdb);
	}

	/*
	 * release token buffers.
	 */
	for(i = 0; i < num_of_tokens; i++) {
		/*free(tokens[i]);*/
	}
	free(tokens);

	return error_code;
}

/*
 * evaluate an expression.
 * in:
 *		num_of_tokens - number of tokens
 *		**tokens - pointer of pointers of tokens
 *		*bdb - BDB file structure.
 * out:
 *		error code(0: no error).
 */
int evaluate_expression(int num_of_tokens, char **tokens, BDB *bdb)
{
	int error_code = ERR_NO_ERROR;
	int bit_ptn;

	KEYWORD* keyword_ptr;
	FONT* font_ptr;

	if(num_of_tokens <= 0) {
		log(LOG_ERROR, "no token is parsed", -1);
		error_code = ERR_SYNTAX_ERROR;
		goto evaluate_expression_finally;
	}

	if(bdb->scanning_bitmap == 0) {
		/* get the first token(command) */
		keyword_ptr = search_keyword_blk(tokens[0]);
		if(keyword_ptr == NULL) {
			error_code = ERR_NO_SUCH_KEYWORD;
			goto evaluate_expression_finally;
		}
	} else {
		/* set current bitmap images. */
		bit_ptn = hex_to_int32(tokens[0]);
		font_ptr = &bdb->font[bdb->font_count];
		int16_to_bytes(bit_ptn, &(font_ptr->bitmap[font_ptr->bitmap_count * MAX_BITMAP_WIDTH_BYTES]));
		font_ptr->bitmap_count++;

		if(font_ptr->bitmap_count >= bdb->pixel_size) {
			/* abort bitmap scanning if received all bitmap lines. */
			/*font_ptr->bitmap_count = 0;*/
			bdb->scanning_bitmap = 0;
		}
		/* skip keyword handler. */
		goto evaluate_expression_finally;
	}

	/* invoke keyword handler function. */
	error_code = (keyword_ptr->func)(num_of_tokens, tokens, bdb);

/* finally block */
evaluate_expression_finally:
	return error_code;
}

/*
 * search a keyword block information.
 * (using linear search)
 * in:
 *		*token - keyword
 * out:
 *		pointer of KEYWORD block(null: not found).
 */
KEYWORD *search_keyword_blk(char *token)
{
	KEYWORD *ptr = NULL;
	int i = 0;

	for(i = 0; i < gs_number_of_keywords; i++) {
		ptr = &gs_keyword_mappings[i];
		if(strlen(ptr->keyword) == strlen(token) && strncmp(ptr->keyword, token, strlen(token)) == 0) {
			return ptr;
		}
	}

	return ptr;
}

/*
 * output BDB data file.
 * in:
 *		filename - output filename.
 *		*bdb - BDB data structure.
 * out:
 *		nothing.
 */
void output_bdb(const char *filename, BDB *bdb)
{
	FILE *fp;
	char font_count[4];
	int i, j;

	if((fp = fopen(filename, "wb")) == NULL) {
		log(LOG_ERROR, "file creation failure.", -1);
		return;
	}

	int32_to_bytes(bdb->font_count, font_count);

	fwrite(bdb->magic_number, sizeof(bdb->magic_number), 1, fp);
	fwrite(&(bdb->major_version), sizeof(bdb->major_version), 1, fp);
	fwrite(&(bdb->minor_version), sizeof(bdb->minor_version), 1, fp);
	fwrite(&(bdb->pixel_size), sizeof(bdb->pixel_size), 1, fp);
	fwrite(&(bdb->high_dpi), sizeof(bdb->high_dpi), 1, fp);
	fwrite(&(bdb->low_dpi), sizeof(bdb->low_dpi), 1, fp);
	fwrite(&(bdb->font_bounding_box_width), sizeof(bdb->font_bounding_box_width), 1, fp);
	fwrite(&(bdb->font_bounding_box_height), sizeof(bdb->font_bounding_box_height), 1, fp);
	fwrite(&(bdb->font_bounding_box_horizon_offset), sizeof(bdb->font_bounding_box_horizon_offset), 1, fp);
	fwrite(&(bdb->font_bounding_box_vertical_offset), sizeof(bdb->font_bounding_box_vertical_offset), 1, fp);
	fwrite(bdb->chars, sizeof(bdb->chars), 1, fp);
/*	fwrite(font_count, sizeof(font_count), 1, fp); */

	/* output bitmap font data. */
	for(i = 0; i < bdb->font_count; i++) {
		fwrite(bdb->font[i].start_char, sizeof(bdb->font[i].start_char), 1, fp);
		fwrite(bdb->font[i].encoding, sizeof(bdb->font[i].encoding), 1, fp);
		fwrite(bdb->font[i].swidth, sizeof(bdb->font[i].swidth), 1, fp);
		fwrite(bdb->font[i].dwidth, sizeof(bdb->font[i].dwidth), 1, fp);
		fwrite(bdb->font[i].bbx, sizeof(bdb->font[i].bbx), 1, fp);

		for(j = 0; j < bdb->pixel_size; j++) {
			fwrite(&(bdb->font[i].bitmap[j * MAX_BITMAP_WIDTH_BYTES]), sizeof(char), MAX_BITMAP_WIDTH_BYTES, fp);
		}
	}

	fclose(fp);
}
