/*
 * BDF2BDB - BDF font data conveter
 * Release version 0.0.0
 * Copyright (c)2006 satoshi akabane
 * http://www.logical-paradox.org/
 *
 * The MIT License
 * Copyright (c) 2006 satoshi akabane(akabane@logical-paradox.org)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights 
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell 
 * copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 * The above copyright notice and this permission notice shall be included in 
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE 
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "bdb.h"
#include "common.h"
#include "procedures.h"

/*
 * handler for 'STARTFONT'.
 * in:
 *		num_of_tokens - number of token
 *		**tokens - pointers of tokens
 *		*bdb - BDB file structure.
 */
int p_startfont(int num_of_tokens, char **tokens, BDB *bdb)
{
	if(num_of_tokens != 2) {
		return ERR_SYNTAX_ERROR;
	}
	if(strncmp("2.1", tokens[1], 3) != 0) {
		/* this program accepts only BDF file format v2.1 */
		return ERR_UNSUPPORTED_VERSION;
	}
	bdb->major_version = '2';
	bdb->minor_version = '1';

	return ERR_NO_ERROR;
}

/*
 * handler for 'COMMENT'.
 * in:
 *		num_of_tokens - number of token
 *		**tokens - pointers of tokens
 *		*bdb - BDB file structure.
 */
int p_comment(int num_of_tokens, char **tokens, BDB *bdb)
{
	/* simply skip all comments */
	return ERR_NO_ERROR;
}

/*
 * handler for 'FONT'.
 * in:
 *		num_of_tokens - number of token
 *		**tokens - pointers of tokens
 *		*bdb - BDB file structure.
 */
int p_font(int num_of_tokens, char **tokens, BDB *bdb)
{
	/* this converter don't support 'FONT' keyword. */
	return ERR_NO_ERROR;
}

/*
 * handler for 'SIZE'.
 * in:
 *		num_of_tokens - number of token
 *		**tokens - pointers of tokens
 *		*bdb - BDB file structure.
 */
int p_size(int num_of_tokens, char **tokens, BDB *bdb)
{
	if(num_of_tokens != 4) {
		return ERR_SYNTAX_ERROR;
	}

	bdb->pixel_size = (char)atoi(tokens[1]);
	bdb->high_dpi = (char)atoi(tokens[2]);
	bdb->low_dpi = (char)atoi(tokens[3]);

	return ERR_NO_ERROR;
}

/*
 * handler for 'FONTBOUNDINGBOX'.
 * in:
 *		num_of_tokens - number of token
 *		**tokens - pointers of tokens
 *		*bdb - BDB file structure.
 */
int p_fontboundingbox(int num_of_tokens, char **tokens, BDB *bdb)
{
	if(num_of_tokens != 5) {
		return ERR_SYNTAX_ERROR;
	}

	bdb->font_bounding_box_width = (char)atoi(tokens[1]);
	bdb->font_bounding_box_height = (char)atoi(tokens[2]);
	bdb->font_bounding_box_horizon_offset = (char)atoi(tokens[3]);
	bdb->font_bounding_box_vertical_offset = (char)atoi(tokens[4]);

	return ERR_NO_ERROR;
}

/*
 * handler for 'STARTPROPERTIES'.
 * in:
 *		num_of_tokens - number of token
 *		**tokens - pointers of tokens
 *		*bdb - BDB file structure.
 */
int p_startproperties(int num_of_tokens, char **tokens, BDB *bdb)
{
	if(num_of_tokens != 2) {
		return ERR_SYNTAX_ERROR;
	}

	return ERR_NO_ERROR;
}

/*
 * handler for 'FONTNAME_REGISTRY'.
 * in:
 *		num_of_tokens - number of token
 *		**tokens - pointers of tokens
 *		*bdb - BDB file structure.
 */
int p_fontname_registry(int num_of_tokens, char **tokens, BDB *bdb)
{
	return ERR_NO_ERROR;
}

/*
 * handler for 'FOUNDRY'.
 * in:
 *		num_of_tokens - number of token
 *		**tokens - pointers of tokens
 *		*bdb - BDB file structure.
 */
int p_foundry(int num_of_tokens, char **tokens, BDB *bdb)
{
	return ERR_NO_ERROR;
}

/*
 * handler for 'FAMILY_NAME'.
 * in:
 *		num_of_tokens - number of token
 *		**tokens - pointers of tokens
 *		*bdb - BDB file structure.
 */
int p_family_name(int num_of_tokens, char **tokens, BDB *bdb)
{
	return ERR_NO_ERROR;
}

/*
 * handler for 'WEIGHT_NAME'.
 * in:
 *		num_of_tokens - number of token
 *		**tokens - pointers of tokens
 *		*bdb - BDB file structure.
 */
int p_weight_name(int num_of_tokens, char **tokens, BDB *bdb)
{
	return ERR_NO_ERROR;
}

/*
 * handler for 'SLANT'.
 * in:
 *		num_of_tokens - number of token
 *		**tokens - pointers of tokens
 *		*bdb - BDB file structure.
 */
int p_slant(int num_of_tokens, char **tokens, BDB *bdb)
{
	return ERR_NO_ERROR;
}

/*
 * handler for 'SETWIDTH_NAME'.
 * in:
 *		num_of_tokens - number of token
 *		**tokens - pointers of tokens
 *		*bdb - BDB file structure.
 */
int p_setwidth_name(int num_of_tokens, char **tokens, BDB *bdb)
{
	return ERR_NO_ERROR;
}

/*
 * handler for 'ADD_STYLE_NAME'.
 * in:
 *		num_of_tokens - number of token
 *		**tokens - pointers of tokens
 *		*bdb - BDB file structure.
 */
int p_add_style_name(int num_of_tokens, char **tokens, BDB *bdb)
{
	return ERR_NO_ERROR;
}

/*
 * handler for 'PIXEL_SIZE'.
 * in:
 *		num_of_tokens - number of token
 *		**tokens - pointers of tokens
 *		*bdb - BDB file structure.
 */
int p_pixel_size(int num_of_tokens, char **tokens, BDB *bdb)
{
	if(num_of_tokens != 2) {
		return ERR_SYNTAX_ERROR;
	}

	bdb->pixel_size = (char)atoi(tokens[1]);
	return ERR_NO_ERROR;
}

/*
 * handler for 'POINT_SIZE'.
 * in:
 *		num_of_tokens - number of token
 *		**tokens - pointers of tokens
 *		*bdb - BDB file structure.
 */
int p_point_size(int num_of_tokens, char **tokens, BDB *bdb)
{
	return ERR_NO_ERROR;
}

/*
 * handler for 'RESOLUTION_X'.
 * in:
 *		num_of_tokens - number of token
 *		**tokens - pointers of tokens
 *		*bdb - BDB file structure.
 */
int p_resolution_x(int num_of_tokens, char **tokens, BDB *bdb)
{
	if(num_of_tokens != 2) {
		return ERR_SYNTAX_ERROR;
	}

	bdb->high_dpi = (char)atoi(tokens[1]);
	return ERR_NO_ERROR;
}

/*
 * handler for 'RESOLUTION_Y'.
 * in:
 *		num_of_tokens - number of token
 *		**tokens - pointers of tokens
 *		*bdb - BDB file structure.
 */
int p_resolution_y(int num_of_tokens, char **tokens, BDB *bdb)
{
	if(num_of_tokens != 2) {
		return ERR_SYNTAX_ERROR;
	}

	bdb->low_dpi = (char)atoi(tokens[1]);
	return ERR_NO_ERROR;
}

/*
 * handler for 'SPACING'.
 * in:
 *		num_of_tokens - number of token
 *		**tokens - pointers of tokens
 *		*bdb - BDB file structure.
 */
int p_spacing(int num_of_tokens, char **tokens, BDB *bdb)
{
	return ERR_NO_ERROR;
}

/*
 * handler for 'AVERAGE_WIDTH'.
 * in:
 *		num_of_tokens - number of token
 *		**tokens - pointers of tokens
 *		*bdb - BDB file structure.
 */
int p_average_width(int num_of_tokens, char **tokens, BDB *bdb)
{
	return ERR_NO_ERROR;
}

/*
 * handler for 'CHARSET_REGISTRY'.
 * in:
 *		num_of_tokens - number of token
 *		**tokens - pointers of tokens
 *		*bdb - BDB file structure.
 */
int p_charset_registry(int num_of_tokens, char **tokens, BDB *bdb)
{
	return ERR_NO_ERROR;
}

/*
 * handler for 'CHARSET_ENCODING'.
 * in:
 *		num_of_tokens - number of token
 *		**tokens - pointers of tokens
 *		*bdb - BDB file structure.
 */
int p_charset_encoding(int num_of_tokens, char **tokens, BDB *bdb)
{
	return ERR_NO_ERROR;
}

/*
 * handler for 'COPYRIGHT'.
 * in:
 *		num_of_tokens - number of token
 *		**tokens - pointers of tokens
 *		*bdb - BDB file structure.
 */
int p_copyright(int num_of_tokens, char **tokens, BDB *bdb)
{
	return ERR_NO_ERROR;
}

/*
 * handler for 'DEFAULT_CHAR'.
 * in:
 *		num_of_tokens - number of token
 *		**tokens - pointers of tokens
 *		*bdb - BDB file structure.
 */
int p_default_char(int num_of_tokens, char **tokens, BDB *bdb)
{
	return ERR_NO_ERROR;
}

/*
 * handler for 'FONT_ASCENT'.
 * in:
 *		num_of_tokens - number of token
 *		**tokens - pointers of tokens
 *		*bdb - BDB file structure.
 */
int p_font_ascent(int num_of_tokens, char **tokens, BDB *bdb)
{
	return ERR_NO_ERROR;
}

/*
 * handler for 'FONT_DESCENT'.
 * in:
 *		num_of_tokens - number of token
 *		**tokens - pointers of tokens
 *		*bdb - BDB file structure.
 */
int p_font_decent(int num_of_tokens, char **tokens, BDB *bdb)
{
	return ERR_NO_ERROR;
}

/*
 * handler for '_XMBDFED_INFO'.
 * in:
 *		num_of_tokens - number of token
 *		**tokens - pointers of tokens
 *		*bdb - BDB file structure.
 */
int p_xmbdfed_info(int num_of_tokens, char **tokens, BDB *bdb)
{
	return ERR_NO_ERROR;
}

/*
 * handler for 'ENDPROPERTIES'.
 * in:
 *		num_of_tokens - number of token
 *		**tokens - pointers of tokens
 *		*bdb - BDB file structure.
 */
int p_endproperties(int num_of_tokens, char **tokens, BDB *bdb)
{
	return ERR_NO_ERROR;
}

/*
 * handler for 'CHARS'.
 * in:
 *		num_of_tokens - number of token
 *		**tokens - pointers of tokens
 *		*bdb - BDB file structure.
 */
int p_chars(int num_of_tokens, char **tokens, BDB *bdb)
{
	int chars;

	if(num_of_tokens != 2) {
		return ERR_SYNTAX_ERROR;
	}

	chars = atoi(tokens[1]);
	int32_to_bytes(chars, bdb->chars);

	/* allocate font description block. */
	bdb->font = (FONT *)malloc(sizeof(FONT) * chars);

	return ERR_NO_ERROR;
}

/*
 * handler for 'STARTCHAR'.
 * in:
 *		num_of_tokens - number of token
 *		**tokens - pointers of tokens
 *		*bdb - BDB file structure.
 */
int p_startchar(int num_of_tokens, char **tokens, BDB *bdb)
{
	int character_code;

	if(num_of_tokens != 2) {
		return ERR_SYNTAX_ERROR;
	}
	if(bdb->font == NULL) {
		/* 'CHARS' is needed before 'STARTCHAR' */
		return ERR_ILLEGAL_FONT_STATE;
	}

	character_code = hex_to_int32(tokens[1]);
	int32_to_bytes(character_code, bdb->font[bdb->font_count].start_char);

	return ERR_NO_ERROR;
}

/*
 * handler for 'ENCODING'.
 * in:
 *		num_of_tokens - number of token
 *		**tokens - pointers of tokens
 *		*bdb - BDB file structure.
 */
int p_encoding(int num_of_tokens, char **tokens, BDB *bdb)
{
	int encoding;

	if(num_of_tokens != 2) {
		return ERR_SYNTAX_ERROR;
	}
	if(bdb->font == NULL) {
		/* 'CHARS' is needed before 'STARTCHAR' */
		return ERR_ILLEGAL_FONT_STATE;
	}

	encoding = atoi(tokens[1]);
	int32_to_bytes(encoding, bdb->font[bdb->font_count].encoding);

	return ERR_NO_ERROR;
}

/*
 * handler for 'SWIDTH'.
 * in:
 *		num_of_tokens - number of token
 *		**tokens - pointers of tokens
 *		*bdb - BDB file structure.
 */
int p_swidth(int num_of_tokens, char **tokens, BDB *bdb)
{
	int swidth_high, swidth_low;

	if(num_of_tokens != 3) {
		return ERR_SYNTAX_ERROR;
	}

	swidth_high = atoi(tokens[1]);
	swidth_low  = atoi(tokens[2]);
	int32_to_bytes(swidth_high, bdb->font[bdb->font_count].swidth[0]);
	int32_to_bytes(swidth_low , bdb->font[bdb->font_count].swidth[1]);

	return ERR_NO_ERROR;
}

/*
 * handler for 'DWIDTH'.
 * in:
 *		num_of_tokens - number of token
 *		**tokens - pointers of tokens
 *		*bdb - BDB file structure.
 */
int p_dwidth(int num_of_tokens, char **tokens, BDB *bdb)
{
	int dwidth_high, dwidth_low;

	if(num_of_tokens != 3) {
		return ERR_SYNTAX_ERROR;
	}

	dwidth_high = atoi(tokens[1]);
	dwidth_low  = atoi(tokens[2]);
	int32_to_bytes(dwidth_high, bdb->font[bdb->font_count].dwidth[0]);
	int32_to_bytes(dwidth_low , bdb->font[bdb->font_count].dwidth[1]);

	return ERR_NO_ERROR;
}

/*
 * handler for 'BBX'.
 * in:
 *		num_of_tokens - number of token
 *		**tokens - pointers of tokens
 *		*bdb - BDB file structure.
 */
int p_bbx(int num_of_tokens, char **tokens, BDB *bdb)
{
	int bbx_1, bbx_2, bbx_3, bbx_4;

	if(num_of_tokens != 5) {
		return ERR_SYNTAX_ERROR;
	}

	bbx_1 = atoi(tokens[1]);
	bbx_2 = atoi(tokens[2]);
	bbx_3 = atoi(tokens[3]);
	bbx_4 = atoi(tokens[4]);
	int32_to_bytes(bbx_1, bdb->font[bdb->font_count].bbx[0]);
	int32_to_bytes(bbx_2, bdb->font[bdb->font_count].bbx[1]);
	int32_to_bytes(bbx_3, bdb->font[bdb->font_count].bbx[2]);
	int32_to_bytes(bbx_4, bdb->font[bdb->font_count].bbx[3]);

	return ERR_NO_ERROR;
}

/*
 * handler for 'BITMAP'.
 * in:
 *		num_of_tokens - number of token
 *		**tokens - pointers of tokens
 *		*bdb - BDB file structure.
 */
int p_bitmap(int num_of_tokens, char **tokens, BDB *bdb)
{
	FONT* font_ptr;

	if(num_of_tokens != 1) {
		return ERR_SYNTAX_ERROR;
	}

	/* allocate bitmap buffer. */
	font_ptr = &bdb->font[bdb->font_count];
	font_ptr->bitmap = (char *)malloc(bdb->pixel_size * MAX_BITMAP_WIDTH_BYTES);
	font_ptr->bitmap_count = 0;

	/* set scanning bitmap flag on. */
	bdb->scanning_bitmap = 1;

	return ERR_NO_ERROR;
}

/*
 * handler for 'ENDCHAR'.
 * in:
 *		num_of_tokens - number of token
 *		**tokens - pointers of tokens
 *		*bdb - BDB file structure.
 */
int p_endchar(int num_of_tokens, char **tokens, BDB *bdb)
{
	if(num_of_tokens != 1) {
		return ERR_SYNTAX_ERROR;
	}

	/* count up */
	bdb->font_count++;
	return ERR_NO_ERROR;
}

/*
 * handler for 'ENDFONT'.
 * in:
 *		num_of_tokens - number of token
 *		**tokens - pointers of tokens
 *		*bdb - BDB file structure.
 */
int p_endfont(int num_of_tokens, char **tokens, BDB *bdb)
{
	return ERR_NO_ERROR;
}
