/*
   php_namazu.c
   This program is free software; you can redistribute it and/or modify
   it under the terms of one of the following licenses:
    - the GNU General Public License version 2
    - the PHP License version 2.02

   Auther: Takuya Tsukada <tsukada@fminn.nagano.nagano.jp>
   History:
     2000.5.19  Release 2.0
     2003.2.14  add functions nmz_set_maxhit(), nmz_set_maxmatch()
                              nmz_get_maxhit(), nmz_get_maxmatch()
                    directive namazu.maxhit, namazu.maxmatch
          (yiwakiri@st.rim.or.jp)
 */

/* $Id $ */


#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "namazu/libnamazu.h"
#include "namazu/field.h"
#include "namazu/hlist.h"
#include "namazu/idxname.h"
#include "namazu/i18n.h"
#include "namazu/l10n-ja.h"
#include "namazu/search.h"
#include "namazu/query.h"
#include "namazu/codeconv.h"

#include "php.h"
#include "php_ini.h"
#include "php_namazu.h"
#include "php_namazu_globals.h"
#ifdef ZTS
#include "TSRM.h"
#endif

/* True globals, no need for thread safety */
static int le_nmzres; 

#ifdef ZTS
static MUTEX_T php_namazu_lock_id = NULL;
#define php_namazu_lock() tsrm_mutex_lock(php_namazu_lock_id)
#define php_namazu_unlock() tsrm_mutex_unlock(php_namazu_lock_id)
#else
#define php_namazu_lock()
#define php_namazu_unlock()
#endif

function_entry namazu_functions[] = {
	PHP_FE(nmz_set_maxhit,		NULL)
	PHP_FE(nmz_get_maxhit,		NULL)
	PHP_FE(nmz_set_maxmatch,		NULL)
	PHP_FE(nmz_get_maxmatch,		NULL)
	PHP_FE(nmz_set_sortorder,		NULL)
	PHP_FE(nmz_set_sortmethod,		NULL)
	PHP_FE(nmz_set_lang,			NULL)
	PHP_FE(nmz_get_lang,			NULL)
	PHP_FE(nmz_get_lang_ctype,		NULL)
	PHP_FE(nmz_is_lang_ja,			NULL)
	PHP_FE(nmz_set_loggingmode,		NULL)
	PHP_FE(nmz_set_debugmode,		NULL)
	PHP_FE(nmz_open,		NULL)
	PHP_FE(nmz_close,		NULL)
	PHP_FE(nmz_search,		NULL)
	PHP_FE(nmz_num_hits,	NULL)
	PHP_FE(nmz_result_score,	NULL)
	PHP_FE(nmz_result_date,		NULL)
	PHP_FE(nmz_result_field,	NULL)
	PHP_FE(nmz_fetch_score,		NULL)
	PHP_FE(nmz_fetch_date,		NULL)
	PHP_FE(nmz_fetch_field,		NULL)
	PHP_FE(nmz_free_result,		NULL)
	PHP_FE(nmz_get_querytoken,	NULL)
	PHP_FE(nmz_get_idxnum,	NULL)
	PHP_FE(nmz_get_idxname,	NULL)
	PHP_FE(nmz_get_idx_totalhitnum,	NULL)
	PHP_FE(nmz_get_idx_hitnumlist,	NULL)
	PHP_FE(nmz_codeconv_query,	NULL)
	PHP_FE(nmz_info,		NULL)
	{NULL, NULL, NULL}
};


zend_module_entry namazu_module_entry = {
	STANDARD_MODULE_HEADER,
	"namazu",
	namazu_functions,
	PHP_MODULE_STARTUP_N(namazu),
	PHP_MODULE_SHUTDOWN_N(namazu),
	PHP_MODULE_ACTIVATE_N(namazu),
	PHP_MODULE_DEACTIVATE_N(namazu),
	PHP_MODULE_INFO_N(namazu),
	PHP_NAMAZU_VERSION,
	STANDARD_MODULE_PROPERTIES
};


ZEND_DECLARE_MODULE_GLOBALS(namazu)


#ifdef COMPILE_DL_NAMAZU
ZEND_GET_MODULE(namazu)
#endif


/* hlist destructor */
static void
php_namazu_free_hlist(zend_rsrc_list_entry *rsrc TSRMLS_DC)
{
	php_nmzresult_t *result = (php_nmzresult_t *)rsrc->ptr;

	nmz_free_hlist(result->hlist);
	efree(result);
}


/* php.ini */
static PHP_INI_MH(OnUpdate_namazu_sortorder)
{
	if (new_value != NULL) {
		if (strcasecmp("ascending", new_value) == 0) {
			NMZG(sortorder) = ASCENDING;
		} else if (strcasecmp("descending", new_value) == 0) {
			NMZG(sortorder) = DESCENDING;
		} else {
			return FAILURE;
		}
	}

	return SUCCESS;
}

static PHP_INI_MH(OnUpdate_namazu_sortmethod)
{
	NMZG(sortfield) = (char *)0;
	if (new_value != NULL) {
		if (strcasecmp("score", new_value) == 0) {
			NMZG(sortmethod) = SORT_BY_SCORE;
		} else if (strcasecmp("date", new_value) == 0) {
			NMZG(sortmethod) = SORT_BY_DATE;
		} else if (strncasecmp("field:", new_value, 6) == 0 && new_value_length < BUFSIZE) {
			NMZG(sortmethod) = SORT_BY_FIELD;
			NMZG(sortfield) = &new_value[6];
		} else {
			return FAILURE;
		}
	}

	return SUCCESS;
}


PHP_INI_BEGIN()
	PHP_INI_ENTRY("namazu.sortorder", NULL, PHP_INI_ALL, OnUpdate_namazu_sortorder)
	PHP_INI_ENTRY("namazu.sortmethod", NULL, PHP_INI_ALL, OnUpdate_namazu_sortmethod)
	STD_PHP_INI_ENTRY("namazu.lang", NULL, PHP_INI_ALL, OnUpdateString, lang, zend_namazu_globals, namazu_globals)
	STD_PHP_INI_BOOLEAN("namazu.loggingmode", "0", PHP_INI_ALL, OnUpdateInt, loggingmode, zend_namazu_globals, namazu_globals)
	STD_PHP_INI_BOOLEAN("namazu.debugmode", "0", PHP_INI_ALL, OnUpdateInt, debugmode, zend_namazu_globals, namazu_globals)
PHP_INI_END()


/* module globals initialize */
static void
php_namazu_init_globals(zend_namazu_globals *pglobals)
{
	pglobals->status = 0;
	pglobals->idxcount = 0;
	pglobals->sortorder = DESCENDING;
	pglobals->sortmethod = SORT_BY_SCORE;
	pglobals->sortfield = (char *)0;
	pglobals->lang = (char *)0;
	pglobals->loggingmode = 0;
	pglobals->debugmode = 0;
}


PHP_MODULE_STARTUP_D(namazu)
{
#ifdef ZTS
	php_namazu_lock_id = tsrm_mutex_alloc();
#endif
	ZEND_INIT_MODULE_GLOBALS(namazu, php_namazu_init_globals, NULL);
	REGISTER_INI_ENTRIES();
	le_nmzres = zend_register_list_destructors_ex(php_namazu_free_hlist, NULL, "Namazu result", module_number);

	return SUCCESS;
}


PHP_MODULE_SHUTDOWN_D(namazu)
{
	UNREGISTER_INI_ENTRIES();
#ifdef ZTS
	tsrm_mutex_free(php_namazu_lock_id);
	php_namazu_lock_id = NULL;
#endif

	return SUCCESS;
}


PHP_MODULE_ACTIVATE_D(namazu)
{
	NMZG(idxcount) = 0;
#ifndef ZTS
	nmz_set_sortmethod(NMZG(sortmethod));
	nmz_set_sortorder(NMZG(sortorder));
	nmz_set_loggingmode((int)NMZG(loggingmode));
	nmz_set_debugmode((int)NMZG(debugmode));
	if (NMZG(sortfield)) {
		nmz_set_sortfield(NMZG(sortfield));
	}
	if (NMZG(lang)) {
		if (nmz_set_lang(NMZG(lang)) == NULL) {
			return FAILURE;
		}
	}
#endif

	return SUCCESS;
}


PHP_MODULE_DEACTIVATE_D(namazu)
{
	if (NMZG(status) & PHP_NAMAZU_STATUS_OPEN) {
		nmz_free_internal();
		php_namazu_unlock();
		NMZG(status) &= ~PHP_NAMAZU_STATUS_OPEN;
	}

	return SUCCESS;
}


PHP_MODULE_INFO_D(namazu)
{
	DISPLAY_INI_ENTRIES();
}


/* {{{ proto bool nmz_set_sortorder(string order)
   Set the sort order */
PHP_FUNCTION(nmz_set_sortorder)
{
	zval **arg_order;

	if (ZEND_NUM_ARGS() != 1 || zend_get_parameters_ex(1, &arg_order) == FAILURE) {
		WRONG_PARAM_COUNT;
	}

	convert_to_string_ex(arg_order);
	if (strcasecmp("ascending", Z_STRVAL_PP(arg_order)) == 0) {
		nmz_set_sortorder(ASCENDING);
	} else if (strcasecmp("descending", Z_STRVAL_PP(arg_order)) == 0) {
		nmz_set_sortorder(DESCENDING);
	} else {
		php_error(E_WARNING, "Illegal argument");
		RETURN_FALSE;
	}
	RETURN_TRUE;
}
/* }}} */


/* {{{ proto bool nmz_set_sortmethod(string method)
   Set the sort method (score, date, field:name) */
PHP_FUNCTION(nmz_set_sortmethod)
{
	zval **arg_method;

	if (ZEND_NUM_ARGS() != 1 || zend_get_parameters_ex(1, &arg_method) == FAILURE) {
		WRONG_PARAM_COUNT;
	}

	convert_to_string_ex(arg_method);
	if (strcasecmp("score", Z_STRVAL_PP(arg_method)) == 0) {
		nmz_set_sortmethod(SORT_BY_SCORE);
	} else if (strcasecmp("date", Z_STRVAL_PP(arg_method)) == 0) {
		nmz_set_sortmethod(SORT_BY_DATE);
	} else if (strncasecmp("field:", Z_STRVAL_PP(arg_method), 6) == 0 &&
	           Z_STRLEN_PP(arg_method) < BUFSIZE) {
		nmz_set_sortmethod(SORT_BY_FIELD);
		nmz_set_sortfield(&(Z_STRVAL_PP(arg_method)[6]));
	} else {
		php_error(E_WARNING, "Illegal argument");
		RETURN_FALSE;
	}

	RETURN_TRUE;
}
/* }}} */


/* {{{ proto bool nmz_set_lang(string lang)
   Set the language */
PHP_FUNCTION(nmz_set_lang)
{
	zval **arg_lang;

	if (ZEND_NUM_ARGS() != 1 || zend_get_parameters_ex(1, &arg_lang) == FAILURE) {
		WRONG_PARAM_COUNT;
	}

	convert_to_string_ex(arg_lang);
	if (nmz_set_lang(Z_STRVAL_PP(arg_lang)) == NULL) {
		RETVAL_FALSE;
	} else {
		RETVAL_TRUE;
	}
}
/* }}} */


/* {{{ proto string nmz_get_lang(void)
   Get the language */
PHP_FUNCTION(nmz_get_lang)
{
	char *lang;

	lang = nmz_get_lang();
	if (lang) {
		RETVAL_STRING(lang, 1);
	} else {
		RETVAL_FALSE;
	}
}
/* }}} */


/* {{{ proto string nmz_get_lang_ctype(void)
   Get the language */
PHP_FUNCTION(nmz_get_lang_ctype)
{
	char *lang;

	lang = nmz_get_lang_ctype();
	if (lang) {
		RETVAL_STRING(lang, 1);
	} else {
		RETVAL_FALSE;
	}
}
/* }}} */


/* {{{ proto bool nmz_is_lang_ja(void)
    */
PHP_FUNCTION(nmz_is_lang_ja)
{
	if (nmz_is_lang_ja()) {
		RETVAL_TRUE;
	} else {
		RETVAL_FALSE;
	}
}
/* }}} */


/* {{{ proto bool nmz_set_maxhit(int mode)
   Set the MaxHits */
PHP_FUNCTION(nmz_set_maxhit)
{
	zval **arg_mode;

	if (ZEND_NUM_ARGS() != 1 || zend_get_parameters_ex(1, &arg_mode) == FAILURE) {
		WRONG_PARAM_COUNT;
	}

	convert_to_long_ex(arg_mode);
	nmz_set_maxhit((int)Z_LVAL_PP(arg_mode));

	RETURN_TRUE;
}
/* }}} */

/* {{{ proto int nmz_get_maxhit(void)
   Get the MaxHits */
PHP_FUNCTION(nmz_get_maxhit)
{
	int	maxhit;

	maxhit = nmz_get_maxhit();
	RETURN_LONG(maxhit);
}
/* }}} */

/* {{{ proto bool nmz_set_maxmatch(int maxmatch)
   Set the MaxMatches */
PHP_FUNCTION(nmz_set_maxmatch)
{
	zval **arg_mode;

	if (ZEND_NUM_ARGS() != 1 || zend_get_parameters_ex(1, &arg_mode) == FAILURE) {
		WRONG_PARAM_COUNT;
	}

	convert_to_long_ex(arg_mode);
	nmz_set_maxmatch((int)Z_LVAL_PP(arg_mode));

	RETURN_TRUE;
}
/* }}} */

/* {{{ proto int nmz_get_maxmatch(void)
   Get the MaxMatches */
PHP_FUNCTION(nmz_get_maxmatch)
{
	int	maxmatch;

	maxmatch = nmz_get_maxmatch();
	RETURN_LONG(maxmatch);
}
/* }}} */

/* {{{ proto bool nmz_set_loggingmode(bool mode)
   Set the logging mode (NMZ.slog) */
PHP_FUNCTION(nmz_set_loggingmode)
{
	zval **arg_mode;

	if (ZEND_NUM_ARGS() != 1 || zend_get_parameters_ex(1, &arg_mode) == FAILURE) {
		WRONG_PARAM_COUNT;
	}

	convert_to_long_ex(arg_mode);
	nmz_set_loggingmode((int)Z_LVAL_PP(arg_mode));

	RETURN_TRUE;
}
/* }}} */

/* {{{ proto bool nmz_set_debugmode(bool mode)
   Set the debug mode */
PHP_FUNCTION(nmz_set_debugmode)
{
	zval **arg_mode;

	if (ZEND_NUM_ARGS() != 1 || zend_get_parameters_ex(1, &arg_mode) == FAILURE) {
		WRONG_PARAM_COUNT;
	}

	convert_to_long_ex(arg_mode);
	nmz_set_debugmode((int)Z_LVAL_PP(arg_mode));

	RETURN_TRUE;
}
/* }}} */


/* {{{ proto int nmz_open(mixed index)
   Open namazu index */
PHP_FUNCTION(nmz_open)
{
	zval **arg_index, **entry;
	HashTable *target_hash;
	int i, err;

	if (NMZG(status) & PHP_NAMAZU_STATUS_OPEN) {
		php_error(E_WARNING, "Unable to open more than one index group");
		RETURN_FALSE;
	}

	if (ZEND_NUM_ARGS() != 1 || zend_get_parameters_ex(1, &arg_index) == FAILURE) {
		WRONG_PARAM_COUNT;
	}

	php_namazu_lock();
	err = 0;
#ifdef ZTS
	nmz_set_sortmethod(NMZG(sortmethod));
	nmz_set_sortorder(NMZG(sortorder));
	nmz_set_loggingmode((int)NMZG(loggingmode));
	nmz_set_debugmode((int)NMZG(debugmode));
	if (NMZG(sortfield)) {
		nmz_set_sortfield(NMZG(sortfield));
	}
	if (NMZG(lang)) {
		if (nmz_set_lang(NMZG(lang)) == NULL) {
			err = 1;
		}
	}
#endif
	switch(Z_TYPE_PP(arg_index)) {
	case IS_ARRAY:
		target_hash = Z_ARRVAL_PP(arg_index);
		zend_hash_internal_pointer_reset(target_hash);
		i = zend_hash_num_elements(target_hash);
		while (i > 0) {
			if (zend_hash_get_current_data(target_hash, (void **) &entry) == FAILURE) {
				break;
			}
			convert_to_string_ex(entry);
			if (nmz_add_index(Z_STRVAL_PP(entry)) != SUCCESS) {
				php_error(E_WARNING, "invalid index name %s", Z_STRVAL_PP(arg_index));
				err = 1;
				break;
			}
			zend_hash_move_forward(target_hash);
			i--;
		}
		break;
	default:
		convert_to_string_ex(arg_index);
		if (nmz_add_index(Z_STRVAL_PP(arg_index)) != SUCCESS) {
			php_error(E_WARNING, "invalid index name %s", Z_STRVAL_PP(arg_index));
			err = 1;
		}
		break;
	}

	if (err) {
		nmz_free_internal();
		php_namazu_unlock();
		RETVAL_FALSE;
	} else {
		NMZG(status) |= PHP_NAMAZU_STATUS_OPEN;
		NMZG(idxcount)++;
		RETVAL_LONG(1);
	}
}


/* {{{ proto bool nmz_close(int identifier)
   Close namazu index */
PHP_FUNCTION(nmz_close)
{
	zval **arg_id;

	if (ZEND_NUM_ARGS() != 1 || zend_get_parameters_ex(1, &arg_id) == FAILURE) {
		WRONG_PARAM_COUNT;
	}

	if (NMZG(status) & PHP_NAMAZU_STATUS_OPEN) {
		nmz_free_internal();
		php_namazu_unlock();
		NMZG(status) &= ~PHP_NAMAZU_STATUS_OPEN;
	}

	RETURN_TRUE;
}


/* {{{ proto int nmz_free_result(int result)
   Free namazu resource */
PHP_FUNCTION(nmz_free_result)
{
	zval **arg_result;
	php_nmzresult_t *nmzres;

	if (ZEND_NUM_ARGS() != 1 || zend_get_parameters_ex(1, &arg_result) == FAILURE) {
		WRONG_PARAM_COUNT;
	}

	if (Z_TYPE_PP(arg_result) == IS_RESOURCE && Z_LVAL_PP(arg_result) == 0) {
		RETURN_FALSE;
	}

	ZEND_FETCH_RESOURCE(nmzres, php_nmzresult_t *, arg_result, -1, "Namazu result", le_nmzres);

	zend_list_delete(Z_RESVAL_PP(arg_result));

	RETURN_TRUE;
}
/* }}} */


/* {{{ proto int nmz_search(int identifier, string query)
   Search on the namazu index */
PHP_FUNCTION(nmz_search)
{
	zval **arg_idxid, **arg_query;
	php_nmzresult_t *nmzres;

	if (!(NMZG(status) & PHP_NAMAZU_STATUS_OPEN)) {
		php_error(E_WARNING, "call nmz_search() befor open the index");
		RETURN_FALSE;
	}

	if (ZEND_NUM_ARGS() != 2 || zend_get_parameters_ex(2, &arg_idxid, &arg_query) == FAILURE) {
		WRONG_PARAM_COUNT;
	}
	convert_to_string_ex(arg_query);

	nmzres = (php_nmzresult_t *)ecalloc(1, sizeof(php_nmzresult_t));
	nmzres->hlist = nmz_search(Z_STRVAL_PP(arg_query));
	if (nmzres->hlist.stat == ERR_FATAL) {
		nmz_free_hlist(nmzres->hlist);
		efree(nmzres);
		php_error(E_WARNING, "libnmz: %s", nmz_get_dyingmsg());
		RETURN_FALSE;
	}
	nmzres->idxcount = NMZG(idxcount);

	ZEND_REGISTER_RESOURCE(return_value, nmzres, le_nmzres);
}
/* }}} */


/* {{{ proto int nmz_num_hits(int result)
   Get number of hits in result */
PHP_FUNCTION(nmz_num_hits)
{
	zval **arg_result;
	php_nmzresult_t *nmzres;

	if (ZEND_NUM_ARGS() != 1 || zend_get_parameters_ex(1, &arg_result) == FAILURE) {
		WRONG_PARAM_COUNT;
	}

	ZEND_FETCH_RESOURCE(nmzres, php_nmzresult_t *, arg_result, -1, "Namazu result", le_nmzres);

	if (nmzres->idxcount != NMZG(idxcount)) {
		php_error(E_WARNING, "Supplied argument is not a result of current index");
		RETURN_FALSE;
	}

	RETURN_LONG(nmzres->hlist.num);
}
/* }}} */


static void
_php_nmz_result(INTERNAL_FUNCTION_PARAMETERS, int type)
{
	zval **arg_result, **arg_row, **arg_field;
	php_nmzresult_t *nmzres;
	char *buf;
	int i;

	if (type == 2) {
		if (ZEND_NUM_ARGS() != 3 ||
		    zend_get_parameters_ex(3, &arg_result, &arg_row, &arg_field) == FAILURE) {
				WRONG_PARAM_COUNT;
		}
		convert_to_string_ex(arg_field);
	} else {
		if (ZEND_NUM_ARGS() != 2 ||
		    zend_get_parameters_ex(2, &arg_result, &arg_row) == FAILURE) {
				WRONG_PARAM_COUNT;
		}
	}

	ZEND_FETCH_RESOURCE(nmzres, php_nmzresult_t *, arg_result, -1, "Namazu result", le_nmzres);

	if (nmzres->idxcount != NMZG(idxcount)) {
		php_error(E_WARNING, "Supplied argument is not a result of current index");
		RETURN_FALSE;
	}

	convert_to_long_ex(arg_row);
	i = Z_LVAL_PP(arg_row);
	if (i < 0 || i >= nmzres->hlist.num) {
		php_error(E_WARNING, "Invalid row number. (%d)", i);
		RETURN_FALSE;
	}

	switch (type) {
	case 1:
		RETVAL_LONG(nmzres->hlist.data[i].date);
		break;
	case 2:
		buf = (char *)emalloc(2*BUFSIZE*sizeof(char));
		buf[0] = '\0';
		buf[1] = '\0';
		nmz_get_field_data(nmzres->hlist.data[i].idxid, nmzres->hlist.data[i].docid, (*arg_field)->value.str.val, buf);
		RETVAL_STRING(buf, 1);
		efree(buf);
		break;
	default:
		RETVAL_LONG(nmzres->hlist.data[i].score);
		break;
	}
}


/* {{{ proto int nmz_result_score(int result, int row)
   Get the score in a result */
PHP_FUNCTION(nmz_result_score)
{
	_php_nmz_result(INTERNAL_FUNCTION_PARAM_PASSTHRU, 0);
}
/* }}} */


/* {{{ proto int nmz_result_date(int result, int row)
   Get the date in a result */
PHP_FUNCTION(nmz_result_date)
{
	_php_nmz_result(INTERNAL_FUNCTION_PARAM_PASSTHRU, 1);
}
/* }}} */


/* {{{ proto string nmz_result_field(int result, int row, string field)
   Get the field of the specified name in a result */
PHP_FUNCTION(nmz_result_field)
{
	_php_nmz_result(INTERNAL_FUNCTION_PARAM_PASSTHRU, 2);
}
/* }}} */


static void
_php_nmz_fetch_intval(INTERNAL_FUNCTION_PARAMETERS, int type)
{
	zval **arg_result, **arg_limit, **arg_offset;
	php_nmzresult_t *nmzres;
	int i, limit, offset;

	switch(ZEND_NUM_ARGS()) {
	case 1:
		if (zend_get_parameters_ex(1, &arg_result) == FAILURE) {
			RETURN_FALSE;
		}
		break;
	case 2:
		if (zend_get_parameters_ex(2, &arg_result, &arg_limit) == FAILURE) {
			RETURN_FALSE;
		}
		break;
	case 3:
		if (zend_get_parameters_ex(3, &arg_result, &arg_limit, &arg_offset) == FAILURE) {
			RETURN_FALSE;
		}
		break;
	default:
		WRONG_PARAM_COUNT;
		break;
	}

	ZEND_FETCH_RESOURCE(nmzres, php_nmzresult_t *, arg_result, -1, "Namazu result", le_nmzres);

	if (nmzres->idxcount != NMZG(idxcount)) {
		php_error(E_WARNING, "Supplied argument is not a result of current index");
		RETURN_FALSE;
	}

	limit = nmzres->hlist.num;
	offset = 0;
	if (ZEND_NUM_ARGS() >= 2) {
		convert_to_long_ex(arg_limit);
		if (limit > Z_LVAL_PP(arg_limit)) {
			limit = Z_LVAL_PP(arg_limit);
		}
	}
	if (ZEND_NUM_ARGS() == 3) {
		convert_to_long_ex(arg_offset);
		if (limit > Z_LVAL_PP(arg_offset)) {
			offset = Z_LVAL_PP(arg_offset);
		}
	}

	/* Initialize return array */
	if (array_init(return_value) == FAILURE) {
		RETURN_FALSE;
	}

	switch (type) {
	case 0:
		for (i = offset; i < limit; i++) {
			add_next_index_long(return_value, nmzres->hlist.data[i].score);
		}
		break;
	default:
		for (i = offset; i < limit; i++) {
			add_next_index_long(return_value, nmzres->hlist.data[i].date);
		}
		break;
	}
}


/* {{{ proto array nmz_fetch_score(int hlist, [int limit[, int offset]])
   Get the score of all row in a result */
PHP_FUNCTION(nmz_fetch_score)
{
	_php_nmz_fetch_intval(INTERNAL_FUNCTION_PARAM_PASSTHRU, 0);
}
/* }}} */


/* {{{ proto array nmz_fetch_date(int hlist, [int limit[, int offset]])
   Get the date of all row in a result */
PHP_FUNCTION(nmz_fetch_date)
{
	_php_nmz_fetch_intval(INTERNAL_FUNCTION_PARAM_PASSTHRU, 1);
}
/* }}} */


/* {{{ proto array nmz_fetch_field(int hlist, string field [, int limit[, int offset]])
   Get the field of all row in a result */
PHP_FUNCTION(nmz_fetch_field)
{
	zval **arg_result, **arg_field, **arg_limit, **arg_offset;
	php_nmzresult_t *nmzres;
	char *buf;
	int i, limit, offset;

	switch(ZEND_NUM_ARGS()) {
	case 2:
		if (zend_get_parameters_ex(2, &arg_result, &arg_field) == FAILURE) {
			RETURN_FALSE;
		}
		break;
	case 3:
		if (zend_get_parameters_ex(3, &arg_result, &arg_field, &arg_limit) == FAILURE) {
			RETURN_FALSE;
		}
		break;
	case 4:
		if (zend_get_parameters_ex(4, &arg_result, &arg_field, &arg_limit, &arg_offset) == FAILURE) {
			RETURN_FALSE;
		}
		break;
	default:
		WRONG_PARAM_COUNT;
		break;
	}

	convert_to_string_ex(arg_field);

	ZEND_FETCH_RESOURCE(nmzres, php_nmzresult_t *, arg_result, -1, "Namazu result", le_nmzres);

	if (nmzres->idxcount != NMZG(idxcount)) {
		php_error(E_WARNING, "Supplied argument is not a result of current index");
		RETURN_FALSE;
	}

	limit = nmzres->hlist.num;
	offset = 0;
	if (ZEND_NUM_ARGS() == 4) {
		convert_to_long_ex(arg_offset);
		if (limit > Z_LVAL_PP(arg_offset)) {
			offset = Z_LVAL_PP(arg_offset);
		}
	}
	if (ZEND_NUM_ARGS() >= 3) {
		convert_to_long_ex(arg_limit);
		if (limit > offset + Z_LVAL_PP(arg_limit)) {
			limit = offset + Z_LVAL_PP(arg_limit);
		}
	}

	/* Initialize return array */
	if (array_init(return_value) == FAILURE) {
		RETURN_FALSE;
	}

	buf = (char *)emalloc(2*BUFSIZE*sizeof(char));
	for (i = offset; i < limit; i++) {
		buf[0] = '\0';
		buf[1] = '\0';
		nmz_get_field_data(nmzres->hlist.data[i].idxid, nmzres->hlist.data[i].docid, (*arg_field)->value.str.val, buf);
		add_next_index_string(return_value, buf, 1);
	}
	efree(buf);
}
/* }}} */


/* {{{ proto array nmz_get_querytoken(int hlist)
   Get query tokens as array */
PHP_FUNCTION(nmz_get_querytoken)
{
	zval **arg_hlist;
	char *query;
	int i, n;

	if (ZEND_NUM_ARGS() != 1 ||zend_get_parameters_ex(1, &arg_hlist) == FAILURE) { 
		WRONG_PARAM_COUNT;
	}

	n = nmz_get_querytokennum();
	if (n <= 0) {
		RETURN_FALSE;
	}

	/* Initialize return array */
	if (array_init(return_value) == FAILURE) {
		RETURN_FALSE;
	}

	for (i = 0; i < n; i++) {
		query = nmz_get_querytoken(i);
		if (query) {
			add_next_index_string(return_value, query, 1);
		}
	}
}
/* }}} */

/* {{{ proto int nmz_get_idxnum(void)
   Get number of idecies */
PHP_FUNCTION(nmz_get_idxnum)
{
	int max_idxid;

	if (ZEND_NUM_ARGS() != 0) {
		WRONG_PARAM_COUNT;
	}

	max_idxid = nmz_get_idxnum();
	RETVAL_LONG(max_idxid);
}

/* {{{ proto string nmz_get_idxname(int index_id)
   Get index name */
PHP_FUNCTION(nmz_get_idxname)
{
	zval **arg_idxid;
	char *indexname;

	if (ZEND_NUM_ARGS() != 1 || zend_get_parameters_ex(1, &arg_idxid) == FAILURE) {
		WRONG_PARAM_COUNT;
	}

	convert_to_long_ex(arg_idxid);
	indexname = nmz_get_idxname((int)Z_LVAL_PP(arg_idxid));
	RETVAL_STRING(indexname, 1);
}

/* {{{ proto int nmz_get_totalhitnum(int index_id)
   Get number of Total hits */
PHP_FUNCTION(nmz_get_idx_totalhitnum)
{
	zval **arg_idxid;
	int totalhitnum;

	if (ZEND_NUM_ARGS() != 1 || zend_get_parameters_ex(1, &arg_idxid) == FAILURE) {
		WRONG_PARAM_COUNT;
	}

	convert_to_long_ex(arg_idxid);
	totalhitnum = nmz_get_idx_totalhitnum((int)Z_LVAL_PP(arg_idxid));
	RETVAL_LONG(totalhitnum);
}

/* {{{ proto array nmz_get_hitnumlist(int index_id)
   Get token/number of hits */
PHP_FUNCTION(nmz_get_idx_hitnumlist)
{
	zval **arg_idxid;
	struct nmz_hitnumlist *hnlist;

	if (ZEND_NUM_ARGS() != 1 || zend_get_parameters_ex(1, &arg_idxid) == FAILURE) {
		WRONG_PARAM_COUNT;
	}

	/* Initialize index_info array */
	if (array_init(return_value) == FAILURE) {
		RETURN_FALSE;
	}

	hnlist = nmz_get_idx_hitnumlist((int)Z_LVAL_PP(arg_idxid));
	if (hnlist->phrase != NULL) {
		add_assoc_long(return_value, "phrase", hnlist->hitnum);
		hnlist = hnlist->phrase;
	} else {
		add_assoc_long(return_value, "no-phrase", 0);
	}

	while (hnlist != NULL) {
		add_assoc_long(return_value, hnlist->word, hnlist->hitnum);
		hnlist = hnlist->next;
	}
}
/* }}} */


/* {{{ proto string nmz_codeconv_query(string query)
   Convert code */
PHP_FUNCTION(nmz_codeconv_query)
{
	zval **arg_query;
	char *query;

	if (ZEND_NUM_ARGS() != 1 || zend_get_parameters_ex(1, &arg_query) == FAILURE) {
		WRONG_PARAM_COUNT;
	}
	convert_to_string_ex(arg_query);
	query = estrndup(Z_STRVAL_PP(arg_query), Z_STRLEN_PP(arg_query));
	if (!query) {
		RETURN_FALSE;
	}
	nmz_codeconv_query(query);
	RETVAL_STRING(query, 0);
}
/* }}} */


/* {{{ proto void nmz_info(void)
   Print namazu infomation */
PHP_FUNCTION(nmz_info)
{
	int n;
	char *p;

	p = nmz_get_lang();
	PUTS("language: ");
	if (p) {
		PUTS(p);
	}
	PUTS("<br>\n");

	n = nmz_get_sortmethod();
	PUTS("sort method: ");
	if (n == SORT_BY_SCORE) {
		PUTS("score");
	} else if (n == SORT_BY_DATE) {
		PUTS("date");
	} else if (n == SORT_BY_FIELD) {
		PUTS("field:");
		PUTS(nmz_get_sortfield());
	}
	PUTS("<br>\n");

	n = nmz_get_sortorder();
	PUTS("sort order: ");
	if (n == ASCENDING) {
		PUTS("ascending");
	} else if (n == DESCENDING) {
		PUTS("descending");
	}
	PUTS("<br>\n");

	RETURN_TRUE;
}
/* }}} */

