<?php
/* vim: set tabstop=4 shiftwidth=4: */

/*
 * ToString Style
 * (based on org.apache.commons.lang.builder.ToStringStyle)
 *
 *
 * PHP version 5
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * 	http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * @category   PHP
 * @package    Commons
 * @author     Yomei Komiya
 * @copyright  2007 the original author or authors.
 * @license    http://www.apache.org/licenses/LICENSE-2.0  Apache License 2.0
 * @version    SVN: $Id: ToStringStyle.php 83 2008-01-02 16:42:00Z whitestar $
 * @link       http://phpcommons.sourceforge.jp/
 * @see        
 * @since      File available since Release 0.8.0
 */

//namespace Commons::Lang::Builder;

require_once 'Commons/Lang/Object.php';
require_once 'Commons/Lang/ClassUtils.php';
require_once 'Commons/Lang/StringBuffer.php';
require_once 'Commons/Lang/StringUtils.php';
require_once 'Commons/Lang/System.php';
require_once 'Commons/Lang/SystemUtils.php';
require_once 'Commons/IO/Serializable.php';
/*
use Commons::Lang::Object;
use Commons::Lang::ClassUtils;
use Commons::Lang::StringBuffer;
use Commons::Lang::StringUtils;
use Commons::Lang::System;
use Commons::Lang::SystemUtils;
use Commons::IO::Serializable;
*/

/**
 * ToStringStyle
 *
 *
 * @category   PHP
 * @package    Commons.Lang.Builder
 * @author     Yomei Komiya
 * @copyright  2007 the original author or authors.
 * @license    http://www.apache.org/licenses/LICENSE-2.0  Apache License 2.0
 * @version    Release: 0.9.0
 * @link       http://phpcommons.sourceforge.jp/
 * @see        
 * @since      Class available since Release 0.8.0
 */
abstract class Commons_Lang_Builder_ToStringStyle
	extends Commons_Lang_Object
	implements Commons_IO_Serializable {

	// Styles
	public static $DEFAULT_STYLE = null;
	public static $MULTI_LINE_STYLE = null;
	public static $NO_FIELD_NAMES_STYLE = null;
	public static $SHORT_PREFIX_STYLE = null;
	public static $SIMPLE_STYLE = null;
	
	
	/**
	 * Initializes the static fields
	 */
	public static function init() {
		self::$DEFAULT_STYLE
			= new Commons_Lang_Builder_DefaultToStringStyle();
		self::$MULTI_LINE_STYLE
			= new Commons_Lang_Builder_MultiLineToStringStyle();
		self::$NO_FIELD_NAMES_STYLE
			= new Commons_Lang_Builder_NoFieldNameToStringStyle();
		self::$SHORT_PREFIX_STYLE
			= new Commons_Lang_Builder_ShortPrefixToStringStyle();
		self::$SIMPLE_STYLE
			= new Commons_Lang_Builder_SimpleToStringStyle();
	}
	
	
	private $_useFieldNames = true;
	private $_useClassName = true;
	private $_useShortClassName = false;
	private $_useIdentityHashCode = true;
	private $_contentStart = '[';
	private $_contentEnd = ']';
	private $_fieldNameValueSeparator = '=';
	private $_fieldSeparatorAtStart = false;
	private $_fieldSeparatorAtEnd = false;
	private $_fieldSeparator = ',';
	private $_arrayStart = '{';
	private $_arraySeparator = ',';
	private $_arrayContentDetail = true;
	private $_arrayEnd = '}';
	private $_defaultFullDetail = true;
	private $_nullText = '<null>';
	private $_sizeStartText = '<size=';
	private $_sizeEndText = '>';
	private $_summaryObjectStartText = '<';
	private $_summaryObjectEndText = '>';
	
	
	/**
	 * Constructor
	 */
	protected function Commons_Lang_Builder_ToStringStyle() {
		parent::__construct();
	}
	
	
	public function __destruct() {
		// do nothing.
	}
	
	
	/**
	 * Appends super's toString
	 *
	 * @param Commons_Lang_StringBuffer $buffer
	 * @param string $superToString
	 */
	public function appendSuper(Commons_Lang_StringBuffer $buffer, $superToString) {
		$this->appendToString($buffer, $superToString);
	}
	
	
	/**
	 * Appends toString
	 *
	 * @param Commons_Lang_StringBuffer $buffer
	 * @param string $toString
	 */
	public function appendToString(Commons_Lang_StringBuffer $buffer, $toString) {
		if (!is_null($toString)) {
			$pos1 = Commons_Lang_StringUtils::indexOf($toString, $this->_contentStart)
				+ Commons_Lang_StringUtils::strlen($this->_contentStart);
			$pos2 = Commons_Lang_StringUtils::lastIndexOf($toString, $this->_contentEnd);
			if ($pos1 !== $pos2 && $pos1 >= 0 && $pos2 >= 0) {
				$data = Commons_Lang_StringUtils::substring($toString, $pos1, $pos2);
				if ($this->_fieldSeparatorAtStart) {
					$this->removeLastFieldSeparator($buffer);
				}
				$buffer->append($data);
				$this->appendFieldSeparator($buffer);
			}
		}
	}
	
	
	/**
	 * Appends the start string
	 *
	 * @param Commons_Lang_StringBuffer $buffer
	 * @param object $object
	 */
	public function appendStart(Commons_Lang_StringBuffer $buffer, $object) {

		if (!is_null($object)) {
			$this->appendClassName($buffer, $object);
			$this->appendIdentityHashCode($buffer, $object);
			$this->appendContentStart($buffer);
			if ($this->_fieldSeparatorAtStart) {
				$this->appendFieldSeparator($buffer);
			}
		}
	}
	
	
	/**
	 * Appends the end string
	 *
	 * @param Commons_Lang_StringBuffer $buffer
	 * @param object $object
	 */
	public function appendEnd(
		Commons_Lang_StringBuffer $buffer, $object) {

		if ($this->_fieldSeparatorAtEnd === false) {
			$this->removeLastFieldSeparator($buffer);
		}
		$this->appendContentEnd($buffer);
	}
	
	
	/**
	 * Removes the last field separator.
	 *
	 * @param Commons_Lang_StringBuffer $buffer String Buffer
	 */
	protected function removeLastFieldSeparator(Commons_Lang_StringBuffer $buffer) {
		$len = $buffer->length();
		$sepLen = Commons_Lang_StringUtils::strlen($this->_fieldSeparator);
		if ($len > 0 && $sepLen > 0 && $len >= $sepLen) {
			$match = true;
			for ($i = 0; $i < $sepLen; $i++) {
				if ($buffer->charAt($len - 1 - $i) !== $this->_fieldSeparator[$sepLen - 1 - $i]) {
					$match = false;
					break;
				}
			}
			if ($match) {
				$buffer->setLength($len - $sepLen);
			}
		}
	}
	
	
	// Public append methods ---------------------------------------------------------------
	/**
	 * Appends the field description.
	 *
	 * @param Commons_Lang_StringBuffer $buffer
	 * @param string $fieldName
	 * @param mixed $value
	 * @param bool $fullDetail
	 */
	public function append(
		Commons_Lang_StringBuffer $buffer, $fieldName, $value, $fullDetail = null) {
		
		$this->appendFieldStart($buffer, $fieldName);
		
		if (is_null($fullDetail)) {
			$this->appendDetail($buffer, $fieldName, $value);
		}
		else {
			if (is_null($value)) {
				$this->appendNullText($buffer, $fieldName);
			}
			else {
				$this->appendInternal(
					$buffer, $fieldName, $value, $this->isFullDetail($fullDetail));
			}
		}
		
		$this->appendFieldEnd($buffer, $fieldName);
	}

	
	/**
	 * Switches the internal append method. 
	 *
	 * @param Commons_Lang_StringBuffer $buffer
	 * @param string $fieldName
	 * @param mixed $value
	 * @param bool $detail
	 */
	protected function appendInternal(
		Commons_Lang_StringBuffer $buffer, $fieldName, $value, $detail) {

		if (is_array($value)) {
			if ($detail) {
				$this->appendDetail($buffer, $fieldName, $value);
			}
			else {
				$this->appendSummary($buffer, $fieldName, $value);
			}
		}
		// in the future
/*		elseif ($value instanceof Commons_Util_Collection) {
			if ($detail) {
				$this->appendDetail($buffer, $fieldName, $value);
			}
			else {
				$this->appendSummarySize($buffer, $fieldName, $value->length());
			}
		}
*/		elseif (is_object($value)) {
			if ($detail) {
				$this->appendDetail($buffer, $fieldName, $value);
			}
			else {
				$this->appendSummary($buffer, $fieldName, $value);
			}
		}
		else {
			if ($detail) {
				$this->appendDetail($buffer, $fieldName, $value);
			}
			else {
				$this->appendSummary($buffer, $fieldName, $value);
			}
		}
	}
	
	
	/**
	 * Appends summary description.
	 *
	 * @param Commons_Lang_StringBuffer $buffer
	 * @param string $fieldName
	 * @param mixed $value
	 */
	protected function appendSummary(
		Commons_Lang_StringBuffer $buffer, $fieldName, $value) {

		if (is_array($value)) {
			$this->appendSummarySize($buffer, $fieldName, count($value));			
		}
		elseif (is_object($value)) {
			$buffer->append($this->_summaryObjectStartText);
			$buffer->append($this->getShortClassName(get_class($value)));
			$buffer->append($this->_summaryObjectEndText);
		}
		else {
			$buffer->append($this->_summaryObjectStartText);
			$buffer->append(gettype($value));
			$buffer->append($this->_summaryObjectEndText);
		}
	}
	

	/**
	 * Appends the details to the buffer.
	 *
	 * @param Commons_Lang_StringBuffer $buffer
	 * @param string $fieldName
	 * @param mixed $value
	 */
	protected function appendDetail(Commons_Lang_StringBuffer $buffer, $fieldName, $value) {
		if (is_bool($value)) {
			$str = $value === true ? 'true' : 'false';
			$buffer->append($str);
		}
		elseif (is_array($value)) {
			$buffer->append($this->_arrayStart);
			$i = 0;
			foreach ($value as $key => $elm) {
				if ($i > 0) {
					$buffer->append($this->_arraySeparator);
				}
				$buffer->append('[' . $key . ']=>');
				$this->appendDetail($buffer, $fieldName, $elm);
				
				$i++;
			}
			$buffer->append($this->_arrayEnd);
		}
		else {
			$buffer->append((string)$value);
		}
	}
	
	
	// Utility methods ---------------------------------------------------------------------
	/**
	 * Appends the class name.
	 *
	 * @param Commons_Lang_StringBuffer $buffer
	 * @param object $object
	 */
	protected function appendClassName(Commons_Lang_StringBuffer $buffer, $object) {
		if ($this->_useClassName && !is_null($object)) {
			if ($this->_useShortClassName) {
				$buffer->append($this->getShortClassName(get_class($object)));
			}
			else {
				$buffer->append(get_class($object));
			}
		}
	}
	
	
	/**
	 * Appends the object ID hash code.
	 *
	 * @param Commons_Lang_StringBuffer $buffer
	 * @param object $object
	 */
	protected function appendIdentityHashCode(
		Commons_Lang_StringBuffer $buffer, $object) {
		if ($this->isUseIdentityHashCode() && !is_null($object)) {
			$buffer->append('@');
			$buffer->append(Commons_Lang_System::identityHashCode($object));
		}
	}

	
	protected function appendContentStart(Commons_Lang_StringBuffer $buffer) {
		$buffer->append($this->_contentStart);
	}
	
	
	protected function appendContentEnd(Commons_Lang_StringBuffer $buffer) {
		$buffer->append($this->_contentEnd);
	}
	
	
	protected function appendNullText(Commons_Lang_StringBuffer $buffer, $fieldName) {
		$buffer->append($this->_nullText);
	}
	
	
	protected function appendFieldSeparator(Commons_Lang_StringBuffer $buffer) {
		$buffer->append($this->_fieldSeparator);
	}
	
	
	protected function appendFieldStart(Commons_Lang_StringBuffer $buffer, $fieldName) {
		if ($this->_useFieldNames && !is_null($fieldName)) {
			$buffer->append($fieldName);
			$buffer->append($this->_fieldNameValueSeparator);
		}
	}
	
	
	protected function appendFieldEnd(Commons_Lang_StringBuffer $buffer, $fieldName) {
		$this->appendFieldSeparator($buffer);
	}
	
	
	protected function appendSummarySize(
		Commons_Lang_StringBuffer $buffer, $fieldName, $size) {
		$buffer->append($this->_sizeStartText);
		$buffer->append($size);
		$buffer->append($this->_sizeEndText);
	}
	
	
	protected function isFullDetail($fullDetailRequest) {
		if (is_null($fullDetailRequest)) {
			return $this->_defaultFullDetail;
		}
		return $fullDetailRequest;
	}
	
	
	protected function getShortClassName($className) {
		return Commons_Lang_ClassUtils::getShortClassName($className);
	}
	
	
	// Accessors ---------------------------------------------------------------------------
	protected function isUseClassName() {
		return $this->_useClassName;
	}
	
	protected function setUseClassName($useClassName) {
		$this->_useClassName = $useClassName;
	}
	
	protected function isUseShortClassName() {
		return $this->_useShortClassName;
	}
	
	protected function setUseShortClassName($useShortClassName) {
		$this->_useShortClassName = $useShortClassName;
	}
	
	protected function isUseIdentityHashCode() {
		return $this->_useIdentityHashCode;
	}

	protected function setUseIdentityHashCode($useIdentityHashCode) {
		$this->_useIdentityHashCode = $useIdentityHashCode;
	}
	
	protected function isUseFieldNames() {
		return $this->_useFieldNames;
	}
	
	protected function setUseFieldNames($useFieldNames) {
		$this->_useFieldNames = $useFieldNames;
	}
	
	protected function isDefaultFullDetail() {
		return $this->_defaultFullDetail;
	}
	
	protected function setDefaultFullDetail($defaultFullDetail) {
		$this->_defaultFullDetail = $defaultFullDetail;
	}
	
	protected function isArrayContentDetail() {
		return $this->_arrayContentDetail;
	}
	
	protected function setArrayContentDetail($arrayContentDetail) {
		$this->_arrayContentDetail = $arrayContentDetail;
	}
	
	protected function getArrayStart() {
		return $this->_arrayStart;
	}
	
	protected function setArrayStart($arrayStart) {
		if (is_null($arrayStart)) {
			$arrayStart = '';
		}
		$this->_arrayStart = $arrayStart;
	}
	
	protected function getArrayEnd() {
		return $this->_arrayEnd;
	}
	
	protected function setArrayEnd($arrayEnd) {
		if (is_null($arrayEnd)) {
			$arrayEnd = '';
		}
		$this->_arrayEnd = $arrayEnd;
	}
	
	protected function getArraySeparator() {
		return $this->_arraySeparator;
	}
	
	protected function setArraySeparator($arraySeparator) {
		if (is_null($arraySeparator)) {
			$arraySeparator = '';
		}
		$this->_arraySeparator = $arraySeparator;
	}
	
	protected function getContentStart() {
		return $this->_contentStart;
	}
	
	protected function setContentStart($contentStart) {
		if (is_null($contentStart)) {
			$contentStart = '';
		}
		$this->_contentStart = $contentStart;
	}
	
	protected function getContentEnd() {
		return $this->_contentEnd;
	}
	
	protected function setContentEnd($contentEnd) {
		if (is_null($contentEnd)) {
			$contentEnd = '';
		}
		$this->_contentEnd = $contentEnd;
	}
	
	protected function getFieldNameValueSeparator() {
		return $this->_fieldNameValueSeparator;
	}
	
	protected function setFieldNameValueSeparator($fieldNameValueSeparator) {
		if (is_null($fieldNameValueSeparator)) {
			$fieldNameValueSeparator = '';
		}
		$this->_fieldNameValueSeparator = $fieldNameValueSeparator;
	}
	
	protected function getFieldSeparator() {
		return $this->_fieldSeparator;
	}
	
	protected function setFieldSeparator($fieldSeparator) {
		if (is_null($fieldSeparator)) {
			$fieldSeparator = '';
		}
		$this->_fieldSeparator = $fieldSeparator;
	}
	
	protected function isFieldSeparatorAtStart() {
		return $this->_fieldSeparatorAtStart;
	}
	
	protected function setFieldSeparatorAtStart($fieldSeparatorAtStart) {
		$this->_fieldSeparatorAtStart = $fieldSeparatorAtStart;
	}
	
	protected function isFieldSeparatorAtEnd() {
		return $this->_fieldSeparatorAtEnd;
	}
	
	protected function setFieldSeparatorAtEnd($fieldSeparatorAtEnd) {
		$this->_fieldSeparatorAtEnd = $fieldSeparatorAtEnd;
	}
	
	public function getNullText() {
		return $this->_nullText;
	}
	
	protected function setNullText($nullText) {
		if (is_null($nullText)) {
			$nullText = '';
		}
		$this->_nullText = $nullText;
	}
	
	protected function getSizeStartText() {
		return $this->_sizeStartText;
	}
	
	protected function setSizeStartText($sizeStartText) {
		if (is_null($sizeStartText)) {
			$sizeStartText = '';
		}
		$this->_sizeStartText = $sizeStartText;
	}
	
	protected function getSizeEndText() {
		return $this->_sizeEndText;
	}
	
	protected function setSizeEndText($sizeEndText) {
		if (is_null($sizeEndText)) {
			$sizeEndText = '';
		}
		$this->_sizeEndText = $sizeEndText;
	}
	
	protected function getSummaryObjectStartText() {
		return $this->_summaryObjectStartText;
	}
	
	protected function setSummaryObjectStartText($summaryObjectStartText) {
		if (is_null($summaryObjectStartText)) {
			$summaryObjectStartText = '';
		}
		$this->_summaryObjectStartText = $summaryObjectStartText;
	}
	
	protected function getSummaryObjectEndText() {
		return $this->_summaryObjectEndText;
	}
	
	protected function setSummaryObjectEndText($summaryObjectEndText) {
		if (is_null($summaryObjectEndText)) {
			$summaryObjectEndText = '';
		}
		$this->_summaryObjectEndText = $summaryObjectEndText;
	}

}


/**
 * DefaultToStringStyle
 * The original class is an inner class of ToStringStyle
 *
 *
 * @category   PHP
 * @package    Commons.Lang.Builder
 * @author     Yomei Komiya
 * @copyright  2007 the original author or authors.
 * @license    http://www.apache.org/licenses/LICENSE-2.0  Apache License 2.0
 * @version    Release: 0.9.0
 * @link       http://phpcommons.sourceforge.jp/
 * @see        
 * @since      Class available since Release 0.8.0
 */
final class Commons_Lang_Builder_DefaultToStringStyle
	extends Commons_Lang_Builder_ToStringStyle {

	public function Commons_Lang_Builder_DefaultToStringStyle() {
		parent::__construct();
	}
	
	
	public function readResolve() {
		return Commons_Lang_Builder_ToStringStyle::$DEFAULT_STYLE;
	}

}


/**
 * NoFieldNameToStringStyle
 * The original class is an inner class of ToStringStyle
 *
 *
 * @category   PHP
 * @package    Commons.Lang.Builder
 * @author     Yomei Komiya
 * @copyright  2007 the original author or authors.
 * @license    http://www.apache.org/licenses/LICENSE-2.0  Apache License 2.0
 * @version    Release: 0.9.0
 * @link       http://phpcommons.sourceforge.jp/
 * @see        
 * @since      Class available since Release 0.8.0
 */
final class Commons_Lang_Builder_NoFieldNameToStringStyle
	extends Commons_Lang_Builder_ToStringStyle {

	public function Commons_Lang_Builder_NoFieldNameToStringStyle() {
		parent::__construct();
		$this->setUseFieldNames(false);
	}

	
	public function readResolve() {
		return Commons_Lang_Builder_ToStringStyle::$NO_FIELD_NAMES_STYLE;
	}
}


/**
 * ShortPrefixToStringStyle
 * The original class is an inner class of ToStringStyle
 *
 *
 * @category   PHP
 * @package    Commons.Lang.Builder
 * @author     Yomei Komiya
 * @copyright  2007 the original author or authors.
 * @license    http://www.apache.org/licenses/LICENSE-2.0  Apache License 2.0
 * @version    Release: 0.9.0
 * @link       http://phpcommons.sourceforge.jp/
 * @see        
 * @since      Class available since Release 0.8.0
 */
final class Commons_Lang_Builder_ShortPrefixToStringStyle
	extends Commons_Lang_Builder_ToStringStyle {
	
	public function Commons_Lang_Builder_ShortPrefixToStringStyle() {
		parent::__construct();
		$this->setUseShortClassName(true);
		$this->setUseIdentityHashCode(false);
	}
	
	
	public function readResolve() {
		return Commons_Lang_Builder_ToStringStyle::$SHORT_PREFIX_STYLE;
	}

}


/**
 * SimpleToStringStyle
 * The original class is an inner class of ToStringStyle
 *
 *
 * @category   PHP
 * @package    Commons.Lang.Builder
 * @author     Yomei Komiya
 * @copyright  2007 the original author or authors.
 * @license    http://www.apache.org/licenses/LICENSE-2.0  Apache License 2.0
 * @version    Release: 0.9.0
 * @link       http://phpcommons.sourceforge.jp/
 * @see        
 * @since      Class available since Release 0.8.0
 */
final class Commons_Lang_Builder_SimpleToStringStyle
	extends Commons_Lang_Builder_ToStringStyle {

	public function Commons_Lang_Builder_SimpleToStringStyle() {
		parent::__construct();
		$this->setUseClassName(false);
		$this->setUseIdentityHashCode(false);
		$this->setUseFieldNames(false);
		$this->setContentStart('');
		$this->setContentEnd('');
	}

	
	public function readResolve() {
		return Commons_Lang_Builder_ToStringStyle::$SIMPLE_STYLE;
	}

}


/**
 * MultiLineToStringStyle
 * The original class is an inner class of ToStringStyle
 *
 *
 * @category   PHP
 * @package    Commons.Lang.Builder
 * @author     Yomei Komiya
 * @copyright  2007 the original author or authors.
 * @license    http://www.apache.org/licenses/LICENSE-2.0  Apache License 2.0
 * @version    Release: 0.9.0
 * @link       http://phpcommons.sourceforge.jp/
 * @see        
 * @since      Class available since Release 0.8.0
 */
final class Commons_Lang_Builder_MultiLineToStringStyle
	extends Commons_Lang_Builder_ToStringStyle {

	public function __construct() {
		parent::__construct();
		$this->setContentStart('[');
		$this->setFieldSeparator(Commons_Lang_SystemUtils::getLineSeparator() . '  ');
		$this->setFieldSeparatorAtStart(true);
		$this->setContentEnd(Commons_Lang_SystemUtils::getLineSeparator() . ']');
	}

	
	public function readResolve() {
		return Commons_Lang_Builder_ToStringStyle::$MULTI_LINE_STYLE;
	}

}


// Pseudo initialization of ToStringStyle's static fields.
Commons_Lang_Builder_ToStringStyle::init();

/*
 * Local variables:
 * tab-width: 4
 * c-basic-offset: 4
 * c-hanging-comment-ender-p: nil
 * End:
 */
?>