<?php
/* vim: set tabstop=4 shiftwidth=4: */

/*
 * String Utilities
 *
 *
 * PHP version 5
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * 	http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * @category   PHP
 * @package    Commons
 * @author     Yomei Komiya
 * @copyright  2007 the original author or authors.
 * @license    http://www.apache.org/licenses/LICENSE-2.0  Apache License 2.0
 * @version    SVN: $Id: StringUtils.php 176 2008-01-28 14:06:29Z whitestar $
 * @link       http://phpcommons.sourceforge.jp/
 * @see        
 * @since      File available since Release 0.8.0
 */

//namespace Commons::Lang;

require_once 'Commons/Lang/Object.php';
require_once 'Commons/Lang/StringIndexOutOfBoundsException.php';
/*
use Commons::Lang::Object;
use Commons::Lang::StringIndexOutOfBoundsException;
*/

/**
 * StringUtils
 *
 *
 * @category   PHP
 * @package    Commons.Lang
 * @author     Yomei Komiya
 * @copyright  2007 the original author or authors.
 * @license    http://www.apache.org/licenses/LICENSE-2.0  Apache License 2.0
 * @version    Release: 0.9.0
 * @link       http://phpcommons.sourceforge.jp/
 * @see        
 * @since      Class available since Release 0.8.0
 */
class Commons_Lang_StringUtils extends Commons_Lang_Object {
	
	private static $_instance = null;
	
	private static $SPLIT_FUNC = null;
	
	private static $STRLEN_FUNC = null;
	
	private static $STRPOS_FUNC = null;
	
	private static $SUBSTR_FUNC = null;
	
	private static $STRTOLOWER_FUNC = null;
	
	private static $STRTOUPPER_FUNC = null;
	

	/**
	 * Constructor
	 */
	public function Commons_Lang_StringUtils() {
		parent::__construct();
	}
	
	
	public function __destruct() {
	}
	
	
	public static function getInstance() {
		if (is_null(self::$_instance)) {
			self::$_instance = new self();
		}
		
		return self::$_instance;
	}
	
	
	/**
	 * Switches functions
	 *
	 * @param string $altFuncName alternative (BETTER) function name
	 * @param string $defaultFuncName default function name
	 * @return string function name. the alternative function name if its function exists.
	 *           otherwise the default function name.
	 */
	protected function defaultFunction($altFuncName, $defaultFuncName) {
		return function_exists($altFuncName) ? $altFuncName : $defaultFuncName;
	}
	
	
	// Multibyte character set sensitive wrappers of string functions ----------------------
	/**
	 * Gets the string length
	 *
	 * @param string $str
	 * @return int String length
	 */
	public static function strlen($str) {
		if (is_null(self::$STRLEN_FUNC)) {
			self::$STRLEN_FUNC = self::defaultFunction('mb_strlen', 'strlen');
		}
		$func = self::$STRLEN_FUNC;

		return $func($str);
	}
	
	
	/**
	 * Gets the first position of needle string within the string
	 *
	 * @param string $haystack
	 * @param string $needle Search string
	 * @param int $offset
	 * @return int The first position
	 */
	public static function strpos($haystack, $needle, $offset = 0) {
		if (is_null(self::$STRPOS_FUNC)) {
			self::$STRPOS_FUNC = self::defaultFunction('mb_strpos', 'strpos');
		}

		$func = self::$STRPOS_FUNC;
		return $func($haystack, $needle, $offset);
	}
	
	
	/**
	 * Gets the last position of needle string within the string
	 *
	 * @param string $haystack
	 * @param string $needle Search string
	 * @return int The last position
	 */
	public static function strrpos($haystack, $needle) {
		if (function_exists('mb_strrpos')) {
			return mb_strrpos($haystack, $needle);
		}
		// We do not use strpos, because strpos is NOT compatible with mb_strrpos
		else {
			$revHaystack = strrev($haystack);
			$revNeedle = strrev($needle);
			$revIndex = strpos($revHaystack, $revNeedle);
			if ($revIndex !== false) {
				return strlen($haystack) - ($revIndex + strlen($needle));
			}
			else {
				return false;
			}
		}
	}
	
	
	/**
	 * Gets the portion of a string
	 *
	 * @param string $str
	 * @param int $start Start index
	 * @param int $length
	 * @return string
	 */
	public static function substr($str, $start, $length = null) {
		if (is_null(self::$SUBSTR_FUNC)) {
			self::$SUBSTR_FUNC = self::defaultFunction('mb_substr', 'substr');
		}
		$func = self::$SUBSTR_FUNC;
		
		if (is_null($length)) {
			return $func($str, $start);
		}
		else {
			return $func($str, $start, $length);
		}
	}
	
	
	/**
	 * Lowercases the passed string
	 *
	 * @param string $str
	 * @return string Lowercased string. An empty ('') if the string is null or empty.
	 */
	public static function strtolower($str) {
		if (is_null(self::$STRTOLOWER_FUNC)) {
			self::$STRTOLOWER_FUNC = self::defaultFunction('mb_strtolower', 'strtolower');
		}
		$func = self::$STRTOLOWER_FUNC;
		
		return $func($str);
	}
	
	
	/**
	 * Uppercases the passed string
	 *
	 * @param string $str
	 * @return string Uppercased string. An empty ('') if the string is null or empty.
	 */
	public static function strtoupper($str) {
		if (is_null(self::$STRTOUPPER_FUNC)) {
			self::$STRTOUPPER_FUNC = self::defaultFunction('mb_strtoupper', 'strtoupper');
		}
		$func = self::$STRTOUPPER_FUNC;
		
		return $func($str);
	}
	
	
	public static function split($pattern, $str, $limit = -1) {
		if (is_null(self::$SPLIT_FUNC)) {
			self::$SPLIT_FUNC = self::defaultFunction('mb_split', 'split');
		}
		$func = self::$SPLIT_FUNC;
		
		return $func($pattern, $str, $limit);
	}
	
	
	/**
	 * Splits the string to some-character-string array
	 *
	 * @param string $str
	 * @param int $splitLength split length >= 1 (default: 1)
	 * @return array New string array. False if the $splitLength is invalid.
	 * Empty array if the passed string is an empty string ''. null if the string is null.
	 */
	public static function str_split($str, $splitLength = 1) {
		if ($splitLength < 1) {
			return false;
		}
		elseif (is_null($str)) {
			return null;
		}
		elseif ($str === '') {
			return array();
		}
		
		if (function_exists('mb_strlen') && function_exists('mb_substr')) {
			$result = array();
			$limit = mb_strlen($str);
			for ($i = 0; $i < $limit; $i += $splitLength) {
				$result[] = mb_substr($str, $i, $splitLength);
			}
			return $result;
		}
		else {
			return str_split($str, $splitLength);
		}
	}
	
	
	// Utility static methods --------------------------------------------------------------
	/**
	 * Returns a character at the index within the string
	 *
	 * @param string $str
	 * @param int $index
	 * @return string The one-character-string. A null if the string is null.
	 * An empty ('') string if the string is empty.
	 * @throws Commons_Lang_StringIndexOutOfBoundsException
	 *  if the index is invalid. 
	 */
	public static function charAt($str, $index) {
		if (is_null($str)) {
			return null;
		}
		elseif ($str === '') {
			return '';
		}
		
		if ($index < 0 || $index >= self::strlen($str)) {
			throw new Commons_Lang_StringIndexOutOfBoundsException($index);
		}
		
		return self::substr($str, $index, 1);		
	}
	
	
	/**
	 * Returns the default string if the passed string is empty (null or '').
	 * 
	 * <pre>
	 * Commons_Lang_StringUtils::defaultIfEmpty(null, 'NULL')  = 'NULL'
	 * Commons_Lang_StringUtils::defaultIfEmpty('', 'NULL')    = 'NULL'
	 * Commons_Lang_StringUtils::defaultIfEmpty('bat', 'NULL') = 'bat'
	 * </pre>
	 *
	 * @param string $str passed string.
	 * @param string $defaultStr default string.
	 * @return string result string.
	 */
	public static function defaultIfEmpty($str, $defaultStr) {
		if (self::isEmpty($str)) {
			return $defaultStr;
		}
		else {
			return $str;
		}
	}
	
	
	/**
	 * Returns the default string if the passed string is null.
	 * 
	 * <pre>
	 * Commons_Lang_StringUtils::defaultString(null)  = ''
	 * Commons_Lang_StringUtils::defaultString('')    = ''
	 * Commons_Lang_StringUtils::defaultString('bat') = 'bat'
	 * Commons_Lang_StringUtils::defaultString(null, 'NULL')  = 'NULL'
	 * Commons_Lang_StringUtils::defaultString('', 'NULL')    = ''
	 * Commons_Lang_StringUtils::defaultString('bat', 'NULL') = 'bat'
	 * </pre>
	 *
	 * @param string $str passed string.
	 * @param string $defaultStr default string (default: empty '').
	 * @return string result string.
	 */
	public static function defaultString($str, $defaultStr = '') {
		if (is_null($str)) {
			return $defaultStr;
		}
		else {
			return $str;
		}
	}
	
	
	/**
	 * Tests empty (null or '').
	 * 
	 * <pre>
	 * Commons_Lang_StringUtils::isEmpty(null)      = true
	 * Commons_Lang_StringUtils::isEmpty('')        = true
	 * Commons_Lang_StringUtils::isEmpty(' ')       = false
	 * Commons_Lang_StringUtils::isEmpty('bob')     = false
	 * Commons_Lang_StringUtils::isEmpty('  bob  ') = false
	 * </pre>
	 * 
	 * @param string $str passed string.
	 * @return bool true if the passed string is null or empty ''.
	 */
	public static function isEmpty($str) {
		if (is_null($str) || $str === '') {
			return true;		
		}
		else {
			return false;
		}
	}
	
	
	/**
	 * Tests not empty (null or '').
	 * 
	 * <pre>
	 * Commons_Lang_StringUtils::isNotEmpty(null)      = false
	 * Commons_Lang_StringUtils::isNotEmpty('')        = false
	 * Commons_Lang_StringUtils::isNotEmpty(' ')       = true
	 * Commons_Lang_StringUtils::isNotEmpty('bob')     = true
	 * Commons_Lang_StringUtils::isNotEmpty('  bob  ') = true
	 * </pre>
	 *
	 * @param string $str passed string.
	 * @return bool true if neither the passed string is null nor empty ''.
	 */
	public static function isNotEmpty($str) {
		return !self::isEmpty($str);
	}
	
	
	/**
	 * Get the leftmost string of the passed string.
	 * 
	 * <pre>
	 * Commons_Lang_StringUtils::left(null, *)    = null
	 * Commons_Lang_StringUtils::left(*, -ve)     = ''
	 * Commons_Lang_StringUtils::left('', *)      = ''
	 * Commons_Lang_StringUtils::left('abc', 0)   = ''
	 * Commons_Lang_StringUtils::left('abc', 2)   = 'ab'
	 * Commons_Lang_StringUtils::left('abc', 4)   = 'abc'
	 * </pre>
	 *
	 * @param string $str passed string
	 * @param int $len result length
	 * @return string result string
	 * @since 0.9.0
	 */
	public static function left($str, $len) {
		if (is_null($str)) {
			return null;	
		}
		elseif ($str === '') {
			return '';
		}
		elseif ($len <= 0) {
			return '';
		}
		
		$currentLen = self::strlen($str);
		if ($currentLen > $len) {
			return self::substr($str, 0, $len);
		}
		else {
			return $str;
		}
	}
	
	
	/**
	 * Left-pads the string with the padding string (default: ' ').
	 * 
	 * <pre>
	 * Commons_Lang_StringUtils::leftPad(null, *)   = null
	 * Commons_Lang_StringUtils::leftPad('', 3)     = '   '
	 * Commons_Lang_StringUtils::leftPad('bat', 3)  = 'bat'
	 * Commons_Lang_StringUtils::leftPad('bat', 5)  = '  bat'
	 * Commons_Lang_StringUtils::leftPad('bat', 1)  = 'bat'
	 * Commons_Lang_StringUtils::leftPad('bat', -1) = 'bat'
	 * Commons_Lang_StringUtils::leftPad(null, *, *)     = null
	 * Commons_Lang_StringUtils::leftPad('', 3, 'z')     = 'zzz'
	 * Commons_Lang_StringUtils::leftPad('bat', 3, 'z')  = 'bat'
	 * Commons_Lang_StringUtils::leftPad('bat', 5, 'z')  = 'zzbat'
	 * Commons_Lang_StringUtils::leftPad('bat', 1, 'z')  = 'bat'
	 * Commons_Lang_StringUtils::leftPad('bat', -1, 'z') = 'bat'
	 * Commons_Lang_StringUtils::leftPad('bat', 3, 'yz')  = 'bat'
	 * Commons_Lang_StringUtils::leftPad('bat', 5, 'yz')  = 'yzbat'
	 * Commons_Lang_StringUtils::leftPad('bat', 8, 'yz')  = 'yzyzybat'
	 * Commons_Lang_StringUtils::leftPad('bat', 1, 'yz')  = 'bat'
	 * Commons_Lang_StringUtils::leftPad('bat', -1, 'yz') = 'bat'
	 * Commons_Lang_StringUtils::leftPad('bat', 5, null)  = '  bat'
	 * Commons_Lang_StringUtils::leftPad('bat', 5, '')    = '  bat'
	 * </pre>
	 *
	 * @param string $str passed string
	 * @param int $len result lenght
	 * @param string $padStr padding string (default: single space ' ')
	 * @return string
	 * @since 0.9.0
	 */
	public static function leftPad($str, $len, $padStr = ' ') {
		if (is_null($padStr) || $padStr === '') {
			$padStr = ' ';
		}
		
		if (is_null($str)) {
			return null;	
		}
		elseif ($str === '' && $len > 0) {
			$padStrLen = self::strlen($padStr);
			if ($padStrLen === 1) {
				return str_repeat($padStr, $len);
			}
			else {	// $padStrLen > 1
				$multiplier = $len / $padStrLen;
				$surplus = $len % $padStrLen;
				return str_repeat($padStr, $multiplier) . self::substr($padStr, 0, $surplus);
			}
		}
		
		$currentLen = self::strlen($str);
		if ($currentLen < $len) {
			$padLen = $len - $currentLen;
			$padStrLen = self::strlen($padStr);
			if ($padStrLen === 1) {
				return str_repeat($padStr, $padLen) . $str;
			}
			else {	// $padStrLen > 1
				$multiplier = $padLen / $padStrLen;
				$surplus = $padLen % $padStrLen;
				return
					str_repeat($padStr, $multiplier) . self::substr($padStr, 0, $surplus)
					. $str;
			}
		}
		else {
			return $str;
		}
	}
	
	
	/**
	 * Get the rightmost string of the passed string.
	 * 
	 * <pre>
	 * Commons_Lang_StringUtils::right(null, *)    = null
	 * Commons_Lang_StringUtils::right(*, -ve)     = ''
	 * Commons_Lang_StringUtils::right('', *)      = ''
	 * Commons_Lang_StringUtils::right('abc', 0)   = ''
	 * Commons_Lang_StringUtils::right('abc', 2)   = 'bc'
	 * Commons_Lang_StringUtils::right('abc', 4)   = 'abc'
	 * </pre>
	 *
	 * @param string $str passed string
	 * @param int $len result length
	 * @return string result string
	 * @since 0.9.0
	 */
	public static function right($str, $len) {
		if (is_null($str)) {
			return null;	
		}
		elseif ($str === '') {
			return '';
		}
		elseif ($len <= 0) {
			return '';
		}
		
		$currentLen = self::strlen($str);
		if ($currentLen > $len) {
			return self::substr($str, $currentLen - $len);
		}
		else {
			return $str;
		}
	}
	

	/**
	 * Right-pads the string with the padding string (default: ' ').
	 * 
	 * <pre>
	 * Commons_Lang_StringUtils::rightPad(null, *)   = null
	 * Commons_Lang_StringUtils::rightPad('', 3)     = '   '
	 * Commons_Lang_StringUtils::rightPad('bat', 3)  = 'bat'
	 * Commons_Lang_StringUtils::rightPad('bat', 5)  = 'bat  '
	 * Commons_Lang_StringUtils::rightPad('bat', 1)  = 'bat'
	 * Commons_Lang_StringUtils::rightPad('bat', -1) = 'bat'
	 * Commons_Lang_StringUtils::rightPad(null, *, *)     = null
	 * Commons_Lang_StringUtils::rightPad('', 3, 'z')     = 'zzz'
	 * Commons_Lang_StringUtils::rightPad('bat', 3, 'z')  = 'bat'
	 * Commons_Lang_StringUtils::rightPad('bat', 5, 'z')  = 'batzz'
	 * Commons_Lang_StringUtils::rightPad('bat', 1, 'z')  = 'bat'
	 * Commons_Lang_StringUtils::rightPad('bat', -1, 'z') = 'bat'
	 * Commons_Lang_StringUtils::rightPad('bat', 3, 'yz')  = 'bat'
	 * Commons_Lang_StringUtils::rightPad('bat', 5, 'yz')  = 'batyz'
	 * Commons_Lang_StringUtils::rightPad('bat', 8, 'yz')  = 'batyzyzy'
	 * Commons_Lang_StringUtils::rightPad('bat', 1, 'yz')  = 'bat'
	 * Commons_Lang_StringUtils::rightPad('bat', -1, 'yz') = 'bat'
	 * Commons_Lang_StringUtils::rightPad('bat', 5, null)  = 'bat  '
	 * Commons_Lang_StringUtils::rightPad('bat', 5, '')    = 'bat  '
	 * </pre>
	 *
	 * @param string $str passed string
	 * @param int $len result lenght
	 * @param string $padStr padding string (default: single space ' ')
	 * @return string
	 * @since 0.9.0
	 */
	public static function rightPad($str, $len, $padStr = ' ') {
		if (is_null($padStr) || $padStr === '') {
			$padStr = ' ';
		}
		
		if (is_null($str)) {
			return null;	
		}
		elseif ($str === '' && $len > 0) {
			$padStrLen = self::strlen($padStr);
			if ($padStrLen === 1) {
				return str_repeat($padStr, $len);
			}
			else {	// $padStrLen > 1
				$multiplier = $len / $padStrLen;
				$surplus = $len % $padStrLen;
				return str_repeat($padStr, $multiplier) . self::substr($padStr, 0, $surplus);
			}
		}
		
		$currentLen = self::strlen($str);
		if ($currentLen < $len) {
			$padLen = $len - $currentLen;
			$padStrLen = self::strlen($padStr);
			if ($padStrLen === 1) {
				return $str . str_repeat($padStr, $padLen);
			}
			else {	// $padStrLen > 1
				$multiplier = $padLen / $padStrLen;
				$surplus = $padLen % $padStrLen;
				return
					$str
					. str_repeat($padStr, $multiplier) . self::substr($padStr, 0, $surplus);
			}
		}
		else {
			return $str;
		}
	}
	
	
	/**
	 * Lowercases the passed string
	 * 
	 * <pre>
	 * Commons_Lang_StringUtils::lowerCase(null)  = null
	 * Commons_Lang_StringUtils::lowerCase('')    = ''
	 * Commons_Lang_StringUtils::lowerCase('aBc') = 'abc'
	 * </pre>
	 *
	 * @param string $str
	 * @return string Lowercased string. A null if the string is null.
	 *         An empty ('') if the string is empty.
	 */
	public static function lowerCase($str) {
		if (is_null($str)) {
			return null;
		}
		elseif ($str === '') {
			return '';
		}
		
		return self::strtolower($str);
	}
	
	
	/**
	 * Uppercases the passed string
	 * 
	 * <pre>
	 * Commons_Lang_StringUtils::upperCase(null)  = null
	 * Commons_Lang_StringUtils::upperCase('')    = ''
	 * Commons_Lang_StringUtils::upperCase('aBc') = 'ABC'
	 * </pre>
	 *
	 * @param string $str
	 * @return string Uppercased string. A null if the string is null.
	 *         An empty ('') if the string is empty.
	 */
	public static function upperCase($str) {
		if (is_null($str)) {
			return null;
		}
		elseif ($str === '') {
			return '';
		}
		
		return self::strtoupper($str);
	}
	
	
	/**
	 * Reverses character order of the string.
	 * 
	 * <pre>
	 * Commons_Lang_StringUtils::reverse(null)  = null
	 * Commons_Lang_StringUtils::reverse('')    = ''
	 * Commons_Lang_StringUtils::reverse('bat') = 'tab'
	 * </pre>
	 *
	 * @param string $str
	 * @return string reversed string.
	 */
	public static function reverse($str) {
		if (is_null($str)) {
			return null;
		}
		
		if (function_exists('mb_strlen') && function_exists('mb_substr')) {
			return join(array_reverse(self::str_split($str, 1)));
		}
		else {
			return strrev($str);
		}
	}
	
	
	/**
	 * Reverses token order of the string parsed by separator.
	 * 
	 * <pre>
	 * Commons_Lang_StringUtils::reverseDelimited(null, *)         = null
	 * Commons_Lang_StringUtils::reverseDelimited('', *)           = ''
	 * Commons_Lang_StringUtils::reverseDelimited('a.b.c', 'x')    = 'a.b.c'
	 * Commons_Lang_StringUtils::reverseDelimited('a.b.c', '.')    = 'c.b.a'
	 * Commons_Lang_StringUtils::reverseDelimited('a::b::c', 'x')  = 'a::b::c'
	 * Commons_Lang_StringUtils::reverseDelimited('a::b::c', '::') = 'c::b::a'
	 * </pre>
	 *
	 * @param string $str
	 * @param string $separatorStr
	 * @return string
	 */
	public static function reverseDelimited($str, $separatorStr) {
		if (is_null($str) || is_null($separatorStr)) {
			return null;
		}
		
		return join(
			array_reverse(self::split(quotemeta($separatorStr), $str)),
			$separatorStr);
	}
	
	
	/**
	 * Returns the new copied one-character-string array of the passed string.
	 *
	 * @param string $str
	 * @return array One-character-string array.
	 * Empty array if the passed string is an empty string ''. null if the string is null.
	 */
	public static function toCharArray($str) {
		return self::str_split($str, 1); 
	}
	
	
	public static function startsWith($str, $prefix, $toffset = 0) {
		if (is_null($str) || is_null($prefix) || $toffset < 0) {
			return false;
		}
		
		return self::strpos($str, $prefix, $toffset) === $toffset;
	}

	
	public static function endsWith($str, $suffix) {
		if (is_null($str) || is_null($suffix)) {
			return false;
		}
		
		$toffset = self::strlen($str) - self::strlen($suffix);
		if ($toffset < 0) {
			return false;
		}
		else {
			return self::substr($str, $toffset) === $suffix;
		}
	}
	
	
	/**
	 * Finds the first index within a string
	 * 
	 * <pre>
	 * Commons_Lang_StringUtils::indexOf(null, *)          = -1
	 * Commons_Lang_StringUtils::indexOf(*, null)          = -1
	 * Commons_Lang_StringUtils::indexOf('', '')           = 0
	 * Commons_Lang_StringUtils::indexOf('aabaabaa', 'a')  = 0
	 * Commons_Lang_StringUtils::indexOf('aabaabaa', 'b')  = 2
	 * Commons_Lang_StringUtils::indexOf('aabaabaa', 'ab') = 1
	 * Commons_Lang_StringUtils::indexOf('aabaabaa', '')   = 0
	 * </pre> 
	 *
	 * @param string $str
	 * @param string $searchStr
	 * @return int the first index of the search string,
	 * -1 if no match or null string input
	 */
	public static function indexOf($str, $searchStr) {
		if (is_null($str) || is_null($searchStr)) {
			return -1;
		}
		if ($searchStr === '') {
			return 0;
		}
		
		$index = self::strpos($str, $searchStr);
		if ($index !== false) {
			return $index;
		}
		else {
			return -1;
		}
	}

	
	/**
	 * Finds the last index within a string
	 *
	 * <pre>
	 * Commons_Lang_StringUtils::lastIndexOf(null, *)          = -1
	 * Commons_Lang_StringUtils::lastIndexOf(*, null)          = -1
	 * Commons_Lang_StringUtils::lastIndexOf('', '')           = 0
	 * Commons_Lang_StringUtils::lastIndexOf('aabaabaa', 'a')  = 0
	 * Commons_Lang_StringUtils::lastIndexOf('aabaabaa', 'b')  = 2
	 * Commons_Lang_StringUtils::lastIndexOf('aabaabaa', 'ab') = 1
	 * Commons_Lang_StringUtils::lastIndexOf('aabaabaa', '')   = 8
	 * </pre>
	 * 
	 * @param string $str
	 * @param string $searchStr
	 * @return int the last index of the search string,
	 * -1 if no match or null string input
	 */
	public static function lastIndexOf($str, $searchStr) {
		if (is_null($str) || is_null($searchStr)) {
			return -1;
		}
		if ($str === '' && $searchStr === '') {
			return 0;
		}
		if ($searchStr === '') {
			return self::strlen($str);
		}
		
		$index = self::strrpos($str, $searchStr);
		if ($index !== false) {
			return $index;
		}
		
		return -1;
	}
	
	
	/**
	 * Returns the part of the string
	 *
	 * @param string $str
	 * @param int $beginIndex
	 * @param int $endIndex
	 * @return string The portion of the string.
	 * A null if the string is null. An empty ('') string if the string is empty.
	 */
	public static function substring($str, $beginIndex, $endIndex = null) {
		if (is_null($str)) {
			return null;
		}
		elseif ($str === '') {
			return '';
		}
		
		if (is_null($endIndex)) {
			return self::substr($str, $beginIndex);
		}
		else {
			return self::substr($str, $beginIndex, $endIndex - $beginIndex);
		}
	}
	
}

/*
 * Local variables:
 * tab-width: 4
 * c-basic-offset: 4
 * c-hanging-comment-ender-p: nil
 * End:
 */
?>