<?php
/* vim: set tabstop=4 shiftwidth=4: */

/*
 * Simple Log
 * (based on org.apache.commons.logging.impl.SimpleLog)
 *
 *
 * PHP version 5
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * 	http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * @category   PHP
 * @package    Commons
 * @author     Yomei Komiya
 * @copyright  2008 the original author or authors.
 * @license    http://www.apache.org/licenses/LICENSE-2.0  Apache License 2.0
 * @version    SVN: $Id: SimpleLog.php 193 2008-02-03 06:38:35Z whitestar $
 * @link       http://phpcommons.sourceforge.jp/
 * @see        
 * @since      File available since Release 0.9.0
 */

//namespace Commons::Logging::Impl;

require_once 'Commons/Exception.php';
require_once 'Commons/Lang/Object.php';
require_once 'Commons/Lang/StdErr.php';
require_once 'Commons/Lang/StringBuffer.php';
require_once 'Commons/Lang/StringUtils.php';
require_once 'Commons/Lang/Time/MicrosecondTime.php';
require_once 'Commons/IO/Serializable.php';
require_once 'Commons/Logging/Log.php';
require_once 'Commons/Logging/Impl/NoOpLog.php';
require_once 'Commons/Util/Properties.php';
/*
use Commons::Exception;
use Commons::Lang::Object;
use Commons::Lang::StdErr;
use Commons::Lang::StringBuffer;
use Commons::Lang::StringUtils;
use Commons::Lang::Time::MicrosecondTime;
use Commons::IO::Serializable;
use Commons::Logging::Log;
use Commons::Util::Properties::Log;
* */

/**
 * SimpleLog
 *
 *
 * @category   PHP
 * @package    Commons.Logging.Impl
 * @author     Yomei Komiya
 * @copyright  2008 the original author or authors.
 * @license    http://www.apache.org/licenses/LICENSE-2.0  Apache License 2.0
 * @version    Release: 0.9.0
 * @link       http://phpcommons.sourceforge.jp/
 * @see        
 * @since      Class available since Release 0.9.0
 */
class Commons_Logging_Impl_SimpleLog extends Commons_Logging_Impl_NoOpLog
//class Commons_Logging_Impl_SimpleLog extends Commons_Lang_Object
	implements Commons_Logging_Log, Commons_IO_Serializable {

	const systemPrefix = 'commons.logging.simplelog.';
	
	const DEFAULT_DATE_TIME_FORMAT = '%G/%m/%d %H:%M:%S %Z';
	
	protected static $simpleLogProps = null;
	
	protected static $showLogName = false;
	
	protected static $showShortName = true;
	
	protected static $showDateTime = false;
	
	protected static $dateTimeFormat = self::DEFAULT_DATE_TIME_FORMAT;
	
	const LOG_LEVEL_TRACE  = 1;
	
	const LOG_LEVEL_DEBUG  = 2;
	
	const LOG_LEVEL_INFO   = 3;
	
	const LOG_LEVEL_WARN   = 4;
	
	const LOG_LEVEL_ERROR  = 5;
	
	const LOG_LEVEL_FATAL  = 6;
	
	const LOG_LEVEL_ALL    = 0;
	
	const LOG_LEVEL_OFF    = 7;
	
	
	private static function getStringProperty($name, $default = null) {
		$prop = null;
		$prop = getenv($name);
		$prop = $prop === false ? self::$simpleLogProps->getProperty($name) : $prop;
		return is_null($prop) ? $default : $prop;
	}
	
	
	private static function getBooleanProperty($name, $default) {
		$prop = Commons_Lang_StringUtils::upperCase(self::getStringProperty($name));
		return is_null($prop) ? $default : 'TRUE' === $prop;
	}
	
	
	public static function init() {
		self::$simpleLogProps = new Commons_Util_Properties();
		
		// Add props from the resource simplelog.properties
		$inStream = fopen('simplelog.properties', 'r', true);
		if ($inStream !== false) {
			self::$simpleLogProps->load($inStream);
			fclose($inStream);
		}
		
		self::$showLogName = self::getBooleanProperty(
			self::systemPrefix . 'showlogname', self::$showLogName);
		self::$showShortName = self::getBooleanProperty(
			self::systemPrefix . 'showShortLogname', self::$showShortName);
		self::$showDateTime = self::getBooleanProperty(
			self::systemPrefix . 'showdatetime', self::$showDateTime);
	}
	
	
	protected $stderr = false;
		
	protected $logName = null;
	
	protected $currentLogLevel;
	
	private $_shortLogName = null;
	
	
	public function Commons_Logging_Impl_SimpleLog($name) {
		$this->logName = $name;
		
		// Set log level from properties
		$lvl = self::getStringProperty(self::systemPrefix . 'log.' . $this->logName);
		$i = Commons_Lang_StringUtils::lastIndexOf($name, '.');
		while (is_null($lvl) && $i > -1) {
			$name = Commons_Lang_StringUtils::substring($name, 0, $i);
			$lvl = self::getStringProperty(self::systemPrefix . 'log.' . $name);
			$i = Commons_Lang_StringUtils::lastIndexOf($name, '.');
		}
		
		if (is_null($lvl)) {
			$lvl = self::getStringProperty(self::systemPrefix . 'defaultlog');
		}
		
		$lvl = strtoupper($lvl);
		switch ($lvl) {
			case 'ALL':   $this->setLevel(self::LOG_LEVEL_ALL); break;
			case 'TRACE': $this->setLevel(self::LOG_LEVEL_TRACE); break;
			case 'DEBUG': $this->setLevel(self::LOG_LEVEL_DEBUG); break;
			case 'INFO':  $this->setLevel(self::LOG_LEVEL_INFO); break;
			case 'WARN':  $this->setLevel(self::LOG_LEVEL_WARN); break;
			case 'ERROR': $this->setLevel(self::LOG_LEVEL_ERROR); break;
			case 'FATAL': $this->setLevel(self::LOG_LEVEL_FATAL); break;
			case 'OFF':   $this->setLevel(self::LOG_LEVEL_OFF); break;
			default:      $this->setLevel(self::LOG_LEVEL_INFO);
		}
	}
	
	
	public function __destruct() {
	}
	
	
	public function setLevel($currentLogLevel) {
		$this->currentLogLevel = $currentLogLevel;
	}
	
	
	public function getLevel() {
		return $this->currentLogLevel;
	}

	
	/**
	 * Logs the message.
	 *
	 * @param int $type log level constants.
	 * @param mixed $message message object.
	 * @param Exception $e exception object.
	 */
	protected function log($type, $message, $e = null) {
		if (is_null($message)) {
			$message = '<null>';
		}
		
		$buf = new Commons_Lang_StringBuffer();
		
		if (self::$showDateTime) {
			$now = new Commons_Lang_Time_MicrosecondTime();
			$buf->append(
				strftime(self::$dateTimeFormat, $now->getSec()) . ' +' . $now->getUsec());
			$buf->append(' ');
		}
		
		switch ($type) {
			case self::LOG_LEVEL_TRACE: $buf->append('[TRACE] '); break;
			case self::LOG_LEVEL_DEBUG: $buf->append('[DEBUG] '); break;
			case self::LOG_LEVEL_INFO:  $buf->append('[INFO]  '); break;
			case self::LOG_LEVEL_WARN:  $buf->append('[WARN]  '); break;
			case self::LOG_LEVEL_ERROR: $buf->append('[ERROR] '); break;
			case self::LOG_LEVEL_FATAL: $buf->append('[FATAL] '); break;
		}
		
		if (self::$showShortName) {
			if (is_null($this->_shortLogName)) {
				// Cut all but the last component of the name for both styles
				$this->_shortLogName
					= Commons_Lang_StringUtils::substring($this->logName,
						Commons_Lang_StringUtils::lastIndexOf($this->logName, '.') + 1);
				$this->_shortLogName
					= Commons_Lang_StringUtils::substring($this->_shortLogName,
						Commons_Lang_StringUtils::lastIndexOf($this->_shortLogName, '/') + 1);
			}
			$buf->append($this->_shortLogName)->append(' - ');
		}
		elseif (self::$showLogName) {
			$buf->append($this->logName)->append(' - ');
		}
		
		$buf->append((string)$message);
		
		// Append stack trace if not null
		if (!is_null($e)) {
			$buf->append(' <');
			$buf->append($e->__toString());
			$buf->append('>');
			$buf->append(Commons_Exception::stackTraceString($e));
		}
		
		$this->write($buf);		
	}
	
	
	protected function write(Commons_Lang_StringBuffer $buffer) {
		Commons_Lang_StdErr::println($buffer->__toString());
	}

	
	protected function isLevelEnabled($logLevel) {
		return ($logLevel >= $this->currentLogLevel);
	}

	
	public final function trace($message, $e = null) {
		if ($this->isLevelEnabled(self::LOG_LEVEL_TRACE)) {
			$this->log(self::LOG_LEVEL_TRACE, $message, $e);
		}
	}

	
	public final function debug($message, $e = null) {
		if ($this->isLevelEnabled(self::LOG_LEVEL_DEBUG)) {
			$this->log(self::LOG_LEVEL_DEBUG, $message, $e);
		}
	}
	
	
	public final function info($message, $e = null) {
		if ($this->isLevelEnabled(self::LOG_LEVEL_INFO)) {
			$this->log(self::LOG_LEVEL_INFO, $message, $e);
		}
	}
	
	
	public final function warn($message, $e = null) {
		if ($this->isLevelEnabled(self::LOG_LEVEL_WARN)) {
			$this->log(self::LOG_LEVEL_WARN, $message, $e);
		}
	}
	
	
	public final function error($message, $e = null) {
		if ($this->isLevelEnabled(self::LOG_LEVEL_ERROR)) {
			$this->log(self::LOG_LEVEL_ERROR, $message, $e);
		}
	}
	
	
	public final function fatal($message, $e = null) {
		if ($this->isLevelEnabled(self::LOG_LEVEL_FATAL)) {
			$this->log(self::LOG_LEVEL_FATAL, $message, $e);
		}
	}
	
	
	public final function isTraceEnabled() {
		return $this->isLevelEnabled(self::LOG_LEVEL_TRACE);
	}
	
	
	public final function isDebugEnabled() {
		return $this->isLevelEnabled(self::LOG_LEVEL_DEBUG);
	}
	
	
	public final function isInfoEnabled() {
		return $this->isLevelEnabled(self::LOG_LEVEL_INFO);
	}
	
	
	public final function isWarnEnabled() {
		return $this->isLevelEnabled(self::LOG_LEVEL_WARN);
	}
	
	
	public final function isErrorEnabled() {
		return $this->isLevelEnabled(self::LOG_LEVEL_ERROR);
	}
	
	
	public final function isFatalEnabled() {
		return $this->isLevelEnabled(self::LOG_LEVEL_FATAL);
	}
	
	
	public function shutdown() {
		// do nothing for stderr.
	}
	

}

Commons_Logging_Impl_SimpleLog::init();

/*
 * Local variables:
 * tab-width: 4
 * c-basic-offset: 4
 * c-hanging-comment-ender-p: nil
 * End:
 */
?>