<?php
/* vim: set tabstop=4 shiftwidth=4: */

/*
 * Duration Format Utilities
 * (based on org.apache.commons.lang.time.DurationFormatUtils)
 * 
 *
 * PHP version 5
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * 	http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * @category   PHP
 * @package    Commons
 * @author     Yomei Komiya
 * @copyright  2007,2008 the original author or authors.
 * @license    http://www.apache.org/licenses/LICENSE-2.0  Apache License 2.0
 * @version    SVN: $Id: DurationFormatUtils.php 282 2008-04-04 14:13:25Z whitestar $
 * @link       http://phpcommons.sourceforge.jp/
 * @see        
 * @since      File available since Release 0.8.0
 */

//namespace Commons::Lang::Time;

require_once 'Commons/Lang/Object.php';
require_once 'Commons/Lang/StringBuffer.php';
require_once 'Commons/Lang/StringUtils.php';
require_once 'Commons/Lang/Time/DateUtils.php';
/*
use Commons::Lang::Object;
use Commons::Lang::StringBuffer;
use Commons::Lang::StringUtils;
use Commons::Lang::Time::DateUtils;
*/

/**
 * DurationFormatToken
 *
 *
 * @category   PHP
 * @package    Commons.Lang.Time
 * @author     Yomei Komiya
 * @copyright  2007,2008 the original author or authors.
 * @license    http://www.apache.org/licenses/LICENSE-2.0  Apache License 2.0
 * @version    Release: 1.0.1
 * @link       http://phpcommons.sourceforge.jp/
 * @see        
 * @since      Class available since Release 0.8.0
 */
class Commons_Lang_Time_DurationFormatToken extends Commons_Lang_Object {
//class DurationFormatToken extends Object {

	/**
	 * @var mixed value
	 */
	private $_value = null;
	
	/**
	 * @var int count
	 */
	private $_count = 0;
	

	/**
	 * Constructor
	 * 
	 * @param object $value value to wrap
	 * @param int $count count to wrap
	 */
	public function Commons_Lang_Time_DurationFormatToken($value, $count = 1) {
		parent::__construct();
		$this->_value = $value;
		$this->_count = $count;
	}
	
	
	/**
	 * Tests whether the collection of tokens contains the value or not.
	 *
	 * @param array $tokens
	 * @param object $value
	 * @return bool true if the collection of tokens contains the value.
	 */
	public static function containsTokenWithValue($tokens, $value) {
		foreach ($tokens as $token) {
			if ($token->getValue() === $value) {
				return true;
			}
		}
		return false;
	}


	public function increment() { 
		$this->_count++;
	}
	
	
	public function getCount() {
		return $this->_count;
	}
	
	
	public function getValue() {
		return $this->_value;
	}


	/**
	 * Equality test.
	 *
	 * @param object $other
	 * @return bool true if this object equals the other.
	 */
	public function equals($other) {
		if ($other instanceof self) {
			if (gettype($this->_value) !== gettype($other->_value)) {
				return false;
			}
			if ($this->_count !== $other->_count) {
				return false;
			}
			if (is_object($this->_value)) {
				if (get_class($this->_value) === get_class($other->_value)) {
					return $this->_value->equals($other->_value);
				}
				else {
					return false;
				}
			}
			else {
				return $this->_value === $other->_value;
			}
		}
		else {
			return false;
		}
	}
	

	public function __toString() {
		return str_repeat($this->_value, $this->_count);
	}
	
}


/**
 * DurationFormatUtils
 *
 *
 * @category   PHP
 * @package    Commons.Lang.Time
 * @author     Yomei Komiya
 * @copyright  2007 the original author or authors.
 * @license    http://www.apache.org/licenses/LICENSE-2.0  Apache License 2.0
 * @version    Release: 1.0.1
 * @link       http://phpcommons.sourceforge.jp/
 * @see        
 * @since      Class available since Release 0.8.0
 */
class Commons_Lang_Time_DurationFormatUtils extends Commons_Lang_Object {
//class DurationFormatUtils extends Object {

	const y = 'y';
	const M = 'M';
	const d = 'd';
	const H = 'H';
	const m = 'm';
	const s = 's';
	const S = 'S';


	public function Commons_Lang_Time_DurationFormatUtils() {
		parent::__construct();
	}
	
	
	public function __destruct() {
		// do nothing.
	}
	
	
	/**
	 * 
	 * @param float $durationMillis
	 * @return string formated string
	 * 
	 */
	public static function formatDurationHMS($durationMillis) {
		return self::formatDuration($durationMillis, 'H:mm:ss.SSS');
	}
	
	
	/**
	 * Formats time duration's string
	 * 
	 * @param float $durationMillis duration millisecond
	 * @param string $format format string
	 * @param bool $padWithZeros zero padding flag
	 */
	public static function formatDuration($durationMillis, $format, $padWithZeros = true) {
		$tokens = self::lexx($format);

		$days         = 0;
		$hours        = 0;
		$minutes      = 0;
		$seconds      = 0;
		$milliseconds = 0;
		
		if (Commons_Lang_Time_DurationFormatToken::containsTokenWithValue($tokens, self::d)) {
			$days = (int) ($durationMillis / Commons_Lang_Time_DateUtils::MILLIS_PER_DAY);
			$durationMillis -= $days * Commons_Lang_Time_DateUtils::MILLIS_PER_DAY;
		}
		if (Commons_Lang_Time_DurationFormatToken::containsTokenWithValue($tokens, self::H)) {
			$hours = (int) ($durationMillis / Commons_Lang_Time_DateUtils::MILLIS_PER_HOUR);
			$durationMillis -= $hours * Commons_Lang_Time_DateUtils::MILLIS_PER_HOUR;
		}
		if (Commons_Lang_Time_DurationFormatToken::containsTokenWithValue($tokens, self::m)) {
			$minutes
				= (int) ($durationMillis / Commons_Lang_Time_DateUtils::MILLIS_PER_MINUTE);
			$durationMillis -= $minutes * Commons_Lang_Time_DateUtils::MILLIS_PER_MINUTE;
		}
		if (Commons_Lang_Time_DurationFormatToken::containsTokenWithValue($tokens, self::s)) {
			$seconds
				= (int) ($durationMillis / Commons_Lang_Time_DateUtils::MILLIS_PER_SECOND);
			$durationMillis -= $seconds * Commons_Lang_Time_DateUtils::MILLIS_PER_SECOND;
		}
		if (Commons_Lang_Time_DurationFormatToken::containsTokenWithValue($tokens, self::S)) {
			$milliseconds = (int) $durationMillis;
		}
		
		return self::format(
			$tokens, 0, 0, $days, $hours, $minutes, $seconds, $milliseconds, $padWithZeros);
	}
	
	
	/**
	 * Formats the value of duration.
	 *
	 * @param array $tokens
	 * @param int $years
	 * @param int $months
	 * @param int $days
	 * @param int $hours
	 * @param int $minutes
	 * @param int $seconds
	 * @param int $milliseconds
	 * @param bool $padWithZeros zero padding flag
	 * @return string formatted string
	 */
	public static function format(
		$tokens,
		$years, $months, $days, $hours, $minutes, $seconds, $milliseconds,
		$padWithZeros) {
			
		$buffer = '';
		$lastOutputSeconds = false;
		foreach ($tokens as $token) {
			$value = $token->getValue();
			$count = $token->getCount();
			if ($value === self::y) {
				$buffer .= $padWithZeros
					? Commons_Lang_StringUtils::leftPad((string)$years, $count, '0')
					: (string)$years;
				$lastOutputSeconds = false;
			}
			elseif ($value === self::M) {
				$buffer .= $padWithZeros
					? Commons_Lang_StringUtils::leftPad((string)$months, $count, '0')
					: (string)$months;
				$lastOutputSeconds = false;
			}
			elseif ($value === self::d) {
				$buffer .= $padWithZeros
					? Commons_Lang_StringUtils::leftPad((string)$days, $count, '0')
					: (string)$days;
				$lastOutputSeconds = false;
			}
			elseif ($value === self::H) {
				$buffer .= $padWithZeros
					? Commons_Lang_StringUtils::leftPad((string)$hours, $count, '0')
					: (string)$hours;
				$lastOutputSeconds = false;
			}
			elseif ($value === self::m) {
				$buffer .= $padWithZeros
					? Commons_Lang_StringUtils::leftPad((string)$minutes, $count, '0')
					: (string)$minutes;
				$lastOutputSeconds = false;
			}
			elseif ($value === self::s) {
				$buffer .= $padWithZeros
					? Commons_Lang_StringUtils::leftPad((string)$seconds, $count, '0')
					: (string)$seconds;
				$lastOutputSeconds = false;
			}
			elseif ($value === self::S) {
				if ($lastOutputSeconds) {
					$milliseconds += 1000;
					$str = $padWithZeros
						? Commons_Lang_StringUtils::leftPad((string)$milliseconds, $count, '0')
						: (string)$milliseconds;
					$buffer .= Commons_Lang_StringUtils::substr($str, 1);
				}
				else {
					$buffer .= $padWithZeros
						? Commons_Lang_StringUtils::leftPad((string)$milliseconds, $count, '0')
						: (string)$milliseconds;
				}
				$lastOutputSeconds = false;
			}
			elseif ($value instanceof Commons_Lang_StringBuffer) {
				$buffer .= $value->__toString();	// for PHP < 5.2.0
				//$buffer .= (string)$value;
			}
		}
		return $buffer;		
	}
	
	
	/**
	 * Parses the format string
	 *
	 * @param string $format
	 * @return array parsed tokens
	 */
	public static function lexx($format) {
		$array = Commons_Lang_StringUtils::toCharArray($format);
		$list = array();
		
		$inLiteral = false;
		$buffer = null;
		$previous = null;
		foreach ($array as $ch) {
			if ($inLiteral && $ch !== "'") {
				$buffer->append($ch);
				continue;
			}
			$value = null;
			switch ($ch) {
				case "'" :
					if ($inLiteral) {
						$buffer = null;
						$inLiteral = false;
					}
					else {
						$buffer = new Commons_Lang_StringBuffer();
						$list[] = new Commons_Lang_Time_DurationFormatToken($buffer);
						$inLiteral = true;
					}
					break;
				case 'y'  : $value = self::y; break;
				case 'M'  : $value = self::M; break;
				case 'd'  : $value = self::d; break;
				case 'H'  : $value = self::H; break;
				case 'm'  : $value = self::m; break;
				case 's'  : $value = self::s; break;
				case 'S'  : $value = self::S; break;
				default   :
					if (is_null($buffer)) {
						$buffer = new Commons_Lang_StringBuffer();
						$list[] = new Commons_Lang_Time_DurationFormatToken($buffer);
					}
					$buffer->append($ch);
			}
			
			if (!is_null($value)) {
				if (!is_null($previous) && $previous->getValue() === $value) {
					$previous->increment();
				}
				else {
					$token = new Commons_Lang_Time_DurationFormatToken($value);
					$list[] = $token; 
					$previous = $token;
				}
				$buffer = null;
			}
		}
		return $list;
	}
	
}

/*
 * Local variables:
 * tab-width: 4
 * c-basic-offset: 4
 * c-hanging-comment-ender-p: nil
 * End:
 */
?>