<?php
/**
 * phpoot - template engine for php
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * @author Haruki Setoyama <haruki@planewave.org>
 * @copyright Copyright &copy; 2003-2004, Haruki SETOYAMA <haruki@planewave.org>
 * @license http://www.opensource.org/licenses/lgpl-license.php LGPL
 * @version 0.4 $Id: phpoot.php,v 1.27 2004/03/04 12:43:56 haruki Exp $
 * @link http://phpoot.sourceforge.jp/
 * @package phpoot
 */
/**
 * Requires PEAR
 */
require_once 'PEAR.php';

// Constants
/**
 * mark for compiled template that mean PHPOOT library has been included
 */
define('PHPOOT', true);
/**
 * directory where PHPOOT library is in
 */
@define('PHPOOT_DIR', dirname(__FILE__));
/**
 * extention added to compiled template
 */
@define('PHPOOT_EXT', '.php');

/**
 * PHPOOT Main Class.
 *
 * @access public
 * @package phpoot
 * @subpackage main_interface
 */
class phpoot extends PEAR {

    /**
     * options
     * <pre>
     * Key                  Meaning
     * 'force_compile'      when true, template will be compiled everytime
     * 'check_timestamp'    when true, modified time of template will be checked
     * 'debug'              ehwn true, display debug information in escape form <!-- -->
     * 'pear_error'         when false, false is returned insted of pearError
     * 'mkdir_mode'         mode for mkdir()
     * </pre>
     * @var array
     */
    var $option = array(
            'force_compile'   => false,
            'check_timestamp' => true,
            'debug'           => false,
            'pear_error'      => true,
            'mkdir_mode'      => 0777
        );

    /**
     * path to directory where compiled files will be stored
     * @var string
     * @access private
     */
    var $_compiled_dir;

    /**
     * @access private
     */
    var $_compiled_dir_use = false;

    /**
     * path to template file
     * @var string
     * @access private
     */
    var $_template_file;

    /**
     * XML_HTMLSax parser object
     * @access private
     */
    var $_parser;

    /**
     * handler object for XML_HTMLSax parser
     * @access private
     */
    var $_handler;

    /**
     * set a path for directory where compiled files will be stored
     *
     * @param stirng $dirname
     * @return void
     */
    function setCompileDir($dirname)
    {
        if ($dirname === null || $dirname === false) {
            $this->_compiled_dir_use = false;
        } else {
            $this->_compiled_dir = $dirname;
            $this->_compiled_dir_use = true;
        }
    }

    /**
     * set a path for template file
     *
     * @param stirng $template
     * @return void
     */
    function setTemplateFile($template)
    {
        $this->_template_file = $template;
        if (! $this->_compiled_dir_use) {
            $this->_compiled_dir = dirname($template);
        }
    }

    /**
     * set option
     * for option for phpoot_handler, see phpoot_handler class in phpoot.parser.php
     * @param stirng $key   key of option
     * @param mixed $val    value of option
     * @return void
     */
    function setOption($key, $val)
    {
        $this->option[$key] = $val;
    }

    /**
     * displays the template results
     *
     * @param array $data   model data
     * @return bool|PEARError
     */
    function display($data)
    {
        $compiled_file = $this->_compiled_dir . '/' . basename($this->_template_file) . PHPOOT_EXT;
        if ($this->option['force_compile'] != false
        || file_exists($compiled_file) != true) {
            $err = $this->_compile($compiled_file);
            if ($err !== true) {
                return $this->raiseError($err);
            }
        }
        elseif ($this->option['check_timestamp'] != false
        && $this->_template_readable() == true
        && $this->_template_update_time() > filemtime($compiled_file)) {
            $err = $this->_compile($compiled_file);
            if ($err !== true) {
                return $this->raiseError($err);
            }
        }

        if (! phpoot_display_compiled($compiled_file, $data, $this->option['debug'])) {
            return $this->raiseError("Error(s) in compiled file '$compiled_file' or in template file '$this->_template_file'. ");
        }
        return true;
    }

    /**
     * gets the template results
     *
     * @param sting $output     variable for output
     * @param array $data       model data
     * @return bool|PEARError
     */
    function expand(&$output, $data)
    {
        ob_start();
        $ret = $this->display($data);
        $output = ob_get_contents();
        ob_end_clean();
        return $ret;
    }

    /**
     * compiles template
     *
     * @access public
     * @return bool|PEARError
     */
    function compile()
    {
        $compiled_file = $this->_compiled_dir . '/' . basename($this->_template_file) . PHPOOT_EXT;
        $err = $this->_compile($compiled_file);
        if ($err !== true) {
            return $this->raiseError($err);
        }
        return true;
    }

    /**
     * compile template file
     *
     * @param stirng $compiled_file
     * @access private
     * @return bool
     */
    function _compile($compiled_file)
    {
        if (! $this->_template_readable()) {
            return "Template file '$this->_template_file' does not exists or is not readable. ";
        }
        if (! file_exists($compiled_file)) {
            $dir = dirname($compiled_file);
            if (! $this->_mkdirs($dir, PHPOOT_DIR_MODE)) {
                return "Fail to create the directory '$dir' for compiled files.";
            }
            if (! is_writable($dir)) {
                return "Directory '$dir' is not writable for compiled file.";
            }
        } else {
            if (! is_writable($compiled_file)) {
                return "File '$compiled_file' for compiled template file is not rewritable.";
            }
        }
        $template = $this->_get_template();
        if (! $template) return "Fail to get template file '$this->_template_file'. ";

        $parsed = $this->_parse($template);
        $parsed = "<?php if(! defined('PHPOOT')) exit(); ?>\n" . $parsed . "<?php \$_check = true; ?>\n"; // <?
        $fp_w = fopen($compiled_file, 'wb');
        if (! $fp_w) return "Fail to open compiled file '$compiled_file'. ";
        if (! fwrite($fp_w, $parsed)) return "Fail to write compiled file '$compiled_file'. ";
        ftruncate($fp_w, ftell($fp_w));
        fclose($fp_w);
        return true;
    }

    /**
     * parse template file
     *
     * @param  string $template
     * @access private
     * @return string parsed PHP code
     */
    function _parse($template)
    {
        if (!isset($this->_parser) || !isset($this->_handler)) {
            include_once PHPOOT_DIR . '/phpoot.parser.php';
            $this->_parser = &new XML_HTMLSax();
            $this->_handler = new phpoot_handler;
            $this->_handler->setOption($this->option);
            $this->_parser->set_object($this->_handler);
            $this->_handler->configParser($this->_parser);
        }

        $this->_handler->init($template);
        $this->_parser->parse($template);
        if (! empty($this->_handler->error)) {
            $this->raiseError("Template has some error. ");
        }

        return $this->_handler->getParsed();
    }

    /**
     * make directory
     *
     * @param stirng $path
     * @access private
     * @return bool
     */
    function _mkdirs($path)
    {
        if (strlen($path) == 0) return false;
        if (is_dir($path)) return true;
        if (is_file($path)) return false;
        if (! $this->_mkdirs(dirname($path))) return false;
        return mkdir($path, $this->option['mkdir_mode']);
    }

    /**
     * get update time of the template file
     *
     * @access private
     * @return int
     */
    function _template_update_time()
    {
        return filemtime($this->_template_file);
    }

    /**
     * check whether the template file is readable
     *
     * @access private
     * @return bool
     */
    function _template_readable()
    {
        return is_readable($this->_template_file);
    }

    /**
     * get text of the template file
     *
     * @access private
     * @return string|false
     */
    function _get_template()
    {
        $fd = fopen($this->_template_file, 'rb');
        if (! $fd) {
            $this->raiseError("Fail to open the template file '$this->_template_file'.");
            return false;
        }
        $contents = fread ($fd, filesize($this->_template_file));
        fclose ($fd);
        return $contents;
    }

    /**
     * if $this->option['pear_error'] is true, false is returned by raiseError()
     *
     * @access private
     * @return false|PEARError
     */
    function raiseError($msg)
    {
        if ($this->option['pear_error']) {
            return parent::raiseError('<b>PHPOOT Error</b>: '.htmlspecialchars($msg));
        } else {
            parent::raiseError('<b>PHPOOT Error</b>: '.htmlspecialchars($msg));
            return false;
        }
    }
}


/**
 * displays the template results with compiled template
 *
 * @param stirng    $compiled_file  path of compiled template file
 * @param array     $_val0          model data
 * @param bool      $_error         if this displayes error or not
 * @access private
 * @return bool
 */
function phpoot_display_compiled($compiled_file, $_val0, $_error =false)
{
    $_check = false; // flag for parse error check
    $_no_error = true;
    include $compiled_file;
    return ($_check && $_no_error);
}

// functions below used in compiled template file.

/**
 * get array of data for nested variabe tag
 *
 * @return array
 * @access private
 */
function phpoot_each($val, $var)
{
    $target = phpoot_val($val, $var);
    if (is_array($target) && isset($target[0])) {
        return $target;
    }
    return array($target);
}

/**
 * make scalar data to array
 * @access private
 */
function phpoot_to_array(&$scalar, $var)
{
    if (is_scalar($scalar)) {
        $arr[$var] = $scalar;
        $scalar = $arr;
    }
}

/**
 * returns string data for display
 * @return string
 * @access private
 */
function phpoot_string($val, $var, $error)
{
    if (is_string($val) || is_numeric($val)) {
        echo phpoot_htmlspecialchars($val);
    }
    elseif (is_array($val) && is_string($val[$var])) {
        echo phpoot_htmlspecialchars($val[$var]);
    }
    else {
        echo ($error) ? "<!-- NON STRING DATA '$var' -->" : '';
    }
}

/**
 * htmlspecialchars() for phpoot
 * @return string
 * @access private
 */
function phpoot_htmlspecialchars($val)
{
    return str_replace('  ', '&nbsp;&nbsp;', htmlspecialchars($val));
}

/**
 * returns string data for display
 * @return string
 * @access private
 */
function phpoot_raw($val, $var, $error)
{
    if (is_string($val) || is_numeric($val)) {
        echo $val;
    }
    elseif (is_array($val) && is_string($val[$var])) {
        echo $val[$var];
    }
    else {
        echo ($error) ? "<!-- NON STRING DATA '$var' -->" : '';
    }
}

/**
 * makes string from data for attribute
 * @return string
 * @access private
 */
function phpoot_attr($attr, $val, $var)
{
    $target = phpoot_val($val, $var);

    if ($target === null || $target === false) {
        return '';
    } elseif ($target === true) {
        return " $attr=\"$attr\"";
    } elseif (! is_string($target)) {
        return '';
    } else {
        return ' ' . $attr . '="' . phpoot_htmlspecialchars($target) . '"';
    }
}

/**
 * get value from $val with variable name
 * @return mixed
 * @access private
 */
function phpoot_val($val, $var)
{
    if (is_object($val)) {
        if (is_callable(array($val, $var))) {
            return $val->$var();
        } else {
            return $val->$var;
        }
    } elseif (is_array($val) && isset($val[$var])) {
        return $val[$var];
    } else {
        return false;
    }
}
?>