#include "config.h"
#include "pipes/pipes.h"
#include <errno.h>
#include <time.h>
#include <stdlib.h>
#include <fcntl.h>
#include <libgen.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <math.h>
#include <oniguruma.h>
#include <sys/stat.h>
#include <signal.h>
#include <limits.h>
#include <dirent.h>
#include <readline/readline.h>
#include <readline/history.h>

#if defined(HAVE_CURSES_H)
#include <curses.h>
#elif defined(HAVE_NCURSES_H)
#include <ncurses.h>
#elif defined(HAVE_NCURSES_NCURSES_H)
#include <ncurses/ncurses.h>
#endif

static void str_cut(enum eKanjiCode code, char* mbs, int termsize, char* dest_mbs, int dest_byte)
{
    if(code == kUtf8) {
        wchar_t* wcs;
        wchar_t* wcs_tmp;
        int i;

        wcs = (wchar_t*)MALLOC(sizeof(wchar_t)*(termsize+1)*MB_CUR_MAX);
        wcs_tmp = (wchar_t*)MALLOC(sizeof(wchar_t)*(termsize+1)*MB_CUR_MAX);
        if(mbstowcs(wcs, mbs, (termsize+1)*MB_CUR_MAX) == -1) {
            mbstowcs(wcs, "?????", (termsize+1)*MB_CUR_MAX);
        }

        for(i=0; i<wcslen(wcs); i++) {
            wcs_tmp[i] = wcs[i];
            wcs_tmp[i+1] = 0;

            if(wcswidth(wcs_tmp, wcslen(wcs_tmp)) > termsize) {
                wcs_tmp[i] = 0;
                break;
            }
        }

        wcstombs(dest_mbs, wcs_tmp, dest_byte);

        FREE(wcs);
        FREE(wcs_tmp);
    }
    else {
        int n;
        BOOL kanji = FALSE;
        for(n=0; n<termsize && n<dest_byte-1; n++) {
            if(!kanji && is_kanji(code, mbs[n])) {
                kanji = TRUE;
            }
            else {
                kanji = FALSE;
            }

            dest_mbs[n] = mbs[n];
        }
        
        if(kanji)
            dest_mbs[n-1] = 0;
        else
            dest_mbs[n] = 0;
    }
}

BOOL cmd_selector(sObject* nextin, sObject* nextout, sRunInfo* runinfo)
{
    if(isatty(0) == 0 || isatty(1) == 0) {
        err_msg("selector: stdin or stdout is not a tty", runinfo->mSName, runinfo->mSLine);
        return FALSE;
    }
    if(tcgetpgrp(0) != getpgid(0)) {
        err_msg("selector: not forground process group", runinfo->mSName, runinfo->mSLine);
        return FALSE;
    }
    char* term_env = getenv("TERM");
    if(term_env == NULL || strcmp(term_env, "") == 0) {
        err_msg("selector: not TERM environment variable setting", runinfo->mSName, runinfo->mSLine);
        return FALSE;
    }

    sCommand* command = runinfo->mCommand;

    enum eLineField lf = gLineField;
    if(sCommand_option_item(command, "-Lw")) {
        lf = kCRLF;
    }
    else if(sCommand_option_item(command, "-Lm")) {
        lf = kCR;
    }
    else if(sCommand_option_item(command, "-Lu")) {
        lf = kLF;
    }
    else if(sCommand_option_item(command, "-La")) {
        lf = kBel;
    }

    enum eKanjiCode code = gKanjiCode;
    if(sCommand_option_item(command, "-byte")) {
        code = kByte;
    }
    else if(sCommand_option_item(command, "-utf8")) {
        code = kUtf8;
    }
    else if(sCommand_option_item(command, "-sjis")) {
        code = kSjis;
    }
    else if(sCommand_option_item(command, "-eucjp")) {
        code = kEucjp;
    }


    BOOL multiple = sCommand_option_item(command, "-multiple");

    if(runinfo->mFilter) {
        fd_split(nextin, lf);

        if(vector_size(SFD(nextin).mLines) > 0) {
#if !defined(__FREEBSD__)
            msave_screen();
#endif
            msave_ttysettings();
            minitscr();

            const int maxx = mgetmaxx();
            const int maxy = mgetmaxy();

            int* markfiles = MALLOC(sizeof(int)*vector_size(SFD(nextin).mLines));
            memset(markfiles, 0, sizeof(int)*vector_size(SFD(nextin).mLines));

            static int scrolltop = 0;
            static int cursor = 0;

            if(!sCommand_option_item(command, "-prereserve-position")) {
                scrolltop = 0;
                cursor = 0;
            }

            if(cursor < 0) {
                cursor = 0;
            }
            if(cursor >= vector_size(SFD(nextin).mLines)) {
                cursor = vector_size(SFD(nextin).mLines)-1;
                if(cursor < 0) cursor = 0;
            }
            if(cursor < scrolltop) {
                int i = cursor;
                int width_sum = 0;
                while(width_sum < maxy) {
                    char* line = vector_item(SFD(nextin).mLines, i);
                    int width = str_termlen(code, line) / maxx + 1;
                    width_sum += width;
                    i--;
                    if(i < 0) {
                        i = -2;
                        break;
                    }
                }
                
                scrolltop = i +2;
            }

            while(1) {
                /// view ///
                mclear();
                int n = scrolltop;
                int y = 0;
                while(y < maxy && n < vector_size(SFD(nextin).mLines)) {
                    int attrs = 0;
                    if(n == cursor) {
                        attrs |= kCAReverse;
                    }
                    else if(markfiles[n]) {
                        attrs |= kCACyan;
                        attrs |= kCABold;
                    }
                    if(attrs) mattron(attrs);

                    char* line = vector_item(SFD(nextin).mLines, n);

                    if(str_termlen(code, line) <= maxx) {
                        mmvprintw(y, 0, "%s", line);
                        y++;
                    }
                    else {
                        char* p = line;
                        while(p < line + strlen(line)) {
                            char one_line[BUFSIZ];
                            str_cut(code, p, maxx, one_line, BUFSIZ);
                            mmvprintw(y, 0, "%s", one_line);
                            y++;

                            p+=strlen(one_line);
                        }
                    }

                    if(attrs) {
                        mattroff();
                    }

                    n++;
                }
                mrefresh();

                /// input ///
                int meta;
                int key = mgetch(&meta);

                if(key == 14 || key == KEY_DOWN) {
                    cursor++;
                }
                else if(key == 16 || key == KEY_UP) {
                    cursor--;
                }
                else if(key == 4 || key == KEY_NPAGE) {
                    cursor += (maxy / 2);
                }
                else if(key == 21 || key == KEY_PPAGE) {
                    cursor -= (maxy /2);
                }
                else if(key == 12) {
                    mclear();
                    mrefresh();
                }
                else if(key == 'q' || key == 3 | key == 27 || key == 7) {
                    mendwin();
#if !defined(__FREEBSD__)
                    mrestore_screen();
#endif
                    mrestore_ttysettings();
                    FREE(markfiles);
                    return FALSE;
                }
                else if(key == 'a' && multiple) {
                    int i;
                    for(i=0; i<vector_size(SFD(nextin).mLines); i++) {
                        markfiles[i] = !markfiles[i];
                    }
                }
                else if(key == ' ' && multiple) {
                    markfiles[cursor] = !markfiles[cursor];
                    cursor++;
                }
                else if(key == 10 || key == 13) {
                    if(multiple) {
                        BOOL flg_mark = FALSE;
                        int k;
                        for(k=0; k<vector_size(SFD(nextin).mLines); k++) {
                            if(markfiles[k]) {
                                if(!fd_write(nextout, vector_item(SFD(nextin).mLines, k))) {
                                    err_msg("interrupt", runinfo->mSName, runinfo->mSLine);
                                    runinfo->mRCode = RCODE_SIGNAL_INTERRUPT;
                                    FREE(markfiles);
                                    return FALSE;
                                }

                                flg_mark = TRUE;
                            }
                        }

                        if(!flg_mark) {
                            if(!fd_write(nextout, vector_item(SFD(nextin).mLines, cursor))) {
                                err_msg("interrupt", runinfo->mSName, runinfo->mSLine);
                                runinfo->mRCode = RCODE_SIGNAL_INTERRUPT;
                                FREE(markfiles);
                                return FALSE;
                            }
                        }
                    }
                    else {
                        if(!fd_write(nextout, vector_item(SFD(nextin).mLines, cursor))) {
                            err_msg("interrupt", runinfo->mSName, runinfo->mSLine);
                            runinfo->mRCode = RCODE_SIGNAL_INTERRUPT;
                            FREE(markfiles);
                            return FALSE;
                        }
                    }
                    runinfo->mRCode = 0;
                    break;
                }

                if(cursor < 0) {
                    cursor = 0;
                }
                if(cursor >= vector_size(SFD(nextin).mLines)) {
                    cursor = vector_size(SFD(nextin).mLines)-1;
                    if(cursor < 0) cursor = 0;
                }
                if(cursor >= n) {
                    scrolltop = n;
                }
                if(cursor < scrolltop) {
                    int i = cursor;
                    int width_sum = 0;
                    while(width_sum < maxy) {
                        char* line = vector_item(SFD(nextin).mLines, i);
                        int width = str_termlen(code, line) / maxx + 1;
                        width_sum += width;
                        i--;
                        if(i < 0) {
                            i = -2;
                            break;
                        }
                    }
                    
                    scrolltop = i +2;
                }
            }

            FREE(markfiles);

            mendwin();
#if !defined(__FREEBSD__)
            mrestore_screen();
#endif
            mrestore_ttysettings();
        }
    }

    return TRUE;
}

BOOL cmd_p(sObject* nextin, sObject* nextout, sRunInfo* runinfo)
{
    if(isatty(0) == 0 || isatty(1) == 0) {
        err_msg("p: stdin or stdout is not a tty", runinfo->mSName, runinfo->mSLine);
        return FALSE;
    }
    if(tcgetpgrp(0) != getpgid(0)) {
        err_msg("p: not forground process group", runinfo->mSName, runinfo->mSLine);
        return FALSE;
    }
    char* term_env = getenv("TERM");
    if(term_env == NULL || strcmp(term_env, "") == 0) {
        err_msg("p: not TERM environment variable setting", runinfo->mSName, runinfo->mSLine);
        return FALSE;
    }

    sCommand* command = runinfo->mCommand;

    enum eLineField lf = kLF;

    enum eKanjiCode code = gKanjiCode;
    if(sCommand_option_item(command, "-byte")) {
        code = kByte;
    }
    else if(sCommand_option_item(command, "-utf8")) {
        code = kUtf8;
    }
    else if(sCommand_option_item(command, "-sjis")) {
        code = kSjis;
    }
    else if(sCommand_option_item(command, "-eucjp")) {
        code = kEucjp;
    }

    if(runinfo->mFilter) {
        sObject* nextin2 = FD_NEW_STACK();

        /// make control characters visible ///
        char* p = SFD(nextin).mBuf;
        while(*p) {
            if(*p == '\n') {
                if(!fd_writec(nextin2, *p)) {
                    return FALSE;
                }
                p++;
            }
            else if(*p >= 1 && *p <= 31) {
                if(!fd_writec(nextin2, '^')) {
                    return FALSE;
                }
                if(!fd_writec(nextin2, *p+'@')) {
                    return FALSE;
                }
                p++;
            }
            else {
                if(!fd_writec(nextin2, *p)) {
                    return FALSE;
                }
                p++;
            }
        }

        /// go ///
        fd_split(nextin2, lf);

        msave_ttysettings();
#if !defined(__FREEBSD__)
        msave_screen();
#endif
        minitscr();

        const int maxx = mgetmaxx();
        const int maxy = mgetmaxy();

        int* markfiles = MALLOC(sizeof(int)*vector_size(SFD(nextin2).mLines));
        memset(markfiles, 0, sizeof(int)*vector_size(SFD(nextin2).mLines));

        static int scrolltop = 0;
        static int cursor = 0;

        if(!sCommand_option_item(command, "-prereserve-position")) {
            scrolltop = 0;
            cursor = 0;
        }

        if(cursor < 0) {
            cursor = 0;
        }
        if(cursor >= vector_size(SFD(nextin2).mLines)) {
            cursor = vector_size(SFD(nextin2).mLines)-1;
            if(cursor < 0) cursor = 0;
        }
        if(cursor < scrolltop) {
            int i = cursor;
            int width_sum = 0;
            while(width_sum < maxy) {
                char* line = vector_item(SFD(nextin2).mLines, i);
                int width = str_termlen(code, line) / maxx + 1;
                width_sum += width;
                i--;
                if(i < 0) {
                    i = -2;
                    break;
                }
            }
            
            scrolltop = i +2;
        }

        while(1) {
            /// view ///
            mclear();
            int n = scrolltop;
            int y = 0;
            while(y < maxy && n < vector_size(SFD(nextin2).mLines)) {
                int attrs = 0;
                if(n == cursor) {
                    attrs |= kCAReverse;
                }
                else if(markfiles[n]) {
                    attrs |= kCACyan;
                    attrs |= kCABold;
                }
                if(attrs) mattron(attrs);

                char* line = vector_item(SFD(nextin2).mLines, n);

                if(str_termlen(code, line) <= maxx) {
                    mmvprintw(y, 0, "%s", line);
                    y++;
                }
                else {
                    char* p = line;
                    while(p < line + strlen(line)) {
                        char one_line[BUFSIZ];
                        str_cut(code, p, maxx, one_line, BUFSIZ);
                        mmvprintw(y, 0, "%s", one_line);
                        y++;

                        p+=strlen(one_line);
                    }
                }

                if(attrs) {
                    mattroff();
                }

                n++;
            }
            mrefresh();

            /// input ///
            int meta;
            int key = mgetch(&meta);

            if(key == 14 || key == KEY_DOWN) {
                cursor++;
            }
            else if(key == 16 || key == KEY_UP) {
                cursor--;
            }
            else if(key == 4 || key == KEY_NPAGE) {
                cursor += (maxy / 2);
            }
            else if(key == 21 || key == KEY_PPAGE) {
                cursor -= (maxy /2);
            }
            else if(key == 12) {
                mclear();
                mrefresh();
            }
            else if(key == 'q' || key == 3 | key == 27 || key == 7) {
                err_msg("p: canceled", runinfo->mSName, runinfo->mSLine);
                mendwin();
#if !defined(__FREEBSD__)
                mrestore_screen();
#endif
                mrestore_ttysettings();
                FREE(markfiles);
                return FALSE;
            }
            else if(key == 10 || key == 13) {
                if(!fd_write(nextout, SFD(nextin).mBuf)) {
                    err_msg("interrupt", runinfo->mSName, runinfo->mSLine);
                    runinfo->mRCode = RCODE_SIGNAL_INTERRUPT;
                    FREE(markfiles);
                    return FALSE;
                }
                runinfo->mRCode = 0;
                break;
            }

            if(cursor < 0) {
                cursor = 0;
            }
            if(cursor >= vector_size(SFD(nextin2).mLines)) {
                cursor = vector_size(SFD(nextin2).mLines)-1;
                if(cursor < 0) cursor = 0;
            }
            if(cursor >= n) {
                scrolltop = n;
            }
            if(cursor < scrolltop) {
                int i = cursor;
                int width_sum = 0;
                while(width_sum < maxy) {
                    char* line = vector_item(SFD(nextin2).mLines, i);
                    int width = str_termlen(code, line) / maxx + 1;
                    width_sum += width;
                    i--;
                    if(i < 0) {
                        i = -2;
                        break;
                    }
                }
                
                scrolltop = i +2;
            }
        }

        FREE(markfiles);

        mendwin();
#if !defined(__FREEBSD__)
        mrestore_screen();
#endif
        mrestore_ttysettings();
    }

    return TRUE;
}

BOOL cmd_readline(sObject* nextin, sObject* nextout, sRunInfo* runinfo)
{
    if(isatty(0) == 0 || isatty(1) == 0) {
        err_msg("readline: stdin or stdout is not a tty", runinfo->mSName, runinfo->mSLine);
        return FALSE;
    }
    if(tcgetpgrp(0) != getpgid(0)) {
        err_msg("readline: not forground process group", runinfo->mSName, runinfo->mSLine);
        return FALSE;
    }
    char* term_env = getenv("TERM");
    if(term_env == NULL || strcmp(term_env, "") == 0) {
        err_msg("readline: not TERM environment variable setting", runinfo->mSName, runinfo->mSLine);
        return FALSE;
    }

    char* prompt;
    sCommand* command = runinfo->mCommand;
    if(command->mArgsNumRuntime >= 2) {
        prompt = command->mArgsRuntime[1];
    }
    else {
        prompt = "> ";
    }

    char* buf = readline(prompt);

    if(!fd_write(nextout, buf)) {
        err_msg("interrupt", runinfo->mSName, runinfo->mSLine);
        runinfo->mRCode = RCODE_SIGNAL_INTERRUPT;
        return FALSE;
    }
    
    runinfo->mRCode = 0;

    return TRUE;
}

