//
// NumberField.java
// This file is part of PosterDivider.
//
package jp.sourceforge.posterdivider;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.TextEvent;
import java.awt.event.TextListener;
import java.text.DecimalFormat;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.swing.JTextField;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;

public class NumberField extends JTextField {
    private static final long serialVersionUID = 1L;
    private static Pattern pattern = Pattern.compile("(-)?[0-9,]+(\\.[0-9,]+)?");
    private static final double initValue = 0;

    private static DecimalFormat createDecimalFormat(int digit) {
	DecimalFormat decimalFormat = new DecimalFormat();
	decimalFormat.setMaximumFractionDigits(digit);
	return decimalFormat;
    }

    private double value;
    private double unitIncrement;// 上下キーを押したときに増減させる値 連続して押し続けると増減する値を増やす
    private int displayDigit;// 小数点以下表示桁数
    private DecimalFormat decimalFormat;
    private int countIncrement = 0;// キーを連続して押した回数

    private double calcValue(String s) {
	Matcher m = pattern.matcher(s);
	return m.find() ? Double.valueOf(m.group().replace(",", "")) : initValue;
    }

    private void updateText() {
	super.setText(decimalFormat.format(this.value));
    }

    private void numberField_actionPerformed() {
	updateText();
    }

    private void numberField_focusLost(FocusEvent e) {
	countIncrement = 0;
	
	//processActionEvent(new ActionEvent(e.getSource(), e.getID(), ""));
	fireActionPerformed();
    }

    private void numberField_textChanged() {
	this.value = calcValue(super.getText());
    }

    private void numberField_keyPressed(KeyEvent e) {
	switch (e.getKeyCode()) {
	case KeyEvent.VK_UP:
	    if (countIncrement >= 19) {
		setValue(this.value + this.unitIncrement * 100);
	    } else if (countIncrement >= 10) {
		setValue(this.value + this.unitIncrement * 10);
	    } else {
		setValue(this.value + this.unitIncrement);
	    }
	    countIncrement++;
	    e.consume();
	    break;
	case KeyEvent.VK_DOWN:
	    if (countIncrement >= 19) {
		setValue(this.value - this.unitIncrement * 100);
	    } else if (countIncrement >= 10) {
		setValue(this.value - this.unitIncrement * 10);
	    } else {
		setValue(this.value - this.unitIncrement);
	    }
	    countIncrement++;
	    e.consume();
	    break;
	default:
	    break;
	}
    }

    private void numberField_keyReleased(KeyEvent e) {
	countIncrement = 0;
    }

    private void numberField_keyTyped(KeyEvent e) {
	switch (e.getKeyChar()) {
	case '.':
	case '-':
	case '0':
	case '1':
	case '2':
	case '3':
	case '4':
	case '5':
	case '6':
	case '7':
	case '8':
	case '9':
	    break;
	default:
	    e.consume();
	    break;
	}
    }

    public NumberField() {
	super();
	this.value = initValue;
	this.unitIncrement = 0;
	this.displayDigit = 0;
	this.decimalFormat = createDecimalFormat(0);

	this.addActionListener(new ActionListener() {
		public void actionPerformed(ActionEvent e) {
		    numberField_actionPerformed();
		}
	    });
	this.addFocusListener(new FocusListener() {
		public void focusGained(FocusEvent e) {
		}

		public void focusLost(FocusEvent e) {
		    numberField_focusLost(e);
		}
	    });
	this.getDocument().addDocumentListener(new DocumentListener() {
		public void insertUpdate(DocumentEvent e) {
		    numberField_textChanged();
		}
		public void removeUpdate(DocumentEvent e) {
		    numberField_textChanged();
		}
		public void changedUpdate(DocumentEvent e) {
		}
	    });
	this.addKeyListener(new KeyListener() {
		public void keyPressed(KeyEvent e) {
		    numberField_keyPressed(e);
		}
		public void keyReleased(KeyEvent e) {
		    numberField_keyReleased(e);
		}
		public void keyTyped(KeyEvent e) {
		    numberField_keyTyped(e);
		}
	    });
	
	updateText();
    }

    public double getValue() {
	return this.value;
    }

    public void setValue(double value) {
	this.value = value;
	updateText();
    }

    @Override
	public void setText(String s) {
	this.value = calcValue(s);
	updateText();
    }

    public double getUnitIncrement() {
	return this.unitIncrement;
    }

    public void setUnitIncrement(double unitIncrement) {
	this.unitIncrement = unitIncrement;
    }

    public int getDisplayDigit() {
	return this.displayDigit;
    }

    public void setDisplayDigit(int displayDigit) {
	this.displayDigit = displayDigit;
	this.decimalFormat = createDecimalFormat(displayDigit);
	updateText();
    }
}
