/* 
 * Copyright (C) since 2008 NTT DATA Corporation 
 *  
 */ 

package org.postgresforest;

import java.sql.*;
import java.util.*;
import java.util.concurrent.*;
import java.lang.ref.WeakReference;

import org.postgresforest.apibase.*;
import org.postgresforest.constant.ErrorStr;
import org.postgresforest.exception.ForestException;
import org.postgresforest.util.*;

import net.jcip.annotations.*;


@NotThreadSafe public final class ForestConnection implements Connection, ForestCloseable {
    
    /** このForestConnectionに紐づくリソースを管理するコントローラ */
    private final EntrypointCommonResource epCommonResource;
    
    private final List<Connection> conList;
    
    /**
     *  このコネクションオブジェクトから生成される子を格納するリンクリスト
     *  （弱参照のリストであり、列挙→get→参照がnullなら要素を消去、という
     *  走査処理を行うためリンクリストの実装を選択している）
     */
    private final List<WeakReference<ForestCloseable>> childList = new LinkedList<WeakReference<ForestCloseable>>();
    /**
     * ForestConnectionの直接の子要素となるForestのJDBCオブジェクトを登録する。
     * この関数で登録したオブジェクトに対しては、特定タイミングで次の操作を行う
     * 1. 縮退が発生した場合に、系の閉塞処理を行う（ｘｘｘｘ関数）
     * 2. リカバリが発生した場合に、オブジェクトそのものの閉塞処理を行う（ｘｘｘｘ関数）
     * 但し、この関数での登録は、WeakReferenceとして保持することになるため、
     * 対応する参照解除の関数は存在しない。
     * （ユーザアプリケーションがStatementなどのオブジェクトを解放する際に
     * Closeなどを呼ぶことを義務付けていないため、登録解除はGCの処理に任せる）
     * @param child
     */
    protected void addChild(final ForestCloseable child) {
        final WeakReference<ForestCloseable> childRef = new WeakReference<ForestCloseable>(child);
        childList.add(childRef);
        
        // リストの無駄な拡大を防ぐため、適当な大きさになった段階でリストの整理を行う
        // 弱参照の参照先がnullになっているエントリを削除する
        // サイズが10の倍数になった時にやっているのは、単に走査作業をサボって高速化している。
        // このエントリが使われる状況は、リカバリ時や縮退時など限られた時であり、
        // 通常のStatement等の作成処理を高速化することを優先するため。
        if (childList.size() % 10 == 0) {
            ListIterator<WeakReference<ForestCloseable>> iter = childList.listIterator();
            while (iter.hasNext()) {
                if (iter.next().get() == null) {
                    iter.remove();
                }
            }
        }
    }
    
    /**
     * ForestConnectionのコンストラクタ<br>
     * このクラスはDriverクラス以外から作成されてはいけないため、protectedとしている
     * @param epCommonResource ForestConStateControllerオブジェクト
     * @param conList 
     */
    protected ForestConnection(final EntrypointCommonResource epCommonResource, final List<Connection> conList) {
        this.epCommonResource = epCommonResource;
        this.conList = conList;
    }
    
    /**
     * ForestConnectionオブジェクトのファイナライザ（オーバーライド）<br>
     * アプリケーションがcloseを閉じ忘れた場合にも、EntrypointCommonResourceの
     * releaseForestEntryPointを実行しなくてはならない（スレッドが永久に終わらない）
     * ため、その処理を実装する。ファイナライザが呼ばれるのはアプリケーション
     * スレッドとは全く別なので、メモリの可視性について細心の注意を払う必要がある
     */
    protected void finalize() throws Throwable {
        close();
        super.finalize();
    }
    
    public void commit() throws SQLException {
        if (isClosed() == true) {
            throw new ForestException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        // トランザクションの境界であることを最初に宣言してAPIを実行する
        epCommonResource.setIsTxBorder(true);
        
        final Callable<Void> task0 = new ConnectionTask.Commit(conList.get(0));
        final Callable<Void> task1 = new ConnectionTask.Commit(conList.get(1));
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void clearWarnings() throws SQLException {
        if (isClosed() == true) {
            throw new ForestException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final Callable<Void> task0 = new ConnectionTask.ClearWarnings(conList.get(0));
        final Callable<Void> task1 = new ConnectionTask.ClearWarnings(conList.get(1));
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public Statement createStatement() throws SQLException {
        if (isClosed() == true) {
            throw new ForestException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        // 行儀のよいアプリケーションのために、リカバリ最中であるならば
        // Statementを作成する直前でリカバリ完了を待ち、リソースを回復する
        epCommonResource.checkRecoveryAndRecoverObjects();
        
        final Callable<Statement> task0 = new ConnectionTask.CreateStatement(conList.get(0));
        final Callable<Statement> task1 = new ConnectionTask.CreateStatement(conList.get(1));
        final Callable<Statement> dummy = new ConnectionTask.CreateStatement(null);
        final List<Statement> stmtList = epCommonResource.executeAllApiWithPreCheck(task0, task1, dummy);
        final ForestStatement newStatement = new ForestStatement(epCommonResource, stmtList);
        // close・縮退の対象として登録する
        addChild(newStatement);
        return newStatement;
    }
    
    public Statement createStatement(int arg0, int arg1) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public Statement createStatement(int arg0, int arg1, int arg2)
            throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public boolean getAutoCommit() throws SQLException {
        if (isClosed() == true) {
            throw new ForestException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final Callable<Boolean> task0 = new ConnectionTask.GetAutoCommit(conList.get(0));
        final Callable<Boolean> task1 = new ConnectionTask.GetAutoCommit(conList.get(1));
        final List<Boolean> results = epCommonResource.executeAnyApi(task0, task1);
        return results.get((results.get(0) != null) ? 0 : 1).booleanValue();
    }
    
    public String getCatalog() throws SQLException {
        if (isClosed() == true) {
            throw new ForestException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final Callable<String> task0 = new ConnectionTask.GetCatalog(conList.get(0));
        final Callable<String> task1 = new ConnectionTask.GetCatalog(conList.get(1));
        final List<String> results = epCommonResource.executeAnyApi(task0, task1);
        return results.get((results.get(0) != null) ? 0 : 1);
    }
    
    public int getHoldability() throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public DatabaseMetaData getMetaData() throws SQLException {
        if (isClosed() == true) {
            throw new ForestException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        // 行儀のよいアプリケーションのために、リカバリ最中であるならば
        // DatabaseMetaDataを作成する直前でリカバリ完了を待ち、リソースを回復する
        epCommonResource.checkRecoveryAndRecoverObjects();
        
        final Callable<DatabaseMetaData> task0 = new ConnectionTask.GetMetaData(conList.get(0));
        final Callable<DatabaseMetaData> task1 = new ConnectionTask.GetMetaData(conList.get(1));
        final Callable<DatabaseMetaData> dummy = new ConnectionTask.GetMetaData(null);
        final List<DatabaseMetaData> dbmds = epCommonResource.executeAllApiWithPreCheck(task0, task1, dummy);
        final ForestDatabaseMetaData newMetaData = new ForestDatabaseMetaData(epCommonResource, dbmds);
        // close・縮退の対象として登録する
        addChild(newMetaData);
        return newMetaData;
    }
    
    public int getTransactionIsolation() throws SQLException {
        if (isClosed() == true) {
            throw new ForestException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final Callable<Integer> task0 = new ConnectionTask.GetTransactionIsolation(conList.get(0));
        final Callable<Integer> task1 = new ConnectionTask.GetTransactionIsolation(conList.get(1));
        final List<Integer> results = epCommonResource.executeAnyApi(task0, task1);
        return results.get((results.get(0) != null) ? 0 : 1).intValue();
    }
    
    public Map<String, Class<?>> getTypeMap() throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public SQLWarning getWarnings() throws SQLException {
        final Callable<SQLWarning> task0 = new ConnectionTask.GetWarnings(conList.get(0));
        final Callable<SQLWarning> task1 = new ConnectionTask.GetWarnings(conList.get(1));
        final List<SQLWarning> results = epCommonResource.executeAnyApi(task0, task1);
        return results.get((results.get(0) != null) ? 0 : 1);
    }
    
    public boolean isClosed() {
        return epCommonResource.getIsClosed();
    }
    
    public boolean isReadOnly() throws SQLException {
        if (isClosed() == true) {
            throw new ForestException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final Callable<Boolean> task0 = new ConnectionTask.IsReadOnly(conList.get(0));
        final Callable<Boolean> task1 = new ConnectionTask.IsReadOnly(conList.get(1));
        final List<Boolean> results = epCommonResource.executeAnyApi(task0, task1);
        return results.get((results.get(0) != null) ? 0 : 1).booleanValue();
    }
    
    public String nativeSQL(String arg0) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_IMPLEMENTED.toString());
    }
    
    public CallableStatement prepareCall(String sql) throws SQLException {
        if (isClosed() == true) {
            throw new ForestException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        // 行儀のよいアプリケーションのために、リカバリ最中であるならば
        // Statementを作成する直前でリカバリ完了を待ち、リソースを回復する
        epCommonResource.checkRecoveryAndRecoverObjects();
        
        final Callable<CallableStatement> task0 = new ConnectionTask.PrepareCall(conList.get(0), sql);
        final Callable<CallableStatement> task1 = new ConnectionTask.PrepareCall(conList.get(1), sql);
        final Callable<CallableStatement> dummy = new ConnectionTask.PrepareCall(null, null);
        final List<CallableStatement> stmtList = epCommonResource.executeAllApiWithPreCheck(task0, task1, dummy);
        final ForestCallableStatement newStatement = new ForestCallableStatement(epCommonResource, stmtList, sql);
        // close・縮退の対象として登録する
        addChild(newStatement);
        return newStatement;
    }
    
    public CallableStatement prepareCall(String arg0, int arg1, int arg2)
            throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public CallableStatement prepareCall(String arg0, int arg1, int arg2,
            int arg3) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public PreparedStatement prepareStatement(String sql) throws SQLException {
        if (isClosed() == true) {
            throw new ForestException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        // 行儀のよいアプリケーションのために、リカバリ最中であるならば
        // Statementを作成する直前でリカバリ完了を待ち、リソースを回復する
        epCommonResource.checkRecoveryAndRecoverObjects();
        
        final Callable<PreparedStatement> task0 = new ConnectionTask.PrepareStatement(conList.get(0), sql);
        final Callable<PreparedStatement> task1 = new ConnectionTask.PrepareStatement(conList.get(1), sql);
        final Callable<PreparedStatement> dummy = new ConnectionTask.PrepareStatement(null, null);
        final List<PreparedStatement> stmtList = epCommonResource.executeAllApiWithPreCheck(task0, task1, dummy);
        final ForestPreparedStatement newStatement = new ForestPreparedStatement(epCommonResource, stmtList, sql);
        // close・縮退の対象として登録する
        addChild(newStatement);
        return newStatement;
    }
    
    public PreparedStatement prepareStatement(String arg0, int arg1)
            throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public PreparedStatement prepareStatement(String arg0, int[] arg1)
            throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public PreparedStatement prepareStatement(String sql, String[] columnNames)
            throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public PreparedStatement prepareStatement(String sql, int resultSetType,
            int resultSetConcurrency) throws SQLException {
        
        if (isClosed() == true) {
            throw new ForestException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        // アップデート可能なResultSetは本JDBCではサポートしない
        if (resultSetConcurrency == ResultSet.CONCUR_UPDATABLE) {
            throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
        }
        // 行儀のよいアプリケーションのために、リカバリ最中であるならば
        // Statementを作成する直前でリカバリ完了を待ち、リソースを回復する
        epCommonResource.checkRecoveryAndRecoverObjects();
        
        final Callable<PreparedStatement> task0 = new ConnectionTask.PrepareStatement_IntInt(conList.get(0), sql, resultSetType, resultSetConcurrency);
        final Callable<PreparedStatement> task1 = new ConnectionTask.PrepareStatement_IntInt(conList.get(1), sql, resultSetType, resultSetConcurrency);
        final Callable<PreparedStatement> dummy = new ConnectionTask.PrepareStatement_IntInt(null, null, resultSetType, resultSetConcurrency);
        final List<PreparedStatement> stmtList = epCommonResource.executeAllApiWithPreCheck(task0, task1, dummy);
        final ForestPreparedStatement newStatement = new ForestPreparedStatement(epCommonResource, stmtList, sql);
        // close・縮退の対象として登録する
        addChild(newStatement);
        return newStatement;
    }
    
    public PreparedStatement prepareStatement(String sql, int resultSetType,
            int resultSetConcurrency, int resultSetHoldability)
            throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void releaseSavepoint(Savepoint savepoint) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void rollback() throws SQLException {
        if (isClosed() == true) {
            throw new ForestException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        // トランザクションの境界であることを最初に宣言してAPIを実行する
        epCommonResource.setIsTxBorder(true);
        
        final Callable<Void> task0 = new ConnectionTask.Rollback(conList.get(0));
        final Callable<Void> task1 = new ConnectionTask.Rollback(conList.get(1));
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void rollback(Savepoint savepoint) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void setAutoCommit(boolean autoCommit) throws SQLException {
        if (isClosed() == true) {
            throw new ForestException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        // setAutoCommit関数は事前のトランザクションをコミットするため、
        // トランザクションの境界であることを最初に宣言してからAPIを実行する
        epCommonResource.setIsTxBorder(true);
        
        final Callable<Void> task0 = new ConnectionTask.SetAutoCommit(conList.get(0), autoCommit);
        final Callable<Void> task1 = new ConnectionTask.SetAutoCommit(conList.get(1), autoCommit);
        epCommonResource.executeAllApi(task0, task1);
        epCommonResource.setIsAutocommit(autoCommit);
        return;
    }
    
    public void setCatalog(String catalog) throws SQLException {
        if (isClosed() == true) {
            throw new ForestException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final Callable<Void> task0 = new ConnectionTask.SetCatalog(conList.get(0), catalog);
        final Callable<Void> task1 = new ConnectionTask.SetCatalog(conList.get(1), catalog);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setHoldability(int holdability) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void setReadOnly(boolean readOnly) throws SQLException {
        if (isClosed() == true) {
            throw new ForestException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final Callable<Void> task0 = new ConnectionTask.SetReadOnly(conList.get(0), readOnly);
        final Callable<Void> task1 = new ConnectionTask.SetReadOnly(conList.get(1), readOnly);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public Savepoint setSavepoint() throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public Savepoint setSavepoint(String name) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void setTransactionIsolation(int level) throws SQLException {
        if (isClosed() == true) {
            throw new ForestException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final Callable<Void> task0 = new ConnectionTask.SetTransactionIsolation(conList.get(0), level);
        final Callable<Void> task1 = new ConnectionTask.SetTransactionIsolation(conList.get(1), level);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setTypeMap(Map<String, Class<?>> map) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    /**
     * Forestコネクションから生成した、現在有効なStatement等のオブジェクトを全て
     * 使用出来ないものとする。これはリカバリ時に
     */
    public void closeActiveStatements() {
        if (isClosed() == true) {
            return;
        }
        
        final ListIterator<WeakReference<ForestCloseable>> iter = childList.listIterator();
        while (iter.hasNext()) {
            final ForestCloseable jdbcObject = iter.next().get();
            if (jdbcObject == null) {
                iter.remove();
            } else {
                try { jdbcObject.close(); } catch (SQLException ignore) { }
            }
        }
    }
    
    public void closeOneSide(int serverId) {
        if (isClosed() == true) {
            return;
        }
        
        // まず子のStatement系を閉じる
        final ListIterator<WeakReference<ForestCloseable>> iter = childList.listIterator();
        while (iter.hasNext()) {
            final ForestCloseable jdbcObject = iter.next().get();
            if (jdbcObject == null) {
                iter.remove();
            } else {
                jdbcObject.closeOneSide(serverId);
            }
        }
        
        // 続いて子のコネクションを閉じる
        final Connection targetCon = conList.get(serverId);
        if (targetCon != null) {
            try {
                targetCon.close();
            } catch (SQLException e) { }
        }
        conList.set(serverId, null);
    }
    
    public void close() throws SQLException {
        if (isClosed() == true) {
            return;
        }
        try {
            // まず子のStatement系を閉じる
            closeActiveStatements();
            // 続いて子のConnectionを閉じる
            final Callable<Void> task0 = new ConnectionTask.Close(conList.get(0));
            final Callable<Void> task1 = new ConnectionTask.Close(conList.get(1));
            // closeは縮退対象としない。これは、close関数の中で仮に縮退などが発生すると
            // 内部から再度closeを呼ばれることになり、無限ループが発生するため
            epCommonResource.executeAllApiWithoutBroken(task0, task1);
        } finally {
            // 最終的に全てのリソースを閉じる
            epCommonResource.releaseEntryPointCommonResource();
        }
    }
    
    /**
     * リカバリ時にEntrypointCommonResourceから呼び出される関数。以前から存在している
     * コネクションの各パラメータを、新規に与えられたコネクションに適用し、以降両系で
     * 使用できるようにする
     * @param recoverServerId
     * @param newCon
     * @return パラメータの取得・適用に成功したならtrue、失敗したならfalse
     * @throws SQLException パラメータの取得・適用中のDBアクセスで失敗した場合
     */
    public boolean recovery(final int recoverServerId, final Connection newCon) throws SQLException {
        if (isClosed() == true) {
            return false;
        }
        
        final Connection srcCon = conList.get(1 - recoverServerId);
        if (srcCon != null) {
            // FIXME 別スレッドでパラメータ等をコピーするべき？
            // XXX （サポートAPI拡張時要チェック）
            // 新規に生成したコネクションの状態を動いている系のコネクションと同じにする
            newCon.setTransactionIsolation(srcCon.getTransactionIsolation());
            newCon.setAutoCommit(srcCon.getAutoCommit());
            // コネクションリストに新規に作ったコネクションを登録する
            conList.set(recoverServerId, newCon);
            return true;
        }
        return false;
    }
}
