/* 
 * Copyright (C) since 2008 NTT DATA Corporation 
 *  
 */ 

package org.postgresforest;

import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;
import java.math.BigDecimal;
import java.net.URL;
import java.sql.Array;
import java.sql.Blob;
import java.sql.Clob;
import java.sql.Date;
import java.sql.ParameterMetaData;
import java.sql.PreparedStatement;
import java.sql.Ref;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Time;
import java.sql.Timestamp;
import java.util.Arrays;
import java.util.Calendar;
import java.util.List;
import java.util.concurrent.Callable;

import net.jcip.annotations.NotThreadSafe;

import org.postgresforest.apibase.*;
import org.postgresforest.constant.ErrorStr;
import org.postgresforest.exception.*;
import org.postgresforest.util.*;

@NotThreadSafe public class ForestPreparedStatement extends ForestStatement implements PreparedStatement {
    
    protected final String executeSql;
    
    public ForestPreparedStatement(EntrypointCommonResource epCommonResource, List<? extends PreparedStatement> stmts, String executeSql) {
        super(epCommonResource, stmts);
        this.executeSql = executeSql;
    }
    
    public void addBatch() throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void clearParameters() throws SQLException {
        checkObjectClosed();
        final Callable<Void> task0 = new PreparedStatementTask.ClearParameters((PreparedStatement) stmts.get(0));
        final Callable<Void> task1 = new PreparedStatementTask.ClearParameters((PreparedStatement) stmts.get(1));
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public boolean execute() throws SQLException {
        checkObjectClosed();
        // 行儀の悪いアプリケーション（トランザクションを跨ってStatementを使いまわすアプリ）
        // のために、リカバリ最中で、かつトランザクションの境界であるならば、executeの
        // 実行直前でリカバリ完了を待ち、リソースの回復をする。この場合、このStatementは
        // 閉塞されているためエラーが返る
        epCommonResource.checkRecoveryAndRecoverObjects();
        
        // FIXME （仕様調整） Tx関連のDCLはexecute系APIでは許可しないようにするべきか？
        
        final Callable<Boolean> task0 = new PreparedStatementTask.Execute((PreparedStatement) stmts.get(0));
        final Callable<Boolean> task1 = new PreparedStatementTask.Execute((PreparedStatement) stmts.get(1));
        final Callable<Boolean> dummy = new PreparedStatementTask.Execute(null);
        final List<Boolean> result = epCommonResource.executeAllApiWithPreCheck(task0, task1, dummy);
        
        // 次のgetUpdateCountなどの固定系実行のAPIのために実行成功した系をセットする
        // 優先実行系で結果が取得できていればその系を次の実行系IDとし、そうでなければ
        // 逆のサーバID（0/1の2値なので1の補数を取ればよい）次の実行系とする
        final int prefServerId = epCommonResource.getPreferentialExecServerId();
        setFixExecuteId( (result.get(prefServerId) != null) ? prefServerId : (1 - prefServerId) );
        return result.get(getFixExecuteId()).booleanValue();
    }
    
    public ResultSet executeQuery() throws SQLException {
        checkObjectClosed();
        // 行儀の悪いアプリケーション（トランザクションを跨ってStatementを使いまわすアプリ）
        // のために、リカバリ最中で、かつトランザクションの境界であるならば、executeの
        // 実行直前でリカバリ完了を待ち、リソースの回復をする。この場合、このStatementは
        // 閉塞されているためエラーが返る
        epCommonResource.checkRecoveryAndRecoverObjects();
        
        // 全系実行か任意一系実行かの振り分けを行う（パース）
        final boolean isAllExec;
        switch (parseQuery(executeSql)) {
            case SELECT: {
                isAllExec = false;
                break;
            }
            case COMMIT: {
                epCommonResource.setIsTxBorder(true);
                isAllExec = true;
                break;
            }
            default: {
                isAllExec = true;
                break;
            }
        }
        
        final Callable<ResultSet> task0 = new PreparedStatementTask.ExecuteQuery((PreparedStatement) stmts.get(0));
        final Callable<ResultSet> task1 = new PreparedStatementTask.ExecuteQuery((PreparedStatement) stmts.get(1));
        final Callable<ResultSet> dummy = new PreparedStatementTask.ExecuteQuery(null);
        final List<ResultSet> results;
        
        if (isAllExec) {
            // 全系実行の場合
            results = epCommonResource.executeAllApiWithPreCheck(task0, task1, dummy);
            
        } else {
            // 任意1系実行の場合
            results = epCommonResource.executeAnyApiWithPreCheck(task0, task1, dummy);
        }
        // 優先実行系の結果が存在していればそちらを使い、結果がない場合は逆を使う。
        // ここで取得したResultSetを基にForestResultSetを作成し、固定実行系を定める。
        final int prefExecId = epCommonResource.getPreferentialExecServerId();
        setFixExecuteId( (results.get(prefExecId) != null) ? prefExecId : 1 - prefExecId );
        final ForestResultSet newResult = new ForestResultSet(epCommonResource, results, getFixExecuteId());
        setForestResultSet(newResult);
        return newResult;
    }
    
    public int executeUpdate() throws SQLException {
        checkObjectClosed();
        // 行儀の悪いアプリケーション（トランザクションを跨ってStatementを使いまわすアプリ）
        // のために、リカバリ最中で、かつトランザクションの境界であるならば、executeの
        // 実行直前でリカバリ完了を待ち、リソースの回復をする。この場合、このStatementは
        // 閉塞されているためエラーが返る
        epCommonResource.checkRecoveryAndRecoverObjects();
        
        // SQLがCOMMITなら、Txの境目であることを宣言する
        if (parseQuery(executeSql) == SQLType.COMMIT) {
            epCommonResource.setIsTxBorder(true);
        }
        
        final Callable<Integer> task0 = new PreparedStatementTask.ExecuteUpdate((PreparedStatement) stmts.get(0));
        final Callable<Integer> task1 = new PreparedStatementTask.ExecuteUpdate((PreparedStatement) stmts.get(1));
        final Callable<Integer> dummy = new PreparedStatementTask.ExecuteUpdate(null);
        final List<Integer> result = epCommonResource.executeAllApiWithPreCheck(task0, task1, dummy);
        
        // 次のgetUpdateCountなどの固定系実行のAPIのために実行系をセットする
        // 優先実行系で結果が取得できていればその系を次の実行系とし、そうでなければ
        // 逆のサーバID（0/1の2値なので1の補数を取ればよい）次の実行系とする
        final int prefServerId = epCommonResource.getPreferentialExecServerId();
        setFixExecuteId( (result.get(prefServerId) != null) ? prefServerId : (1 - prefServerId) );
        return result.get(getFixExecuteId());
    }
    
    public ResultSetMetaData getMetaData() throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public ParameterMetaData getParameterMetaData() throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void setArray(int arg0, Array arg1) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void setAsciiStream(int arg0, InputStream arg1, int arg2) throws SQLException {
        checkObjectClosed();
        
        final List<InputStream> streamList;
        try {
            streamList = SimpleCopyParallelInputStream.createParallelInputStream(arg1, arg2);
        } catch (IOException e) {
            final ForestException exp = new ForestException(ErrorStr.STREAM_DUPLICATE_FAILED.toString());
            exp.setMultipleCause( Arrays.<Exception>asList(e, null) );
            throw exp;
        }
        
        final Callable<Void> task0 = new PreparedStatementTask.SetAsciiStream((PreparedStatement) stmts.get(0), arg0, streamList.get(0), arg2);
        final Callable<Void> task1 = new PreparedStatementTask.SetAsciiStream((PreparedStatement) stmts.get(1), arg0, streamList.get(1), arg2);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setBigDecimal(int arg0, BigDecimal arg1) throws SQLException {
        checkObjectClosed();
        
        final Callable<Void> task0 = new PreparedStatementTask.SetBigDecimal((PreparedStatement) stmts.get(0), arg0, arg1);
        final Callable<Void> task1 = new PreparedStatementTask.SetBigDecimal((PreparedStatement) stmts.get(1), arg0, arg1);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setBinaryStream(int arg0, InputStream arg1, int arg2) throws SQLException {
        checkObjectClosed();
        
        final List<InputStream> streamList;
        try {
            streamList = SimpleCopyParallelInputStream.createParallelInputStream(arg1, arg2);
        } catch (IOException e) {
            final ForestException exp = new ForestException(ErrorStr.STREAM_DUPLICATE_FAILED.toString());
            exp.setMultipleCause( Arrays.<Exception>asList(e, null) );
            throw exp;
        }
        
        final Callable<Void> task0 = new PreparedStatementTask.SetBinaryStream((PreparedStatement) stmts.get(0), arg0, streamList.get(0), arg2);
        final Callable<Void> task1 = new PreparedStatementTask.SetBinaryStream((PreparedStatement) stmts.get(1), arg0, streamList.get(1), arg2);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setBlob(int arg0, Blob arg1) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void setBoolean(int arg0, boolean arg1) throws SQLException {
        checkObjectClosed();
        
        final Callable<Void> task0 = new PreparedStatementTask.SetBoolean((PreparedStatement) stmts.get(0), arg0, arg1);
        final Callable<Void> task1 = new PreparedStatementTask.SetBoolean((PreparedStatement) stmts.get(1), arg0, arg1);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setByte(int arg0, byte arg1) throws SQLException {
        checkObjectClosed();
        
        final Callable<Void> task0 = new PreparedStatementTask.SetByte((PreparedStatement) stmts.get(0), arg0, arg1);
        final Callable<Void> task1 = new PreparedStatementTask.SetByte((PreparedStatement) stmts.get(1), arg0, arg1);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setBytes(int arg0, byte[] arg1) throws SQLException {
        checkObjectClosed();
        
        final Callable<Void> task0 = new PreparedStatementTask.SetBytes((PreparedStatement) stmts.get(0), arg0, arg1);
        final Callable<Void> task1 = new PreparedStatementTask.SetBytes((PreparedStatement) stmts.get(1), arg0, arg1);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setCharacterStream(int arg0, Reader arg1, int arg2)
            throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void setClob(int arg0, Clob arg1) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void setDate(int arg0, Date arg1) throws SQLException {
        checkObjectClosed();
        
        final Callable<Void> task0 = new PreparedStatementTask.SetDate((PreparedStatement) stmts.get(0), arg0, arg1);
        final Callable<Void> task1 = new PreparedStatementTask.SetDate((PreparedStatement) stmts.get(1), arg0, arg1);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setDate(int arg0, Date arg1, Calendar arg2) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void setDouble(int arg0, double arg1) throws SQLException {
        checkObjectClosed();
        
        final Callable<Void> task0 = new PreparedStatementTask.SetDouble((PreparedStatement) stmts.get(0), arg0, arg1);
        final Callable<Void> task1 = new PreparedStatementTask.SetDouble((PreparedStatement) stmts.get(1), arg0, arg1);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setFloat(int arg0, float arg1) throws SQLException {
        checkObjectClosed();
        
        final Callable<Void> task0 = new PreparedStatementTask.SetFloat((PreparedStatement) stmts.get(0), arg0, arg1);
        final Callable<Void> task1 = new PreparedStatementTask.SetFloat((PreparedStatement) stmts.get(1), arg0, arg1);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setInt(int arg0, int arg1) throws SQLException {
        checkObjectClosed();
        
        final Callable<Void> task0 = new PreparedStatementTask.SetInt((PreparedStatement) stmts.get(0), arg0, arg1);
        final Callable<Void> task1 = new PreparedStatementTask.SetInt((PreparedStatement) stmts.get(1), arg0, arg1);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setLong(int arg0, long arg1) throws SQLException {
        checkObjectClosed();
        
        final Callable<Void> task0 = new PreparedStatementTask.SetLong((PreparedStatement) stmts.get(0), arg0, arg1);
        final Callable<Void> task1 = new PreparedStatementTask.SetLong((PreparedStatement) stmts.get(1), arg0, arg1);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setNull(int arg0, int arg1) throws SQLException {
        checkObjectClosed();
        
        final Callable<Void> task0 = new PreparedStatementTask.SetNull((PreparedStatement) stmts.get(0), arg0, arg1);
        final Callable<Void> task1 = new PreparedStatementTask.SetNull((PreparedStatement) stmts.get(1), arg0, arg1);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setNull(int arg0, int arg1, String arg2) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void setObject(int arg0, Object arg1) throws SQLException {
        checkObjectClosed();
        
        final Callable<Void> task0 = new PreparedStatementTask.SetObject_IntObj((PreparedStatement) stmts.get(0), arg0, arg1);
        final Callable<Void> task1 = new PreparedStatementTask.SetObject_IntObj((PreparedStatement) stmts.get(1), arg0, arg1);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setObject(int arg0, Object arg1, int arg2) throws SQLException {
        checkObjectClosed();
        
        final Callable<Void> task0 = new PreparedStatementTask.SetObject_IntObjInt((PreparedStatement) stmts.get(0), arg0, arg1, arg2);
        final Callable<Void> task1 = new PreparedStatementTask.SetObject_IntObjInt((PreparedStatement) stmts.get(1), arg0, arg1, arg2);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setObject(int arg0, Object arg1, int arg2, int arg3)
            throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void setRef(int arg0, Ref arg1) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void setShort(int arg0, short arg1) throws SQLException {
        checkObjectClosed();
        
        final Callable<Void> task0 = new PreparedStatementTask.SetShort((PreparedStatement) stmts.get(0), arg0, arg1);
        final Callable<Void> task1 = new PreparedStatementTask.SetShort((PreparedStatement) stmts.get(1), arg0, arg1);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setString(int arg0, String arg1) throws SQLException {
        checkObjectClosed();
        
        final Callable<Void> task0 = new PreparedStatementTask.SetString((PreparedStatement) stmts.get(0), arg0, arg1);
        final Callable<Void> task1 = new PreparedStatementTask.SetString((PreparedStatement) stmts.get(1), arg0, arg1);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setTime(int arg0, Time arg1) throws SQLException {
        checkObjectClosed();
        
        final Callable<Void> task0 = new PreparedStatementTask.SetTime((PreparedStatement) stmts.get(0), arg0, arg1);
        final Callable<Void> task1 = new PreparedStatementTask.SetTime((PreparedStatement) stmts.get(1), arg0, arg1);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setTime(int arg0, Time arg1, Calendar arg2) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void setTimestamp(int arg0, Timestamp arg1) throws SQLException {
        checkObjectClosed();
        
        final Callable<Void> task0 = new PreparedStatementTask.SetTimestamp((PreparedStatement) stmts.get(0), arg0, arg1);
        final Callable<Void> task1 = new PreparedStatementTask.SetTimestamp((PreparedStatement) stmts.get(1), arg0, arg1);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setTimestamp(int arg0, Timestamp arg1, Calendar arg2)
            throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void setURL(int arg0, URL arg1) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void setUnicodeStream(int arg0, InputStream arg1, int arg2)
            throws SQLException {
        checkObjectClosed();
        
        final List<InputStream> streamList;
        try {
            streamList = SimpleCopyParallelInputStream.createParallelInputStream(arg1, arg2);
        } catch (IOException e) {
            final ForestException exp = new ForestException(ErrorStr.STREAM_DUPLICATE_FAILED.toString());
            exp.setMultipleCause( Arrays.<Exception>asList(e, null) );
            throw exp;
        }
        
        final Callable<Void> task0 = new PreparedStatementTask.SetUnicodeStream((PreparedStatement) stmts.get(0), arg0, streamList.get(0), arg2);
        final Callable<Void> task1 = new PreparedStatementTask.SetUnicodeStream((PreparedStatement) stmts.get(1), arg0, streamList.get(1), arg2);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
}
