/* 
 * Copyright (C) since 2008 NTT DATA Corporation 
 *  
 */ 

package org.postgresforest.exception;

import java.sql.SQLException;
import java.io.*;
import java.util.*;

/**
 * ForestJDBCドライバが出す例外の基底クラス<br>
 * <br>
 * 本例外クラスは、multipleCauseメンバを持つ。基本的にForestJDBCドライバは
 * ２台のデータベースにアクセスをし、その結果を得る。そのため、２台の
 * データベースから異なる例外が返却される場合がありうる。
 * これらの例外を「setMultipleCause()」で格納し、printStackTraceの中では
 * 両方の例外を出力することで、２つの系で発生した異なる問題を、両方共に
 * 見ることができる。
 */
@SuppressWarnings("serial")
public class ForestException extends SQLException {
    
    public ForestException() {
    }
    
    public ForestException(String arg0) {
        super(arg0);
    }
    
    public ForestException(String reason, String SQLState) {
        super(reason, SQLState);
    }
    
    public ForestException(String reason, String SQLState, int vendorCode) {
        super(reason, SQLState, vendorCode);
    }
    
    private volatile String subMessage = "";
    public String toString() {
        return super.toString() + subMessage;
    }
    
    /**
     * Forestとして２台のDBから同時に例外が返ってきた場合などに
     * その例外を格納するためのメソッド。printStackTrace関数は、ここで
     * 指定された例外を「Caused by」の形でスタックトレースの最後に出力する。
     * @param exceptions
     */
    public void setMultipleCause(List<Exception> exceptions) {
        
        try {
            // this.toString()メソッド用に、子の例外のtoStringのみを取り出す
            final StringBuilder sb = new StringBuilder();
            for (int i = 0; i < exceptions.size(); i++) {
                sb.append("\n\t(ForestJDBC multiple cause ");
                sb.append(i);
                sb.append(") ");
                if (exceptions.get(i) == null) {
                    sb.append(" not exception");
                } else {
                    sb.append(exceptions.get(i).toString());
                }
            }
            subMessage = sb.toString();
            
            // 現在のスタック状態を取り出し、後ろから順次参照するためにLinkedListにする
            final LinkedList<StackTraceElement> mainElem = new LinkedList<StackTraceElement>();
            Collections.addAll(mainElem, getStackTrace());
            
            for(Throwable e : exceptions) {
                if (e == null)
                    continue;
                while (true) {
                    // 保持している例外からもスタック情報を取り出し、LinkedListにする
                    final LinkedList<StackTraceElement> expElem = 
                        new LinkedList<StackTraceElement>(Arrays.<StackTraceElement>asList(e.getStackTrace()));
                    
                    // スタックトレースの最後の位置（つまりMain文に当たる部分）から順に
                    // 比較を行い、同じである限り、対象の例外のスタックトレースを削り続ける
                    final ListIterator<StackTraceElement> mainIter = mainElem.listIterator(mainElem.size());
                    while (mainIter.hasPrevious()) {
                        if (mainIter.previous().equals(expElem.getLast())) {
                            expElem.removeLast();
                        } else {
                            break;
                        }
                    }
                    
                    // 不要部分を削り取ったスタックトレースから、再度配列を構成し、例外に再セットする
                    final StackTraceElement[] newStackTrace = expElem.toArray(new StackTraceElement[0]);
                    e.setStackTrace(newStackTrace);
                    
                    // さらにcauseがあれば、causeからもスタックを削り取る
                    if (e.getCause() != null && e.getCause() != e) {
                        e = e.getCause();
                        continue;
                    } else {
                        break;
                    }
                }
            }
            
            // 削り取ったスタック情報を基に、MultipleCauseのtoStringに含めるための
            // 原因事象のスタックトレースを生成する
            String multipleCauseStr = "";
            for (int i = 0; i < exceptions.size(); i++) {
                final ByteArrayOutputStream baos = new ByteArrayOutputStream();
                final PrintWriter pw = new PrintWriter(baos);
                if (exceptions.get(i) == null) {
                    pw.println("not exception");
                } else {
                    exceptions.get(i).printStackTrace(pw);
                }
                pw.flush();
                final String stackTrace = baos.toString();
                multipleCauseStr += stackTrace.replaceAll("\n", "\n\t(ForestJDBC multiple cause " + i + ") ")
                                              .replaceAll("^",  "\n\t(ForestJDBC multiple cause " + i + ") ");
            }
            initCause(new ForestException.ForestMultipleCause(multipleCauseStr));
        } catch (RuntimeException e) {
            // メッセージ文字列生成中に例外が発生した場合、
            // 例外文字列の格納は諦めざるを得ない・・
            // initCause(new ForestException.ForestMultipleCause(exceptions));
        }
        
    }
    
    /**
     * Forestオプションとして、2台のDBから同時に例外が返ってきた場合などに
     * その例外を格納するためのメソッド。printStackTrace関数は、ここで
     * 指定された例外を「Caused by」の形でスタックトレースの最後に出力する。
     * @param exceptions
     */
    public void setMultipleCause(Exception[] exceptions) {
        setMultipleCause(java.util.Arrays.<Exception>asList(exceptions));
    }
    
    @SuppressWarnings("serial")
    public static class ForestMultipleCause extends Exception {
        private final String message;
        public ForestMultipleCause(final String causeString) {
            super();
            final StringBuilder sb = new StringBuilder();
            sb.append("複数例外を原因としたForestの例外");
            sb.append(causeString);
            message = sb.toString();
        }
        public String getMessage() {
            return message;
        }
    }
}
