/* 
 * Copyright (C) since 2008 NTT DATA Corporation 
 *  
 */ 

package org.postgresforest.util;

import java.util.*;
import java.util.concurrent.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.postgresforest.constant.ConstStr;
import org.postgresforest.constant.ErrorStr;
import org.postgresforest.exception.NonForestUrlException;
import org.postgresforest.exception.ForestInvalidUrlException;

import net.jcip.annotations.*;

/**
 * Forest用URLの各要素（IP:PORT、DBNAME、OPTION）を格納するImmutableクラス。
 * クラスメソッドのparseUrlを使用してインスタンスを作成する。
 * 接続先が等しいかどうかは、本クラスのインスタンス同士を比較（equals()）
 * することで判断できる。
 */
@Immutable
public final class ForestUrl {
    private final String ipports_dbname;
    
    private final List<PgUrl> mngDbs = new CopyOnWriteArrayList<PgUrl>();
    /** 管理情報DBへのPgUrlオブジェクトの、並行アクセス可能・変更不可能なリストを返す */
    public List<PgUrl> getMngDbUrls() {
        return Collections.unmodifiableList(mngDbs);
    }
    
    /** このForestUrlが保持するIP:PORT,IP:PORT/DBの文字列表現 */
    @Override public String toString() {
        return ipports_dbname;
    }
    
    /**
     * このインスタンス同士の論理的な同値性を比較をする。<br>
     * "IP1:PORT1,IP2:PORT2/DBNAME" の文字列が等しいことを、
     * ForestUrlが等しいと定義する（設計書参照）
     */
    @Override public boolean equals(Object obj) {
        if (obj instanceof ForestUrl) {
            return ipports_dbname.equals(((ForestUrl) obj).ipports_dbname);
        } else {
            return false;
        }
    }
    
    /** 
     * ハッシュコードを返却する<br>
     * equlasで等しいと判断されるインスタンス同士では、必ず同じハッシュ
     * コードが返る。<br>
     * （EffectiveJava等、Javaの情報参照）equalsをオーバーライドしている
     * ため、hashCodeのオーバーライドは必須
     */
    @Override public int hashCode() {
        return ipports_dbname.hashCode();
    }
    
    /** パース用パターン （IP/PORT/DB名/オプション分割用）*/
    private static final Pattern patternForestUrl = Pattern.compile("^(.+):(.+),(.+):(.+)/([^\\?]+)(\\?(.+))?$");
    /** パース用パターン （IP確認用）*/
    private static final Pattern patternIp = Pattern.compile("[A-Za-z0-9\\.\\-_]+");
    /** パース用パターン （DB名確認用。先頭はアルファベット、２文字目以降アルファベット＋数字＋＿＋＄）*/
    private static final Pattern patternDb = Pattern.compile("[A-Za-z][A-Za-z0-9\\$_]*");
    
    /**
     * 接続文字列のパースを行い、ForestUrlオブジェクトとオプションを格納した
     * Propertiesをペアで返却する
     * @param url このJDBCドライバの接続文字列（外部仕様定義書参照）
     * @return 生成したForestUrlインスタンスと、オプションのペア
     * @throws NonForestUrlException 接続文字列がForestのものではなかった場合
     * @throws ForestInvalidUrlException 与えられた接続文字列が適切でなかった場合
     */
    public static Pair2<ForestUrl,Properties> parseUrl(final String url) throws ForestInvalidUrlException, NonForestUrlException {
        if (!url.startsWith(ConstStr.FOREST_URL_PREFIX.toString())) {
            throw new NonForestUrlException(ErrorStr.INVALID_FORESTURL.toString() + " : " + url);
        }
        final String ipports_db_opt = url.substring(ConstStr.FOREST_URL_PREFIX.toString().length());
        // IP,PORT,IP,PORT,DBNAME,OPTIONへの分解
        final Matcher matcherForest = patternForestUrl.matcher(ipports_db_opt);
        if (matcherForest.matches() == false) {
            throw new ForestInvalidUrlException(ErrorStr.INVALID_FORESTURL.toString() + " : " + url);
        }
        final String[] ips = {matcherForest.group(1), matcherForest.group(3)};
        final String[] ports = {matcherForest.group(2), matcherForest.group(4)};
        final String dbname = matcherForest.group(5);
        // OPTIONは存在していれば頭の？を除いた部分、存在していなければ空文字列とする
        final String options = (matcherForest.group(7) != null) ? matcherForest.group(7) : "";
        
        // IPのチェック（無効な文字列が存在しないかのチェックをするレベル）
        for (final String ip : ips) {
            final Matcher matcherIp = patternIp.matcher(ip);
            if (matcherIp.matches() == false) {
                throw new ForestInvalidUrlException(ErrorStr.INVALID_IPADDR.toString() + " : " + ip);
            }
        }
        // Portのチェック
        for (final String port : ports) {
            try {
                final int intPort = Integer.parseInt(port);
                if (0 < intPort && intPort < 65536) {
                    continue;
                }
            } catch (NumberFormatException e) {
            }
            throw new ForestInvalidUrlException(ErrorStr.INVALID_PORT.toString() + " : " + port);
        }
        
        // dbnameのチェック
        final Matcher matcherDb = patternDb.matcher(dbname);
        if (matcherDb.matches() == false) {
            throw new ForestInvalidUrlException(ErrorStr.INVALID_DBNAME.toString() + " : " + dbname);
        }
        
        // IP,PORTを接続してIP:PORTの形にし、List化する
        final List<String> ipports = new ArrayList<String>(2);
        for (int i = 0; i < ips.length; i++) {
            ipports.add(ips[i] + ":" + ports[i]);
        }
        // IP,PORTの重複チェック
        final Set<String> tmpIpportsSet = new HashSet<String>(2);
        for (final String ipport : ipports) {
            tmpIpportsSet.add(ipport);
        }
        if (ipports.size() != tmpIpportsSet.size()) {
            throw new ForestInvalidUrlException(ErrorStr.SAME_ADDRESS.toString());
        }
        // ForestUrlを生成する
        final ForestUrl forestUrl = new ForestUrl(ipports, dbname);
        
        // optionを分解してPropertyを構成する
        final Properties prop = new Properties();
        if (options.length() != 0) {
            for (final String opt : options.split("&")) {
                final String[] elem = opt.split("=");
                if (elem.length == 2 && prop.getProperty(elem[0]) == null) {
                    prop.setProperty(elem[0], elem[1]);
                } else {
                    // xxx=yyyの形ではない場合、あるいは既にキーがプロパティに存在している場合は例外
                    throw new ForestInvalidUrlException(ErrorStr.INVALID_OPTION.toString() + " : " + opt);
                }
            }
        }
        
        // 生成したForestUrlとPropertiesを返却する
        return new Pair2<ForestUrl, Properties>(forestUrl, prop);
    }
    
    /** コンストラクタ。このコンストラクタはparseUrlからのみ呼ばれる */
    private ForestUrl(final List<String> ipport, final String dbname) {
        Collections.sort(ipport);
        this.ipports_dbname = ipport.get(0) + "," + ipport.get(1) + "/" + dbname;
        
        /**
         * 現在管理DB接続の際のユーザ・パスは仕様変更されているためここのユーザ名や
         * パスワードは仮のもの（IP・ポートをmngDbsに投入するための仮ユーザ名・パスワード。
         * ユーザ名・パスワードについては実際には別の値が現在は使われている）
         * // TODO （管理情報user/pass）管理DBへのユーザ・パスワードを変更
         */
        // 管理データベースへ接続するためのURLオブジェクトを作成
        // 管理データベースに接続する際のUSER・PASSWORDは、管理DB名と同じ（外部設計書参照）
        final String mngdbname = dbname + ConstStr.MNGDB_SUFFIX.toString();
        mngDbs.add(new PgUrl(ipport.get(0), mngdbname, mngdbname, mngdbname));
        mngDbs.add(new PgUrl(ipport.get(1), mngdbname, mngdbname, mngdbname));
    }
    
}
