#!/bin/bash
#
# Online recovery script for PostgreSQL HA
#
# <description>
#  This script is designed for PostgreSQL HA to recover database instance faulted.
#
# <usage>
#  PGHARecovery.sh -all | -base | -closevm | -sync | -recovery | -openvm | -cleanup | -cancel configfile
#
# <return>
#  0 = OK, 1 = FATAL
#
# <dependency>
#  ~/.pgpass PG_HOME/share/recovery.conf.sample
#
# $Id: PGHARecovery.sh 2959 2010-02-19 09:53:42Z takahatat $

PROG=`basename $0`
DIR=$(cd $(dirname $0) && pwd)/$(basename "$FILE")
USER=`/usr/bin/whoami`
HOST=`hostname`

########################################
# Definition
########################################

PGCONFIG_ONLINE="postgresql.conf.online"
MAX_DURATION_ADD=10

#
# messages
#
MSG_I001="online recovery (base) started"
MSG_I002="- reloading postgresql.conf for recovery..."
MSG_I003="- executing checkpoint..."
MSG_I004="- executing pg_start_backup()..."
MSG_I005="- executing base_recovery_command..."
MSG_I006="online recovery (base) successful"
MSG_I007="online recovery (sync) started"
MSG_I008="- changing server_info..."
MSG_I009="- waiting for pghainfo cache to be refreshed..."
MSG_I010="- waiting for busy transaction(s) to idle..."
MSG_I011="pid(s) remaining"
MSG_I012="- executing pg_stop_backup()..."
MSG_I013="start WAL location"
MSG_I014="stop WAL location"
MSG_I015="- waiting for stop WAL location to be archived..."
MSG_I016="- executing sync_recovery_command..."
MSG_I017="- executing instance_startup_command..."
MSG_I018="- setting recovery.conf..."
MSG_I019="- removing recovery.conf..."
MSG_I020="online recovery (sync) successful"
MSG_I021="- resetting postgresql.conf..."
MSG_I022="updated server_info.udb_validity"
MSG_I023="online recovery (cleanup) started"
MSG_I024="online recovery (cleanup) successful"
MSG_I025="- reloading postgresql.conf for online..."
MSG_I026="- removing archived WAL file(s)..."
MSG_I027="current configuration"
MSG_I028="reloaded configuration"
MSG_I029="kill SIGHUP process"
MSG_I030="- getting max serverinfo_read_duration..."
MSG_I031="max serverinfo_read_duration"
MSG_I032="loading serverinfo_read_duration"
MSG_I033="online recovery (closevm) started"
MSG_I034="online recovery (closevm) successful"
MSG_I035="online recovery (openvm) started"
MSG_I036="online recovery (openvm) successful"
MSG_I037="online recovery (recovery) started"
MSG_I038="online recovery (recovery) successful"
MSG_I039="loading udb_url"
MSG_I040="NOTICE : execute -cleanup on recovery node"
MSG_I041="- removing ${PGCONFIG_ONLINE}..."
MSG_I042="online recovery (cancel) started"
MSG_I043="online recovery (cancel) successful"
MSG_I044="- executing env_check_command..."

MSG_E001="usage : ${PROG} -all | -base | -closevm | -sync | -recovery | -openvm | -cleanup | -cancel configfile"
MSG_E002="failed to copy file(s)"
MSG_E003="failed to kill SIGHUP process" 
MSG_E004="file not found"
MSG_E005="failed to execute checkpoint"
MSG_E006="failed to configure file(s)"
MSG_E007="failed to execute pg_start_backup()"
MSG_E008="failed to execute base_recovery_command"
MSG_E009="online recovery (base) failed"
MSG_E010="failed to get database name like %_pghainfo"
MSG_E011="failed to change server_info.udb_validity"
MSG_E012="timeout. there is busy transaction(s)"
MSG_E013="failed to execute pg_stop_backup()"
MSG_E014="timeout. WAL file is not archived."
MSG_E015="failed to execute sync_recovery_command"
MSG_E016="failed to execute instance_startup_command"
MSG_E017="failed to remove file(s)"
MSG_E018="online recovery (sync) failed"
MSG_E019="online recovery (cleanup) failed"
MSG_E020="failed to load serverinfo_read_duration"
MSG_E021="online recovery (closevm) failed"
MSG_E022="online recovery (openvm) failed"
MSG_E023="online recovery (recovery) failed"
MSG_E024="online recovery (cancel) failed"
MSG_E025="failed to execute env_check_command"
MSG_E026="invalid file exist. (maybe forgot to execute this script with '-cleanup' or '-cancel' option)"
MSG_E027="invalid HA status. there are many master database instances."
MSG_E028="invalid host. this database instance is not master."
MSG_E029="failed to execute psql"
MSG_E030="invalid file permission"

########################################
# Function
########################################

# base recovery
function BaseRecovery() {

    # if pg_start_backup() already is already executed
    if [ ! -e ${dbcluster_dir}/backup_label ]
    then
        #
        # reload postgresql.conf for base recovery
        #
        Logging "${MSG_I002}"

        # backup postgresql.conf & edit postgresql.conf (replace archive_command)
        if [ ! -e ${dbcluster_dir}/${PGCONFIG_ONLINE} ]
        then
            cp ${dbcluster_dir}/postgresql.conf ${dbcluster_dir}/${PGCONFIG_ONLINE} && \
            sed -e "s/^archive_command/#archive_command/" ${dbcluster_dir}/${PGCONFIG_ONLINE} > ${dbcluster_dir}/postgresql.conf && \
            echo "archive_command = '${archive_command}'" >> ${dbcluster_dir}/postgresql.conf
            if [ $? -ne 0 ]
            then
                EchoNG
                Logging "${MSG_E006} : ${dbcluster_dir}/postgresql.conf"
                return 1
            fi
        else
            Logging "${MSG_E026} : ${dbcluster_dir}/${PGCONFIG_ONLINE}"
            return 1
        fi

        ReloadConfiguration || return 1

        #
        # execute CHECKPOINT
        #
        Logging "${MSG_I003}"
        ${pg_dir}/bin/psql -U ${pg_user} -d ${pg_db} -p ${pg_port} -q -c "CHECKPOINT" > /dev/null
        if [ $? -ne 0 ]
        then
            EchoNG
            Logging "${MSG_E005}"
            return 1
        fi
        EchoOK

        #
        # execute pg_start_backup
        #
        Logging "${MSG_I004}"
        XLOG_START=`${pg_dir}/bin/psql -U ${pg_user} -d ${pg_db} -p ${pg_port} -q -A -t -c "SELECT pg_xlogfile_name(pg_start_backup('pgha_recovery'))" | awk '{print $1}' | head -n 1`
        if [ $? -ne 0 ] || [ "x${XLOG_START}" == "x" ]
        then
            EchoNG
            Logging "${MSG_E007}"
            return 1
        fi
        Logging "${MSG_I013} : ${XLOG_START}"
        EchoOK
    fi

    #
    # set recovery.conf
    #
    Logging "${MSG_I018}"
    sed -e "s|^#restore_command.\+$|restore_command = '${restore_command}'|" ${pg_dir}/share/recovery.conf.sample >> ${dbcluster_dir}/recovery.conf
    if [ $? -ne 0 ]
    then
        EchoNG
        Logging "${MSG_E006} : recovery.conf append restore_command"
        return 1
    fi
    EchoOK

    #
    # execute base recovery
    #
    Logging "${MSG_I005}"
    ${base_recovery_command}
    if [ $? -ne 0 ]
    then
        EchoNG
        Logging "${MSG_E008}"
        return 1
    fi
    EchoOK

    #
    # move recovery.conf
    #
    Logging "${MSG_I019}"
    rm ${dbcluster_dir}/recovery.conf
    if [ $? -ne 0 ]
    then
        EchoNG
        Logging "${MSG_E017} : ${dbcluster_dir}/recovery.conf"
        return 1
    fi
    EchoOK

    return 0
}

# suppress JDBC access
function CloseVM() {
    #
    # change server_info.udb_validity (suppress query from JDBC)
    #
    ChangeUDBValidity wait online online
    if [ $? -ne 0 ]
    then
        return 1
    fi

    #
    # wait for busy transaction to idle
    #
    GetMaxDuration && WaitTransactionToIdle
    if [ $? -ne 0 ]
    then
        return 1
    fi

    return 0
}

# sync recovery (redo xlog)
function SyncRecovery() {

    if [ -e ${dbcluster_dir}/backup_label ]
    then
        #
        # execute pg_stop_backup
        #
        Logging "${MSG_I012}"
        XLOG_STOP=`${pg_dir}/bin/psql -U ${pg_user} -d ${pg_db} -p ${pg_port} -q -A -t -c "SELECT pg_xlogfile_name(pg_stop_backup())" | awk '{print $1}' | head -n 1`
        if [ $? -ne 0 ] || [ "x${XLOG_STOP}" == "x" ]
        then
            EchoNG
            Logging "${MSG_E013}"
            return 1
        fi
        Logging "${MSG_I014} : ${XLOG_STOP}"
        EchoOK
    
        #
        # wait for stop WAL location to be archived
        #
        Logging "${MSG_I015}"
        TMP=1
        for CNT in `seq 1 ${archive_wait_timeout}`
        do
            sleep 1
            echo -n "."
            if [ -e ${archivexlog_dir}/${XLOG_STOP} ] && [ `du -b ${archivexlog_dir}/${XLOG_STOP} | awk '{print $1}' | head -n 1` -eq 16777216 ]
            then
                TMP=0
                break
            fi
        done
        echo
        if [ ${TMP} -ne 0 ]
        then
            EchoNG
            Logging "${MSG_E014}"
            return 1
        fi
        EchoOK
    fi

    #
    # execute sync recovery
    #
    Logging "${MSG_I016}"
    ${sync_recovery_command}
    if [ $? -ne 0 ]
    then
        EchoNG
        Logging "${MSG_E015}"
        return 1
    fi
    EchoOK

    return 0
}

# recovery instance
function RecoverInstance() {
    #
    # execute instance startup
    #
    Logging "${MSG_I017}"
    ${instance_startup_command} > /dev/null
    if [ $? -ne 0 ]
    then
        EchoNG
        Logging "${MSG_E016}"
        return 1
    fi
    EchoOK

    return 0
} 

# unsuppress JDBC access
function OpenVM() {
    #
    # change server_info.udb_validity (suppress query from JDBC)
    #
    ChangeUDBValidity online all all
    if [ $? -ne 0 ]
    then
        return 1
    fi

    return 0
}

# cleanup recovery
function CleanupRecovery() {
    #
    # reset postgresql.conf
    #
    if [ -e ${dbcluster_dir}/${PGCONFIG_ONLINE} ]
    then
        Logging "${MSG_I021}"
        cp ${dbcluster_dir}/${PGCONFIG_ONLINE} ${dbcluster_dir}/postgresql.conf
        if [ $? -ne 0 ]
        then
            EchoNG
            Logging "${MSG_E002} : ${dbcluster_dir}/${PGCONFIG_ONLINE} ${dbcluster_dir}/postgresql.conf"
            return 1
        fi
        EchoOK
    fi

    #
    # reload postgresql.conf
    #
    Logging "${MSG_I025}"
    ReloadConfiguration || return 1

    #
    # remove postgresql.conf for recovery
    #
    if [ -e ${dbcluster_dir}/${PGCONFIG_ONLINE} ] && [ "${dbcluster_dir}/${PGCONFIG_ONLINE}" != "/" ]
    then
        Logging "${MSG_I041}"
        rm ${dbcluster_dir}/${PGCONFIG_ONLINE}
        if [ $? -ne 0 ]
        then
            EchoNG
            Logging "${MSG_E017} : ${dbcluster_dir}/${PGCONFIG_ONLINE}"
            return 1
        fi
        EchoOK
    fi

    #
    # remove archived WAL
    #
    if [ "${archivexlog_dir}/" != "/" ]
    then
        Logging "${MSG_I026}"
        rm -f ${archivexlog_dir}/*
        if [ $? -ne 0 ]
        then
            EchoNG
            Logging "${MSG_E017} : ${archivexlog_dir}/*"
            return 1
        fi
        EchoOK
    fi

    return 0
}

# cancel recovery
function CancelRecovery() {
    if [ -e ${dbcluster_dir}/backup_label ]
    then
        #
        # execute pg_stop_backup
        #
        Logging "${MSG_I012}"
        XLOG_STOP=`${pg_dir}/bin/psql -U ${pg_user} -d ${pg_db} -p ${pg_port} -q -A -t -c "SELECT pg_xlogfile_name(pg_stop_backup())" | awk '{print $1}' | head -n 1`
        if [ $? -ne 0 ] || [ "x${XLOG_STOP}" == "x" ]
        then
            EchoNG
            Logging "${MSG_E013}"
            return 1
        fi
        Logging "${MSG_I014} : ${XLOG_STOP}"
        EchoOK
    fi
    
    #
    # change server_info.udb_validity (suppress query from JDBC)
    #
    ChangeUDBValidity online wait wait
    if [ $? -ne 0 ]
    then
        return 1
    fi
    
    #
    # cleanup recovery temporally files
    #
    CleanupRecovery
    if [ $? -ne 0 ]
    then
        return 1
    fi

    return 0
}

# reload postgresql.conf
function ReloadConfiguration() {
    #
    # reload postgresql.conf
    #
    Logging "${MSG_I027} : archive_mode = `${pg_dir}/bin/psql -U ${pg_user} -d ${pg_db} -p ${pg_port} -q -A -t -c 'SHOW archive_mode' | head -n 1`"
    Logging "${MSG_I027} : archive_command = `${pg_dir}/bin/psql -U ${pg_user} -d ${pg_db} -p ${pg_port} -q -A -t -c 'SHOW archive_command' | head -n 1`"
    PID=`cat ${dbcluster_dir}/postmaster.pid | head -n 1` && Logging "${MSG_I029} : `ps -f --no-header -p ${PID}`" && kill -s SIGHUP `cat ${dbcluster_dir}/postmaster.pid | head -n 1`
    if [ $? -ne 0 ]
    then
        EchoNG
        Logging "${MSG_E003}"
        return 1
    fi
    Logging "${MSG_I028} : archive_mode = `${pg_dir}/bin/psql -U ${pg_user} -d ${pg_db} -p ${pg_port} -q -A -t -c 'SHOW archive_mode' | head -n 1`"
    Logging "${MSG_I028} : archive_command = `${pg_dir}/bin/psql -U ${pg_user} -d ${pg_db} -p ${pg_port} -q -A -t -c 'SHOW archive_command' | head -n 1`"
    EchoOK
    
    return 0
}

# change server_info.udb_validity (suppress query from JDBC)
# arg1 : updated udb_validity, arg2 : current udb_validity, arg3 : target server's current udb_validity
function ChangeUDBValidity() {
    UPDATED_VALIDITY=$1
    CURRENT_VALIDITY=$2
    TARGET_SERVER=$3
    
    case ${UPDATED_VALIDITY} in
        online)
            UPDATED_VALIDITY_VAL=1
            ;;
        wait)
            UPDATED_VALIDITY_VAL=0
            ;;
        *)
            UPDATED_VALIDITY_VAL=0
            ;;
    esac
    
    Logging "${MSG_I008}"
    DBLIST=`${pg_dir}/bin/psql -U ${pg_user} -d ${pg_db} -p ${pg_port} -q -A -t -c "SELECT datname FROM pg_database WHERE datname LIKE '%_pghainfo'" | awk '{print $1}'`
    if [ $? -ne 0 ] || [ "x${DBLIST}" == "x" ]
    then
        EchoNG
        Logging "${MSG_E010}"
        return 1
    fi
    for DB in ${DBLIST}
    do
        Logging "${MSG_I039} : ${DB}"
        case ${TARGET_SERVER} in
            online)
                URLLIST=`${pg_dir}/bin/psql -U ${pg_user} -d ${DB} -p ${pg_port} -q -A -t -c "SELECT udb_url FROM server_info WHERE udb_validity = 1" | awk '{print $1}'`
                ;;
            wait)
                URLLIST=`${pg_dir}/bin/psql -U ${pg_user} -d ${DB} -p ${pg_port} -q -A -t -c "SELECT udb_url FROM server_info WHERE udb_validity = 0" | awk '{print $1}'`
                ;;
            *)
                URLLIST=`${pg_dir}/bin/psql -U ${pg_user} -d ${DB} -p ${pg_port} -q -A -t -c "SELECT udb_url FROM server_info"| awk '{print $1}'`
                ;;
        esac
        
        
        for URL in ${URLLIST}
        do
            HOST=`echo ${URL} | cut -d "/" -f 1 | cut -d ":" -f 1`
            PORT=`echo ${URL} | cut -d "/" -f 1 | cut -d ":" -f 2`
            UDB=`echo ${URL} | cut -d "/" -f 2`
            
            case ${CURRENT_VALIDITY} in
                online)
                    ${pg_dir}/bin/psql -U ${pg_user} -d ${DB} -h ${HOST} -p ${PORT} -q -c "UPDATE server_info SET udb_validity = ${UPDATED_VALIDITY_VAL} WHERE udb_validity = 1"
                    ;;
                wait)
                    ${pg_dir}/bin/psql -U ${pg_user} -d ${DB} -h ${HOST} -p ${PORT} -q -c "UPDATE server_info SET udb_validity = ${UPDATED_VALIDITY_VAL} WHERE udb_validity = 0"
                    ;;
                *)
                    ${pg_dir}/bin/psql -U ${pg_user} -d ${DB} -h ${HOST} -p ${PORT} -q -c "UPDATE server_info SET udb_validity = ${UPDATED_VALIDITY_VAL}"
                    ;;
            esac
            if [ $? -eq 0 ]
            then
                Logging "${MSG_I022} : ${DB} at ${HOST} (${URL} from ${CURRENT_VALIDITY} to ${UPDATED_VALIDITY})"
            else
                EchoNG
                Logging "${MSG_E011}"
                return 1
            fi
        done
    done
    EchoOK
    
    return 0
}

# get max serverinfo_read_duration
function GetMaxDuration() {
    DURATION=0

    Logging "${MSG_I030}"
    DBLIST=`${pg_dir}/bin/psql -U ${pg_user} -d ${pg_db} -p ${pg_port} -q -A -t -c "SELECT datname FROM pg_database WHERE datname LIKE '%_pghainfo'" | awk '{print $1}'`
    for DB in ${DBLIST}
    do
        Logging "${MSG_I032} : ${DB}"

        TMP=`${pg_dir}/bin/psql -U ${pg_user} -d ${DB} -p ${pg_port} -q -A -t -c "SELECT value FROM global_config WHERE key = 'mngdb_read_duration'" | awk '{print $1}' | head -n 1`
        if [ "x${TMP}" == "x" ]
        then
            Logging "${MSG_E020}"
        fi

        if [ ${TMP} -gt ${DURATION} ]
        then
            DURATION=${TMP}
        fi
    done
    
    Logging "${MSG_I031} : ${DURATION}"
    EchoOK
    
    DURATION=`expr ${DURATION} + ${MAX_DURATION_ADD}`
    
    return 0
}

# wait for busy transaction to idle
function WaitTransactionToIdle() {
    Logging "${MSG_I009}"
    for CNT in `seq 1 ${DURATION}`
    do
        sleep 1
        echo -n "."
    done
    echo
    EchoOK
    Logging "${MSG_I010}"
    TMP=1
    for CNT in `seq 1 ${idle_wait_timeout}`
    do
        sleep 1
        PIDS=`${pg_dir}/bin/psql -U ${pg_user} -d ${DB} -p ${pg_port} -q -A -t -c "SELECT procpid FROM pg_stat_activity WHERE procpid <> pg_backend_pid() AND current_query <> '<IDLE>' AND current_query NOT LIKE '%autovacuum:%'" | awk '{print $1}' | tr "\n" " "`
        if [ "x${PIDS}" == "x" ]
        then
            TMP=0
            break
        fi
        Logging "${MSG_I011} : ${PIDS}"
    done
    if [ ${TMP} -ne 0 ]
    then
        EchoNG
        Logging ${MSG_E012}
        return 1
    fi
    EchoOK
    
    return 0
}

# check environment
function CheckEnvironment() {
    Logging "${MSG_I044}"
    ${env_check_command}
    if [ $? -ne 0 ]
    then
        EchoNG
        Logging "${MSG_E025}"
        return 1
    fi
    EchoOK
    
    # check psql password setting
    if [ ! -e ~/.pgpass ]
    then
        Logging "${MSG_E004} (~/.pgpass)"
        return 1
    fi
    if [ `/usr/bin/stat --format="%a" ~/.pgpass` != "600" ]
    then
        Logging "${MSG_E030}. ~/.pgpass must be -rw-------."
        return 1
    fi
    
    # check ha system catalog
    ${pg_dir}/bin/psql -U ${pg_user} -p ${pg_port} -d ${pg_db} -q -A -t -l > /dev/null 2>&1
    if [ $? -ne 0 ]
    then
        Logging "${MSG_E029}"
        return 1
    fi
    DBLIST=`${pg_dir}/bin/psql -U ${pg_user} -d ${pg_db} -p ${pg_port} -q -A -t -c "SELECT datname FROM pg_database WHERE datname LIKE '%_pghainfo'" | awk '{print $1}'`
    if [ $? -ne 0 ] || [ "x${DBLIST}" == "x" ]
    then
        EchoNG
        Logging "${MSG_E010}"
        return 1
    fi
    for DB in ${DBLIST}
    do
        ROWS=`${pg_dir}/bin/psql -U ${pg_user} -d ${DB} -p ${pg_port} -q -A -t -c "SELECT udb, count(*) AS cnt FROM (SELECT udb_url, udb_validity, split_part(udb_url, '/', 2) AS udb FROM server_info WHERE udb_validity IN (0, 1)) AS udb_info GROUP BY udb"`
        for ROW in ${ROWS}
        do
            UDB=`echo ${ROW} | cut -d "|" -f 1`
            CNT=`echo ${ROW} | cut -d "|" -f 2`
            if [ ${CNT} -ne 1 ]
            then
                Logging "${MSG_E027} (${UDB})"
                return 1
            fi
        done
        ROWS=`${pg_dir}/bin/psql -U ${pg_user} -d ${DB} -p ${pg_port} -q -A -t -c "SELECT udb, count(*) AS cnt FROM (SELECT udb_url, udb_validity, split_part(udb_url, '/', 2) AS udb FROM server_info WHERE udb_validity IN (0, 1) AND udb_url LIKE '${src_pg_host}%') AS udb_info GROUP BY udb"`
        for ROW in ${ROWS}
        do
            UDB=`echo ${ROW} | cut -d "|" -f 1`
            CNT=`echo ${ROW} | cut -d "|" -f 2`
            if [ ${CNT} -ne 1 ]
            then
                Logging "${MSG_E028} (${UDB})"
                return 1
            fi
        done
    done
    
    return 0
}

function Logging() {
    MODE=$2
    OPTS=""

    case ${MODE} in
        withline)
            EchoLine
            Logging "$1"
            EchoLine
            return 0
            ;;
        noreturn)
            OPTS="-n"
            ;;
        *)
            ;;
    esac

    echo ${OPTS} "$1"
    /usr/bin/logger "${PROG} $1"
}

function EchoOK() {
    Logging "[ OK ]"
}

function EchoNG() {
    Logging "[ NG ]"
}

function EchoLine() {
    Logging "----------------------------------------"
}

########################################
# SHELL
########################################

#
# init
#

# check argument count
if [ $# -ne 2 ]
then
    Logging "${MSG_E001}"
    exit 1
fi

# set argument
PHASE=$1
CONFIG=$2

# check configuration file
if [ ! -e ${CONFIG} ]
then
    Logging "${MSG_E004} : ${CONFIG}"
    exit 1
fi

# convert configfile path
if [ `echo ${CONFIG} | grep "^/" | wc -l` -ne 1 ]
then
    CONFIG=${PWD}/${CONFIG}
fi

# set configuration
. ${CONFIG}

#
# main
#
RET=9
case ${PHASE} in
    -all)
        $0 -base ${CONFIG} || exit 1
        Logging ""
        $0 -closevm ${CONFIG} || exit 1
        Logging ""
        $0 -sync ${CONFIG} || exit 1
        Logging ""
        $0 -recovery ${CONFIG} || exit 1
        Logging ""
        $0 -openvm ${CONFIG} || exit 1
        Logging ""
        $0 -cleanup ${CONFIG} || exit 1
        Logging ""
        Logging "${MSG_I040}"
        RET=0
        ;;
    -base)
        Logging "${MSG_I001}" withline
        
        CheckEnvironment && BaseRecovery
        RET=$?

        if [ ${RET} -eq 0 ]
        then
            Logging "${MSG_I006}" withline
        else
            Logging "${MSG_E009}" withline
        fi
        ;;
    -closevm)
        Logging "${MSG_I033}" withline
        
        CheckEnvironment && CloseVM
        RET=$?

        if [ ${RET} -eq 0 ]
        then
            Logging "${MSG_I034}" withline
        else
            Logging "${MSG_E021}" withline
        fi
        ;;
    -sync)
        Logging "${MSG_I007}" withline

        CheckEnvironment && SyncRecovery
        RET=$?

        if [ ${RET} -eq 0 ]
        then
            Logging "${MSG_I020}" withline
        else
            Logging "${MSG_E018}" withline
        fi
        ;;
    -recovery)
        Logging "${MSG_I037}" withline

        CheckEnvironment && RecoverInstance
        RET=$?

        if [ ${RET} -eq 0 ]
        then
            Logging "${MSG_I038}" withline
        else
            Logging "${MSG_E023}" withline
        fi
        ;;
    -openvm)
        Logging "${MSG_I035}" withline

        CheckEnvironment && OpenVM
        RET=$?

        if [ ${RET} -eq 0 ]
        then
            Logging "${MSG_I036}" withline
        else
            Logging "${MSG_E022}" withline
        fi
        ;;
    -cleanup)
        Logging "${MSG_I023}" withline

        CleanupRecovery
        RET=$?

        if [ ${RET} -eq 0 ]
        then
            Logging "${MSG_I024}" withline
        else
            Logging "${MSG_E019}" withline
        fi
        ;;
    -cancel)
        Logging "${MSG_I042}" withline

        CheckEnvironment && CancelRecovery
        RET=$?

        if [ ${RET} -eq 0 ]
        then
            Logging "${MSG_I043}" withline
        else
            Logging "${MSG_E024}" withline
        fi
        ;;
    *)
        Logging "${MSG_E001}"
        exit 1
        ;;
esac

#
# final
#
exit ${RET}


