#!/bin/bash
#
# Data transfer script for PostgreSQL HA
#
# <description>
#  This script is designed for PostgreSQL HA
#
# <usage>
#  recovery_util_rsync.sh -base_[rsync|scp] | -sync_[rsync|scp] | -recovery | -envcheck configfile
#
# <return>
#  0 = OK, 1 = FATAL
#
# <dependency>
#
# $Id$

PROG=`basename $0`
DIR=$(cd $(dirname $0) && pwd)/$(basename "$FILE")
USER=`/usr/bin/whoami`
HOST=`hostname`

########################################
# Definition
########################################

#
# messages
#
MSG_I001="rsync command finished"
MSG_I002="scp command finished"

MSG_E001="usage : ${PROG} -base | -sync | -recovery | -envcheck configfile"
MSG_E002="failed to execute rsync command"
MSG_E003="invalid user"
MSG_E004="file not found"
MSG_E005="failed to execute ssh without passphase"
MSG_E006="failed to execute scp command"

########################################
# Function
########################################

function Logging() {
    MODE=$2
    OPTS=""

    case ${MODE} in
        withline)
            EchoLine
            Logging "$1"
            EchoLine
            return 0
            ;;
        noreturn)
            OPTS="-n"
            ;;
        *)
            ;;
    esac

    echo ${OPTS} "$1"
    /usr/bin/logger "${PROG} $1"
}

function EchoLine() {
    Logging "----------------------------------------"
}

########################################
# SHELL
########################################

#
# init
#

# check argument count
if [ $# -ne 2 ]
then
    Logging "${MSG_E001}"
    exit 1
fi

# set argument
PHASE=$1
CONFIG=$2

# check configuration file
if [ ! -e ${CONFIG} ]
then
    Logging "${MSG_E004} : ${CONFIG}"
    exit 1
fi

# set configuration
. ${CONFIG}

if [ "${transfer_limit_rate}x" != "0x" ]
then
    TRANSFER_LIMIT_RATE_KB=`expr ${transfer_limit_rate} / 1024` || exit 1
fi

dbcluster_dirparent=`dirname ${dbcluster_dir}`
archivexlog_dirparent=`dirname ${archivexlog_dir}`

RET=9
case ${PHASE} in
    -base_init)
        if [ -e ${dbcluster_dir}/postgresql.conf ]
        then
            cp -pL ${dbcluster_dir}/postgresql.conf ${dbcluster_dirparent}/postgresql.conf.online || exit 1
        fi
        if [ -e ${dbcluster_dir} ]
        then
            rm -rf ${dbcluster_dir} || exit 1
        fi
        if [ -e ${archivexlog_dir} ]
        then
            rm -rf ${archivexlog_dir} || exit 1
        fi
        ;;
        
    -base_rsync)
        /usr/bin/ssh -24q ${USER}@${dst_pg_host} ${DIR}/recovery_util.sh -base_init ${CONFIG} || exit 1
        
        dbcluster_dirname=`basename ${dbcluster_dir}`
        
        rsync -arq --bwlimit=${TRANSFER_LIMIT_RATE_KB} --include="/${dbcluster_dirname}/pg_xlog/archive_status/" --exclude="/${dbcluster_dirname}/pg_xlog/archive_status/*" --exclude="/${dbcluster_dirname}/pg_xlog/*" --exclude="/${dbcluster_dirname}/postgresql.conf.online" ${dbcluster_dir} ${USER}@${dst_pg_host}:${dbcluster_dirparent}
        RET=$?
        Logging "${MSG_I001} (return code : ${RET})"
        case ${RET} in
            0)
                # success
                ;;
            23)
                # skip partial transfer due to error because of xlog files
                ;;
            24)
                # skip partial transfer due to vanished source files because of sort work files
                ;;
            *)
                Logging "${MSG_E002}"
                exit 1
                ;;
        esac
        ;;
        
    -base_scp)
        /usr/bin/ssh -24q ${USER}@${dst_pg_host} ${DIR}/recovery_util.sh -base_init ${CONFIG} || exit 1
        
        /usr/bin/scp -24qp -r ${dbcluster_dir} ${USER}@${dst_pg_host}:${dbcluster_dir%/*}
        RET=$?
        Logging "${MSG_I002} (return code : ${RET})"
        case ${RET} in
            0)
                # success
                ;;
            *)
                Logging "${MSG_E006}"
                exit 1
                ;;
        esac
        ;;
        
    -sync_rsync)
        rsync -arq --bwlimit=${TRANSFER_LIMIT_RATE_KB} ${archivexlog_dir} ${USER}@${dst_pg_host}:${archivexlog_dirparent}
        RET=$?
        Logging "${MSG_I001} (return code : ${RET})"
        case ${RET} in
            0)
                # success
                ;;
            23)
                # skip partial transfer due to error because of archived xlog files
                ;;
            *)
                Logging "${MSG_E002}"
                exit 1
                ;;
        esac
        ;;
        
    -sync_scp)
        /usr/bin/scp -24qp -r ${archivexlog_dir} ${USER}@${dst_pg_host}:${archivexlog_dir%/*}
        RET=$?
        Logging "${MSG_I002} (return code : ${RET})"
        case ${RET} in
            0)
                # success
                ;;
            *)
                Logging "${MSG_E006}"
                exit 1
                ;;
        esac
        ;;
        
    -recovery_remote)
        cp -p ${dbcluster_dirparent}/postgresql.conf.online ${dbcluster_dir}/postgresql.conf || exit 1
        rm ${dbcluster_dirparent}/postgresql.conf.online || exit 1
        if [ -e ${dbcluster_dir}/postgresql.conf.online ]
        then
            rm ${dbcluster_dir}/postgresql.conf.online
        fi
        ${pg_dir}/bin/pg_ctl start -w -D ${dbcluster_dir} ${pg_args} || exit 1
        if [ "${archivexlog_dir}/" != "/" ]
        then
            rm -f ${archivexlog_dir}/* || exit 1
        fi
        ;;
        
    -recovery)
        /usr/bin/ssh -24q ${USER}@${dst_pg_host} ${DIR}/recovery_util.sh -recovery_remote ${CONFIG} || exit 1
        ;;
        
    -envcheck)
        # check exec user
        if [ ! -O ${dbcluster_dir} ]
        then
            Logging "${MSG_E003} (${USER})"
            exit 1
        fi
        
        # check ssh key setting
        if [ "x${check_ssh}" == "xtrue" ]
        then
            /usr/bin/ssh -24q -o "BatchMode yes" ${USER}@${dst_pg_host} true
            if [ $? -ne 0 ]
            then
                Logging "${MSG_E005} (${USER}@${dst_pg_host})"
                exit 1
            fi
        fi
        ;;
    *)
        Logging "${MSG_E001}"
        exit 1
        ;;
        
esac

exit 0


