<?php
/*! @file
 *@brief
 * PSGraph base
 *  共通利用クラス定義
 *
 *@author hirohito
 *@note
 */
/* This file is distributed under BSD licence. Please read the COPYRIGHT file. */


//================================================
/*!@brief
 * 描画系クラスのベースクラス
 *
 */
abstract class PSGraph_base
{
    protected $Width;		//!< 占める領域の幅
    protected $Height;		//!< 占める領域の高さ
    protected $DataSeries = array();	//!< 管理するデータコンテナオブジェクト配列
    protected $Output;		//!< 出力制御オブジェクト


    //================================================================
    /*! constructor
     *
     *@param	width	幅
     *@param	height	高さ
     *@param	output	出力制御オブジェクト
     */
    protected function __construct( $width, $height, PSGraph_output &$output )
    {
	$this->Width = (int)$width;
	$this->Height = (int)$height;
	$this->Output = &$output;
    }


    //================================================================
    /*! データコンテナ追加
     *
     *@param	data_obj	データコンテナオブジェクト
     */
    protected function add_data_series( $data_obj )
    {
	$this->DataSeries[] = $data_obj;
    }


    //================================================================
    /*! アトリビュート文字列生成
     *
     *@param	attr	アトリビュート連想配列
     *@retval	string	アトリビュート文字列
     *@note
       引数で与えた連想配列 (array) の、特定のキーから xml アトリビュート文字列を生成して返す。
     */
    protected function make_common_attribute_string( array $attr )
    {
	$s = "";

	// area rectangle.
	if( isset( $attr['X'] ) ) $s .= 'x="'. $attr['X']. '" ';
	if( isset( $attr['Y'] ) ) $s .= 'y="'. $attr['Y']. '" ';
	if( isset( $attr['Width'] ) ) $s .= 'width="'. $attr['Width']. '" ';
	if( isset( $attr['Height'] ) ) $s .= 'height="'. $attr['Height']. '" ';

	// stroke styles
	if( isset( $attr['Fill'] ) ) $s .= 'fill="'. $attr['Fill']. '" ';
	if( isset( $attr['Stroke'] ) ) $s .= 'stroke="'. $attr['Stroke']. '" ';
	if( isset( $attr['StrokeWidth'] ) ) $s .= 'stroke-width="'. $attr['StrokeWidth']. '" ';
	if( isset( $attr['StrokeDasharray'] ) ) $s .= 'stroke-dasharray="'. $attr['StrokeDasharray']. '" ';

	// fonts
	if( isset( $attr['FontSize'] ) ) $s .= 'font-size="'. $attr['FontSize']. 'px" ';
	if( isset( $attr['FontFamily'] ) ) $s .= 'font-family="'. $attr['FontFamily']. '" ';
	if( isset( $attr['FontWeight'] ) ) $s .= 'font-weight="'. $attr['FontWeight']. '" ';
	if( isset( $attr['FontStyle'] ) ) $s .= 'font-style="'. $attr['FontStyle']. '" ';
	if( isset( $attr['TextAnchor'] ) ) $s .= 'text-anchor="'. $attr['TextAnchor']. '" ';
	if( isset( $attr['TextDecoration'] ) ) $s .= 'text-decoration="'. $attr['TextDecoration']. '" ';

	// opacity
	if( isset( $attr['Opacity'] ) ) $s .= 'opacity="'. $attr['Opacity']. '" ';
	if( isset( $attr['StrokeOpacity'] ) ) $s .= 'stroke-opacity="'. $attr['StrokeOpacity']. '" ';

	return $s;
    }


    abstract function draw();
}



//================================================
/*!@brief
 * グラフ本体クラスのベースクラス
 *
 */
abstract class PSGraph_util extends PSGraph_base
{
    //! 色リスト
    static protected $COLORLIST = array( '#0084d1', '#004586', '#ff420e', '#ffd320', '#579d1c', '#7e0021',
					 '#83caff', '#314004', '#aecf00', '#4b1f6f', '#ff950e', '#c5000b' );
    
    public $AtGraphArea = array( 'StrokeWidth'=>1, 'Stroke'=>'black', 'Fill'=>'white' );
					//!< グラフエリアアトリビュート
    public $AtPlotArea = array();	//!< プロットエリアアトリビュート
    public $AtMainTitle = null;		//!< メインタイトルアトリビュート
    public $AtLegend = null;		//!< 凡例アトリビュート

    protected $AuxTags = array();	//!< 追加任意タグ
    protected $WorkMode = array();	//!< 動作モード (see setMode() function.)


    //================================================================
    /*! constructor
     *
     *@param	width	幅
     *@param	height	高さ
     */
    function __construct( $width, $height )
    {
	if( $width <= 0 || $height <= 0 ) {
	    trigger_error( "width or height < 0", E_USER_ERROR );
	}

	$output = new PSGraph_output_stdout;
	parent::__construct( $width, $height, $output );

	/*
	 * make default
	 */
	$this->AtGraphArea['Width'] = $this->Width;
	$this->AtGraphArea['Height'] = $this->Height;
    }


    //================================================================
    /*! 動作モード指定
     *
     *@param	mode	動作モード
     *@note
     *<pre>
     * 設定可能モード
     *  NO_CONTENT_TYPE		ContentType ヘッダを出力しない
     *  NO_XML_DECLARATION	XML宣言およびDOCTYPE宣言を出力しない
     *  NO_SVG_TAG		SVGタグを出力しない
     *  NO_SVG_TAG_CLOSE	SVG終了タグのみを出力しない
     *</pre>
     */
    function setMode( $mode )
    {
	$this->WorkMode[$mode] = true;
    }


    //================================================================
    /*! プロットエリアのマージン設定
     *
     *@param	top	上マージン
     *@param	right	右マージン
     *@param	bottom	下マージン
     *@param	left	左マージン
     *@note
     * 上下左右個別に設定できる。
     * 設定値を変えない場合は、そのパラメータをnullにしてcallする。
     */
    function setMargin( $top, $right, $bottom, $left )
    {
	// calculate insufficiency parameters.
	if( $top === null ) $top = $this->AtPlotArea['Y'];
	if( $right === null ) $right = $this->Width - $this->AtPlotArea['Width']
				  - $this->AtPlotArea['X'];
	if( $bottom === null ) $bottom = $this->Height - $this->AtPlotArea['Height']
				   - $this->AtPlotArea['Y'];
	if( $left === null ) $left = $this->AtPlotArea['X'];

	// set lot area's parameters
	$this->AtPlotArea['X'] = $left;
	$this->AtPlotArea['Y'] = $top;
	$this->AtPlotArea['Width'] = $this->Width - $left - $right;
	$this->AtPlotArea['Height'] = $this->Height - $top - $bottom;
	if( $this->AtPlotArea['Width'] < 0 ) $this->AtPlotArea['Width'] = 0;
	if( $this->AtPlotArea['Height'] < 0 ) $this->AtPlotArea['Height'] = 0;
    }


    //================================================================
    /*! バッファーへ描画
     *
     */
    function drawBuffer()
    {
	$this->Output = new PSGraph_output_buffer;
	$this->draw();
	return $this->Output->getOutput();
    }


    //================================================================
    /*! メインタイトルの追加
     *
     *@param	title_string	タイトル文字列
     */
    function addMainTitle( $title_string )
    {
	$this->setMargin( 25, null, null, null );
	$this->AtMainTitle = array( 'Title'=>$title_string, 'Y'=>20, 'FontSize'=>16,
				    'TextAnchor'=>'middle' );
    }


    //================================================================
    /*! 凡例表示追加
     *
     *@note
     * 自動追加されるので、たいていの場合、ユーザがこのメソッドを使うことはないかもしれない。
     */
    function addLegend()
    {
	if( $this->AtLegend !== null ) return;

	$right = $this->Width - $this->AtPlotArea['Width'] - $this->AtPlotArea['X'] + 70;
	$this->setMargin( null, $right, null, null );
	$this->AtLegend = array( 'X'=>($this->Width-60), 'FontSize'=>10 );
    }


    //================================================================
    /*! 凡例を表示しない
     *
     */
    function clearLegend()
    {
	if( $this->AtLegend === null ) return;

	$right = $this->Width - $this->AtPlotArea['Width'] - $this->AtPlotArea['X'] - 70;
	$this->setMargin( null, $right, null, null );
	$this->AtLegend = null;
    }


    //================================================================
    /*! 任意タグを追加
     *
     *@param	text	タグテキスト
     */
    function addAuxTag( $text )
    {
	$this->AuxTags[] = $text;
    }


    //================================================================
    /*! テキスト追加
     *
     *@param	x	X座標
     *@param	y	Y座標
     *@param	text	テキスト
     *@note
     * addAuxTag()の簡易テキスト版。
     * フォントサイズの指定などは、<tspan>要素を使える。
     */
    function addText( $x, $y, $text )
    {
	$this->AuxTags[] = "<text x=\"$x\" y=\"$y\">$text</text>\n";
    }


    //================================================================
    /*! 描画共通部１
     *
     */
    protected function draw_common1()
    {
	/*
	 * draw headers. (http header, xml header, and others)
	 */
	if( ! isset( $this->WorkMode['NO_CONTENT_TYPE'] ) ) {
	    Header( "Content-Type: image/svg+xml" );
	}

	if( ! isset( $this->WorkMode['NO_XML_DECLARATION'] ) ) {
	    $this->Output->printf( "<?xml version=\"1.0\" standalone=\"no\" ?>\n" );
	    $this->Output->printf( "<!DOCTYPE svg PUBLIC \"-//W3C//DTD SVG 20010904//EN\"".
				   " \"http://www.w3.org/TR/2001/REC-SVG-20010904/DTD/svg10.dtd\">\n" );
	}

	if( ! isset( $this->WorkMode['NO_SVG_TAG'] ) ) {
	    $this->Output->printf( "<svg width=\"{$this->Width}px\" height=\"{$this->Height}px\"".
			" xmlns=\"http://www.w3.org/2000/svg\" xmlns:xlink=\"http://www.w3.org/1999/xlink\">\n\n" );
	}

	/*
	 * draw background and border.
	 */
	$this->Output->printf( "<rect %s />\n", $this->make_common_attribute_string( $this->AtGraphArea ) );
	if( isset($this->AtGraphArea['Image']) ) {
	    $this->Output->printf( "<image x=\"%d\" y=\"%d\" width=\"%d\" height=\"%d\" xlink:href=\"%s\" />\n",
				   0, 0, $this->Width, $this->Height, $this->AtGraphArea['Image'] );
	}
	
	/*
	 * draw plot area.
	 */
	$this->Output->printf( "<rect %s />\n", $this->make_common_attribute_string( $this->AtPlotArea ) );
	if( isset($this->AtPlotArea['Image']) ) {
	    $this->Output->printf( "<image x=\"%d\" y=\"%d\" width=\"%d\" height=\"%d\" xlink:href=\"%s\" />\n",
		    $this->AtPlotArea['X'], $this->AtPlotArea['Y'],
		    $this->AtPlotArea['Width'], $this->AtPlotArea['Height'], $this->AtPlotArea['Image'] );
	}
    }


    //================================================================
    /*! 描画共通部2
     *
     */
    protected function draw_common2()
    {
	/*
	 * draw main title.
	 */
	if( $this->AtMainTitle ) {
	    if( ! isset($this->AtMainTitle['X']) ) $this->AtMainTitle['X'] = $this->Width / 2;
	    $this->Output->printf( "<text %s>%s</text>\n",
		    $this->make_common_attribute_string( $this->AtMainTitle ),
		    htmlspecialchars( $this->AtMainTitle['Title'] ) );
	}

	/*
	 * auxiliary tags.
	 */
	foreach( $this->AuxTags as $atag ) {
	    $this->Output->printf( $atag );
	}

	if( ! isset( $this->WorkMode['NO_SVG_TAG'] ) && ! isset( $this->WorkMode['NO_SVG_TAG_CLOSE'] ) ) {
	    $this->Output->printf( "\n</svg>\n" );
	}
    }
}



//================================================
/*!@brief
 * 出力制御クラス　スーパークラス
 *
 */
abstract class PSGraph_output
{
    abstract function printf();
}



//================================================
/*!@brief
 * 出力制御クラス　標準出力
 *
 */
class PSGraph_output_stdout extends PSGraph_output
{
    //================================================================
    /*! 出力
     *
     *@param	format	出力フォーマット
     *@param	...	出力データ
     */
    function printf()
    {
	$arg_list = func_get_args();
	$format = array_shift( $arg_list );

	vprintf( $format, $arg_list );
    }
}



//================================================
/*!@brief
 * 出力制御クラス　バッファ
 *
 */
class PSGraph_output_buffer extends PSGraph_output
{
    protected $OutBuf = "";	//!< 出力保存用


    //================================================================
    /*! 出力
     *
     *@param	format	出力フォーマット
     *@param	...	出力データ
     */
    function printf()
    {
	$arg_list = func_get_args();
	$format = array_shift( $arg_list );

	$this->OutBuf .= vsprintf( $format, $arg_list );
    }


    //================================================================
    /*! 出力結果取得
     *
     *@retval	str	出力結果
     */
    function getOutput()
    {
	return $this->OutBuf;
    }
}
