<?php
/**
 * Mp3Audio  mp3ファイルのID3タグ情報を読み取り、変更するクラス
 *
 * @version $Id: mp3.func.php,v 1.5 2004/10/22 17:23:45 haniwa Exp $
 * @since 2004/10/20
 *
 * 実行に際しては id3lib          http://id3lib.sourceforge.net/
 * 及び           id3v2           http://id3v2.sourceforge.net/
 * がそれぞれセットアップされている必要があります。
 *
 *
 * @see      setId3tag		タグ情報の書き込み
 * @see      getId3tag		タグ情報の取得
 * @see      clearId3tag	タグ情報のクリア
 *
 <pre>
  ～使い方～
 
  //書き込みたいタグ。（省略可能だが最低１つは指定)
  $data = array(
      'ARTIST'=>"アーティスト名",
      'ALBUM'=>"アルバム情報",
      'TITLE'=>"タイトル名",
      'COMMENT'=>"コメント",
      'YEAR'=>"発表年",
      'GENRE'=>"ジャンルコード",
      'TRACK'=>"曲番"
  );
 
  //ファイル名(フルパスが望ましい)を指定してオブジェクト生成。ファイルが存在しない場合はエラー。
  $ma = new Mp3Audio($file);
 
  // パラメータのハッシュを引き渡して書き込み。
  $ma->setId3tag($data);
 </pre>
 *
 *
 **/
class Mp3Audio {

    var $filename = "";

    /**
     * Mp3Audio function
     *
     * Mp3Audioオブジェクトを生成します。
     * @access    public
     * @param     String    $filename          操作対象ファイル名（フルパス）
     */

    function  Mp3Audio($filename){
    
        //ファイルの存在を確認する。存在しなければエラーとする。
        if ( !file_exists($filename) ) trigger_error("Mp3file:{$filename} is not found.",E_USER_ERROR);
        
        $this->filename = $filename;
        
    }


    /**
     * setId3tag function
     *
     * 引き渡されたデータをID3タグとしてmp3ファイルに書き出します。
     * mp3ファイルに書き込める権限がないとエラーとなります。
     * 
     * @access    public
     * @param     Array     $src_data    書き出しに利用する配列。
     * @param     string    $encoding    渡した文字列のエンコーディング
     * @return    boolean   結果
     */
    function setId3tag($src_data,$encoding="SJIS"){

        //ファイルが書き込み可能でない場合はエラー
        if ( !is_writable($this->filename) ) trigger_error("Mp3file:File is not Writable",E_USER_ERROR);

        //コマンドに渡すオプションをためるバッファ
        $opt = "";
        
        /**********************オプションを組み立てる。*********************/

        //エンコードをid3タグ用のSJISに統一する。
        foreach ( $src_data as $key => $val ) {
        
            $src_data[$key] = mb_convert_encoding($val,"SJIS",$encoding);
        
        }

        //アーティスト情報（未指定時は埋め込まない）
        if ( $src_data['ARTIST'] != "" ) $opt .= " -a '{$src_data['ARTIST']}' ";
        
        //アルバム情報（未指定時は埋め込まない）
        if ( $src_data['ALBUM'] != "" ) $opt .= " -A '{$src_data['ALBUM']}' ";
        
        //タイトル情報（未指定時は埋め込まない）
        if ( $src_data['TITLE'] != "" ) $opt .= " -t '{$src_data['TITLE']}' ";
        
        //コメント 項目:内容　の形式である必要がある。（未指定時は埋め込まない）
        if ( $src_data['COMMENT'] != "" ) $opt .= " -c '{$src_data['COMMENT']}' ";
        
        //ジャンル情報(数値で指定)（未指定時は埋め込まない）
        if ( $src_data['GENRE'] != "" ) $opt .= " -g '{$src_data['GENRE']}' ";
        
        //発表年情報(数値で指定)（未指定時は埋め込まない）
        if ( $src_data['YEAR'] != "" ) $opt .= " -y '{$src_data['YEAR']}' ";
        
        //曲番情報(数値で指定)（未指定時は埋め込まない）
        if ( $src_data['TRACK'] != "" ) $opt .= " -T '{$src_data['TRACK']}' ";
        
        //オプションが最終的に指定されていない場合はエラーとする。
        if ( $opt == "" ) trigger_error("Mp3file:No Option set",E_USER_ERROR);


        //コマンドを組み立てる
        $cmd = "id3v2 ".$opt.$this->filename;
    
        //コマンドを実行し、結果のステータスコードと出力を変数に確保する。
        exec($cmd,$retval,$stat);
        
        //以上終了時はエラーとする。
        if ( $stat != 0 ) trigger_error("Mp3file:WriteError:{$cmd}",E_USER_ERROR);
    
    }
    
    /**
     * getId3tag function
     *
     * mp3ファイルから情報を取得します。
     * @access    public
     */
    function getId3tag(){
    
        //コマンドを組み立てる
        $cmd = "id3info ".$this->filename;
    
        //コマンドを実行し、結果のステータスコードと出力を変数に確保する。
        exec($cmd,$retval,$stat);
        
        //以上終了時はエラーとする。
        if ( $stat != 0 ) trigger_error("Mp3file:ReadError:{$cmd}",E_USER_ERROR);
    
        //返却用のハッシュ。
        $result = array();
    
        //正規表現エンコードをSJISに固定
        mb_regex_encoding("SJIS");
    
        //結果を整形して配列に得る。
        foreach ( $retval as $key => $val ) {
        
            /****** タグの情報の取得 ******/

            //アーティスト情報を発見したら取り分ける
            if ( mb_ereg("^=== TPE1",$val) ) {
                $result['ARTIST'] = mb_ereg_replace("^.*:\s","",$val);
            }

            //アルバム情報を発見したら取り分ける
            if ( mb_ereg("^=== TALB",$val) ) {
                $result['ALBUM'] = mb_ereg_replace("^.*:\s","",$val);
            }

            //タイトル情報を発見したら取り分ける
            if ( mb_ereg("^=== TIT2",$val) ) {
                $result['TITLE'] = mb_ereg_replace("^.*:\s","",$val);
            }

            //コメント情報を発見したら取り分ける
            if ( mb_ereg("^=== COMM",$val) ) {
                $result['COMMENT'] = mb_ereg_replace("^.*:\s","",$val);
            }

            //ジャンル情報を発見したら取り分ける
            if ( mb_ereg("^=== TCON",$val) ) {
                $result['GENRE'] = mb_ereg_replace("^.*:\s","",$val);
            }

            //発表年情報を発見したら取り分ける
            if ( mb_ereg("^=== TYER",$val) ) {
                $result['YEAR'] = mb_ereg_replace("^.*:\s","",$val);
            }

            //曲番情報を発見したら取り分ける
            if ( mb_ereg("^=== TRCK",$val) ) {
                $result['TRACK'] = mb_ereg_replace("^.*:\s","",$val);
            }

            /****** ｍｐ３の情報の取得 ******/

            //ビットレート発見したら取り分ける
            if ( mb_ereg("^Bitrate:\s",$val) ) {
                $result['Bitrate'] = mb_ereg_replace("^Bitrate:\s","",$val);
            }

            //サンプリングレートを発見したら取り分ける
            if ( mb_ereg("^Frequency:\s",$val) ) {
                $result['Frequency'] = mb_ereg_replace("^Frequency:\s","",$val);
            }


        }
        
        //$result = $retval;
    
        
        //結果を返却
        return $result;
    
    }
    
    /**
     * clearId3tag function
     *
     * 引き渡されたデータを設定ファイルに書き出します。
     * @access    public
     */
    function clearId3tag(){

        //ファイルが書き込み可能でない場合はエラー
        if ( !is_writable($this->filename) ) trigger_error("Mp3file:File is not Writable",E_USER_ERROR);

        //コマンドを組み立てる
        $cmd = "id3v2 -D ".$this->filename;
    
        //コマンドを実行し、結果のステータスコードと出力を変数に確保する。
        exec($cmd,$retval,$stat);
        
        //以上終了時はエラーとする。
        if ( $stat != 0 ) trigger_error("Mp3file:ClearError:{$cmd}",E_USER_ERROR);

    }

}


?>